﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WrappedSoundHandle.h"
#include "SoundArchiveContext.h"
#include "FlightRecorder.h"

namespace
{
    //  サウンドをフェードアウトするときのフレーム数
#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    //  グラフィクス表示のときは徐々にフェードアウト
    const int FadeOutFrames = 8;
#else
    //  コンソール表示のときはすぐに反映させるために 0
    const int FadeOutFrames = 0;
#endif

    //  ある float の値 val が
    //  -ThresholdConsiderAsZero < val < ThresholdConsiderAsZero
    //  ならば 0 とみなす
    const float ThresholdConsiderAsZero = 0.0001f;
}
namespace
{
    //  パラメータのデフォルト値
    const float    ParameterDefault_Volume          = 1.0f;
    const float    ParameterDefault_Pitch           = 1.0f;
    const int      ParameterDefault_StartOffset     = 0;
    const float    ParameterDefault_Pan             = 0.0f;
    const float    ParameterDefault_SPan            = 0.0f;
    const float    ParameterDefault_MainSend        = 0.0f;
    const float    ParameterDefault_EffectSendA     = 0.0f;
    const float    ParameterDefault_EffectSendB     = 0.0f;
    const float    ParameterDefault_EffectSendC     = 0.0f;
    const float    ParameterDefault_LpfFrequency    = 0.0f;
    const int      ParameterDefault_BqfType         = nn::atk::BiquadFilterType_Inherit;
    const float    ParameterDefault_BqfValue        = 0.5f;
    const float    ParameterDefault_MainOutVolume   = 1.0f;
    const float    ParameterDefault_PanMain         = 0.0f;
    const float    ParameterDefault_SPanMain        = 0.0f;
    const float    ParameterDefault_MainSendMain    = 0.0f;
    const float    ParameterDefault_EffectSendAMain = 0.0f;
    const float    ParameterDefault_EffectSendBMain = 0.0f;
    const float    ParameterDefault_EffectSendCMain = 0.0f;
    const uint32_t ParameterDefault_OutLine         = nn::atk::OutputLine_Main;

    //  パラメータの最大値
    const float ParameterMax_Volume          = 4.0f;
    const float ParameterMax_Pitch           = 4.0f;
    const int   ParameterMax_StartOffset     = 600000;
    const float ParameterMax_Pan             = 2.0f;
    const float ParameterMax_SPan            = 2.0f;
    const float ParameterMax_MainSend        = 1.0f;
    const float ParameterMax_EffectSendA     = 1.0f;
    const float ParameterMax_EffectSendB     = 1.0f;
    const float ParameterMax_EffectSendC     = 1.0f;
    const float ParameterMax_LpfFrequency    = 0.0f;
    const int   ParameterMax_BqfType         = nn::atk::BiquadFilterType_BandPassFilter2048;
    const float ParameterMax_BqfValue        = 1.0f;
    const float ParameterMax_MainOutVolume   = 4.0f;
    const float ParameterMax_PanMain         = 2.0f;
    const float ParameterMax_SPanMain        = 2.0f;
    const float ParameterMax_MainSendMain    = 1.0f;
    const float ParameterMax_EffectSendAMain = 1.0f;
    const float ParameterMax_EffectSendBMain = 1.0f;
    const float ParameterMax_EffectSendCMain = 1.0f;

    //  パラメータの最小値
    const float ParameterMin_Volume          =  0.0f;
    const float ParameterMin_Pitch           =  0.01f;
    const int   ParameterMin_StartOffset     =  0;
    const float ParameterMin_Pan             = -2.0f;
    const float ParameterMin_SPan            = -2.0f;
    const float ParameterMin_MainSend        = -1.0f;
    const float ParameterMin_EffectSendA     = -1.0f;
    const float ParameterMin_EffectSendB     = -1.0f;
    const float ParameterMin_EffectSendC     = -1.0f;
    const float ParameterMin_LpfFrequency    = -1.0f;
    const int   ParameterMin_BqfType         =  nn::atk::BiquadFilterType_Inherit;
    const float ParameterMin_BqfValue        =  0.0f;
    const float ParameterMin_MainOutVolume   =  0.0f;
    const float ParameterMin_PanMain         = -2.0f;
    const float ParameterMin_SPanMain        = -2.0f;
    const float ParameterMin_MainSendMain    = -1.0f;
    const float ParameterMin_EffectSendAMain = -1.0f;
    const float ParameterMin_EffectSendBMain = -1.0f;
    const float ParameterMin_EffectSendCMain = -1.0f;
}
namespace
{
    //  パラメータをセットします
    void SetParameter(int& value, int newValue, int min, int max, WrappedSoundHandle& pHandle) NN_NOEXCEPT
    {
        newValue = nn::atk::detail::fnd::Clamp( newValue, min, max );

        pHandle.SetParameterIsDirty( value != newValue );
        value = newValue;
    }
    //  パラメータをセットします  float 版
    void SetParameter(float& value, float newValue, float min, float max, WrappedSoundHandle& pHandle) NN_NOEXCEPT
    {
        newValue = nn::atk::detail::fnd::Clamp( newValue, min, max );
        if( -ThresholdConsiderAsZero < newValue && newValue < ThresholdConsiderAsZero )
        {
            newValue = 0.0f;
        }

        pHandle.SetParameterIsDirty( value != newValue );
        value = newValue;
    }
}


//  コンストラクタです
WrappedSoundHandle::WrappedSoundHandle() NN_NOEXCEPT
    : m_pSoundArchiveContext( nullptr )
    , m_pSoundHandle( nullptr )
    , m_IsDirtyParameter( true )
    , m_HandleIndex( 0 )
    , m_SoundIndex( 0 )
    , m_Volume         ( ParameterDefault_Volume )
    , m_Pitch          ( ParameterDefault_Pitch )
    , m_StartOffset    ( ParameterDefault_StartOffset )
    , m_Pan            ( ParameterDefault_Pan )
    , m_SPan           ( ParameterDefault_SPan )
    , m_MainSend       ( ParameterDefault_MainSend )
    , m_EffectSendA    ( ParameterDefault_EffectSendA )
    , m_EffectSendB    ( ParameterDefault_EffectSendB )
    , m_EffectSendC    ( ParameterDefault_EffectSendC )
    , m_LpfFrequency   ( ParameterDefault_LpfFrequency )
    , m_BqfType        ( ParameterDefault_BqfType )
    , m_BqfValue       ( ParameterDefault_BqfValue )
    , m_MainOutVolume  ( ParameterDefault_MainOutVolume )
    , m_PanMain        ( ParameterDefault_PanMain )
    , m_SPanMain       ( ParameterDefault_SPanMain )
    , m_MainSendMain   ( ParameterDefault_MainSendMain )
    , m_EffectSendAMain( ParameterDefault_EffectSendAMain )
    , m_EffectSendBMain( ParameterDefault_EffectSendBMain )
    , m_EffectSendCMain( ParameterDefault_EffectSendCMain )
    , m_OutputLine     ( ParameterDefault_OutLine )
{
}
//  初期化します
void WrappedSoundHandle::Initialize(SoundArchiveContext* pSoundArchiveContext, nn::atk::SoundHandle* pSoundHandle, int handleIndex) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pSoundArchiveContext );
    NN_ABORT_UNLESS_NOT_NULL( pSoundHandle );

    m_pSoundArchiveContext = pSoundArchiveContext;
    m_pSoundHandle = pSoundHandle;
    m_HandleIndex = handleIndex;
}
//  終了処理をします
void WrappedSoundHandle::Finalize() NN_NOEXCEPT
{
    m_pSoundHandle = nullptr;
    m_pSoundArchiveContext = nullptr;
}
//  更新します
void WrappedSoundHandle::Update() NN_NOEXCEPT
{
    if( m_IsDirtyParameter )
    {
        m_IsDirtyParameter = false;

        ApplyParameters();
    }
}

//  再生します
void WrappedSoundHandle::Start() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundArchiveContext );

    nn::atk::SoundArchivePlayer* pPlayer = m_pSoundArchiveContext->GetSoundArchivePlayer();
    if( pPlayer != nullptr )
    {
        Stop();

        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
        info.startOffset = m_StartOffset;
        info.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;

        const nn::atk::SoundArchive::ItemId itemId = nn::atk::SoundArchive::GetSoundIdFromIndex( m_SoundIndex );
        m_pSoundArchiveContext->LoadSound( itemId );
        pPlayer->StartSound( m_pSoundHandle, itemId, &info );

        ApplyParameters();
    }
}
//  停止します
void WrappedSoundHandle::Stop() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );

    m_pSoundHandle->Stop( FadeOutFrames );
}
//  一時停止または再開します
void WrappedSoundHandle::Pause() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );

    const bool isPaused = IsPaused();
    m_pSoundHandle->Pause( !isPaused, FadeOutFrames );
}

//  再生中であるか
bool WrappedSoundHandle::IsPlaying() const NN_NOEXCEPT
{
    return !IsStoppped() && !IsPaused();
}
//  一時停止しているか
bool WrappedSoundHandle::IsPaused() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );

    return m_pSoundHandle->IsPause();
}
//  停止しているか
bool WrappedSoundHandle::IsStoppped() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );

    return !m_pSoundHandle->IsAttachedSound();
}

//  サウンドのラベルを取得します
const char* WrappedSoundHandle::GetSoundLabel() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundArchiveContext );

    const nn::atk::SoundArchive::ItemId itemId = nn::atk::SoundArchive::GetSoundIdFromIndex( m_SoundIndex );
    return m_pSoundArchiveContext->GetSoundLabel( itemId );
}
//  パラメータを適用する必要があるかどうかを設定します
void WrappedSoundHandle::SetParameterIsDirty(bool isDirty) NN_NOEXCEPT
{
    m_IsDirtyParameter = m_IsDirtyParameter || isDirty;
}

//  パラメータを適用します
void WrappedSoundHandle::ApplyParameters() NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[SndHdl] Apply Params : index= %d", GetHandleIndex() );
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundHandle );
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundArchiveContext );

    m_pSoundHandle->SetVolume( m_Volume );
    m_pSoundHandle->SetPitch( m_Pitch );
    m_pSoundHandle->SetPan( m_Pan );
    m_pSoundHandle->SetSurroundPan( m_SPan );
    m_pSoundHandle->SetMainSend( m_MainSend );
    m_pSoundHandle->SetEffectSend( nn::atk::AuxBus_A, m_EffectSendA );
    m_pSoundHandle->SetEffectSend( nn::atk::AuxBus_B, m_EffectSendB );
    m_pSoundHandle->SetEffectSend( nn::atk::AuxBus_C, m_EffectSendC );
    m_pSoundHandle->SetLowPassFilterFrequency( m_LpfFrequency );
    m_pSoundHandle->SetBiquadFilter( m_BqfType, m_BqfValue );
    m_pSoundHandle->SetOutputVolume( nn::atk::OutputDevice::OutputDevice_Main, m_MainOutVolume );
    m_pSoundHandle->SetOutputPan( nn::atk::OutputDevice::OutputDevice_Main, m_PanMain );
    m_pSoundHandle->SetOutputSurroundPan( nn::atk::OutputDevice::OutputDevice_Main, m_SPanMain );
    m_pSoundHandle->SetOutputMainSend( nn::atk::OutputDevice::OutputDevice_Main, m_MainSendMain );
    m_pSoundHandle->SetOutputEffectSend( nn::atk::OutputDevice::OutputDevice_Main, nn::atk::AuxBus_A, m_EffectSendAMain );
    m_pSoundHandle->SetOutputEffectSend( nn::atk::OutputDevice::OutputDevice_Main, nn::atk::AuxBus_B, m_EffectSendBMain );
    m_pSoundHandle->SetOutputEffectSend( nn::atk::OutputDevice::OutputDevice_Main, nn::atk::AuxBus_C, m_EffectSendCMain );
    m_pSoundHandle->SetOutputLine( m_OutputLine );
}

//--  パラメータの Set, Get 関数  --//
int WrappedSoundHandle::GetHandleIndex() const NN_NOEXCEPT
{
    return m_HandleIndex;
}
void WrappedSoundHandle::SetSoundIndex(uint32_t index) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( m_pSoundArchiveContext );
    const uint32_t soundCount = m_pSoundArchiveContext->GetSoundCount();

    //  clamp
    if( index >= soundCount )
    {
        index = soundCount;
    }

    SetParameterIsDirty( m_SoundIndex != index );
    m_SoundIndex = index;
}
uint32_t WrappedSoundHandle::GetSoundIndex() const NN_NOEXCEPT
{
    return m_SoundIndex;
}
void WrappedSoundHandle::SetVolume(float volume) NN_NOEXCEPT
{
    SetParameter(
        m_Volume,
        volume,
        ParameterMin_Volume,
        ParameterMax_Volume,
        *this
    );
}
float WrappedSoundHandle::GetVolume() const NN_NOEXCEPT
{
    return m_Volume;
}
void WrappedSoundHandle::SetPitch(float pitch) NN_NOEXCEPT
{
    SetParameter(
        m_Pitch,
        pitch,
        ParameterMin_Pitch,
        ParameterMax_Pitch,
        *this
    );
}
float WrappedSoundHandle::GetPitch() const NN_NOEXCEPT
{
    return m_Pitch;
}
void WrappedSoundHandle::SetStartOffset(int startOffset) NN_NOEXCEPT
{
    SetParameter(
        m_StartOffset,
        startOffset,
        ParameterMin_StartOffset,
        ParameterMax_StartOffset,
        *this
    );
}
int WrappedSoundHandle::GetStartOffset() const NN_NOEXCEPT
{
    return m_StartOffset;
}
void WrappedSoundHandle::SetPan(float pan) NN_NOEXCEPT
{
    SetParameter(
        m_Pan,
        pan,
        ParameterMin_Pan,
        ParameterMax_Pan,
        *this
    );
}
float WrappedSoundHandle::GetPan() const NN_NOEXCEPT
{
    return m_Pan;
}
void WrappedSoundHandle::SetSPan(float pan) NN_NOEXCEPT
{
    SetParameter(
        m_SPan,
        pan,
        ParameterMin_SPan,
        ParameterMax_SPan,
        *this
    );
}
float WrappedSoundHandle::GetSPan() const NN_NOEXCEPT
{
    return m_SPan;
}
void WrappedSoundHandle::SetMainSend(float send) NN_NOEXCEPT
{
    SetParameter(
        m_MainSend,
        send,
        ParameterMin_MainSend,
        ParameterMax_MainSend,
        *this
    );
}
float WrappedSoundHandle::GetMainSend() const NN_NOEXCEPT
{
    return m_MainSend;
}
void WrappedSoundHandle::SetEffectSendA(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendA,
        send,
        ParameterMin_EffectSendA,
        ParameterMax_EffectSendA,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendA() const NN_NOEXCEPT
{
    return m_EffectSendA;
}
void WrappedSoundHandle::SetEffectSendB(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendB,
        send,
        ParameterMin_EffectSendB,
        ParameterMax_EffectSendB,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendB() const NN_NOEXCEPT
{
    return m_EffectSendB;
}
void WrappedSoundHandle::SetEffectSendC(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendC,
        send,
        ParameterMin_EffectSendC,
        ParameterMax_EffectSendC,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendC() const NN_NOEXCEPT
{
    return m_EffectSendC;
}
void WrappedSoundHandle::SetLpfFrequency(float value) NN_NOEXCEPT
{
    SetParameter(
        m_LpfFrequency,
        value,
        ParameterMin_LpfFrequency,
        ParameterMax_LpfFrequency,
        *this
    );
}
float WrappedSoundHandle::GetLpfFrequency() const NN_NOEXCEPT
{
    return m_LpfFrequency;
}
void WrappedSoundHandle::SetBqfType(int type) NN_NOEXCEPT
{
    SetParameter(
        m_BqfType,
        type,
        ParameterMin_BqfType,
        ParameterMax_BqfType,
        *this
    );
}
int WrappedSoundHandle::GetBqfType() const NN_NOEXCEPT
{
    return m_BqfType;
}
void WrappedSoundHandle::SetBqfValue(float value) NN_NOEXCEPT
{
    SetParameter(
        m_BqfValue,
        value,
        ParameterMin_BqfValue,
        ParameterMax_BqfValue,
        *this
    );
}
float WrappedSoundHandle::GetBqfValue() const NN_NOEXCEPT
{
    return m_BqfValue;
}
void WrappedSoundHandle::SetMainOutVolume(float volume) NN_NOEXCEPT
{
    SetParameter(
        m_MainOutVolume,
        volume,
        ParameterMin_MainOutVolume,
        ParameterMax_MainOutVolume,
        *this
    );
}
float WrappedSoundHandle::GetMainOutVolume() const NN_NOEXCEPT
{
    return m_MainOutVolume;
}
void WrappedSoundHandle::SetPanMain(float pan) NN_NOEXCEPT
{
    SetParameter(
        m_PanMain,
        pan,
        ParameterMin_PanMain,
        ParameterMax_PanMain,
        *this
    );
}
float WrappedSoundHandle::GetPanMain() const NN_NOEXCEPT
{
    return m_PanMain;
}
void WrappedSoundHandle::SetSPanMain(float pan) NN_NOEXCEPT
{
    SetParameter(
        m_SPanMain,
        pan,
        ParameterMin_SPanMain,
        ParameterMax_SPanMain,
        *this
    );
}
float WrappedSoundHandle::GetSPanMain() const NN_NOEXCEPT
{
    return m_SPanMain;
}
void WrappedSoundHandle::SetMainSendMain(float send) NN_NOEXCEPT
{
    SetParameter(
        m_MainSendMain,
        send,
        ParameterMin_MainSendMain,
        ParameterMax_MainSendMain,
        *this
    );
}
float WrappedSoundHandle::GetMainSendMain() const NN_NOEXCEPT
{
    return m_MainSendMain;
}
void WrappedSoundHandle::SetEffectSendAMain(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendAMain,
        send,
        ParameterMin_EffectSendAMain,
        ParameterMax_EffectSendAMain,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendAMain() const NN_NOEXCEPT
{
    return m_EffectSendAMain;
}
void WrappedSoundHandle::SetEffectSendBMain(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendBMain,
        send,
        ParameterMin_EffectSendBMain,
        ParameterMax_EffectSendBMain,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendBMain() const NN_NOEXCEPT
{
    return m_EffectSendBMain;
}
void WrappedSoundHandle::SetEffectSendCMain(float send) NN_NOEXCEPT
{
    SetParameter(
        m_EffectSendCMain,
        send,
        ParameterMin_EffectSendCMain,
        ParameterMax_EffectSendCMain,
        *this
    );
}
float WrappedSoundHandle::GetEffectSendCMain() const NN_NOEXCEPT
{
    return m_EffectSendCMain;
}
void WrappedSoundHandle::SetOutputLine(uint32_t lineFlag) NN_NOEXCEPT
{
    SetParameterIsDirty( m_OutputLine != lineFlag );
    m_OutputLine = lineFlag;
}
uint32_t WrappedSoundHandle::GetOutputLine() const NN_NOEXCEPT
{
    return m_OutputLine;
}
