﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk.h>
#include "gfxutil/GfxContext.h"
#include "SoundOutputPanel.h"
#include "FlightRecorder.h"

namespace
{
    //  パネルのタイトル
    const char* PanelTitle = "SoundOutput";
    //  情報ラベルの文字のスケール
    const float InfoLanelScaleX = 0.75f;
    const float InfoLanelScaleY = 0.8f;

    //  波形と波形の間のマージン
    const float WaveDrawerMargin = 4.0f;
    //  波形のラベルの背景色
    const nn::util::Uint8x4 WaveDrawerLabelColor = GetUint8x4( 32, 32, 32, 255 );
    //  波形のラベルの文字のスケール
    const float WaveDrawerLabelScaleX = 0.8f;
    const float WaveDrawerLabelScaleY = 0.7f;
    //  波形の線の色
    nn::util::Uint8x4 WaveDrawerLineColor = GetUint8x4( 32, 160, 160, 255 );
    //  波形の背景の色
    nn::util::Uint8x4 WaveDrawerBackColor = GetUint8x4( 16, 16, 16, 255 );

    //  アプリケーションフレーム数
    const size_t ApplicationFrameCount = 60;

    //  WaveDrawer で描画する波形の最大, 最小時間
    const nn::TimeSpan WaveDrawerTimeScaleMax = nn::TimeSpan::FromSeconds( 4 );           // 4s
    const nn::TimeSpan WaveDrawerTimeScaleMin = nn::TimeSpan::FromMicroSeconds( 31250 );  // 1/32s

    //  WaveDrawer で描画する波形の振幅のスケールの最大, 最小, 増減値
    const float WaveDrawerAmpScaleMax = 2.0f;
    const float WaveDrawerAmpScaleMin = 0.1f;
    const float WaveDrawerAmpScaleIncrement = 0.1f;
}

//  ------------------------------------------------------------
//
//                      SoundOutputPanel
//
//  ------------------------------------------------------------

//  必要なメモリ量を取得します
size_t SoundOutputPanel::GetRequiredMemorySize() NN_NOEXCEPT
{
    size_t memSize = 0;

    for( auto& wd : m_WaveDrawer )
    {
        memSize += wd.GetRequiredMemorySize( m_Recorder.GetSampleCountPerSecond() );
    }

    memSize += sizeof(int16_t) * GetSampleCountPerApplicationFrame();

    memSize += m_Recorder.GetRequiredMemorySize();

    return memSize;
}
//  初期化します
void SoundOutputPanel::Initialize(void* buffer, size_t bufferSize, float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_EQUAL( bufferSize, GetRequiredMemorySize() );

    m_Panel.SetTitle( PanelTitle );
    m_Panel.SetPosition( positionX, positionY );
    m_Panel.SetSize( sizeX, sizeY );

    m_InfoLabel.SetDrawAlign( gfxutil::Label::DrawAlign_Left );
    m_InfoLabel.SetScale( InfoLanelScaleX, InfoLanelScaleY );

    m_TimeScale = nn::TimeSpan::FromSeconds( 1 );
    m_AmpScale = 1.0f;

    nn::util::BytePtr ptr( buffer );
    for( auto& wd : m_WaveDrawer )
    {
        const size_t size = wd.GetRequiredMemorySize( m_Recorder.GetSampleCountPerSecond() );
        wd.Initialize( ptr.Get(), size, m_Recorder.GetSampleCountPerSecond() );
        ptr.Advance( size );
    }
    m_WaveDrawer[ChannelIndex_FL] .SetText( "FL" );
    m_WaveDrawer[ChannelIndex_FR] .SetText( "FR" );
    m_WaveDrawer[ChannelIndex_FC] .SetText( "FC" );
    m_WaveDrawer[ChannelIndex_LFE].SetText( "LFE" );
    m_WaveDrawer[ChannelIndex_RL] .SetText( "RL" );
    m_WaveDrawer[ChannelIndex_RR] .SetText( "RR" );

    m_SampleBuffer = ptr.Get<int16_t>();
    ptr.Advance( sizeof(int16_t) * GetSampleCountPerApplicationFrame() );
    NN_ABORT_UNLESS_NOT_NULL( m_SampleBuffer );

    {
        const size_t size = m_Recorder.GetRequiredMemorySize();
        m_Recorder.Initialize( ptr.Get(), size );
        ptr.Advance( size );
        m_Recorder.Start();
    }
}
//  終了します
void SoundOutputPanel::Finalize() NN_NOEXCEPT
{
    m_Recorder.Finalize();
}
//  更新します
void SoundOutputPanel::Update() NN_NOEXCEPT
{
    const size_t sampleCountPerFrame = GetSampleCountPerApplicationFrame();

    while( m_Recorder.ReadSamples( m_SampleBuffer, sampleCountPerFrame ) == sampleCountPerFrame )
    {
        const size_t sampleCountPerChannel = sampleCountPerFrame / ChannelIndex_Count;

        m_WaveDrawer[ChannelIndex_FL] .Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_FrontLeft, ChannelIndex_Count );
        m_WaveDrawer[ChannelIndex_FR] .Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_FrontRight, ChannelIndex_Count );
        m_WaveDrawer[ChannelIndex_FC] .Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_FrontCenter, ChannelIndex_Count );
        m_WaveDrawer[ChannelIndex_LFE].Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_Lfe, ChannelIndex_Count );
        m_WaveDrawer[ChannelIndex_RL] .Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_RearLeft, ChannelIndex_Count );
        m_WaveDrawer[ChannelIndex_RR] .Push( m_SampleBuffer, sampleCountPerChannel, nn::atk::ChannelIndex_RearRight, ChannelIndex_Count );
    }

    m_InfoLabel.SetText( "[%s]  Time:%.2fsec  Amp:x%.1f", m_Recorder.IsRecording() ? "Play" : "Stop", m_TimeScale.GetMilliSeconds() / 1000.0f, m_AmpScale );
}
//  入力による更新を行います
void SoundOutputPanel::UpdateByHid(const HidPad& hidPad) NN_NOEXCEPT
{
    if( m_Panel.IsFocused() )
    {
        if( hidPad.IsContinue( HidPad::Button_Left ) )
        {
            const auto next = std::max( nn::TimeSpan::FromMicroSeconds( m_TimeScale.GetMicroSeconds() / 2 ), WaveDrawerTimeScaleMin );
            FlightRecorder::GetInstance().WriteLog( "[SndOut] TimeScale : %lld -> %lld", m_TimeScale.GetMilliSeconds(), next.GetMilliSeconds() );
            m_TimeScale = next;
        }
        else if( hidPad.IsContinue( HidPad::Button_Right ) )
        {
            const auto next = std::min( nn::TimeSpan::FromMicroSeconds( m_TimeScale.GetMicroSeconds() * 2 ), WaveDrawerTimeScaleMax );
            FlightRecorder::GetInstance().WriteLog( "[SndOut] TimeScale : %lld -> %lld", m_TimeScale.GetMilliSeconds(), next.GetMilliSeconds() );
            m_TimeScale = next;
        }

        if( hidPad.IsContinue( HidPad::Button_Up ) )
        {
            const auto next = std::min( m_AmpScale + WaveDrawerAmpScaleIncrement, WaveDrawerAmpScaleMax );
            FlightRecorder::GetInstance().WriteLog( "[SndOut] AmpScale : %f -> %f", m_AmpScale, next );
            m_AmpScale = next;

            for( auto& wd : m_WaveDrawer )
            {
                wd.SetAmpScale( m_AmpScale );
            }
        }
        else if( hidPad.IsContinue( HidPad::Button_Down ) )
        {
            const auto next = std::max( m_AmpScale - WaveDrawerAmpScaleIncrement, WaveDrawerAmpScaleMin );
            FlightRecorder::GetInstance().WriteLog( "[SndOut] AmpScale : %f -> %f", m_AmpScale, next );
            m_AmpScale = next;

            for( auto& wd : m_WaveDrawer )
            {
                wd.SetAmpScale( m_AmpScale );
            }
        }

        if( hidPad.IsTrigger( HidPad::Button_A ) )
        {
            FlightRecorder::GetInstance().WriteLog( "[SndOut] Start" );
            if( !m_Recorder.IsRecording() )
            {
                for( auto& wd : m_WaveDrawer )
                {
                    wd.Clear();
                }
            }
            m_Recorder.Start();
        }
        else if( hidPad.IsTrigger( HidPad::Button_B ) )
        {
            FlightRecorder::GetInstance().WriteLog( "[SndOut] Stop" );
            m_Recorder.Stop();
        }
    }
}
//  描画します
void SoundOutputPanel::Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT
{
    m_Panel.Draw( gfxContext );

    gfxutil::FontRenderer& fontRenderer = gfxContext.GetFontRenderer();
    nn::util::Float2 infoPos = m_Panel.GetClientPositionRightBottom();
    infoPos.y -= m_InfoLabel.CalculateDrawSize( fontRenderer ).y;

    m_InfoLabel.SetPosition( infoPos.x, infoPos.y );
    m_InfoLabel.Draw( gfxContext );

    const nn::util::Float2 waveLeftTop = m_Panel.GetClientPositionLeftTop();
    const nn::util::Float2 waveRightBottom = infoPos;

    nn::util::Float2 position = waveLeftTop;
    nn::util::Float2 size = GetFloat2( waveRightBottom.x - waveLeftTop.x, ( waveRightBottom.y - waveLeftTop.y - ( ChannelIndex_Count - 1 ) * WaveDrawerMargin ) / ChannelIndex_Count );

    for( auto& wd : m_WaveDrawer )
    {
        wd.DrawBackGround( gfxContext, position, size );
        position.y += size.y + WaveDrawerMargin;
    }

    position.y = waveLeftTop.y;
    for( auto& wd : m_WaveDrawer )
    {
        wd.DrawWave( gfxContext, position, size, m_TimeScale );
        position.y += size.y + WaveDrawerMargin;
    }
}

//  フォーカスを設定します
void SoundOutputPanel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_Panel.SetFocused( isFocused );
}

//  1 アプリケーションフレームで読むサンプル数を取得します
size_t SoundOutputPanel::GetSampleCountPerApplicationFrame() const NN_NOEXCEPT
{
    return ChannelIndex_Count * m_Recorder.GetSampleCountPerSecond() / ApplicationFrameCount;
}


//  ------------------------------------------------------------
//
//                        WaveDrawer
//
//  ------------------------------------------------------------
NN_DEFINE_STATIC_CONSTANT( const int SoundOutputPanel::WaveDrawer::LineCount );
//  必要なメモリ量を取得します
size_t SoundOutputPanel::WaveDrawer::GetRequiredMemorySize(uint32_t sampleCountPerSec) NN_NOEXCEPT
{
    size_t memSize = 0;

    memSize += GetRequiredMemorySizeForWaveDrawerPoints();

    memSize += GetRequiredMemorySizeForWaveDrawerSampleBuffer( sampleCountPerSec );

    return memSize;
}
//  初期化します
void SoundOutputPanel::WaveDrawer::Initialize(void* buffer, size_t bufferSize, uint32_t sampleCountPerSec) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_GREATER_EQUAL( bufferSize, GetRequiredMemorySize( sampleCountPerSec ) );

    m_Label.SetBackColor( WaveDrawerLabelColor );
    m_Label.SetScale( WaveDrawerLabelScaleX, WaveDrawerLabelScaleY );

    nn::util::BytePtr ptr( buffer );
    {
        const size_t size = GetRequiredMemorySizeForWaveDrawerPoints();
        m_pPoints = ptr.Get<nn::util::Float3>();
        NN_ABORT_UNLESS_NOT_NULL( m_pPoints );
        ptr.Advance( size );

        const int pointCount = LineCount * 2;
        for( int i = 0; i < pointCount; i++ )
        {
            m_pPoints[i].z = 0.0f;
        }
    }

    {
        const size_t size = GetRequiredMemorySizeForWaveDrawerSampleBuffer( sampleCountPerSec );
        m_pSample = ptr.Get<int16_t>();
        NN_ABORT_UNLESS_NOT_NULL( m_pSample );
        ptr.Advance( size );

        m_WritePosition = 0;
        m_SampleCount = static_cast<int>( size / sizeof(int16_t) );
        m_BlockSampleSize = static_cast<int>( sampleCountPerSec * WaveDrawerTimeScaleMin.GetMicroSeconds() / nn::TimeSpan::FromSeconds( 1 ).GetMicroSeconds() );

        //  割り切れないと Draw の time に設定した時間分の波形を描画できません
        NN_ABORT_UNLESS( m_BlockSampleSize % LineCount == 0 );
        m_BlockSampleSize /= LineCount;

        //  m_SampleCount を m_BlockSampleSize の倍数にします
        m_SampleCount = m_SampleCount / m_BlockSampleSize * m_BlockSampleSize;
    }

    m_AmpScale = 1.0f;
}
//  サンプルを追加します
void SoundOutputPanel::WaveDrawer::Push(const int16_t* pSamples, size_t sampleCount, int offset, int stride) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pSamples );
    NN_ABORT_UNLESS_GREATER_EQUAL( offset, 0 );
    NN_ABORT_UNLESS_GREATER_EQUAL( stride, 0 );

    pSamples += offset;
    for( size_t i = 0; i < sampleCount; ++i )
    {
        m_pSample[m_WritePosition++] = *pSamples;
        if( m_WritePosition == m_SampleCount )
        {
            m_WritePosition = 0;
        }

        pSamples += stride;
    }
}
//  波形を無音状態にします
void SoundOutputPanel::WaveDrawer::Clear() NN_NOEXCEPT
{
    for( int i = 0; i < m_SampleCount; i++ )
    {
        m_pSample[i] = 0;
    }
}
//  背景を描画します
void SoundOutputPanel::WaveDrawer::DrawBackGround(gfxutil::GfxContext& gfxContext, const nn::util::Float2& position, const nn::util::Float2& size) NN_NOEXCEPT
{
    gfxContext.DrawQuad( nn::util::Vector3f( position.x, position.y, 0.0f ), size, WaveDrawerBackColor );
}
//  time に設定された時間分の波形を描画します
void SoundOutputPanel::WaveDrawer::DrawWave(gfxutil::GfxContext& gfxContext, const nn::util::Float2& position, const nn::util::Float2& size, const nn::TimeSpan& time) NN_NOEXCEPT
{
    const int blockSize = static_cast<int>( m_BlockSampleSize * time.GetMicroSeconds()  / WaveDrawerTimeScaleMin.GetMicroSeconds() );
    int seek = m_WritePosition / blockSize * blockSize;

    const float mulX = size.x / LineCount;
    const float mulY = m_AmpScale * 0.5f * size.y / INT16_MAX;
    const float positionX = position.x + size.x;
    const float positionY = position.y + 0.5f * size.y;
    const float clampYMax =  0.5f * size.y;
    const float clampYMin = -0.5f * size.y;
    const int pointCount = LineCount * 2;

    for( int i = 0; i < pointCount; i += 2 )
    {
        int16_t max = INT16_MIN;
        int16_t min = INT16_MAX;
        int foundMax = 0;
        int foundMin = 0;

        //  ある区間の最大, 最小値を求めます
        for( int k = 0; k < blockSize; k++ )
        {
            seek--;
            if( seek < 0 )
            {
                seek = m_SampleCount - 1;
            }

            if( max < m_pSample[seek] )
            {
                max = m_pSample[seek];
                foundMax = k;
            }
            if( min > m_pSample[seek] )
            {
                min = m_pSample[seek];
                foundMin = k;
            }
        }

        m_pPoints[i].x     = positionX - mulX * ( i / 2 );
        m_pPoints[i + 1].x = m_pPoints[i].x;

        if( foundMin < foundMax )
        {
            //  min が max よりも先に見つかったので
            //  min が先に描画されるようにします
            m_pPoints[i].y     = nn::atk::detail::fnd::Clamp( mulY * min, clampYMin, clampYMax ) + positionY;
            m_pPoints[i + 1].y = nn::atk::detail::fnd::Clamp( mulY * max, clampYMin, clampYMax ) + positionY;
        }
        else
        {
            //  max が min よりも先に見つかったので
            //  max が先に描画されるようにします
            m_pPoints[i].y     = nn::atk::detail::fnd::Clamp( mulY * max, clampYMin, clampYMax ) + positionY;
            m_pPoints[i + 1].y = nn::atk::detail::fnd::Clamp( mulY * min, clampYMin, clampYMax ) + positionY;
        }
    }

    gfxContext.DrawLinkedPoint( m_pPoints, pointCount, WaveDrawerLineColor );

    m_Label.SetPosition( position.x, position.y );
    m_Label.Draw( gfxContext );
}

//  ラベルの文字を設定します
void SoundOutputPanel::WaveDrawer::SetText(const char* text) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( text );
    m_Label.SetText( text );
}
//  波形の振幅のスケールを設定します
void SoundOutputPanel::WaveDrawer::SetAmpScale(float scale) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX( scale, WaveDrawerAmpScaleMin, WaveDrawerAmpScaleMax );
    m_AmpScale = scale;
}

//  波形を構成する点のバッファサイズを取得します
size_t SoundOutputPanel::WaveDrawer::GetRequiredMemorySizeForWaveDrawerPoints() const NN_NOEXCEPT
{
    const int pointCount = LineCount * 2;
    return pointCount * sizeof(nn::util::Float3);
}
//  サンプルを保存するバッファのサイズを取得します
size_t SoundOutputPanel::WaveDrawer::GetRequiredMemorySizeForWaveDrawerSampleBuffer(uint32_t sampleCountPerSec) const NN_NOEXCEPT
{
    const auto sampleCount = WaveDrawerTimeScaleMax.GetSeconds() * sampleCountPerSec;
    const auto memSize = sizeof(int16_t) * ( sampleCount + sampleCountPerSec );  //  少し大きめに確保します
    return static_cast<size_t>( memSize );
}
