﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "Util.h"
#include "SoundArchiveContext.h"
#include "FlightRecorder.h"

namespace
{
    //  atk::SoundHeap のヒープサイズ
    const size_t SoundHeapSize = 320 * 1024 * 1024;
    //  SoundArchivePlayer の STRM 1 つあたりのキャッシュのサイズ
    const size_t SoundArchivePlayerStreamCacheSize = 16 * 1024;
}
namespace
{
#if defined( ENABLE_HTCS_CONNECTION )
    //  SoundEdit から index 番目の SoundController の実装を取得します
    nn::atk::viewer::detail::SoundControllerImpl& GetSoundControllerImpl(const nn::atk::viewer::SoundEdit& soundEdit, int index)
    {
        nn::atk::viewer::SoundController* pSoundController = const_cast<nn::atk::viewer::SoundEdit&>( soundEdit ).GetSoundObjectController().GetSoundController( index );
        NN_ABORT_UNLESS_NOT_NULL( pSoundController );
        return *reinterpret_cast<nn::atk::viewer::detail::SoundControllerImpl*>( pSoundController );
    }
#endif
    //  SoundArchivePlayer に設定する STRM キャッシュサイズを計算します
    size_t CalcStreamCacheSize(const nn::atk::SoundArchive& soundArchive) NN_NOEXCEPT
    {
        nn::atk::SoundArchive::SoundArchivePlayerInfo info;
        if ( soundArchive.ReadSoundArchivePlayerInfo( &info ) )
        {
            return info.streamSoundCount * SoundArchivePlayerStreamCacheSize;
        }
        else
        {
            return 0u;
        }
    }
}

NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveContext::SoundHandleCountMax );


//  初期化します
void SoundArchiveContext::Initialize(void* pMemory, size_t memorySize) NN_NOEXCEPT
{
    m_IsFileOpened = false;

    m_pMemoryForInfoBlock = nullptr;
    m_pMemoryForSoudDataManager = nullptr;
    m_pMemoryForSoundArchivePlayer = nullptr;
    m_pMemoryForStreamBuffer = nullptr;

    //  Allocator の初期化
    m_Allocator.Initialize( pMemory, memorySize );

    //  SoundHeap の初期化
    m_pMemoryForSoundHeap = m_Allocator.Allocate( SoundHeapSize );
    NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundHeap );
    const bool isSuccess = m_SoundHeap.Create( m_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

#if defined( ENABLE_HTCS_CONNECTION )
    m_IsConnected = false;
    m_pMemoryForSoundEdit = nullptr;

    //  SoundEdit の初期化
    {
        nn::atk::viewer::SoundEdit::Option option;
        const size_t memSize = m_SoundEdit.GetRequiredMemorySize( option );
        m_pMemoryForSoundEdit = m_Allocator.Allocate( memSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundEdit );

        nn::atk::viewer::SoundEdit::InitializeArg args;
        args.buffer = m_pMemoryForSoundEdit;
        args.bufferSize = memSize;
        args.soundArchive = &m_SoundArchive;
        args.soundArchivePlayer = &m_SoundArchivePlayer;
        m_SoundEdit.Initialize( args, option );
        m_SoundEdit.Start();

        for( int i = 0; i < SoundHandleCountMax; i++ )
        {
            m_LoadSoundCallBackParameter[i].pSoundArchiveContext = this;
            m_LoadSoundCallBackParameter[i].pWrappedSoundHandle = &m_WrappedSoundHandle[i];
            GetSoundControllerImpl( m_SoundEdit, i ).SetPreplayAction( LoadSoundCallBack, &m_LoadSoundCallBackParameter[i] );
        }
    }

    //  m_WrappedSoundHandle の初期化
    for( int i = 0 ; i < SoundHandleCountMax; i++ )
    {
        nn::atk::viewer::SoundController* pSoundController = const_cast<nn::atk::viewer::SoundEdit&>( m_SoundEdit ).GetSoundObjectController().GetSoundController( i );
        NN_ABORT_UNLESS_NOT_NULL( pSoundController );

        m_WrappedSoundHandle[i].Initialize( this, pSoundController->GetSoundHandle(), i );
    }

#else
    //  m_WrappedSoundHandle の初期化
    for( int i = 0 ; i < SoundHandleCountMax; i++ )
    {
        m_WrappedSoundHandle[i].Initialize( this, &m_SoundHandle[i], i );
    }

#endif
}
//  終了処理をします
void SoundArchiveContext::Finalize() NN_NOEXCEPT
{
    Close();

#if defined( ENABLE_HTCS_CONNECTION )
    m_IsConnected = false;

    m_SoundEdit.Stop();
    m_SoundEdit.Finalize();

    m_Allocator.Free( m_pMemoryForSoundEdit );
    m_pMemoryForSoundEdit = nullptr;
#endif

    for( int i = 0; i< SoundHandleCountMax; i++ )
    {
        m_WrappedSoundHandle[i].Finalize();
    }

    m_SoundHeap.Destroy();
    m_Allocator.Free( m_pMemoryForSoundHeap );
    m_pMemoryForSoundHeap = nullptr;
}
//  更新処理をします
void SoundArchiveContext::Update() NN_NOEXCEPT
{
    if( m_IsFileOpened )
    {
        m_SoundArchivePlayer.Update();

        for( int i = 0 ; i < SoundHandleCountMax; i++ )
        {
            m_WrappedSoundHandle[i].Update();
        }

#if defined( ENABLE_HTCS_CONNECTION )
        m_SoundEdit.Update();

        const bool connected = m_SoundEdit.IsConnected();
        if( m_IsConnected != connected )
        {
            //  接続状態が変わったとき、すべてのサウンドを停止させる
            for( int i = 0; i < SoundHandleCountMax; i++ )
            {
                GetSoundHandle(i).Stop();
            }

            m_IsConnected = connected;

            if( connected )
            {
                FlightRecorder::GetInstance().WriteLog( "[Info] Connected." );
                NN_LOG( "connected.\n" );
            }
            else
            {
                FlightRecorder::GetInstance().WriteLog( "[Info] Disconnected." );
                NN_LOG( "disconnected.\n" );
            }
        }
#endif
    }
}

//  サウンドアーカイブ内のサウンドの数を取得します
uint32_t SoundArchiveContext::GetSoundCount() const NN_NOEXCEPT
{
    if( m_IsFileOpened )
    {
        return m_SoundArchive.GetSoundCount();
    }
    else
    {
        return 0;
    }
}
//  soundId のサウンドのラベルを取得します
const char* SoundArchiveContext::GetSoundLabel(nn::atk::SoundArchive::ItemId itemId) const NN_NOEXCEPT
{
    if( m_IsFileOpened )
    {
        return m_SoundArchive.GetItemLabel( itemId );
    }
    else
    {
        return nullptr;
    }
}
//  handleIndex 番目の SoundHandle を取得します
WrappedSoundHandle& SoundArchiveContext::GetSoundHandle(int handleIndex) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( handleIndex, 0, SoundHandleCountMax );
    return m_WrappedSoundHandle[handleIndex];
}
//  handleIndex 番目の SoundHandle を取得します
const WrappedSoundHandle& SoundArchiveContext::GetSoundHandle(int handleIndex) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( handleIndex, 0, SoundHandleCountMax );
    return m_WrappedSoundHandle[handleIndex];
}
//  SoundArchivePlayer を取得します
nn::atk::SoundArchivePlayer* SoundArchiveContext::GetSoundArchivePlayer() NN_NOEXCEPT
{
    if( m_IsFileOpened )
    {
        return &m_SoundArchivePlayer;
    }
    else
    {
        return nullptr;
    }
}

//  接続されているかを取得します
bool SoundArchiveContext::IsConnected() const NN_NOEXCEPT
{
#if defined( ENABLE_HTCS_CONNECTION )
    return m_IsConnected;
#else
    return false;
#endif
}

//  ファイルを開きます
void SoundArchiveContext::Open(const char* filePath) NN_NOEXCEPT
{
    //  RELOAD_SOUND_ARCHIVE_TEMP: サウンドアーカイブの再読み込みのための暫定処理
    //  2 度めの Open() ならば、この時点で m_IsFileOpened は true
    FlightRecorder::GetInstance().WriteLog( "[SndArcCtxt] Open %s", filePath );
    const bool isOpened = m_IsFileOpened;

    Close();

    //  ファイルの有無の確認
    {
        nn::fs::DirectoryEntryType type;
        const nn::Result result = nn::fs::GetEntryType( &type, filePath );

        if( result.IsFailure() )
        {
            NN_LOG( "cannot find \"%s\".\n", filePath );
            return ;
        }
        if( type != nn::fs::DirectoryEntryType_File )
        {
            NN_LOG( "\"%s\" is not a file.\n", filePath );
            return ;
        }
    }

    // SoundArchive の初期化
    {
        m_SoundArchive.SetFileAccessMode( nn::atk::FsSoundArchive::FileAccessMode_InFunction );

        const bool isSuccess = m_SoundArchive.Open( filePath );
        NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)", filePath );
    }

    // SoundArchive のパラメータ情報をメモリにロード
    {
        const size_t infoBlockSize = m_SoundArchive.GetHeaderSize();
        m_pMemoryForInfoBlock = m_Allocator.Allocate( infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForInfoBlock );

        bool isSuccess = m_SoundArchive.LoadHeader( m_pMemoryForInfoBlock, infoBlockSize );
        NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

        //  TODO: ラベル文字列データがないことを判断する良い方法が出来たら差し替え
        const size_t labelBlockSize = m_SoundArchive.GetLabelStringDataSize();
        const uint32_t LabelStringDataIsNothing = static_cast<uint32_t>( -1 );   //  ラベル文字列データがないときの返り値
        if( labelBlockSize == LabelStringDataIsNothing )
        {
            m_pMemoryForLabelBlock = nullptr;
        }
        else
        {
            m_pMemoryForLabelBlock = m_Allocator.Allocate( labelBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
            NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForLabelBlock );

            isSuccess = m_SoundArchive.LoadLabelStringData( m_pMemoryForLabelBlock, labelBlockSize );
            NN_ABORT_UNLESS( isSuccess, "cannot load LabelBlock" );
        }
    }

    // SoundDataManager の初期化
    {
        const size_t memSize = m_SoundDataManager.GetRequiredMemSize( &m_SoundArchive );
        m_pMemoryForSoudDataManager = m_Allocator.Allocate( memSize, nn::atk::SoundDataManager::BufferAlignSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoudDataManager );

        const bool isSuccess = m_SoundDataManager.Initialize( &m_SoundArchive, m_pMemoryForSoudDataManager, memSize );
        NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );
    }

    // SoundArchivePlayer の初期化
    {
        const size_t memSizeSoundArchive = m_SoundArchivePlayer.GetRequiredMemSize( &m_SoundArchive );
        m_pMemoryForSoundArchivePlayer = m_Allocator.Allocate( memSizeSoundArchive, nn::atk::SoundArchivePlayer::BufferAlignSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSoundArchivePlayer );

        size_t memSizeStreamBuffer = m_SoundArchivePlayer.GetRequiredStreamBufferSize( &m_SoundArchive );
        memSizeStreamBuffer = nn::util::align_up( memSizeStreamBuffer, nn::audio::MemoryPoolType::SizeGranularity );
        m_pMemoryForStreamBuffer = m_Allocator.Allocate( memSizeStreamBuffer, nn::audio::MemoryPoolType::AddressAlignment );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForStreamBuffer );
        AttachMemoryPool( &m_MemoryPoolForStreamBuffer, m_pMemoryForStreamBuffer, memSizeStreamBuffer );

        const size_t memSizeStreamCache = CalcStreamCacheSize( m_SoundArchive );
        m_pMemoryForStreamCacheBuffer = m_Allocator.Allocate( memSizeStreamCache, nn::atk::FsSoundArchive::BufferAlignSize );

        nn::atk::SoundArchivePlayer::InitializeParam param;
        param.pSoundArchive = &m_SoundArchive;
        param.pSoundDataManager =  &m_SoundDataManager;
        param.pSetupBuffer = m_pMemoryForSoundArchivePlayer;
        param.setupBufferSize = memSizeSoundArchive;
        param.pStreamBuffer = m_pMemoryForStreamBuffer;
        param.streamBufferSize = memSizeStreamBuffer;
        param.pStreamCacheBuffer = m_pMemoryForStreamCacheBuffer;
        param.streamCacheSize = memSizeStreamCache;

        const bool isSuccess = m_SoundArchivePlayer.Initialize( param );
        NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );
    }

    m_IsFileOpened = true;

    //  RELOAD_SOUND_ARCHIVE_TEMP: サウンドアーカイブの再読み込みのための暫定処理
    //  2 度めの Open() ならば、FileAccessEnd() を呼び出す必要がある
    if( isOpened )
    {
        m_SoundArchive.FileAccessEnd();
    }
}
//  soundId のサウンドを読み込みます
void SoundArchiveContext::LoadSound(nn::atk::SoundArchive::ItemId itemId) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[SndArcCtxt] LoadSound : %d", itemId );

    //  ストリームサウンドはロード不要
    if( m_SoundArchive.GetSoundType( itemId ) != nn::atk::SoundArchive::SoundType_Stream )
    {
        const bool isSuccess = m_SoundDataManager.LoadData( itemId, &m_SoundHeap );
        if( !isSuccess )
        {
            //  メモリプールのデタッチを反映するために Update します
            m_SoundArchivePlayer.Update();

            //  ヒープ容量が足りなかった可能性があるため、ヒープをクリアして再度ロード
            m_SoundHeap.Clear();
            m_SoundDataManager.LoadData( itemId, &m_SoundHeap );
        }
    }
}

//  ファイルを閉じます
void SoundArchiveContext::Close() NN_NOEXCEPT
{
    if( m_IsFileOpened )
    {
        FlightRecorder::GetInstance().WriteLog( "[SndArcCtxt] Close" );
        m_IsFileOpened = false;

        for( int i = 0; i < SoundHandleCountMax; i++ )
        {
            GetSoundHandle( i ).Stop();
        }

        m_SoundArchivePlayer.Finalize();
        DetachMemoryPool( &m_MemoryPoolForStreamBuffer );
        m_Allocator.Free( m_pMemoryForStreamCacheBuffer );
        m_Allocator.Free( m_pMemoryForStreamBuffer );
        m_Allocator.Free( m_pMemoryForSoundArchivePlayer );
        m_pMemoryForStreamCacheBuffer = nullptr;
        m_pMemoryForStreamBuffer = nullptr;
        m_pMemoryForSoundArchivePlayer = nullptr;

        m_SoundDataManager.Finalize();
        m_Allocator.Free( m_pMemoryForSoudDataManager );
        m_pMemoryForSoudDataManager = nullptr;

        m_SoundArchive.Close();

        if( m_pMemoryForLabelBlock != nullptr )
        {
            m_Allocator.Free( m_pMemoryForLabelBlock );
        }
        m_Allocator.Free( m_pMemoryForInfoBlock );
        m_pMemoryForLabelBlock = nullptr;
        m_pMemoryForInfoBlock = nullptr;
    }
}

#if defined( ENABLE_HTCS_CONNECTION )
//  soundId のサウンドを読み込みます
void SoundArchiveContext::LoadSoundCallBack(nn::atk::viewer::SoundController& target, nn::atk::SoundArchive::ItemId soundId, void* userParam) NN_NOEXCEPT
{
    FlightRecorder::GetInstance().WriteLog( "[SndArcCtxt] LoadSoundCallBack : %d", soundId );

    NN_ABORT_UNLESS_NOT_NULL( userParam );
    LoadSoundCallBackParameter* pLoadSoundCallBackParameter = reinterpret_cast<LoadSoundCallBackParameter*>( userParam );

    SoundArchiveContext* pSoundArchiveContext = pLoadSoundCallBackParameter->pSoundArchiveContext;
    NN_ABORT_UNLESS_NOT_NULL( pSoundArchiveContext );
    pSoundArchiveContext->LoadSound( soundId );

    //  HACK : soundId から soundIndex を mask を用いて取得します
    const uint32_t mask = ( 1 << 24 ) - 1;
    uint32_t soundIndex = soundId & mask;

    WrappedSoundHandle* pWrappedSoundHandle = pLoadSoundCallBackParameter->pWrappedSoundHandle;
    NN_ABORT_UNLESS_NOT_NULL( pWrappedSoundHandle );
    const uint32_t outputLine = pWrappedSoundHandle->GetOutputLine();
    pWrappedSoundHandle->SetSoundIndex( soundIndex );

    NN_STATIC_ASSERT( nn::atk::viewer::Limits::EffectBusCountMax == nn::atk::AuxBus_Count );
    nn::atk::viewer::SoundParameters& parameter = target.GetParameters();
    parameter.volume = pWrappedSoundHandle->GetVolume();
    parameter.pitch = pWrappedSoundHandle->GetPitch();
    parameter.startOffset = static_cast<uint32_t>( pWrappedSoundHandle->GetStartOffset() );
    parameter.pan = pWrappedSoundHandle->GetPan();
    parameter.surroundPan = pWrappedSoundHandle->GetSPan();
    parameter.mainSend = pWrappedSoundHandle->GetMainSend();
    parameter.effectSend[ nn::atk::AuxBus_A ] = pWrappedSoundHandle->GetEffectSendA();
    parameter.effectSend[ nn::atk::AuxBus_B ] = pWrappedSoundHandle->GetEffectSendB();
    parameter.effectSend[ nn::atk::AuxBus_C ] = pWrappedSoundHandle->GetEffectSendC();
    parameter.lowPassFilterFrequency = pWrappedSoundHandle->GetLpfFrequency();
    parameter.biquadFilterType = static_cast<uint32_t>( pWrappedSoundHandle->GetBqfType() );
    parameter.biquadFilterValue = pWrappedSoundHandle->GetBqfValue();
    parameter.mainOutVolume = pWrappedSoundHandle->GetMainOutVolume();
    parameter.mainPan = pWrappedSoundHandle->GetPanMain();
    parameter.mainSurroundPan = pWrappedSoundHandle->GetSPanMain();
    parameter.mainMainSend = pWrappedSoundHandle->GetMainSendMain();
    parameter.mainEffectSend[ nn::atk::AuxBus_A ] = pWrappedSoundHandle->GetEffectSendAMain();
    parameter.mainEffectSend[ nn::atk::AuxBus_B ] = pWrappedSoundHandle->GetEffectSendBMain();
    parameter.mainEffectSend[ nn::atk::AuxBus_C ] = pWrappedSoundHandle->GetEffectSendCMain();
    parameter.isMainOutEnabled = ( ( outputLine & nn::atk::OutputLine_Main ) != 0 );
}
#endif
