﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/settings/settings_DebugPad.h>
#include "HidPad.h"

namespace
{
    // Button と DebugPad の対応関係
    const int DebugPadMap[HidPad::Button_Max] =
    {
        nn::hid::DebugPadButton::A::Index,
        nn::hid::DebugPadButton::B::Index,
        nn::hid::DebugPadButton::X::Index,
        nn::hid::DebugPadButton::Y::Index,
        nn::hid::DebugPadButton::R::Index,
        nn::hid::DebugPadButton::L::Index,
        nn::hid::DebugPadButton::ZR::Index,
        nn::hid::DebugPadButton::Start::Index,
        nn::hid::DebugPadButton::Left::Index,
        nn::hid::DebugPadButton::Up::Index,
        nn::hid::DebugPadButton::Right::Index,
        nn::hid::DebugPadButton::Down::Index,
    };

    // Button と NPad の対応関係
    const int NpadMap[HidPad::Button_Max] =
    {
        nn::hid::NpadButton::A::Index,
        nn::hid::NpadButton::B::Index,
        nn::hid::NpadButton::X::Index,
        nn::hid::NpadButton::Y::Index,
        nn::hid::NpadButton::R::Index,
        nn::hid::NpadButton::L::Index,
        nn::hid::NpadButton::ZR::Index,
        nn::hid::NpadButton::Plus::Index,
        nn::hid::NpadButton::Left::Index,
        nn::hid::NpadButton::Up::Index,
        nn::hid::NpadButton::Right::Index,
        nn::hid::NpadButton::Down::Index,
    };

    struct PadState
    {
        nn::hid::DebugPadState debugPad;
        nn::hid::NpadFullKeyState npadFullKey;
        nn::hid::NpadHandheldState npadHandHeld;
    };

    bool Test(const PadState& state, const HidPad::Button button)
    {
        NN_ABORT_UNLESS_RANGE( button, 0, HidPad::Button_Max );
        bool isTested = false;
        isTested |= state.debugPad.buttons.Test(DebugPadMap[button]);
        isTested |= state.npadFullKey.buttons.Test(NpadMap[button]);
        isTested |= state.npadHandHeld.buttons.Test(NpadMap[button]);
        return isTested;
    }

    // PadState に対する押し続けているフレームの更新を行います
    void UpdateFrameCount(const HidPad::Button button, const PadState& state, int64_t* pFrameCount) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL( pFrameCount );

        if( Test(state, button) )
        {
            if( *pFrameCount < INT64_MAX )
            {
                (*pFrameCount)++;
            }
        }
        else
        {
            *pFrameCount = 0;
        }
    }
}

//  初期化します
void HidPad::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeDebugPad();

    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap( &map );

    map.buttonA      = nn::hid::KeyboardKey::A::Index;
    map.buttonB      = nn::hid::KeyboardKey::B::Index;
    map.buttonX      = nn::hid::KeyboardKey::X::Index;
    map.buttonY      = nn::hid::KeyboardKey::Y::Index;
    map.buttonL      = nn::hid::KeyboardKey::L::Index;
    map.buttonR      = nn::hid::KeyboardKey::R::Index;
    map.buttonZR     = nn::hid::KeyboardKey::Z::Index;
    map.buttonLeft   = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight  = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp     = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown   = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart  = nn::hid::KeyboardKey::Space::Index;

    nn::settings::SetDebugPadKeyboardMap( map );

    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    for( int i = 0; i < Button_Max; i++ )
    {
        m_HoldingFrameCount[i] = 0;
    }
}
//  更新処理をします
void HidPad::Update() NN_NOEXCEPT
{
    PadState state;
    nn::hid::GetDebugPadStates( &(state.debugPad), 1 );
    // nn::hid 側で NN_DEFINE_STATIC_CONST 対応されると、一時変数を使用する必要がなくなります
    const nn::hid::NpadIdType fullKeyId = nn::hid::NpadId::No1;
    nn::hid::GetNpadStates( &(state.npadFullKey), 1, fullKeyId );
    const nn::hid::NpadIdType handHeldId = nn::hid::NpadId::Handheld;
    nn::hid::GetNpadStates( &(state.npadHandHeld), 1, handHeldId );

    UpdateFrameCount( Button::Button_A    , state, &m_HoldingFrameCount[Button::Button_A] );
    UpdateFrameCount( Button::Button_B    , state, &m_HoldingFrameCount[Button::Button_B] );
    UpdateFrameCount( Button::Button_X    , state, &m_HoldingFrameCount[Button::Button_X] );
    UpdateFrameCount( Button::Button_Y    , state, &m_HoldingFrameCount[Button::Button_Y] );
    UpdateFrameCount( Button::Button_R    , state, &m_HoldingFrameCount[Button::Button_R] );
    UpdateFrameCount( Button::Button_L    , state, &m_HoldingFrameCount[Button::Button_L] );
    UpdateFrameCount( Button::Button_Z    , state, &m_HoldingFrameCount[Button::Button_Z] );
    UpdateFrameCount( Button::Button_Left , state, &m_HoldingFrameCount[Button::Button_Left] );
    UpdateFrameCount( Button::Button_Up   , state, &m_HoldingFrameCount[Button::Button_Up] );
    UpdateFrameCount( Button::Button_Right, state, &m_HoldingFrameCount[Button::Button_Right] );
    UpdateFrameCount( Button::Button_Down , state, &m_HoldingFrameCount[Button::Button_Down] );
    UpdateFrameCount( Button::Button_Start, state, &m_HoldingFrameCount[Button::Button_Start] );
}

//  ボタンを押しているかを返します
bool HidPad::IsHold(Button button) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( button, 0, Button_Max );
    return m_HoldingFrameCount[button] > 0;
}
//  ボタンを長押ししているときに適度に true を返します
bool HidPad::IsContinue(Button button) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( button, 0, Button_Max );
    const int64_t count = m_HoldingFrameCount[button];

    //  長押ししている時間が長くなるほど、
    //  短い周期で true を返すようにします
    const int64_t Term1 = 36;
    const int64_t Term1Circle = 12;
    const int64_t Term2 = 36 + Term1;
    const int64_t Term2Circle = 6;
    const int64_t Term3 = 36 + Term2;
    const int64_t Term3Circle = 3;

    if( count < Term1 )
    {
        return count % Term1Circle == 1;
    }
    else if( count < Term2 )
    {
        return count % Term2Circle == 1;
    }
    else if( count < Term3 )
    {
        return count % Term3Circle == 1;
    }
    else
    {
        return true;
    }
}
//  ボタンを押した瞬間かを返します
bool HidPad::IsTrigger(Button button) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RANGE( button, 0, Button_Max );
    return m_HoldingFrameCount[button] == 1;
}

//  いずれかのボタンを押しているかを返します
bool HidPad::IsHoldAnyKey() const NN_NOEXCEPT
{
    for( int i = 0; i < Button_Max; i++ )
    {
        if( IsHold( static_cast<Button>( i ) ) )
        {
            return true;
        }
    }
    return false;
}
//  いずれかのいずれかのボタンの IsContinue() が true であるかを返します
bool HidPad::IsContinueAnyKey() const NN_NOEXCEPT
{
    for( int i = 0; i < Button_Max; i++ )
    {
        if( IsContinue( static_cast<Button>( i ) ) )
        {
            return true;
        }
    }
    return false;
}
//  いずれかのボタンを押した瞬間かを返します
bool HidPad::IsTriggerAnyKey() const NN_NOEXCEPT
{
    for( int i = 0; i < Button_Max; i++ )
    {
        if( IsTrigger( static_cast<Button>( i ) ) )
        {
            return true;
        }
    }
    return false;
}
