﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/atk.h>
#include "FlightRecorder.h"

namespace
{
    //  AtkPlayer と同じディレクトリに新しいファイルを開きます
    bool OpenNewFile(nn::fs::FileHandle* pFile, const char* name, int mode) NN_NOEXCEPT
    {
        char path[nn::fs::EntryNameLengthMax + 1];  //  + 1 は終端文字

        //  AtkPlayer と同じディレクトリに name ファイルを作成するためにパスを取得します
        {
            const char* atkPlayerPath = nn::os::GetHostArgv()[0];
            strcpy( path, atkPlayerPath );

            char* separator = strrchr( path, '\\' );
            if( separator == nullptr )
            {
                separator = strrchr( path, '/' );
                if( separator == nullptr )
                {
                    NN_LOG( "can't find file path separator [%s]\n", path );
                    return false;
                }
            }

            sprintf( separator + 1, "%s", name );
        }

        //  ファイルが存在すれば消します
        {
            const nn::Result result = nn::fs::DeleteFile( path );
            if( result.IsFailure() && !nn::fs::ResultPathNotFound::Includes( result ) )
            {
                NN_LOG( "can't remove file [%s]\n", path );
                return false;
            }
        }

        //  ファイルを新しく作成します
        {
            const nn::Result result = nn::fs::CreateFile( path, 0 );
            if( result.IsFailure() )
            {
                NN_LOG( "can't create file [%s]\n", path );
                return false;
            }
        }

        //  ファイルを開きます
        {
            const nn::Result result = nn::fs::OpenFile( pFile, path, mode );
            if( result.IsFailure() )
            {
                NN_LOG( "can't open file [%s]\n", path );
                return false;
            }
        }

        return true;
    }
}

//  ログを書き出します
void FlightRecorder::WriteLog(const char* str, ...) NN_NOEXCEPT
{
    if( !m_IsEnabled ) return ;

    char buf[1024];
    va_list args;

    va_start( args, str );
    int len = vsprintf( buf, str, args );
    va_end( args );

    //  改行コードを直に代入します
    buf[len++] = 0x0d;
    buf[len++] = 0x0a;

    nn::fs::WriteFile( m_LogFile, m_LogFileWriteIndex, buf, len, nn::fs::WriteOption::MakeValue( nn::fs::WriteOptionFlag_Flush ) );
    m_LogFileWriteIndex += len;
}

//  初期化します
void FlightRecorder::Initialize(bool isEnabledFlightRecorder, const char* logName) NN_NOEXCEPT
{
    m_IsEnabled = false;
    m_LogFileWriteIndex = 0;

    if( !isEnabledFlightRecorder ){
        return ;
    }

    //  ファイルを開きます
    {
        if( !OpenNewFile( &m_LogFile, logName, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend ) )
        {
            return ;
        }
    }

    m_IsEnabled = true;
}
//  終了処理します
void FlightRecorder::Finalize() NN_NOEXCEPT
{
    if( !m_IsEnabled ) return ;
    nn::fs::CloseFile( m_LogFile );
}
//  インスタンスを取得します
FlightRecorder& FlightRecorder::GetInstance() NN_NOEXCEPT
{
    static FlightRecorder inst;
    return inst;
}
