#/usr/bin/perl -w
#---------------------------------------------------------------------------------*
# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
# 
# The content herein is highly confidential and should be handled accordingly.
#---------------------------------------------------------------------------------*/

use strict;
use Math::Trig; # sinh/cosh/asinh

use constant PI => 4 * atan2(1, 1);
use enum qw(LPF HPF BPF);

my $sampleRate = 48000;
my $Ap = 0.5;

#/**
# * @brief Q14 に変換します。
# */
sub Float2Int16
{
    my ($value) = @_;
    $value = int( $value * 16384 + 0.5);
    $value += 65536 if ($value < 0); # 正の値で返す (perl の16進数表示仕様の都合上)
    return $value;
}

#/**
# * @brief 係数をダンプします。
# */
sub ShowCoefficients
{
    my ($pCoefficientTable, $labelString) = @_;
    print "[$labelString]\n" if (defined $labelString);

    for (my $k = 0; $k < scalar @{$pCoefficientTable}; ++$k)
    {
        my $function = \&Float2Int16;
        my @entry = @{$pCoefficientTable->[$k]};
        my @coefficients;
        my $frequencyString;
        # BPF
        if (scalar @entry > 6)
        {
            $frequencyString = sprintf("//( %f - %f Hz )", $entry[0], $entry[1]);
            @coefficients = @entry[2 ... 6];
        }
        # LPF/HPF
        else
        {
            $frequencyString = sprintf("//( %fHz )", $entry[0]);
            @coefficients = @entry[1 ... 5];
        }
        print sprintf("    { 0x%04x, 0x%04x, 0x%04x, 0x%04x, 0x%04x },   %s\n",
            $function->($coefficients[0]),
            $function->($coefficients[1]),
            $function->($coefficients[2]),
            $function->($coefficients[3]),
            $function->($coefficients[4]),
            $frequencyString
        );
    }
}

#/**
# * @brief 係数を計算します。
# */
sub CalculateCoefficients
{
    my $returnValues = [];
    my @a = (0, 0, 0);
    my @b = (0, 0, 0);
    my ($p) = @_;
=pod
    $p->{Fc};            #//!< カットオフ周波数 (LPF/HPF)
    $p->{Fc1};           #//!< 低い方の周波数 (BPF)
    $p->{Fc2};           #//!< 高い方の周波数 (BPF)
    $p->{Fs};            #//!< サンプルレート
    $p->{FilterType};    #//!< フィルタタイプ (LPF/HPF/BPF)
    $p->{Ap};            #//!< リプル [dB]
    $p->{N};             #//!< フィルタ次数
=cut
    #-----------------------------------------------------------
    # 共通部分
    #-----------------------------------------------------------
    my $epsiron = sqrt(10 ** (0.1 * $p->{Ap}) - 1);
    my $gamma = asinh(1 / $epsiron) / $p->{N};
    #-----------------------------------------------------------

    if (1 == $p->{N})
    {
        if (BPF == $p->{FilterType})
        {
            my $p0 = -sinh($gamma);
            my $wc1 = tan(2 * PI() * $p->{Fc1} / (2 * $p->{Fs}));
            my $wc2 = tan(2 * PI() * $p->{Fc2} / (2 * $p->{Fs}));
            my $W = ($wc2 - $wc1);
            my $M = ($wc2 * $wc1);
            my $denom = $M + 1 - $p0 * $W;
            my $B = -$p0 * $W / $denom;
            $a[0] = 1.0;
            $a[1] = 2 * ($M - 1) / $denom * -1;         # 演算の都合上、符号反転
            $a[2] = ($W * $p0 + $M + 1 ) / $denom * -1; # 演算の都合上、符号反転
            $b[0] = $B;
            $b[1] = 0;
            $b[2] = -$B;
            push (@{$returnValues}, $p->{Fc1});
            push (@{$returnValues}, $p->{Fc2});
        }
    }
    elsif (2 == $p->{N})
    {
        #-----------------------------------------------------------
        # N=2 共通部分
        #-----------------------------------------------------------
        my $wa = tan(2 * PI() * $p->{Fc} / (2 * $p->{Fs}));
        my $k = 1;
        my $sigmai = -sinh($gamma) * sin( (2 * $k - 1) * PI() / (2 * $p->{N}) );
        my $omegai =  cosh($gamma) * cos( (2 * $k - 1) * PI() / (2 * $p->{N}) );
        my $wa2 = $wa ** 2;
        my $alphai = -2 * $sigmai;
        my $betai = $sigmai ** 2 + $omegai ** 2;
        #-----------------------------------------------------------
        if (LPF == $p->{FilterType})
        {
            my $denom = $alphai * $wa + $betai * $wa2 + 1;
            my $H0 = 10 ** (-0.05 * $p->{Ap}) / $denom;
            my $B = $H0 * $betai * $wa2;

            $a[0] = 1.0;
            $a[1] = 2 * ($betai * $wa2 - 1) / $denom * -1;             # 演算の都合上、符号反転
            $a[2] = (1 - $alphai * $wa + $betai * $wa2) / $denom * -1; # 演算の都合上、符号反転
            $b[0] = $B;
            $b[1] = 2 * $B;
            $b[2] = $B;
            push (@{$returnValues}, $p->{Fc});
        }
        elsif (HPF == $p->{FilterType})
        {
            my $denom = $wa2 + $alphai * $wa + $betai;
            my $H0 = 10 ** (-0.05 * $p->{Ap}) / $denom;
            my $B = $H0 * $betai;

            $a[0] = 1.0;
            $a[1] = 2 * ($wa2 - $betai) / $denom * -1;             # 演算の都合上、符号反転
            $a[2] = ($wa2 - $alphai * $wa + $betai) / $denom * -1; # 演算の都合上、符号反転
            $b[0] = $B;
            $b[1] = -2 * $B;
            $b[2] = $B;
            push (@{$returnValues}, $p->{Fc});
        }
    }
    push (@{$returnValues}, @b);
    push (@{$returnValues}, @a[1, 2]);
    return $returnValues;
}

sub main()
{
    #---------------------------------------------------------------
    # LPF/HPF
    #---------------------------------------------------------------
    {
        my @coefficientTableForLpf = ();
        my @coefficientTableForHpf = ();
        # 浮動小数点の乗算誤差を避けるため、
        # オクターブのループとオクターブ内のループを分ける。
        for (my $frequencyBase = 16; $frequencyBase < $sampleRate / 2; $frequencyBase *= 2)
        {
            my $steps = 16;
            my $scale = 2 ** (1 / $steps); # オクターブ16分割
            for (my $k = 0; $k < $steps; ++$k)
            {
                my $frequency = $frequencyBase * ($scale ** $k);
                last if ($frequency >= $sampleRate / 2); # ナイキスト周波数以上は処理しない。

                #-------------------------------
                # LPF
                #-------------------------------
                my %parameters = (
                    Ap => $Ap,
                    Fs => $sampleRate,
                    Fc => $frequency,
                    N => 2,
                    FilterType => LPF
                );
                my $pCoefficients = CalculateCoefficients(\%parameters);
                push (@coefficientTableForLpf, $pCoefficients);

                #-------------------------------
                # HPF
                #-------------------------------
                # フィルタタイプのみ変更
                $parameters{FilterType} = HPF;
                $pCoefficients = CalculateCoefficients(\%parameters);
                push (@coefficientTableForHpf, $pCoefficients);
            }
        }
        
        # 係数テーブルの並びの都合上、逆順にしておく。
        @coefficientTableForLpf = reverse @coefficientTableForLpf;

        ShowCoefficients(\@coefficientTableForLpf, "Lpf (\@$sampleRate Hz)");
        ShowCoefficients(\@coefficientTableForHpf, "Hpf (\@$sampleRate Hz)");
    }

    #---------------------------------------------------------------
    # BPF
    #---------------------------------------------------------------
    # [中心周波数, 中心周波数のずれ, 低い側周波数の増減の加減]
    # * 中心周波数 [Hz]
    # * 中心周波数のずれ
    #   オクターブの 1/16 を単位として、低い側周波数に対するずれ分
    # * 低い側周波数の増減の加減
    #   オクターブ 1/16 分割を基準とし、1 ならそのまま、0.5 なら基準の 1/2 (オクターブ 1/32 分割) の割合で低い側周波数を動かす。
    #   高い側周波数は、オクターブ 1/32 で動かす。
    foreach my $p ([512, 1, 0.5], [1024, 2, 1], [2048, 2, 1])
    {
        my @coefficientTable = ();
        my $frequencyCenter = $p->[0];
        my $offset = $p->[1];
        my $rate = $p->[2];
        my $pTable = \@coefficientTable;
        for (my $octave = 0; $octave < 10 ; ++$octave)
        {
            my $steps = 16;
            my $scale = 2 ** (1 / $steps);
            my $frequencyLowerBase = $frequencyCenter / 2 ** ($octave * $rate) / ($scale ** $offset);
            my $frequencyUpperBase = $frequencyCenter * 2 ** ($octave * 0.5);
            for (my $k = 0; $k < $steps; ++$k)
            {
                my $scaleLower = $scale ** ($k * $rate);
                my $scaleUpper = $scale ** ($k / 2);
                my $frequencyLower = $frequencyLowerBase / $scaleLower;
                my $frequencyUpper = $frequencyUpperBase * $scaleUpper;
                next if ($frequencyUpper >= $sampleRate / 2);
                my %parameters = (
                    Ap => $Ap,
                    Fs => $sampleRate,
                    Fc1 => $frequencyLower,
                    Fc2 => $frequencyUpper,
                    N => 1,
                    FilterType => BPF
                );
                my $pCoefficients = CalculateCoefficients(\%parameters);
                push (@{$pTable}, $pCoefficients);
            }
        }
        @{$pTable} = reverse @{$pTable};
        ShowCoefficients($pTable, "Bpf $frequencyCenter (\@$sampleRate Hz)");
    }
}

$sampleRate = $ARGV[0] if (defined $ARGV[0]);
$Ap = $ARGV[1] if (defined $ARGV[1]);

main;

__END__
