﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/viewer/vfx_Preview.h>
#include <nn/vfx/vfx_Misc.h>

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
Preview::Preview() NN_NOEXCEPT
{
}

//------------------------------------------------------------------------------
//  初期化処理
//------------------------------------------------------------------------------
void Preview::Initialize( nn::vfx::Heap* pHeap, VfxPreviewType type, nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    m_Guid = guid;
    m_PreviewType = type;
    m_pNext = NULL;
    m_pPrev = NULL;
    m_pHeap = pHeap;
    m_Time = 0.0f;
    m_ReleaseCounter = -1;
    nn::util::MatrixIdentity( &m_AutoMoveMatrix );
    nn::util::MatrixIdentity( &m_ResMatrix );
    nn::util::MatrixIdentity( &m_DrawMatrix );
    m_IsSetResPreview = false;
}

//---------------------------------------------------------------------------
//  終了処理です。
//---------------------------------------------------------------------------
void Preview::Finalize() NN_NOEXCEPT
{
}

//------------------------------------------------------------------------------
//  計算処理
//------------------------------------------------------------------------------
bool Preview::Calculate( bool isPause, float frameRate, const nn::util::Matrix4x3fType& centerMatrix, const nn::util::Matrix4x3fType& viewMatrix ) NN_NOEXCEPT
{
    if( !isPause )
    {
        m_Time += frameRate;
    }
    else
    {
        return true;
    }

    if ( m_ReleaseCounter > 0 )
    {
        m_ReleaseCounter--;
        if ( m_ReleaseCounter == 0 )
        {
            return false;
        }

        return true;
    }

    // プレビューリソースからリソースマトリクスを生成する
    if( m_IsSetResPreview )
    {
        // Bone Matrix
        nn::util::Matrix4x3fType boneMatrix;
        nn::util::MatrixIdentity( &boneMatrix );

        Preview* pPreview = GetPreview( m_ResPreview.constrain.modelGuid );
        if( pPreview )
        {
            if( pPreview->GetPreviewType() == VfxPreviewType_ModelPreview ||
                pPreview->GetPreviewType() == VfxPreviewType_UserModel )
            {
                pPreview->GetMatrix( &boneMatrix, m_ResPreview.constrain.boneIndex );

                if ( m_ResPreview.constrain.matrixApplyType == MatrixApplyType_T )
                {
                    nn::util::Vector3fType w;
                    nn::util::MatrixGetAxisW( &w, boneMatrix );
                    nn::util::MatrixIdentity( &boneMatrix );
                    nn::util::MatrixSetAxisW( &boneMatrix, w );
                }
            }
        }

        // SRT制御 Matrix
        const nn::util::Vector3fType rotate =
        {{
            m_ResPreview.constrain.offsetRotateX,
            m_ResPreview.constrain.offsetRotateY,
            m_ResPreview.constrain.offsetRotateZ,
        }};
        const nn::util::Vector3fType offsetScale =
        {{
            m_ResPreview.constrain.offsetScale.x,
            m_ResPreview.constrain.offsetScale.y,
            m_ResPreview.constrain.offsetScale.z,
        }};
        const nn::util::Vector3fType offsetPosition =
        {{
            m_ResPreview.constrain.offsetPosition.x,
            m_ResPreview.constrain.offsetPosition.y,
            m_ResPreview.constrain.offsetPosition.z,
        }};

        nn::vfx::detail::MatrixCreateSrtXyz( &m_ResMatrix, offsetScale, rotate, offsetPosition );

        // 自動移動 mAutoMoveMtx
        CalculateAutoMove();

        // マトリクスを更新
        nn::util::MatrixMultiply( &m_DrawMatrix, boneMatrix, centerMatrix );
        nn::util::MatrixMultiply( &m_DrawMatrix, m_ResMatrix, m_DrawMatrix );
        nn::util::MatrixMultiply( &m_DrawMatrix, m_AutoMoveMatrix, m_DrawMatrix );

        // エミッタビルボード
        if( m_ResPreview.constrain.isEmitterBillboard == 1 )
        {
            nn::util::Matrix4x3fType dst;
            nn::util::MatrixInverse( &dst, viewMatrix );
            nn::util::Vector3fType trans;
            nn::util::MatrixGetAxisW( &trans, m_DrawMatrix );
            nn::util::MatrixSetAxisW( &dst, trans );
            m_DrawMatrix = dst;
        }
        else if ( m_ResPreview.constrain.isEmitterBillboard == 2 )
        {
            nn::util::Vector3fType   pos;
            nn::util::Vector3fType   eye;
            nn::util::Matrix4x3fType dst;
            nn::util::Float4x3       matrix;

            nn::util::MatrixGetAxisW( &pos, m_DrawMatrix );

            nn::util::MatrixStore( &matrix, viewMatrix );
            eye._v[0] = -matrix.m[0][0] * matrix.m[3][0] - matrix.m[0][1] * matrix.m[3][1] - matrix.m[0][2] * matrix.m[3][2];
            eye._v[1] = -matrix.m[1][0] * matrix.m[3][0] - matrix.m[1][1] * matrix.m[3][1] - matrix.m[1][2] * matrix.m[3][2];
            eye._v[2] = -matrix.m[2][0] * matrix.m[3][0] - matrix.m[2][1] * matrix.m[3][1] - matrix.m[2][2] * matrix.m[3][2];

            float billboardY = nn::util::Atan2Est( -( nn::util::VectorGetX( eye ) - nn::util::VectorGetX( pos ) ), ( nn::util::VectorGetZ( eye ) - nn::util::VectorGetZ( pos ) ) );

            nn::util::Vector3fType rot = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, -billboardY, 0.f );

            nn::util::MatrixIdentity( &dst );
            nn::util::MatrixSetRotateXyz( &dst, rot );
            nn::util::MatrixSetAxisW( &dst, pos );
            m_DrawMatrix = dst;
        }
    }
    else
    {
        m_DrawMatrix = m_ResMatrix;
    }

    return true;
}


//---------------------------------------------------------------------------
//  プレビューをリストに追加します。
//---------------------------------------------------------------------------
void Preview::AddPreview( Preview* pPreview ) NN_NOEXCEPT
{
    if( !m_pNext )
    {
        m_pNext = pPreview;
        pPreview->m_pPrev = this;
        pPreview->m_pNext = NULL;
    }
    else
    {
        Preview* previewTail = GetTailPreview();
        previewTail->m_pNext = pPreview;
        pPreview->m_pPrev = previewTail;
        pPreview->m_pNext = NULL;
    }
}

//---------------------------------------------------------------------------
//  このプレビューをリストから削除します。
//---------------------------------------------------------------------------
void Preview::RemovePreview() NN_NOEXCEPT
{
    if( !m_pNext )
    {
        if( m_pPrev )
        {
            m_pPrev->m_pNext = NULL;
        }
    }
    else
    {
        m_pPrev->m_pNext = m_pNext;
        m_pNext->m_pPrev = m_pPrev;
    }

    m_pPrev = NULL;
    m_pNext = NULL;
}

//---------------------------------------------------------------------------
//  guidを指定してプレビューを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetPreview( nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    Preview* pPreview = GetHeadPreview();
    while( pPreview )
    {
        if( pPreview->GetGuid() == guid && pPreview->IsAlive() )
        {
            return pPreview;
        }

        pPreview = pPreview->GetNextPreview();
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  最後尾のPreviewを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetTailPreview() NN_NOEXCEPT
{
    if( !m_pNext )
    {
        return this;
    }
    else
    {
        Preview* pPreview = this->GetNextPreview();
        while( pPreview )
        {
            if( !pPreview->GetNextPreview() )
            {
                break;
            }
            pPreview = pPreview->GetNextPreview();
        }
        return pPreview;
    }
}

//---------------------------------------------------------------------------
//  先頭のPreviewを取得します。
//---------------------------------------------------------------------------
Preview* Preview::GetHeadPreview() NN_NOEXCEPT
{
    Preview* pHead = this;
    while( pHead->m_pPrev )
    {
        pHead = pHead->m_pPrev;
    }
    return pHead;
}

//---------------------------------------------------------------------------
//! @brief        自動移動を計算します。
//---------------------------------------------------------------------------
void Preview::CalculateAutoMove() NN_NOEXCEPT
{
    nn::util::MatrixIdentity( &m_AutoMoveMatrix );

    if( m_ResPreview.autoMove.moveType == 0 )
    {
        return;
    }

    // パラメータを取得
    float autoMoveRadius = ( m_ResPreview.autoMove.moveType == 2 ) ?
                                 m_ResPreview.autoMove.distance :
                                 m_ResPreview.autoMove.radiusXZ;                               //automtic->autoMoveRadius;
    float autoMoveRotSpeed = ( m_ResPreview.autoMove.moveType == 2 ) ?
                                 m_ResPreview.autoMove.moveSpeed :
                                 nn::util::DegreeToRadian( m_ResPreview.autoMove.speedXZ );    //DegToIdx( automtic->autoMoveRotVel );
    float autoYSwing = m_ResPreview.autoMove.amplitudeY;                                       //automtic->autoYSwing;
    float autoYSwingSpeed = nn::util::DegreeToRadian( m_ResPreview.autoMove.speedY );          //DegToIdx( automtic->autoYSwingSpeed );
    float autoZRollSpeed = nn::util::DegreeToRadian( m_ResPreview.autoMove.rollZ );            //DegToIdx( automtic->autoZRollSpeed );

    // 使用するマトリクスを初期化
    nn::util::Matrix4x3fType matrixAutoMove;
    nn::util::Matrix4x3fType matrixRotate;
    nn::util::Matrix4x3fType matrixZRoll;

    nn::util::MatrixIdentity( &matrixAutoMove );
    nn::util::MatrixIdentity( &matrixRotate );
    nn::util::MatrixIdentity( &matrixZRoll );

    float autoMoveRot = autoMoveRotSpeed * m_Time;
    float autoYSwingRot = autoYSwingSpeed * m_Time;
    float autoZRoll = autoZRollSpeed * m_Time;

    nn::util::Vector3fType pos;
    nn::util::Vector3fType prePos;

    //　XZ平面を移動
    if( m_ResPreview.autoMove.moveType == 1 )
    {
        // カレント
        {
            // XZ位置
            float sinV, cosV;
            nn::util::SinCosEst( &sinV, &cosV, autoMoveRot );
            nn::util::VectorSetX( &pos, sinV * autoMoveRadius );
            nn::util::VectorSetZ( &pos, cosV * autoMoveRadius );

            // Y位置
            nn::util::VectorSetY( &pos, nn::util::SinEst( autoYSwingRot ) * autoYSwing );
        }

        // １フレーム前
        {
            float preTime = m_Time - 1.0f;

            autoMoveRot = autoMoveRotSpeed * preTime;
            autoYSwingRot = autoYSwingSpeed  * preTime;
            autoZRoll = autoZRollSpeed * preTime;

            // XZ位置
            float sinV, cosV;
            nn::util::SinCosEst( &sinV, &cosV, autoMoveRot );
            nn::util::VectorSetX( &prePos, sinV * autoMoveRadius );
            nn::util::VectorSetZ( &prePos, cosV * autoMoveRadius );

            // Y位置
            nn::util::VectorSetY( &prePos, nn::util::SinEst( autoYSwingRot ) * autoYSwing );
        }

        // 回転行列作成
        nn::util::Vector3fType basisZ;
        nn::util::VectorSubtract( &basisZ, pos, prePos );

        if( !nn::util::VectorIsZero( basisZ ) )
        {
            nn::util::VectorNormalize( &basisZ, basisZ );

            nn::util::Vector3fType up = {{ 0, 1, 0 }};
            nn::util::Vector3fType basisX;
            nn::util::VectorCross( &basisX, up, basisZ );

            if( !nn::util::VectorIsZero( basisX ) )
            {
                nn::util::VectorNormalize( &basisX, basisX );

                nn::util::Vector3fType basisY;
                nn::util::VectorCross( &basisY, basisZ, basisX );

                nn::util::MatrixSetAxisX( &matrixAutoMove, basisX );
                nn::util::MatrixSetAxisY( &matrixAutoMove, basisY );
                nn::util::MatrixSetAxisZ( &matrixAutoMove, basisZ );

                // ZRoll
                nn::util::Vector3fType rotate = {{ 0, 0, autoZRoll }};
                nn::util::MatrixSetRotateXyz( &matrixZRoll, rotate );

                // 合成
                nn::util::MatrixMultiply( &m_AutoMoveMatrix, matrixAutoMove, matrixZRoll );
                nn::util::MatrixSetTranslate( &m_AutoMoveMatrix, pos );
            }
        }
    } // 指定方向に直進移動
    else if ( m_ResPreview.autoMove.moveType == 2 )
    {
        autoMoveRot = fmod( autoMoveRot, autoMoveRadius);

        nn::util::AngleIndex dirV = nn::util::DegreeToAngleIndex( m_ResPreview.autoMove.straightDirection );
        nn::util::AngleIndex dirH = nn::util::DegreeToAngleIndex( m_ResPreview.autoMove.straightAngle );

        float sinV, cosV, sinH, cosH;
        nn::util::SinCosTable( &sinV, &cosV, dirV );
        nn::util::SinCosTable( &sinH, &cosH, dirH );

        // 揺れY位置
        float swing = nn::util::SinEst( autoYSwingRot ) * autoYSwing;

        nn::util::VectorSetX( &pos, autoMoveRot * sinV * cosH );
        nn::util::VectorSetY( &pos, autoMoveRot * sinH + swing );
        nn::util::VectorSetZ( &pos, autoMoveRot * cosH * cosV );

        // ZRoll
        nn::util::Vector3fType rotate = {{ 0, 0, autoZRoll }};
        nn::util::MatrixSetRotateXyz( &matrixZRoll, rotate );

        // 合成
        nn::util::MatrixMultiply( &m_AutoMoveMatrix, matrixAutoMove, matrixZRoll );
        nn::util::MatrixSetTranslate( &m_AutoMoveMatrix, pos );
    }
}

} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn

