﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/viewer/vfx_EffectPreview.h>
#include <nn/vfx/vfx_System.h>

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

//---------------------------------------------------------------------------
//  EffectPreview クラスのインスタンスを生成します。
//---------------------------------------------------------------------------
EffectPreview* EffectPreview::CreateEffectPreview( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    // プレビュー生成用のメモリを確保します。
    void* buffer = pHeap->Alloc( sizeof( nn::vfx::viewer::detail::EffectPreview ) );
    if( !buffer )
    {
        nn::vfx::detail::OutputWarning( "Memory Allocation Error!! : %d\n", sizeof( nn::vfx::viewer::detail::EffectPreview ) );
        return NULL;
    }

    // クラスの生成を行います。
    nn::vfx::viewer::detail::EffectPreview* pEffectPreview = new ( buffer )nn::vfx::viewer::detail::EffectPreview();
    if( !pEffectPreview )
    {
        return NULL;
    }

    // 初期化処理を行います。
    pEffectPreview->Initialize( pHeap, VfxPreviewType_EffectPreview, guid );
    pEffectPreview->InitializeEffectPreview( pSystem );

    // 破棄コールバックをセット
    pEffectPreview->SetDestroyPreviewCallback( EffectPreview::DestroyEffectPreview );

    return pEffectPreview;
}

//---------------------------------------------------------------------------
//  EffectPreview クラスのインスタンスを破棄します。
//---------------------------------------------------------------------------
void EffectPreview::DestroyEffectPreview( nn::vfx::Heap* pHeap, Preview* pPreview ) NN_NOEXCEPT
{
    EffectPreview* effectPreview = reinterpret_cast< EffectPreview* >( pPreview );

    // 再生中のエミッタセットを破棄
    effectPreview->DestroyEmitterSet();

    // プレビューの終了処理を行います。
    effectPreview->Finalize();

    // メモリを解放
    pHeap->Free( effectPreview );
}

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
EffectPreview::EffectPreview() NN_NOEXCEPT
    : Preview()
{
    m_pSystem = NULL;
    m_ResId = -1;

    m_IsSetResEffectPreview = false;
    m_IsEmitterSetVisible = true;

    memset( &m_ResEffectPreview, 0, sizeof( ResEffectPreview ) );

    for( int i = 0; i < nn::vfx::SystemParameters_MaxEmitterInclusionCount; ++i )
    {
        m_IsEmitterVisible[ i ] = true;
    }

    for( int i = 0; i < PreviewSettings_MaxFadingPreviewCount; ++i )
    {
        m_pFadeEmitterSetList[ i ] = NULL;
    }

    m_pReservationList = nullptr;
    m_IsManualEmitter  = false;
}

//---------------------------------------------------------------------------
//  エフェクトプレビューを初期化します。
//---------------------------------------------------------------------------
void EffectPreview::InitializeEffectPreview( nn::vfx::System* pSystem ) NN_NOEXCEPT
{
    m_pSystem = pSystem;
}

//---------------------------------------------------------------------------
//  計算処理です。
//---------------------------------------------------------------------------
bool EffectPreview::Calculate( bool isPause, float frameRate, const nn::util::Matrix4x3fType& centerMatrix, const nn::util::Matrix4x3fType& viewMatrix ) NN_NOEXCEPT
{
    // 規定クラスの計算処理を行う
    bool ret = Preview::Calculate( isPause, frameRate, centerMatrix, viewMatrix );
    if ( !ret )
    {
        return ret;
    }

    // エフェクトハンドル取得
    if ( !m_CurrentHandle.IsValid() )
    {
        return true;
    }

    // エフェクトプレビューリソースの内容を適用する
    if ( m_IsSetResEffectPreview )
    {
        ApplyEffectPreviewParameter();
    }

    nn::vfx::EmitterSet* pEmitterSet = m_CurrentHandle.GetEmitterSet();

    // 描画マトリクスをセット
    // mDrawMatrixは規定クラスの計算処理で計算される。(マニュアルエミッタ放出時はスキップする)
    if ( ( !m_IsManualEmitter ) || ( !m_ResEffectPreview.basicSetting.isManualEmitter ) )
    {
        pEmitterSet->SetMatrix( GetDrawMatrix() );
    }


    // エミッタカラー
    // 仕様検討中。

    // 再生開始フレームに到達した場合は再生開始する
    if( m_ResEffectPreview.basicSetting.startFrame == GetTime() )
    {
        pEmitterSet->SetCalculationEnable( true );
        pEmitterSet->SetVisible( m_IsEmitterSetVisible );
    }

    // フェード処理が終わったエミッタセットは NULL とする
    for( int i = 0; i < PreviewSettings_MaxFadingPreviewCount; ++i )
    {
        if( m_pFadeEmitterSetList[ i ] )
        {
            if( m_pFadeEmitterSetList[ i ]->IsAlive() )
            {
                // マニュアルエミッタ放出時はスキップする。
                if ( ( !m_IsManualEmitter ) || ( !m_ResEffectPreview.basicSetting.isManualEmitter ) )
                {
                    m_pFadeEmitterSetList[ i ]->SetMatrix( GetDrawMatrix() );
                }
            }
            else
            {
                m_pFadeEmitterSetList[ i ] = NULL;
            }
        }
    }

    // マニュアルエミッタ放出
    if ( ( m_IsManualEmitter ) && (  m_ResEffectPreview.basicSetting.isManualEmitter ) )
    {
        if ( ( m_CurrentHandle.IsValid() &&
             ( m_ResEffectPreview.basicSetting.emitterInverval > 0 ) &&
             ( ( (int)(GetTime() ) - m_ResEffectPreview.basicSetting.startFrame ) % m_ResEffectPreview.basicSetting.emitterInverval ) == 0 ) )
        {
            for ( int i = 0; i < m_ResEffectPreview.basicSetting.maxEmitCountPerFrame; i++ )
            {
                m_CurrentHandle.GetEmitterSet()->EmitParticle( GetDrawMatrix(), nullptr );
            }
        }
    }
    return true;
} //NOLINT(impl/function_size)

//---------------------------------------------------------------------------
//  エミッタセットを生成します。
//---------------------------------------------------------------------------
bool EffectPreview::CreateEmitterSet() NN_NOEXCEPT
{
    if( m_CurrentHandle.IsValid() )
    {
        return false;
    }

    if ( m_ResId  == -1 )
    {
        return false;
    }

    // エミッタセット生成します。
    if ( m_ResEffectPreview.basicSetting.isManualEmitter )
    {
        if ( m_pReservationList )
        {
            GetHeap()->Free( m_pReservationList );
        }

        // 放出可能パーティクル数を算出
        nn::vfx::Resource* resource = m_pSystem->GetResource( m_ResId );
        nn::vfx::EmitterSetResource* emitterSetRes = resource->GetEmitterSetResource( 0 );

        // 長寿命に合わせて最大パーティクル数を決定する
        int life = 0;
        for ( int i = 0; i < emitterSetRes->emitterCount; i++ )
        {
            life = std::max ( life, emitterSetRes->pEmitterResource[ i ].m_pResEmitter->ptcl.life );
        }
        int maxParticleCount = ( m_ResEffectPreview.basicSetting.emitterInverval > 0 ) ? ( ( life / m_ResEffectPreview.basicSetting.emitterInverval ) + 1 ) * m_ResEffectPreview.basicSetting.maxEmitCountPerFrame : 0;

        // パーティクル放出予約リスト用メモリを確保
        m_pReservationList = reinterpret_cast<nn::vfx::EmitReservationInfo *>( GetHeap()->Alloc( m_ResEffectPreview.basicSetting.maxEmitCountPerFrame * sizeof(nn::vfx::EmitReservationInfo ) ) );

        m_pSystem->CreateManualEmitterSetId( &m_CurrentHandle,
                                             0,
                                             m_ResId,
                                             EffectPreviewGroupId,
                                             maxParticleCount,
                                             m_ResEffectPreview.basicSetting.maxEmitCountPerFrame,
                                             m_pReservationList,
                                             nullptr );
        m_IsManualEmitter = true;
    }
    else
    {
        m_pSystem->CreateEmitterSetId( &m_CurrentHandle, 0, m_ResId, EffectPreviewGroupId, GetHeap() );
    }
    if ( !m_CurrentHandle.IsValid() )
    {
        return false;
    }

    nn::vfx::EmitterSet* emitterSet = m_CurrentHandle.GetEmitterSet();

    // 強制開始フレームをセット
    if( m_ResEffectPreview.basicSetting.forceStartFrame != 0 )
    {
        ForceCalculate( m_ResEffectPreview.basicSetting.forceStartFrame );
    }

    // 再生開始フレームが指定されていれば、処理を停止
    if( m_ResEffectPreview.basicSetting.startFrame != 0 )
    {
        emitterSet->SetCalculationEnable( false );
        emitterSet->SetVisible( false );
    }

    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットを破棄します。
//---------------------------------------------------------------------------
bool EffectPreview::DestroyEmitterSet() NN_NOEXCEPT
{
    if( m_CurrentHandle.IsValid() )
    {
        // TODO : なるべく : 現状、即時KILLが無い
        m_pSystem->KillEmitterSet( m_CurrentHandle.GetEmitterSet() );
        m_CurrentHandle.Invalidate();
        if ( m_pReservationList )
        {
            GetHeap()->Free( m_pReservationList );
            m_pReservationList = nullptr;
        }
        if ( m_IsManualEmitter ) m_IsManualEmitter = false;
    }

    for( int i = 0; i < PreviewSettings_MaxFadingPreviewCount; ++i )
    {
        if( m_pFadeEmitterSetList[ i ] && m_pFadeEmitterSetList[ i ]->IsAlive() )
        {
            m_pFadeEmitterSetList[ i ]->Kill();
            m_pFadeEmitterSetList[ i ] = NULL;
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//  指定フレーム再生位置を進めます。
//---------------------------------------------------------------------------
bool EffectPreview::ForceCalculate( int frame ) NN_NOEXCEPT
{
    if( !m_CurrentHandle.IsValid() )
    {
        return false;
    }
    m_CurrentHandle.GetEmitterSet()->ForceCalculate( frame );
    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットをフェードします。
//---------------------------------------------------------------------------
bool EffectPreview::Fade() NN_NOEXCEPT
{
    if( !m_CurrentHandle.IsValid() )
    {
        return false;
    }

    // 「無限寿命」かつ「放出停止」のみ指定されたエミッタを見つけたら個別に消す
    m_CurrentHandle.GetEmitterSet()->KillInfinityEmitter();
    m_CurrentHandle.GetEmitterSet()->Fade();

    // Fade中のエミッタセットを操作する為に保持する。
    for( int i = 0; i < PreviewSettings_MaxFadingPreviewCount; ++i )
    {
        if( !m_pFadeEmitterSetList[ i ] )
        {
            m_pFadeEmitterSetList[ i ] = m_CurrentHandle.GetEmitterSet();
            break;
        }
    }

    m_CurrentHandle.Invalidate();

    return true;
}

//---------------------------------------------------------------------------
//  プレビュー再生をリセットします。
//---------------------------------------------------------------------------
void EffectPreview::ResetPreview( bool doFade ) NN_NOEXCEPT
{
    NN_UNUSED( doFade );
    if( m_CurrentHandle.IsValid() )
    {
        if( doFade )
        {
            Fade();
        }
        else
        {
            DestroyEmitterSet();
        }
        m_CurrentHandle.Invalidate();
    }

    // 表示・非表示を再セット
    SetVisible( m_IsEmitterSetVisible );

    // エミッタセットを再生成
    CreateEmitterSet();

    SetTime( 0.0f );

    ApplyEffectPreviewParameter();
}

//---------------------------------------------------------------------------
//  描画の表示/非表示を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetVisible( bool visibility ) NN_NOEXCEPT
{
    if( !m_CurrentHandle.IsValid() )
    {
        return;
    }

    nn::vfx::EmitterSet* pEmitterSet = m_CurrentHandle.GetEmitterSet();
    if( !pEmitterSet )
    {
        return;
    }

    pEmitterSet->SetVisible( visibility );

    // Fade中のエミッタセットにも適用
    for( int i = 0; i < PreviewSettings_MaxFadingPreviewCount; ++i )
    {
        if( m_pFadeEmitterSetList[ i ] && m_pFadeEmitterSetList[ i ]->IsAlive() )
        {
            m_pFadeEmitterSetList[ i ]->SetStopDraw( !visibility );
        }
    }

    m_IsEmitterSetVisible = visibility;
}


//---------------------------------------------------------------------------
//  エフェクトプレビューリソースを適用する。
//---------------------------------------------------------------------------
void EffectPreview::ApplyEffectPreviewParameter() NN_NOEXCEPT
{
    if ( !m_CurrentHandle.IsValid() ) return;

    nn::vfx::EmitterSet* pEmitterSet = m_CurrentHandle.GetEmitterSet();

    // 描画マトリクスをセット
    // mDrawMatrixは規定クラスの計算処理で計算される。(マニュアルエミッタ放出時はスキップする)
    if ( ( !m_IsManualEmitter ) || ( !m_ResEffectPreview.basicSetting.isManualEmitter ) )
    {
        pEmitterSet->SetMatrix( GetDrawMatrix() );
    }

    // エミッタセットカラー
    nn::util::Vector4fType color;
    nn::util::VectorLoad( &color, m_ResEffectPreview.color.color );
    pEmitterSet->SetColor( color );

    // 放出レートスケール
    pEmitterSet->SetEmissionRatioScale( m_ResEffectPreview.emission.emissionRate );

    // 放出間隔スケール
    pEmitterSet->SetEmissionIntervalScale( m_ResEffectPreview.emission.emissionInterval );

    // ライフスケール
    pEmitterSet->SetParticleLifeScale( m_ResEffectPreview.ptclControl.life );

    // パーティクル放出時スケール
    nn::util::Vector3fType emScale;
    nn::util::VectorLoad( &emScale, m_ResEffectPreview.ptclScale.emissionParticleScale );
    pEmitterSet->SetEmissionParticleScale( emScale );

    // パーティクル放出時スケール
    nn::util::Vector3fType pScale;
    nn::util::VectorLoad( &pScale, m_ResEffectPreview.ptclScale.particleScale );
    pEmitterSet->SetParticleScale( pScale );

    // エミッタ形状
    nn::util::Vector3fType volScale;
    nn::util::VectorLoad( &volScale, m_ResEffectPreview.ptclScale.emitterVolumeScale );
    pEmitterSet->SetEmitterVolumeScale( volScale );

    // 初速
    pEmitterSet->SetAllDirectionalVel( m_ResEffectPreview.ptclControl.allDirectionVel );
    pEmitterSet->SetDirectionalVel( m_ResEffectPreview.ptclControl.directionalVel );
    pEmitterSet->SetRandomVel( m_ResEffectPreview.ptclControl.randomVel );

    nn::util::Vector3fType addVel;
    nn::util::VectorLoad( &addVel, m_ResEffectPreview.ptclControl.addVel );
    pEmitterSet->SetAddVel( addVel );

    // エミッタカラー設定
    if ( m_ResEffectPreview.ptclControl.enableEmitterColor )
    {
        const nn::util::Vector4fType color0 =
        { {
                m_ResEffectPreview.ptclControl.emitterColor0.x,
                m_ResEffectPreview.ptclControl.emitterColor0.y,
                m_ResEffectPreview.ptclControl.emitterColor0.z,
                1.0f,
            } };
        pEmitterSet->SetEmitterColor0( color0 );
        const nn::util::Vector4fType color1 =
        { {
                m_ResEffectPreview.ptclControl.emitterColor1.x,
                m_ResEffectPreview.ptclControl.emitterColor1.y,
                m_ResEffectPreview.ptclControl.emitterColor1.z,
                1.0f,
            } };
        pEmitterSet->SetEmitterColor1( color1 );
    }
    else
    {
        const nn::util::Vector4fType color01 =
        { {
            1.0f,
            1.0f,
            1.0f,
            1.0f,
        } };
        pEmitterSet->SetEmitterColor0( color01 );
        pEmitterSet->SetEmitterColor1( color01 );
    }
}

} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn
