﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_TextureSampler.h>

namespace nn {
namespace vfx {
namespace detail {

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
bool TextureSamplerSlot::Initialize( nn::gfx::Shader* pShader, const char* samplerName ) NN_NOEXCEPT
{
    vertexShaderSamplerLocation = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Sampler, samplerName );
    pixelShaderSamplerLocation  = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, samplerName );
    return true;
}


int TextureSampler::g_SamplerPatternCount       = 0;
TextureSampler* TextureSampler::g_SamplerBuffer = NULL;

//---------------------------------------------------------------------------
// 必要となる全てのサンプラを生成する
//---------------------------------------------------------------------------
void TextureSampler::InitializeSamplerTable( nn::gfx::Device* pGfxDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    //NN_SDK_ASSERT_NOT_NULL( !g_SamplerArray );
    if ( g_SamplerBuffer )
    {
        detail::OutputWarning( "Texture Sampler Table Already Created.\n" );
        return;
    }

    // 全パターン数
    g_SamplerPatternCount = 4 * 4 * 2;

    g_SamplerBuffer = static_cast<TextureSampler*>(
                    pHeap->Alloc( sizeof( TextureSampler ) * g_SamplerPatternCount ) );
    uint8_t* pSamplerWork = reinterpret_cast< uint8_t * >( g_SamplerBuffer );

    int index = 0;

    for ( int k = 0; k < 2; k++ )
    {
        for ( int i = 0; i < 4; i++ )
        {
            for ( int j = 0; j < 4; j++ )
            {
                ResTextureSampler resSampler;
                memset( &resSampler, 0, sizeof(ResTextureSampler) );
                resSampler.wrapU    = static_cast<uint8_t>( i );
                resSampler.wrapV    = static_cast<uint8_t>( j );
                resSampler.filter   = static_cast<uint8_t>( k );

                TextureSampler* pSampler = new ( pSamplerWork ) TextureSampler();
                pSampler->Initialize( pGfxDevice, &resSampler );
                pSamplerWork += sizeof( TextureSampler );
                index++;
            }
        }
    }
}

//---------------------------------------------------------------------------
// 全てのサンプラを破棄する
//---------------------------------------------------------------------------
void TextureSampler::FinalizeSamplerTable( nn::gfx::Device* pGfxDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( g_SamplerBuffer );

    for ( int i = 0; i < g_SamplerPatternCount; i++ )
    {
        g_SamplerBuffer[i].Finalize( pGfxDevice );
    }

    pHeap->Free( g_SamplerBuffer );

    g_SamplerPatternCount   = 0;
    g_SamplerBuffer         = NULL;
}


//---------------------------------------------------------------------------
//! @brief  生成済みサンプラを取得する
//---------------------------------------------------------------------------
TextureSampler* TextureSampler::GetSamplerFromTable( detail::ResTextureSampler* pResTextureSampler ) NN_NOEXCEPT
{
    int index = pResTextureSampler->wrapU * 4 + pResTextureSampler->wrapV + pResTextureSampler->filter * 16;
    NN_SDK_ASSERT( index < g_SamplerPatternCount );
    return &g_SamplerBuffer[index];
}


//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void TextureSampler::Initialize( nn::gfx::Device* device, ResTextureSampler* resSampler ) NN_NOEXCEPT
{
    // TODO:ミップマップLOD指定
    // TODO:ミラーワンス指定

    m_TextureSamplerResource = *resSampler;

    nn::gfx::Sampler::InfoType info;
    info.SetDefault();

    if( m_TextureSamplerResource.filter == TextureFilterMode_Linear )
    {
        info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear );
    }
    else
    {
        info.SetFilterMode( nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint );
    }

    switch( m_TextureSamplerResource.wrapU )
    {
    case TextureWrapMode_Mirror:      info.SetAddressU( nn::gfx::TextureAddressMode_Mirror ); break;
    case TextureWrapMode_Repeat:      info.SetAddressU( nn::gfx::TextureAddressMode_Repeat ); break;
    case TextureWrapMode_Clamp:       info.SetAddressU( nn::gfx::TextureAddressMode_ClampToEdge ); break;
    case TextureWrapMode_MirrorOnce:  info.SetAddressU( nn::gfx::TextureAddressMode_MirrorClampToEdge ); break;
    default:break;
    }

    switch( m_TextureSamplerResource.wrapV )
    {
    case TextureWrapMode_Mirror:      info.SetAddressV( nn::gfx::TextureAddressMode_Mirror ); break;
    case TextureWrapMode_Repeat:      info.SetAddressV( nn::gfx::TextureAddressMode_Repeat ); break;
    case TextureWrapMode_Clamp:       info.SetAddressV( nn::gfx::TextureAddressMode_ClampToEdge ); break;
    case TextureWrapMode_MirrorOnce:  info.SetAddressV( nn::gfx::TextureAddressMode_MirrorClampToEdge ); break;
    default:break;
    }

    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );

    m_Sampler.Initialize( device, info );

    m_IsInitialized  = true;
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void TextureSampler::Finalize( nn::gfx::Device* device ) NN_NOEXCEPT
{
    m_Sampler.Finalize( device );
}

//---------------------------------------------------------------------------
//  テクスチャーサンプラをディスクリプタプールに登録します。
//---------------------------------------------------------------------------
void TextureSampler::RegisterSamplerToDescriptorPool( void* pRegisterCallback, void* pUserData ) NN_NOEXCEPT
{
    nn::vfx::RegisterSamplerSlot pRegisterSamplerSlotCallback = reinterpret_cast<nn::vfx::RegisterSamplerSlot>( pRegisterCallback );

    for ( int i = 0; i < g_SamplerPatternCount; i++ )
    {
        pRegisterSamplerSlotCallback( &g_SamplerBuffer[i].m_DescriptorSlot, g_SamplerBuffer[i].GetTextureSampler(), pUserData );
    }
}

//---------------------------------------------------------------------------
//  テクスチャーサンプラをディスクリプタプールから開放します。
//---------------------------------------------------------------------------
void TextureSampler::UnregisterSamplerFromDescriptorPool( void* pUnregisterCallback, void* pUserData ) NN_NOEXCEPT
{
    nn::vfx::UnregisterSamplerSlot pUnregisterSamplerSlotCallback = reinterpret_cast<nn::vfx::UnregisterSamplerSlot>( pUnregisterCallback );

    for ( int i = 0; i < g_SamplerPatternCount; i++ )
    {
        pUnregisterSamplerSlotCallback( &g_SamplerBuffer[i].m_DescriptorSlot, g_SamplerBuffer[i].GetTextureSampler(), pUserData );

        g_SamplerBuffer[i].m_DescriptorSlot.Invalidate();
    }
}

} // namespace detail
} // namespace vfx
} // namespace nn

