﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_StripeConnection.h>
#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_Misc.h>

namespace nn {
namespace vfx {
namespace detail {

static const CustomShaderConstantBufferIndex ConnectedStripeConstantBufferIndex = CustomShaderConstantBufferIndex_4;
static const EmitterPluginCallbackIndex ConnectedStripeCallbackIndex = EmitterPluginCallbackIndex_1;

//------------------------------------------------------------------------------
//  パーティクルのワールド座標を取得
//------------------------------------------------------------------------------
inline void GetParticleWorldPos( nn::util::Vector3fType* pOutPos, Emitter* pEmitter, int particleIndex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutPos );
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );

    nn::util::Vector3fType localPos;
    const nn::util::Float4* lpos = pEmitter->GetParticleLocalPosition( particleIndex );
    if ( lpos )
    {
        nn::util::VectorLoad( &localPos, lpos->v );
    }

    pEmitter->TransformToWorldPos( pOutPos, localPos, particleIndex );
}

//------------------------------------------------------------------------------
//  パーティクルのスケールを取得
//------------------------------------------------------------------------------
inline void GetParticleScale( nn::util::Vector3fType* pOutScale, Emitter* pEmitter, int particleindex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutScale );
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );
    const detail::ParticleData* pParticleData = pEmitter->GetParticleData( particleindex );
    ParticleCalculateArg arg( pEmitter, pParticleData->GetTime( pEmitter->GetTime() ), pParticleData->GetLife(), particleindex );
    arg.CalculateWorldScale( pOutScale );
}

//------------------------------------------------------------------------------
//  エミッタ行列Y軸を取得
//------------------------------------------------------------------------------
void GetParticleEmitterMatrixY( nn::util::Vector3fType* pOutEmatY, Emitter* pEmitter, ConnectionStripeSystem::ResourceType* pStripeData, int particleIndex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutEmatY );
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );
    NN_SDK_REQUIRES_NOT_NULL( pStripeData );

    if( pStripeData->calcType == StripeOrientationType_Billboard )
    {
        //------------------------------------------------------------------------------
        // ビルボードストライプ（※使わないので何もしない）
        //------------------------------------------------------------------------------
        return;
    }
    else if( pStripeData->calcType == StripeOrientationType_EmitterMatrix || pStripeData->calcType == StripeOrientationType_EmitterUpright )
    {
        //------------------------------------------------------------------------------
        // エミッタ行列ストライプ／エミッタ上下に伸ばす
        //------------------------------------------------------------------------------
        if( pEmitter->GetResEmitter()->emitter.followType == ParticleFollowType_EmitterFull )
        {
            const nn::util::Matrix4x3fType& matrixSrt = pEmitter->GetMatrixSrt();
            nn::util::MatrixGetAxisY( pOutEmatY, matrixSrt );
        }
        else
        {
            nn::util::VectorSetX( pOutEmatY, pEmitter->GetParticleEmitterMatrixSrtRow0( particleIndex )->y );
            nn::util::VectorSetY( pOutEmatY, pEmitter->GetParticleEmitterMatrixSrtRow1( particleIndex )->y );
            nn::util::VectorSetZ( pOutEmatY, pEmitter->GetParticleEmitterMatrixSrtRow2( particleIndex )->y );
        }
    }
}

//---------------------------------------------------------------------------
//  頂点バッファに頂点情報をセットします。
//---------------------------------------------------------------------------
void SetVertexAttribute(
    nn::vfx::detail::ConnectionStripeSystem::VertexAttribute* pOutVertexBuffer,
    int index,
    int maxIndex,
    const nn::util::Vector3fType& pos,
    const nn::util::Vector3fType& dir,
    const nn::util::Vector3fType& emat,
    float scale ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutVertexBuffer );

    const int indexL = index * 2;
    const int indexR = indexL + 1;

    nn::util::VectorStore( pOutVertexBuffer[ indexL ].pos.v, pos );
    nn::util::VectorStore( pOutVertexBuffer[ indexR ].pos.v, pos );

    nn::util::VectorStore( pOutVertexBuffer[ indexL ].dir.v, dir );
    nn::util::VectorStore( pOutVertexBuffer[ indexR ].dir.v, dir );

    nn::util::VectorStore( pOutVertexBuffer[ indexL ].emat.v, emat );
    nn::util::VectorStore( pOutVertexBuffer[ indexR ].emat.v, emat );

    // MEMO: 連結式ストライプのバッファの詰め順は履歴式とは逆なので、スケール方向も逆にしておく。
#ifdef _VFX_NSDK01215_COMPATIBLE
    pOutVertexBuffer[ indexL ].pos.w = scale;
    pOutVertexBuffer[ indexR ].pos.w = -scale;
#else
    pOutVertexBuffer[ indexL ].pos.w = -scale;
    pOutVertexBuffer[ indexR ].pos.w = scale;
#endif
    pOutVertexBuffer[ indexR ].dir.w = pOutVertexBuffer[ indexL ].dir.w = ( static_cast< float >( index ) ) / ( maxIndex - 1 );    // テクスチャ座標
    pOutVertexBuffer[ indexR ].emat.w = pOutVertexBuffer[ indexL ].emat.w = static_cast< float >( index );
}

ConnectionStripeSystem* ConnectionStripeSystem::g_pStripeSystem = NULL;

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
ConnectionStripeSystem::ConnectionStripeSystem( nn::vfx::System* pSystem, BufferingMode bufferMode ) NN_NOEXCEPT
    : m_pSystem( pSystem )
    , m_BufferMode( bufferMode )
    , m_StripeWorkSize( 0 )
{
    m_DefaultColor.x = 1.0f;
    m_DefaultColor.y = 1.0f;
    m_DefaultColor.z = 1.0f;
    m_DefaultColor.w = 1.0f;

    // ストライプコールバックを設定する
    CallbackSet set;
    set.emitterInitialize = ConnectionStripeSystem::InitializeStripeEmitter;
    set.emitterPostCalculate = ConnectionStripeSystem::EmitterPostCalculateCallback;
    set.emitterDraw = ConnectionStripeSystem::EmitterDrawCallback;
    set.emitterFinalize = ConnectionStripeSystem::FinalizeStripeEmitter;
    set.renderStateSet = detail::DummyRenderStateSetCallback;
    m_pSystem->SetEmitterPluginCallbackSet( detail::ConnectedStripeCallbackIndex, set );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
ConnectionStripeSystem::~ConnectionStripeSystem() NN_NOEXCEPT
{
}

//------------------------------------------------------------------------------
//  連結式ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void ConnectionStripeSystem::InitializeSystem( nn::vfx::Heap* pHeap, nn::vfx::System* pSystem, BufferingMode bufferMode ) NN_NOEXCEPT
{
    if( g_pStripeSystem )
    {
        OutputError( "ConnectionStripeSystem is already initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* pBuffer = pHeap->Alloc( sizeof( ConnectionStripeSystem ) );
    if( pBuffer == NULL )
    {
        nn::vfx::detail::OutputWarning( "[ConnectedStripe] Memory Allocate Error!! : %d\n", sizeof( ConnectionStripeSystem ) );
        return;
    }

    // システムのコンストラクタを走らせる
    g_pStripeSystem = new ( pBuffer )ConnectionStripeSystem( pSystem, bufferMode );
}

//------------------------------------------------------------------------------
//  連結式ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void ConnectionStripeSystem::FinalizeSystem( nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    g_pStripeSystem->~ConnectionStripeSystem();
    pHeap->Free( g_pStripeSystem );
    g_pStripeSystem = NULL;
}

//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool ConnectionStripeSystem::InitializeStripeEmitter( EmitterInitializeArg& arg ) NN_NOEXCEPT
{
    //------------------------------------------------------------------------------
    // シェーダアトリビュートの方は ShaderManager 側で初期化済みという前提。
    // （代わりにアトリビュート名が決め打ちかつ VEC4 縛りになっている）
    //------------------------------------------------------------------------------
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return false;
    }

    // エミッタ単位での記憶領域を確保
    ResStripeConnection* pStripeData = GetStripeData( arg.pEmitter );
    EmitterPluginUserData* pEPUserData = reinterpret_cast< EmitterPluginUserData* >( arg.pEmitter->GetDynamicHeap().Alloc( sizeof( EmitterPluginUserData ) ) );
    if( pEPUserData == NULL )
    {
        return false;
    }
    MemUtil::FillZero( pEPUserData, sizeof( EmitterPluginUserData ) ); // ゼロ埋めで初期化
    pEPUserData->bufferSide = BufferSide_FrontBuffer;
    arg.pEmitter->SetEmitterPluginUserData( pEPUserData );

    //------------------------------------------------------------------------------
    // エミッタ単位で記憶する情報を保存
    //------------------------------------------------------------------------------
    {
        pEPUserData->staticParam0.x = pStripeData->headAlpha;
        pEPUserData->staticParam0.y = pStripeData->tailAlpha;

        // エミッタ単位の乱数決定
        pEPUserData->random.x = arg.pEmitter->GetRandomGenerator()->GetFloat();
        pEPUserData->random.y = arg.pEmitter->GetRandomGenerator()->GetFloat();
        pEPUserData->random.z = arg.pEmitter->GetRandomGenerator()->GetFloat();
        pEPUserData->random.w = arg.pEmitter->GetRandomGenerator()->GetFloat();
    }

    // 定数バッファ を初期化
    arg.pEmitter->InitializeCustomConstantBuffer( detail::ConnectedStripeConstantBufferIndex, sizeof( ConstantBufferObject ) );

    // ポリゴン用バッファを確保
    g_pStripeSystem->AllocStripeSystemVertexBuffer( arg.pEmitter );
    return true;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void ConnectionStripeSystem::FinalizeStripeEmitter( EmitterFinalizeArg& arg ) NN_NOEXCEPT
{
    if( arg.pEmitter->GetCalculationType() != EmitterCalculationMode_Cpu )
    {
        return;
    }

    EmitterPluginUserData* pEPUserData = ConnectionStripeSystem::GetEmitterUserData( arg.pEmitter );
    if ( pEPUserData )
    {
        pEPUserData->vertexBufferInitialized = false;

        // エミッタユーザーデータの領域を解放
        detail::FreeFromDynamicHeapSafely( reinterpret_cast< void** >( &pEPUserData ), arg.pEmitter );
        arg.pEmitter->SetEmitterPluginUserData( NULL );
    }
}

//---------------------------------------------------------------------------
//  ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
bool ConnectionStripeSystem::AllocStripeSystemVertexBuffer( Emitter* pEmitter ) NN_NOEXCEPT
{
    //---------------------------------------------------------------------------
    // ここでエミッタ全体のストライプ用頂点バッファを初期化。
    //---------------------------------------------------------------------------
    const ResStripeConnection* pStripeData = GetStripeData( pEmitter );
    EmitterPluginUserData* pEPUserData = ConnectionStripeSystem::GetEmitterUserData( pEmitter );

    // 頂点バッファの個数を求めるのに必要な計算
    // パーティクルの最大数：「エミッタに接続」「先端につなぐ」で一つ点が増えるので、その分1つ追加。
    const int maxParticleCount = pEmitter->GetParticleAttrFillMax() + 1;
    const int divCount = static_cast< int >( pStripeData->numDivide );
    const int vertexBufferCount = ( maxParticleCount + ( maxParticleCount - 1 ) * divCount ) * 2;

    // 頂点バッファサイズ（シングルバッファの分）
    const int bufferSize = vertexBufferCount * sizeof( VertexAttribute );

    //---------------------------------------------------------------------------
    // Attribute の初期化
    //---------------------------------------------------------------------------
    if( bufferSize != 0 )
    {
        if ( pEmitter->InitializeCustomAttribute( m_BufferMode, bufferSize ) )
        {
            pEPUserData->vertexBufferInitialized = true;
        }
    }

    return pEPUserData->vertexBufferInitialized;
}

//---------------------------------------------------------------------------
//  頂点バッファに頂点情報をセット＋カウンタの更新のユーティリティ
//---------------------------------------------------------------------------
void ConnectionStripeSystem::AddVertexDataToBuffer(
    VertexAttribute* pOutVertexBuffer,
    int* pOutAttrIndex,
    int* pOutRenderVertexCount,
    int vertexCount,
    const nn::util::Vector3fType& vertexWorldPos,
    const nn::util::Vector3fType& vertexWorldDir,
    const nn::util::Vector3fType& emitterMatY,
    float scaleValue ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutVertexBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pOutAttrIndex );
    NN_SDK_REQUIRES_NOT_NULL( pOutRenderVertexCount );

    SetVertexAttribute( pOutVertexBuffer, *pOutAttrIndex, vertexCount, vertexWorldPos, vertexWorldDir, emitterMatY, scaleValue );
    ( *pOutRenderVertexCount ) += 2;
    ( *pOutAttrIndex )++;
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::UpdateStripePolygon( Emitter* pEmitter ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pEmitter );

    // MEMO: 場合によっては、この関数自体を分割数ありなし版で分岐させても良いかも？
    EmitterPluginUserData* pEPUserData = ConnectionStripeSystem::GetEmitterUserData( pEmitter );
    ResStripeConnection* pStripeData = ConnectionStripeSystem::GetStripeData( pEmitter );

    // 描画頂点数をリセット
    pEPUserData->renderVertexCount = 0;
    if( pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.IsValidate() == false )
    {
        // アトリビュートの初期化に失敗しているような場合は何もできない
        return;
    }

    // パーティクルのソート
    // ソートした配列の先頭＆配列サイズ
    int particleNum = 0;
    detail::SortData* pSortIndexArrayHead = NULL;
    {
        // 現状必ず存在する処理インデックス 0 のソートバッファを利用する
        const int processingIndex = 0;
        m_pSystem->GetSortedParticleList( &pSortIndexArrayHead, &particleNum, pEmitter, detail::ParticleSortType_DescendingOrder, pEmitter->GetTime(), processingIndex );
        if( particleNum == 0 )
        {
            // 生きているパーティクルが無いので、連結式ストライプの描画はできない
            return;
        }
    }

    VertexAttribute* pVertexBuffer = reinterpret_cast< VertexAttribute* >( pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Map( pEPUserData->bufferSide ) );

    // ポリゴン生成（共通）
    int attrIndex = 0;

    // 描画する背骨の数
    int baseVertexCount = particleNum;
    const TailConnectionType tailConnectionType = static_cast< TailConnectionType >( pStripeData->connectionType );
    if( tailConnectionType == TailConnectionType_ConnectToEmitter || tailConnectionType == TailConnectionType_Loop )
    {
        baseVertexCount += 1;
    }

    // 分割数を込にした、描画するすべての頂点数
    const int numDivide = pStripeData->numDivide;
    const float invDelta = 1.0f / ( numDivide + 1 );
    const int wholeVertexCount = baseVertexCount + ( baseVertexCount - 1 ) * numDivide;

    // スケールフェードイン／アウト
    const float scaleFadeValue = pEmitter->CalculateCurrentScaleFadeValue();
    const int loopCount = particleNum - 1;

    //---------------------------------------------------------------------------
    // 二回以上ループしないとポリゴンが成立しないので処理しない
    // 「先端に繋ぐ」の場合は、粒が1つの時は異常系になるので個別に弾く
    //---------------------------------------------------------------------------
    if( ( numDivide == 0 && baseVertexCount >= 2 && !( tailConnectionType == TailConnectionType_Loop && particleNum <= 1 ) ) ||
        ( numDivide > 0 && particleNum >= 3 ) )
    {
        //---------------------------------------------------------------------------
        // 先端処理：結合タイプによる特殊処理
        //---------------------------------------------------------------------------
        {
            nn::util::Vector3fType nowPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nowDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nextPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nextDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType ematY = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            float scaleValue = 0.0f;

            if( tailConnectionType == TailConnectionType_Loop )
            {
                //---------------------------------------------------------------------------
                // 「先端に繋ぐ」
                //---------------------------------------------------------------------------
                const int idxNow = pSortIndexArrayHead[ loopCount ].index;
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                const int idxNext2 = pSortIndexArrayHead[ 1 ].index;

                detail::GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                detail::GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
                GetParticleScale( &scale, pEmitter, idxNow );
                scaleValue = nn::util::VectorGetX( scale ) * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // 分割あり
                    //---------------------------------------------------------------------------
                    nn::util::Vector3fType next2Pos;
                    detail::GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    nn::util::VectorSubtract( &nowDir, nextPos, nowPos );   // MEMO: これだとほんのちょっと歪む
                    GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    // 方向は前のパーティクルと比較した方がまだ自然。
                    nn::util::Vector3fType next2Pos;
                    detail::GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    nn::util::VectorSubtract( &nowDir, nextPos, nowPos );
                    nn::util::VectorSubtract( &nextDir, next2Pos, nextPos );
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
            else if( tailConnectionType == TailConnectionType_ConnectToEmitter )
            {
                //---------------------------------------------------------------------------
                // 「エミッタに繋ぐ」：エミッタ位置にも頂点を追加。
                //---------------------------------------------------------------------------
                // TODO: 分割無し時、エミッタにつなぐ部分が捻じれている
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                const int idxNext2 = pSortIndexArrayHead[ 1 ].index;

                detail::GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNext );
                GetParticleScale( &scale, pEmitter, idxNext );
                scaleValue = nn::util::VectorGetX( scale ) * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // 分割あり
                    //---------------------------------------------------------------------------
                    const nn::util::Matrix4x3fType& matrixSrt = pEmitter->GetMatrixSrt();
                    nn::util::MatrixGetAxisW( &nowPos, matrixSrt );

                    nn::util::Vector3fType next2Pos;
                    detail::GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                    GetQuarterVectorFrom3Pos( &nowDir, nowPos, nextPos, next2Pos, 0.25f );
                    GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    // 完全にが一周した
                    //「エミッタにつなぐ」
                    const nn::util::Matrix4x3fType& matrixSrt = pEmitter->GetMatrixSrt();
                    nn::util::MatrixGetAxisW( &nowPos, matrixSrt );

                    // 一つ前の位置
                    nn::util::VectorSubtract( &nowDir, nextPos, nowPos );  // 頂点方向（ワールド系）
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }

            // 分割数がある場合、間の分割数分の頂点を埋める。
            if( numDivide > 0 &&
                ( tailConnectionType == TailConnectionType_Loop || tailConnectionType == TailConnectionType_ConnectToEmitter ))
            {
                for( int i = 1; i <= numDivide; ++i )
                {
                    const float innerPos = invDelta * i;
                    nn::util::Vector3fType curvePos;
                    nn::util::Vector3fType curveDir;

                    GetInterpolatedDirVector( &curveDir, nowDir, nextDir, innerPos );
                    detail::HermiteInterpolationOnCubic( &curvePos, nowPos, nowDir, nextPos, nextDir, innerPos );

                    AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, curvePos, curveDir, ematY, scaleValue );
                }
            }
        }

        //---------------------------------------------------------------------------
        // 共通処理：ソートされた順にパーティクルをポリゴンで繋ぐ
        //---------------------------------------------------------------------------
        {
            nn::util::Vector3fType nowPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nowDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nextPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nextDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType ematY = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );

            // 最後の頂点のひとつ前までループして処理する
            for( int idx = 0; idx < loopCount; ++idx )
            {
                const int idxNow = pSortIndexArrayHead[ idx ].index;
                const int idxNext = pSortIndexArrayHead[ idx + 1 ].index;
                detail::GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                detail::GetParticleWorldPos( &nextPos, pEmitter, idxNext );

                GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
                GetParticleScale( &scale, pEmitter, idxNow );
                const float scaleValue = nn::util::VectorGetX( scale ) * scaleFadeValue;

                if( numDivide > 0 )
                {
                    //---------------------------------------------------------------------------
                    // エルミート補間を行うための処理分岐。
                    // ここに来た時点で、粒が3つ以上あるのは確定。
                    // この部分の処理は、粒が3つでも4つ以上でも正しく動作する想定。
                    //---------------------------------------------------------------------------
                    if( idx == 0 )
                    {
                        // 先頭部分（3点で近似補間）
                        const int idxNext2 = pSortIndexArrayHead[ idx + 2 ].index;
                        nn::util::Vector3fType next2Pos;
                        detail::GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                        nn::util::VectorSubtract( &nowDir, nextPos, nowPos );
                        GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                    }
                    else if( idx == loopCount - 1 )
                    {
                        // ストライプの末端のひとつ前（3点で近似補間）
                        const int idxPrev = pSortIndexArrayHead[ idx - 1 ].index;
                        nn::util::Vector3fType prevPos;
                        detail::GetParticleWorldPos( &prevPos, pEmitter, idxPrev );

                        GetSubAndHalfVector( &nowDir, nextPos, prevPos );
                        GetQuarterVectorFrom3Pos( &nextDir, nextPos, nowPos, prevPos, -0.25f );
                    }
                    else
                    {
                        // 中間部（4点で補間）
                        // 前に1点、次に2点存在するのが保証されている。
                        const int idxPrev = pSortIndexArrayHead[ idx - 1 ].index;
                        const int idxNext2 = pSortIndexArrayHead[ idx + 2 ].index;
                        nn::util::Vector3fType prevPos;
                        nn::util::Vector3fType next2Pos;
                        detail::GetParticleWorldPos( &prevPos, pEmitter, idxPrev );
                        detail::GetParticleWorldPos( &next2Pos, pEmitter, idxNext2 );

                        GetSubAndHalfVector( &nowDir, nextPos, prevPos );
                        GetSubAndHalfVector( &nextDir, next2Pos, nowPos );
                    }
                }
                else
                {
                    //---------------------------------------------------------------------------
                    // 分割無し
                    //---------------------------------------------------------------------------
                    detail::GetParticleWorldPos( &nowPos, pEmitter, idxNow );
                    detail::GetParticleWorldPos( &nextPos, pEmitter, idxNext );
                    nn::util::VectorSubtract( &nowDir, nextPos, nowPos );
                }

                // まず基準となる頂点について埋めておく。
                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );

                // 分割数がある場合、間の分割数分の頂点を埋める。
                if( numDivide > 0 )
                {
                    for( int i = 1; i <= numDivide; ++i )
                    {
                        const float innerPos = invDelta * i;
                        nn::util::Vector3fType curvePos;
                        nn::util::Vector3fType curveDir;

                        GetInterpolatedDirVector( &curveDir, nowDir, nextDir, innerPos );
                        detail::HermiteInterpolationOnCubic( &curvePos, nowPos, nowDir, nextPos, nextDir, innerPos );

                        AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, curvePos, curveDir, ematY, scaleValue );
                    }
                }
            }
        }

        //---------------------------------------------------------------------------
        // 端処理（最後尾）
        //---------------------------------------------------------------------------
        if( particleNum >= 2 )
        {
            nn::util::Vector3fType nowPos = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType nowDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType ematY = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
            nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );

            const int idxNow = pSortIndexArrayHead[ loopCount ].index;
            GetParticleEmitterMatrixY( &ematY, pEmitter, pStripeData, idxNow );
            GetParticleScale( &scale, pEmitter, idxNow );
            const float scaleValue = nn::util::VectorGetX( scale ) * scaleFadeValue;

            detail::GetParticleWorldPos( &nowPos, pEmitter, idxNow );

            if( tailConnectionType == TailConnectionType_NoConnection ||
                tailConnectionType == TailConnectionType_ConnectToEmitter )
            {
                //---------------------------------------------------------------------------
                // 「通常」＆「エミッタに繋ぐ」
                //---------------------------------------------------------------------------
                nn::util::Vector3fType prevPos;
                const int idxPrev = pSortIndexArrayHead[ loopCount - 1 ].index;
                detail::GetParticleWorldPos( &prevPos, pEmitter, idxPrev );
                nn::util::VectorSubtract( &nowDir, nowPos, prevPos );

                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
            else if( tailConnectionType == TailConnectionType_Loop )
            {
                //---------------------------------------------------------------------------
                // 「先端につなぐ」
                //---------------------------------------------------------------------------
                const int idxNext = pSortIndexArrayHead[ 0 ].index;
                nn::util::Vector3fType nextPos;
                detail::GetParticleWorldPos( &nextPos, pEmitter, idxNext );
                nn::util::VectorSubtract( &nowDir, nextPos, nowPos );

                AddVertexDataToBuffer( pVertexBuffer, &attrIndex, &pEPUserData->renderVertexCount, wholeVertexCount, nowPos, nowDir, ematY, scaleValue );
            }
        }
    }

    pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.Unmap();
} // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  ストライプ描画処理を行います。
//---------------------------------------------------------------------------
void ConnectionStripeSystem::Draw( nn::gfx::CommandBuffer* pCommandBuffer, Emitter* pEmitter, ShaderType shaderType, void* pUserParam, DrawParameterArg* drawParameterArg ) NN_NOEXCEPT
{
    NN_UNUSED( pUserParam );

    const EmitterPluginUserData* pEPUserData = ConnectionStripeSystem::GetEmitterUserData( pEmitter );
    if( pEPUserData->renderVertexCount == 0 )
    {
        // 描画するポリゴンバッファが無いので処理を戻す
        return;
    }

    const ResStripeConnection* pStripeData = GetStripeData( pEmitter );
    System* const              pSystem = g_pStripeSystem->m_pSystem;
    nn::gfx::Device*           pDevice = pSystem->GetGfxDevice();
    TemporaryBuffer*           pTempBuffer = drawParameterArg->m_pDrawTempBuffer;
    Shader* const              pShader = pEmitter->GetShader( shaderType );

    //---------------------------------------------------------------------------
    // シェーダのストライプ共通設定のセットアップ
    //---------------------------------------------------------------------------
    if( !StripeSystemUtility::SetupDefaultShaderSetting( pCommandBuffer, pEmitter, shaderType, pShader, pUserParam, drawParameterArg ) )
    {
        return;
    }

    //---------------------------------------------------------------------------
    // ストライプが使用する頂点バッファを設定
    //---------------------------------------------------------------------------
    int bufferIndex = pEmitter->GetEmitterResource()->m_ParticlePropertyBufferSlot[ detail::ParticlePropertyAttributeBufferIndex_EmitterPlugin ];

    pCommandBuffer->SetVertexBuffer( bufferIndex,
        pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.GetGpuAddress( pEPUserData->bufferSide ),
        sizeof( VertexAttribute ),
        pEmitter->GetGfxObjects()->m_ParticleEmitterPluginAttribute.GetBlockSize() );

    //---------------------------------------------------------------------------
    // 連結式ストライプを描画
    //---------------------------------------------------------------------------
    {
        ConstantBufferObject constantBuffer;
        MakeConstantBufferObject(&constantBuffer, pEPUserData, pEmitter, StripeMeshType_StandardStripe);
        bool result = StripeSystemUtility::SetConstantBufferObject< ConstantBufferObject >(pCommandBuffer, pDevice, pTempBuffer, pShader, &constantBuffer);
        if( result )
        {
            pCommandBuffer->Draw( nn::gfx::PrimitiveTopology_TriangleStrip, pEPUserData->renderVertexCount, 0 );
        }
    }

    if( pStripeData->option == StripeMeshType_CrossStripe )
    {
        ConstantBufferObject constantBuffer;
        MakeConstantBufferObject( &constantBuffer, pEPUserData, pEmitter, StripeMeshType_CrossStripe );
        bool result = StripeSystemUtility::SetConstantBufferObject< ConstantBufferObject >( pCommandBuffer, pDevice, pTempBuffer, pShader, &constantBuffer );
        if( result )
        {
            pCommandBuffer->Draw( nn::gfx::PrimitiveTopology_TriangleStrip, pEPUserData->renderVertexCount, 0 );
        }
    }
}

} // namespace detail
} // namespace vfx
} // namespace nn
