﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_CurlNoiseData.h>
#include <nn/vfx/vfx_FieldRandom.h>
#include <nn/vfx/vfx_MemUtil.h>
#include <nn/vfx/vfx_System.h>

namespace nn {
namespace vfx {
namespace detail {

// Srtマトリクスから Rtマトリクスを生成する
void MakrRtMatrix( nn::util::Matrix4x3fType* matrixRt, const nn::util::Matrix4x3fType& matrixSrt ) NN_NOEXCEPT
{
    // matrixRT
    nn::util::Vector3fType basisX;
    nn::util::Vector3fType basisY;
    nn::util::Vector3fType basisZ;
    nn::util::MatrixGetAxisX( &basisX, matrixSrt );
    nn::util::MatrixGetAxisY( &basisY, matrixSrt );
    nn::util::MatrixGetAxisZ( &basisZ, matrixSrt );

    float scaleX = nn::util::VectorLength( basisX );
    float scaleY = nn::util::VectorLength( basisY );
    float scaleZ = nn::util::VectorLength( basisZ );

    if ( scaleX > 0.0f )
    {
        const float inv = 1.0f / scaleX;
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisX( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisX( matrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisX( matrixRt, zero );
    }

    if ( scaleY > 0.0f )
    {
        const float inv = 1.0f / scaleY;
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisY( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisY( matrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisY( matrixRt, zero );
    }

    if ( scaleZ > 0.0f )
    {
        float inv = 1.0f / scaleZ;
        nn::util::Vector3fType v;
        nn::util::MatrixGetAxisZ( &v, matrixSrt );
        nn::util::VectorMultiply( &v, v, inv );
        nn::util::MatrixSetAxisZ( matrixRt, v );
    }
    else
    {
        const nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
        nn::util::MatrixSetAxisZ( matrixRt, zero );
    }

    nn::util::Vector3fType trans;
    nn::util::MatrixGetAxisW( &trans, matrixSrt );
    nn::util::MatrixSetAxisW( matrixRt, trans );
}

//------------------------------------------------------------------------------
//  8キーアニメーションの計算（フィールド）
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateField8KeyAnim(
    nn::util::Float3* pOutValue,
    const Emitter* pEmitter,
    const nn::vfx::detail::ResAnim8KeyParamSet& animData,
    int particleIndex,
    float time ) NN_NOEXCEPT
{
    // TODO : なるべく : CPUでも最適な状態で動作するように

    nn::util::Vector3fType start = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    nn::util::Vector3fType end   = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    int  term = 0;
    float dis = 0;
    const int lastKeyIndex = animData.keyNum - 1;

    // キーなし
    if( animData.keyNum == 0 )
    {
        pOutValue->x = pOutValue->y = pOutValue->z = 0;
        return;
    }
    // キーが一つのみ
    if( animData.keyNum == 1 )
    {
        detail::Float3Copy( pOutValue, animData.keyValue[ 0 ] );
        return;
    }

    // ループを加味した、再生タイミング[0, 1]に計算し直す
    float localTime = 0;
    if( animData.loop > 0 )
    {
        localTime = time + animData.startRandom * pEmitter->m_CpuParticleProperty.random[particleIndex].x * animData.loopNum;
        localTime = ( static_cast< float >( ::std::fmodf( localTime, static_cast< float >( animData.loopNum ) ) ) ) / static_cast< float >( animData.loopNum );
    }
    else
    {
        localTime = time / static_cast< float >( pEmitter->m_pEmitterData->ptcl.life );
    }
    //float rate = ( anim.loop ) ? ( 100.0f / anim.loopNum ) : 1.0f;
    //float localTime = (time / life) * rate + static_cast<float>( anim.startRandom ) * emitter->particleAttr->random.x;
    //終端処理（ループ終端を0.0には飛ばさない）
    //localTime = ( localTime == rate ) ? 1.0f : fmod( localTime, 1.0f );

    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if( animData.keyValue[ lastKeyIndex ].w <= localTime )
    {
        detail::Float3Copy( pOutValue, animData.keyValue[ lastKeyIndex ] );
        return;
    }
    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if( localTime < animData.keyValue[ 0 ].w )
    {
        detail::Float3Copy( pOutValue, animData.keyValue[ 0 ] );
        return;
    }

    // TODO：平たく書いた方がよいかも。
    for( term = 0; term < animData.keyNum; term++ )
    {
        int index1 = term;
        int index2 = term + 1;
        float t1 = animData.keyValue[ index1 ].w;
        float t2 = animData.keyValue[ index2 ].w;

        if( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            nn::util::VectorLoad( &start, animData.keyValue[ index1 ].v );
            nn::util::VectorLoad( &end, animData.keyValue[ index2 ].v );
            dis = t2 - t1;

            //---------------------------------------------------------------------------
            // ( start + ( end - start ) / dis * ( localTime - t1 ) );
            //---------------------------------------------------------------------------
            nn::util::Vector3fType ret;
            nn::util::VectorSubtract( &ret, end, start );
            nn::util::VectorMultiply( &ret, ret, ( localTime - t1 ) / dis );
            nn::util::VectorAdd( &ret, start, ret );

            nn::util::VectorStore( pOutValue, ret );
            return;
        }
    }

    pOutValue->x = 0.0f;
    pOutValue->y = 0.0f;
    pOutValue->z = 0.0f;
    return;
}

//------------------------------------------------------------------------------
//  8キーアニメーションの計算
//------------------------------------------------------------------------------
void EmitterCalculator::Calculate8KeyAnim(
    nn::util::Float3*                           pOutValue,
    const nn::vfx::detail::ResAnim8KeyParam&    animData,
    int                                         keyCount,
    float                                       random,
    float                                       time,
    float                                       loopRate,
    float                                       isLoopRandom,
    float                                       life ) NN_NOEXCEPT
{
    // TODO : なるべく : CPUでも最適な状態で動作するように

    nn::util::Vector3fType start = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    nn::util::Vector3fType end   = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    int  term = 0;
    float dis = 0;
    const int lastKeyIndex = keyCount - 1;

    // キーなし
    if( keyCount == 0 )
    {
        pOutValue->x = 1.f;
        pOutValue->y = 1.f;
        pOutValue->z = 1.f;
        return;
    }
    // キーが一つのみ
    if( keyCount == 1 )
    {
        detail::Float3Copy( pOutValue, animData.value[ 0 ] );
        return;
    }

    // ループを加味した、再生タイミング[0, 1]に計算し直す
    float localTime = 0;
    if( loopRate > 0 )
    {
        localTime = time + isLoopRandom * random * loopRate;
        localTime = ( ::std::fmodf( localTime, loopRate ) ) / static_cast< float >( loopRate );
    }
    else
    {
        localTime = time / static_cast< float >( life );
    }

    //float rate = ( 100.0f / loopRate );
    //float localTime = (time / life) * rate + isLoopRandom * emitter->particleAttr->random.x;
    ////終端処理（ループ終端を0.0には飛ばさない）
    //localTime = ( localTime == rate ) ? 1.0f : fmod( localTime, 1.0f );

    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if( animData.value[ lastKeyIndex ].w <= localTime )
    {
        detail::Float3Copy( pOutValue, animData.value[ lastKeyIndex ] );
        return;
    }
    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if( localTime < animData.value[ 0 ].w )
    {
        detail::Float3Copy( pOutValue, animData.value[ 0 ] );
        return;
    }

    for( term = 0; term < keyCount; term++ )
    {
        int index1 = term;
        int index2 = term + 1;
        float t1 = animData.value[ index1 ].w;
        float t2 = animData.value[ index2 ].w;

        if( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            nn::util::VectorLoad( &start, animData.value[ index1 ].v );
            nn::util::VectorLoad( &end, animData.value[ index2 ].v );

            dis = t2 - t1;
            //---------------------------------------------------------------------------
            // ( start + ( end - start ) / dis * ( localTime - t1 ) );
            //---------------------------------------------------------------------------
            nn::util::Vector3fType ret;
            nn::util::VectorSubtract( &ret, end, start );
            nn::util::VectorMultiply( &ret, ret, ( localTime - t1 ) / dis );
            nn::util::VectorAdd( &ret, start, ret );

            nn::util::VectorStore( pOutValue, ret );
            return;
        }
    }

    pOutValue->x = 0.0f;
    pOutValue->y = 0.0f;
    pOutValue->z = 0.0f;
    return;
}

//------------------------------------------------------------------------------
//  フィールド：カスタムフィールド
//------------------------------------------------------------------------------
void CalculateParticleBehaviorCustomField( nn::util::Vector3fType* pOutPos, nn::util::Vector3fType* pOutVec, float* pOutLife, float* pOutBirthTime, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex ) NN_NOEXCEPT
{
    CustomFieldCallback Field = pEmitter->GetEmitterSet()->GetSystem()->GetCustomFieldCallback();
    if( Field )
    {
        const ResFieldCustom* pFieldCustomData = pEmitter->GetEmitterResource()->m_pFieldCustomData;
        Field( pOutPos, pOutVec, pOutLife, pOutBirthTime, pEmitter, pParticleProperty, pFieldCustomData, particleIndex );
    }
}

//------------------------------------------------------------------------------
//  フィールド：GpuNoise
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldGpuNoise( nn::util::Vector3fType* pOutPos, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex, float time ) NN_NOEXCEPT
{
    ResFieldRandom* resFieldRandom = pEmitter->m_pEmitterRes->m_pFieldGpuNoiseData;

    nn::util::Float3 randomVel;
    if( resFieldRandom->animParam.enable )
    {
        CalculateField8KeyAnim( &randomVel, pEmitter, resFieldRandom->animParam, particleIndex, time );
    }
    else
    {
        randomVel = resFieldRandom->randomVel;
    }

    nn::util::Vector3fType v = NN_UTIL_VECTOR_3F_INITIALIZER( randomVel.x, randomVel.y, randomVel.z );
    CalculateGpuNoise( pOutPos, pEmitter, pParticleProperty, particleIndex, v, time );
}


//------------------------------------------------------------------------------
//  ランダム（FE1のバージョン）
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldRandomSimple( nn::util::Vector3fType* pOutVec, Emitter* pEmitter, int particleIndex, float time ) NN_NOEXCEPT
{
    ResFieldRandomFe1* pFieldRandomData = pEmitter->m_pEmitterRes->m_pFieldRandomSimpleData;

    nn::util::Float3 randomVel;
    if( pFieldRandomData->animParam.enable )
    {
        CalculateField8KeyAnim( &randomVel, pEmitter, pFieldRandomData->animParam, particleIndex, time );
    }
    else
    {
        randomVel = pFieldRandomData->randomVel;
    }

    if( ( static_cast< uint32_t >( time ) % pFieldRandomData->blank ) == 0 )
    {
        const nn::util::Vector3fType& randomVec3f = pEmitter->m_Random.GetVector3();

        nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) + nn::util::VectorGetX( randomVec3f ) * randomVel.x );
        nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) + nn::util::VectorGetY( randomVec3f ) * randomVel.y );
        nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) + nn::util::VectorGetZ( randomVec3f ) * randomVel.z );
    }
}

//------------------------------------------------------------------------------
//  磁力
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldMagnet( nn::util::Vector3fType* pOutPos, nn::util::Vector3fType* pOutVec, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex, float time ) NN_NOEXCEPT
{
    NN_UNUSED( pParticleProperty );

    enum FieldMagnetFlag
    {
        FieldMagnetFlag_X = ( 1 << 0 ),                        //!< X軸に適応
        FieldMagnetFlag_Y = ( 1 << 1 ),                        //!< Y軸に適応
        FieldMagnetFlag_Z = ( 1 << 2 ),                        //!< Z軸に適応
    };

    ResFieldMagnet* pFieldMagnetData = pEmitter->m_pEmitterRes->m_pFieldMagnetData;

    // ８キーアニメーション適用
    nn::util::Float3 fieldMagnetPower;
    {
        const ResAnim8KeyParamSet& anim = pEmitter->m_pEmitterRes->m_pFieldMagnetData->animParam;

        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldMagnetPower, pEmitter, anim, particleIndex, time );
        }
        else
        {
            fieldMagnetPower.x = pFieldMagnetData->fieldMagnetPower;
        }
    }
    const float magnetPower = fieldMagnetPower.x;

    if( pFieldMagnetData->isFollowEmitter )
    {
        // エミッタに追従
        nn::util::Vector3fType local;
        nn::util::Vector3fType src;
        nn::util::MatrixGetAxisW( &src, pEmitter->m_MatrixSrt );

        // ローカルのベクトルを算出
        //emitter->TransformLocalVec( &local, &src );
        {
            nn::util::Vector3fType* pDst = &local;
            const nn::util::Vector3fType* pSrc = &src;

            nn::util::Matrix4x3fType invMatrix;
            if( pEmitter->m_pEmitterData->emitter.followType == ParticleFollowType_None )
            {
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt0 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt1 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt2 );

                // 追従しない
                nn::util::Matrix4x3fType matrix;
                pEmitter->GetParticleEmitterMatrixSrt( &matrix, particleIndex );
                nn::util::MatrixInverse( &invMatrix, matrix );
            }
            else
            {
                // 完全追従 or 位置のみ追従
                nn::util::MatrixInverse( &invMatrix, pEmitter->m_MatrixSrt );
            }
            nn::util::VectorTransform( pDst, *pSrc, invMatrix );
        }

        // 速度を更新
        if( pFieldMagnetData->fieldMagnetFlagX )
        {
            nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) + ( nn::util::VectorGetX( local ) + pFieldMagnetData->fieldMagnetPos.x - nn::util::VectorGetX( *pOutPos ) - nn::util::VectorGetX( *pOutVec ) ) * magnetPower );
        }
        if( pFieldMagnetData->fieldMagnetFlagY )
        {
            nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) + ( nn::util::VectorGetY( local ) + pFieldMagnetData->fieldMagnetPos.y - nn::util::VectorGetY( *pOutPos ) - nn::util::VectorGetY( *pOutVec ) ) * magnetPower );
        }
        if( pFieldMagnetData->fieldMangetFlagZ )
        {
            nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) + ( nn::util::VectorGetZ( local ) + pFieldMagnetData->fieldMagnetPos.z - nn::util::VectorGetZ( *pOutPos ) - nn::util::VectorGetZ( *pOutVec ) ) * magnetPower );
        }
    }
    else
    {
        // エミッタに追従しない
        // 速度を更新
        if( pFieldMagnetData->fieldMagnetFlagX )
        {
            nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) + ( pFieldMagnetData->fieldMagnetPos.x - nn::util::VectorGetX( *pOutPos ) - nn::util::VectorGetX( *pOutVec ) ) * magnetPower );
        }
        if( pFieldMagnetData->fieldMagnetFlagY )
        {
            nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) + ( pFieldMagnetData->fieldMagnetPos.y - nn::util::VectorGetY( *pOutPos ) - nn::util::VectorGetY( *pOutVec ) ) * magnetPower );
        }
        if( pFieldMagnetData->fieldMangetFlagZ )
        {
            nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) + ( pFieldMagnetData->fieldMagnetPos.z - nn::util::VectorGetZ( *pOutPos ) - nn::util::VectorGetZ( *pOutVec ) ) * magnetPower );
        }
    }
}


//------------------------------------------------------------------------------
//  スピン
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldSpin( nn::util::Vector3fType* pOutPos, const Emitter* pEmitter, ParticleProperty* pParticleProperty, int particleIndex, float time ) NN_NOEXCEPT
{
    ResFieldSpinData* pFieldSpinData = pEmitter->m_pEmitterRes->m_pFieldSpinData;
    float sinV, cosV;

    float frameRate = pEmitter->m_FrameRate;
    float dynamicsRand = pParticleProperty->scale[particleIndex].w;
    //dynamicsRand = 1.0f;

    // ８キーアニメーション適用
    nn::util::Float3 fieldSpinRotate;
    {
        const ResAnim8KeyParamSet& anim = pEmitter->m_pEmitterRes->m_pFieldSpinData->rotateAnimParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldSpinRotate, pEmitter, anim, particleIndex, time );
            fieldSpinRotate.x = nn::util::DegreeToRadian( fieldSpinRotate.x );    //ラジアンに変換
        }
        else
        {
            fieldSpinRotate.x = pFieldSpinData->fieldSpinRotate;
        }
    }
    nn::util::Float3 fieldSpinOuter;
    {
        const ResAnim8KeyParamSet& anim = pEmitter->m_pEmitterRes->m_pFieldSpinData->outerAnimParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldSpinOuter, pEmitter, anim, particleIndex, time );
            fieldSpinOuter.x = nn::util::DegreeToRadian( fieldSpinOuter.x );    //ラジアンに変換
        }
        else
        {
            fieldSpinOuter.x = pFieldSpinData->fieldSpinOuter;
        }
    }

    nn::util::SinCosEst( &sinV, &cosV, fieldSpinRotate.x * dynamicsRand * frameRate );
    const float spinOuter = fieldSpinOuter.x;

    switch( pFieldSpinData->fieldSpinAxis )
    {
    case 0://X軸
    {
        const float v0 = nn::util::VectorGetY( *pOutPos ) *  cosV + nn::util::VectorGetZ( *pOutPos ) * sinV;
        const float v1 = nn::util::VectorGetY( *pOutPos ) * -sinV + nn::util::VectorGetZ( *pOutPos ) * cosV;
        nn::util::VectorSetY( pOutPos, v0 );
        nn::util::VectorSetZ( pOutPos, v1 );

        // 拡散速度
        if( spinOuter == 0.0f )
        {
            break;
        }

        const float length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        const float r = 1.0f / ::std::sqrtf( length2 ) * spinOuter * dynamicsRand * frameRate;
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + ( v0 * r ) );
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + ( v1 * r ) );
    }
        break;

    case 1://Y軸
    {
        const float v0 = nn::util::VectorGetZ( *pOutPos ) *  cosV + nn::util::VectorGetX( *pOutPos ) * sinV;
        const float v1 = nn::util::VectorGetZ( *pOutPos ) * -sinV + nn::util::VectorGetX( *pOutPos ) * cosV;
        nn::util::VectorSetZ( pOutPos, v0 );
        nn::util::VectorSetX( pOutPos, v1 );

        // 拡散速度
        if( spinOuter == 0.0f )
        {
            break;
        }

        const float length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        const float r = 1.0f / ::std::sqrtf( length2 ) * spinOuter * dynamicsRand * frameRate;
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + ( v0 * r ) );
        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + ( v1 * r ) );
    }
        break;

    case 2://Z軸
    {
        const float v0 = nn::util::VectorGetX( *pOutPos ) *  cosV + nn::util::VectorGetY( *pOutPos ) * sinV;
        const float v1 = nn::util::VectorGetX( *pOutPos ) * -sinV + nn::util::VectorGetY( *pOutPos ) * cosV;
        nn::util::VectorSetX( pOutPos, v0 );
        nn::util::VectorSetY( pOutPos, v1 );

        // 拡散速度
        if( spinOuter == 0.0f )
        {
            break;
        }

        const float length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        const float r = 1.0f / ::std::sqrtf( length2 ) * spinOuter * dynamicsRand * frameRate;
        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + ( v0 * r ) );
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + ( v1 * r ) );
    }
        break;
    default:
        break;
    }
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
//  コリジョン
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldCollision( nn::util::Vector3fType* pOutPos, nn::util::Vector3fType* pOutVec, float* pOutLife, Emitter* pEmitter, ParticleProperty* pParticleProperty, int particleIndex, ParticleData* pParticleData, float time ) NN_NOEXCEPT
{
    enum FieldCollisionReactionType
    {
        FieldCollisionReactionType_Reflect = 0,     //!< 反射
        FieldCollisionReactionType_Disappear,       //!< 消滅
        FieldCollisionReactionType_Custom,          //!< カスタム
    };

    static const float Epsilon = 0.0001f;

    ResFieldCollisionData* pFieldCollisionData = pEmitter->m_pEmitterRes->m_pFieldCollisionData;

    // コリジョン回数判定
    if( pFieldCollisionData->fieldCollisionCnt != -1 && pFieldCollisionData->fieldCollisionCnt <= static_cast< int32_t >( pParticleData->collisionCount ) )
    {
        return;
    }

    // XZ平面のY座標
    float y = pFieldCollisionData->fieldCollisionCoord;
    {
        // データ上のコリジョン平面を利用する

        //----------------------------------------------------------
        //      ワールド座標系
        //----------------------------------------------------------
        if( pFieldCollisionData->fieldCollisionIsWorld )
        {
            // エミッタ座標系へ
            nn::util::Vector3fType worldPos;
            nn::util::Matrix4x3fType matrix;

            // エミッタ→ワールド変換行列
            nn::util::Matrix4x3fType *pEmitterMatrix;

            if( pEmitter->m_pEmitterData->emitter.followType == ParticleFollowType_EmitterFull )
            {
                pEmitterMatrix = &pEmitter->m_MatrixSrt;
            }
            else
            {
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt0 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt1 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt2 );

                nn::util::MatrixSet( &matrix,
                    pParticleProperty->emitterMatrixSrt0[particleIndex].x, pParticleProperty->emitterMatrixSrt1[particleIndex].x, pParticleProperty->emitterMatrixSrt2[particleIndex].x,
                    pParticleProperty->emitterMatrixSrt0[particleIndex].y, pParticleProperty->emitterMatrixSrt1[particleIndex].y, pParticleProperty->emitterMatrixSrt2[particleIndex].y,
                    pParticleProperty->emitterMatrixSrt0[particleIndex].z, pParticleProperty->emitterMatrixSrt1[particleIndex].z, pParticleProperty->emitterMatrixSrt2[particleIndex].z,
                    pParticleProperty->emitterMatrixSrt0[particleIndex].w, pParticleProperty->emitterMatrixSrt1[particleIndex].w, pParticleProperty->emitterMatrixSrt2[particleIndex].w );

                pEmitterMatrix = &matrix;
            }

            nn::util::VectorTransform( &worldPos, *pOutPos, *pEmitterMatrix );

            switch( pFieldCollisionData->fieldCollisionType )
            {
                // 消滅
            case FieldCollisionReactionType_Disappear:

                // 上から下への侵入を許さない
                if( nn::util::VectorGetY( worldPos ) < y )
                {
                    nn::util::VectorSetY( &worldPos, y );
                    *pOutLife = time; // 寿命をここまでにする
                }
                break;

                // 反射
            case FieldCollisionReactionType_Reflect:

                // 上から下への侵入を許さない
                if( nn::util::VectorGetY( worldPos ) < y )
                {
                    nn::util::VectorSetY( &worldPos, y + Epsilon );

                    // 速度をワールドに持っていく
                    nn::util::Vector3fType worldVel;
                    nn::util::VectorTransformNormal( &worldVel, *pOutVec, *pEmitterMatrix );

                    // 反射
                    nn::util::VectorSetY( &worldVel, nn::util::VectorGetY( worldVel ) * -pFieldCollisionData->fieldCollisionCoef );

                    // 逆行列
                    nn::util::Matrix4x3fType invEmitterMatrix;
                    nn::util::MatrixInverse( &invEmitterMatrix, *pEmitterMatrix );

                    // 速度と位置をエミッタ座標系に持っていく
                    nn::util::Vector3fType temp = worldPos;
                    nn::util::VectorTransform( &temp, worldPos, invEmitterMatrix );
                    {
                        nn::util::Vector3fType localVel;
                        nn::util::Vector3fType zero = NN_UTIL_VECTOR_3F_INITIALIZER( 0.0f, 0.0f, 0.0f );
                        nn::util::MatrixSetAxisW( &invEmitterMatrix, zero );

                        nn::util::VectorTransform( &localVel, worldVel, invEmitterMatrix );
                        nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( localVel ) * pFieldCollisionData->fieldCollisionRegist );
                        nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( localVel ) * pFieldCollisionData->fieldCollisionRegist );
                        nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( localVel ) * pFieldCollisionData->fieldCollisionRegist );
                    }

                    pParticleData->collisionCount += 1; //コリジョン回数を増やす
                }
                break;
            default:
                break;
            }
        }

        //----------------------------------------------------------
        //      エミッタ座標系
        //----------------------------------------------------------
        else
        {
            switch( pFieldCollisionData->fieldCollisionType )
            {
                // 消滅
            case FieldCollisionReactionType_Disappear:

                // 上から下への侵入を許さない
                if( nn::util::VectorGetY( *pOutPos ) < y )
                {
                    nn::util::VectorSetY( pOutPos, y );
                    *pOutLife = time;
                }
                break;

                // 反射
            case FieldCollisionReactionType_Reflect:

                // 上から下への侵入を許さない
                if( nn::util::VectorGetY( *pOutPos ) < y )
                {
                    nn::util::VectorSetY( pOutPos, y );
                    nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) * -pFieldCollisionData->fieldCollisionCoef );

                    nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) * pFieldCollisionData->fieldCollisionRegist );
                    nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) * pFieldCollisionData->fieldCollisionRegist );
                    nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) * pFieldCollisionData->fieldCollisionRegist );

                    pParticleData->collisionCount += 1; //コリジョン回数を増やす
                }
                break;
            default:
                break;
            }
        }
    }

}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
//  収束
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldConvergence( nn::util::Vector3fType* pOutPos, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex, float time ) NN_NOEXCEPT
{
    // TODO : なるべく : CPUでも最適な状態で動作するように

    enum FieldConvergenceType
    {
        FieldConvergenceType_ToAssignedPosition = 0,           //!< 指定された位置に収束
        FieldConvergenceType_ToEmitterPosition,                //!< エミッタ位置に収束
    };

    // 仮打ちのパラメータ
    ResFieldConvergenceData* pFieldConvergenceData = pEmitter->m_pEmitterRes->m_pFieldConvergenceData;

    nn::util::Float3 fieldConvergenceRatio;
    float frameRate = pEmitter->m_FrameRate;
    float dynamicsRand = pParticleProperty->scale[particleIndex].w;
    //dynamicsRand = 1.0f;

    // ８キーアニメーション適用
    {
        const ResAnim8KeyParamSet& anim = pEmitter->m_pEmitterRes->m_pFieldConvergenceData->animParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldConvergenceRatio, pEmitter, anim, particleIndex, time );
        }
        else
        {
            fieldConvergenceRatio.x = pFieldConvergenceData->fieldConvergenceRatio;
        }
    }
    const float convergenceRatio = fieldConvergenceRatio.x;

    if( pFieldConvergenceData->fieldConvergenceType == FieldConvergenceType_ToAssignedPosition )
    {
        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + ( pFieldConvergenceData->fieldConvergencePos.x - nn::util::VectorGetX( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + ( pFieldConvergenceData->fieldConvergencePos.y - nn::util::VectorGetY( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + ( pFieldConvergenceData->fieldConvergencePos.z - nn::util::VectorGetZ( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
    }
    else
    {
        nn::util::Vector3fType local;
        nn::util::Vector3fType src;

        nn::util::MatrixGetAxisW( &src, pEmitter->m_MatrixRt );

        {
            nn::util::Vector3fType* pDst = &local;
            const nn::util::Vector3fType* pSrc = &src;

            nn::util::Matrix4x3fType invMat;
            if( pEmitter->m_pEmitterData->emitter.followType == ParticleFollowType_None )
            {
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt0 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt1 );
                NN_SDK_ASSERT_NOT_NULL( pParticleProperty->emitterMatrixSrt2 );

                // 追従しない
                nn::util::Matrix4x3fType emitterMatrix;
                pEmitter->GetParticleEmitterMatrixSrt( &emitterMatrix, particleIndex );
                nn::util::MatrixInverse( &invMat, emitterMatrix );
            }
            else
            {
                // 完全追従 or 位置のみ追従
                nn::util::MatrixInverse( &invMat, pEmitter->m_MatrixSrt );
            }
            nn::util::VectorTransform( pDst, *pSrc, invMat );
        }

        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + ( nn::util::VectorGetX( local ) + pFieldConvergenceData->fieldConvergencePos.x - nn::util::VectorGetX( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + ( nn::util::VectorGetY( local ) + pFieldConvergenceData->fieldConvergencePos.y - nn::util::VectorGetY( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + ( nn::util::VectorGetZ( local ) + pFieldConvergenceData->fieldConvergencePos.z - nn::util::VectorGetZ( *pOutPos ) ) * convergenceRatio * dynamicsRand * frameRate );
    }
}

//------------------------------------------------------------------------------
//  位置に加算
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehaviorFieldPosAdd( nn::util::Vector3fType* pOutPos, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex, float time ) NN_NOEXCEPT
{
    // 仮打ちパラメータ
    ResFieldPosAddData* pAddPosData = pEmitter->m_pEmitterRes->m_pFieldPosAddData;

    const float frameRate = pEmitter->m_FrameRate;
    const float dynamicsRand = pParticleProperty->scale[particleIndex].w;
    nn::util::Float3 fieldPosAdd;

    // ８キーアニメーション適用
    {
        const ResAnim8KeyParamSet& anim = pAddPosData->animParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldPosAdd, pEmitter, anim, particleIndex, time );
        }
        else
        {
            fieldPosAdd = pAddPosData->fieldPosAdd;
        }
    }

    if( !pAddPosData->isFieldPosAddGlobal )
    {
        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + ( fieldPosAdd.x * dynamicsRand * frameRate ) );
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + ( fieldPosAdd.y * dynamicsRand * frameRate ) );
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + ( fieldPosAdd.z * dynamicsRand * frameRate ) );
    }
    else
    {
        const float posAddX = fieldPosAdd.x * dynamicsRand * frameRate;
        const float posAddY = fieldPosAdd.y * dynamicsRand * frameRate;
        const float posAddZ = fieldPosAdd.z * dynamicsRand * frameRate;

        nn::util::Vector3fType temp = NN_UTIL_VECTOR_3F_INITIALIZER( posAddX, posAddY, posAddZ );
        const int index = particleIndex;
        if( pEmitter->m_pEmitterData->emitter.followType == ParticleFollowType_None )
        {
            // pParticleProperty->emitterMatrixSrt0/1/2 からテンポラリで RTマトリクスを生成する
            nn::util::Matrix4x3fType rtMatrix;
            nn::util::Matrix4x3fType srtMatrix = NN_UTIL_MATRIX_4X3F_INITIALIZER(
                pParticleProperty->emitterMatrixSrt0[ index ].x, pParticleProperty->emitterMatrixSrt1[ index ].x, pParticleProperty->emitterMatrixSrt2[ index ].x,
                pParticleProperty->emitterMatrixSrt0[ index ].y, pParticleProperty->emitterMatrixSrt1[ index ].y, pParticleProperty->emitterMatrixSrt2[ index ].y,
                pParticleProperty->emitterMatrixSrt0[ index ].z, pParticleProperty->emitterMatrixSrt1[ index ].z, pParticleProperty->emitterMatrixSrt2[ index ].z,
                pParticleProperty->emitterMatrixSrt0[ index ].w, pParticleProperty->emitterMatrixSrt1[ index ].w, pParticleProperty->emitterMatrixSrt2[ index ].w );
            MakrRtMatrix( &rtMatrix, srtMatrix );
            nn::util::MatrixTranspose( &rtMatrix, rtMatrix );
            nn::util::VectorTransformNormal( &temp, temp, rtMatrix );
        }
        else
        {
            nn::util::Matrix4x3fType matrix;
            nn::util::MatrixTranspose( &matrix, pEmitter->m_MatrixRt );
            nn::util::VectorTransformNormal( &temp, temp, matrix );
        }

        nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + nn::util::VectorGetX( temp ) );
        nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + nn::util::VectorGetY( temp ) );
        nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + nn::util::VectorGetZ( temp ) );
    }
}

//------------------------------------------------------------------------------
// 揺らぎ : Sin波
//------------------------------------------------------------------------------
float CalculateFluctuationSineWave( float time, float amplitude, float cycle, float initialPhase, float phaseRandom, const nn::util::Vector4fType& random ) NN_NOEXCEPT
{
    const float t = ( ( time + initialPhase ) / cycle ) + ( phaseRandom * nn::util::VectorGetX( random ) );             //現在フレーム位置（割合）
    const float flucValue = 1.0f - ( ( nn::util::CosEst( t * nn::util::FloatPi * 2.0f ) + 1.0f ) * 0.5f * amplitude );  //揺らぎ量
    return flucValue;
}

//------------------------------------------------------------------------------
// 揺らぎ : のこぎり波
//------------------------------------------------------------------------------
float CalculateFluctuationSawToothWave( float time, float amplitude, float cycle, float initialPhase, float phaseRandom, const nn::util::Vector4fType& random ) NN_NOEXCEPT
{
    const float t = ( ( time + initialPhase ) / cycle ) + ( phaseRandom * nn::util::VectorGetX( random ) );     //現在フレーム位置（割合）
    const float flucValue = ::std::fabs( ( 1 - Fract( t ) * amplitude ) );                                      //揺らぎ量
    return flucValue;
}

//------------------------------------------------------------------------------
// 揺らぎ : 矩形波
//------------------------------------------------------------------------------
float CalculateFluctuationRectangleWave( float time, float amplitude, float cycle, float initialPhase, float phaseRandom, const nn::util::Vector4fType& random ) NN_NOEXCEPT
{
    const float t = ( ( time + initialPhase ) / cycle ) + ( phaseRandom * nn::util::VectorGetX( random ) );     //現在フレーム位置（割合）
    const float flucValue = ::std::fabs( 1.0f - ( 1.0f - Step( 0.5f, Fract( t ) ) ) * amplitude );              //揺らぎ量
    return flucValue;
}


//---------------------------------------------------------------------------
//  パーティクルの挙動計算を行う 積算版
//---------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleBehavior(
    nn::util::Vector3fType* pOutPos,
    nn::util::Vector3fType* pOutVec,
    float* pOutBirthTime,
    float* pOutLife,
    Emitter* pEmitter,
    int index,
    float time,
    const nn::util::Vector3fType& posBack,
    const nn::util::Vector3fType& vecBack ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEmitter->m_pEmitterRes->m_ResEmitterStaticConstantBuffer );

    const float frameRate = pEmitter->m_FrameRate;

    ParticleProperty* pParticleProperty = &pEmitter->m_CpuParticleProperty;
    ParticleData*     pParticleData     = &pEmitter->m_pParticleData[ index ];

    // 速度＋運動量ランダム
    const float rndRatio = pParticleProperty->scale[ index ].w * frameRate;
    {
        nn::util::Vector3fType temp;
        nn::util::VectorMultiply( &temp, vecBack, rndRatio );
        nn::util::VectorAdd( pOutPos, temp, posBack );
    }

    // 空気抵抗
    // MEMO: 1より大の場合は加速するという旧来の仕様がある。
    if( pEmitter->m_pEmitterRes->m_ResEmitterStaticConstantBuffer->airRegist != 1.0f )
    {
        const float airRegist = ::std::powf( pEmitter->m_pEmitterRes->m_ResEmitterStaticConstantBuffer->airRegist, frameRate );
        nn::util::VectorMultiply( pOutVec, vecBack, airRegist );
    }
    else
    {
        // 空気抵抗が無ければ値をコピー
        //if( time > 0 )
        {
            // 初回フレームのみ
            *pOutVec = vecBack;
        }
    }
    // この時点でpos,vecのフロントバッファへの更新処理が完了。

    // 重力
    const float gravScaleAnim = pEmitter->GetEmitterAnimValue().gravityScale;
    if( gravScaleAnim > 0.f )
    {
        const float ubGravityX = pEmitter->m_pEmitterData->emission.gravityDir.x * gravScaleAnim;
        const float ubGravityY = pEmitter->m_pEmitterData->emission.gravityDir.y * gravScaleAnim;
        const float ubGravityZ = pEmitter->m_pEmitterData->emission.gravityDir.z * gravScaleAnim;

        if( pEmitter->m_pEmitterData->emission.isWorldGravity )
        {
            nn::util::Vector3fType gravity = NN_UTIL_VECTOR_3F_INITIALIZER( ubGravityX, ubGravityY, ubGravityZ );
            if( pEmitter->m_pEmitterData->emitter.followType == ParticleFollowType_None )
            {
                // pParticleProperty->emitterMatrixSrt0/1/2 からテンポラリで RTマトリクスを生成する
                nn::util::Matrix4x3fType rtMatrix;
                nn::util::Matrix4x3fType srtMatrix = NN_UTIL_MATRIX_4X3F_INITIALIZER(
                    pParticleProperty->emitterMatrixSrt0[ index ].x, pParticleProperty->emitterMatrixSrt1[ index ].x, pParticleProperty->emitterMatrixSrt2[ index ].x,
                    pParticleProperty->emitterMatrixSrt0[ index ].y, pParticleProperty->emitterMatrixSrt1[ index ].y, pParticleProperty->emitterMatrixSrt2[ index ].y,
                    pParticleProperty->emitterMatrixSrt0[ index ].z, pParticleProperty->emitterMatrixSrt1[ index ].z, pParticleProperty->emitterMatrixSrt2[ index ].z,
                    pParticleProperty->emitterMatrixSrt0[ index ].w, pParticleProperty->emitterMatrixSrt1[ index ].w, pParticleProperty->emitterMatrixSrt2[ index ].w );
                MakrRtMatrix( &rtMatrix, srtMatrix  );
                nn::util::MatrixTranspose( &rtMatrix, rtMatrix );
                nn::util::VectorTransformNormal( &gravity, gravity, rtMatrix );
            }
            else
            {
                nn::util::Matrix4x3fType matrix;
                nn::util::MatrixTranspose( &matrix, pEmitter->m_MatrixRt );
                nn::util::VectorTransformNormal( &gravity, gravity, matrix );
            }
            nn::util::VectorMultiply( &gravity, gravity, frameRate );

            nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) + nn::util::VectorGetX( gravity ) );
            nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) + nn::util::VectorGetY( gravity ) );
            nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) + nn::util::VectorGetZ( gravity ) );
        }
        else
        {
            nn::util::VectorSetX( pOutVec, nn::util::VectorGetX( *pOutVec ) + frameRate * ubGravityX );
            nn::util::VectorSetY( pOutVec, nn::util::VectorGetY( *pOutVec ) + frameRate * ubGravityY );
            nn::util::VectorSetZ( pOutVec, nn::util::VectorGetZ( *pOutVec ) + frameRate * ubGravityZ );
        }
    }

    // フィールド
    if( pEmitter->m_pEmitterRes->m_IsUseField )
    {
        //------------------------------------------------------------------------------
        //  フィールド処理の結果、パーティクルが消える可能性のあるものを先に実行
        //------------------------------------------------------------------------------
        if( pEmitter->m_pEmitterRes->m_pFieldCollisionData )
        {
            // コリジョンはパーティクルの消滅も扱うので先に実行する
            CalculateParticleBehaviorFieldCollision( pOutPos, pOutVec, pOutLife, pEmitter, pParticleProperty, index, pParticleData, time );
        }
        //------------------------------------------------------------------------------
        //  パーティクルが消滅するなら以降の計算は端折って良い（カスタムフィールド以外）
        //------------------------------------------------------------------------------

        if( pEmitter->m_pEmitterRes->m_pFieldGpuNoiseData )
        {
            CalculateParticleBehaviorFieldGpuNoise( pOutPos, pEmitter, pParticleProperty, index, time );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldRandomSimpleData )
        {
            CalculateParticleBehaviorFieldRandomSimple( pOutVec, pEmitter, index, time );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldMagnetData )
        {
            CalculateParticleBehaviorFieldMagnet( pOutPos, pOutVec, pEmitter, pParticleProperty, index, time );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldSpinData )
        {
            CalculateParticleBehaviorFieldSpin( pOutPos, pEmitter, pParticleProperty, index, time );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldConvergenceData )
        {
            CalculateParticleBehaviorFieldConvergence( pOutPos, pEmitter, pParticleProperty, index, time );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldCurlNoiseData )
        {
            CalculateParticleBehavior_FieldCurlNoise( pOutPos, pOutVec, pEmitter, pParticleProperty, index );
        }
        if( pEmitter->m_pEmitterRes->m_pFieldPosAddData )
        {
            CalculateParticleBehaviorFieldPosAdd( pOutPos, pEmitter, pParticleProperty, index, time );
        }

        //------------------------------------------------------------------------------
        //  カスタムフィールド
        //------------------------------------------------------------------------------
        if( pEmitter->m_pEmitterRes->m_pFieldCustomData )
        {
            CalculateParticleBehaviorCustomField( pOutPos, pOutVec, pOutLife, pOutBirthTime, pEmitter, pParticleProperty, index );
        }
    }
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
// 現在のパーティクルスケールを計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleScaleVecFromTime(
    nn::util::Vector3fType*         pOutScale,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   initialScale,
    const nn::util::Vector4fType&   random,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    detail::Vector3fCopy( pOutScale, initialScale );

    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    //アニメーション
    if( pEmitterResource->m_ResEmitterStaticConstantBuffer->scaleAnimKeyNum > 1 )
    {
        float loopRate;
        if( pEmitterResource->m_pResEmitter->ptcl.scaleAnimIsLoop )
        {
            loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.scaleAnimLoopRate );
        }
        else
        {
            loopRate = 0.0f;
        }

        nn::util::Float3 sAnim;
        Calculate8KeyAnim( &sAnim,
            pEmitterResource->m_ResEmitterStaticConstantBuffer->scale8keyAnim,
            pEmitterResource->m_ResEmitterStaticConstantBuffer->scaleAnimKeyNum,
            nn::util::VectorGetX( random ), time, loopRate,
            static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.scaleAnimIsLoopInitRandom ),
            life );

        nn::util::VectorSetX( pOutScale, nn::util::VectorGetX( *pOutScale ) * sAnim.x );
        nn::util::VectorSetY( pOutScale, nn::util::VectorGetY( *pOutScale ) * sAnim.y );
        nn::util::VectorSetZ( pOutScale, nn::util::VectorGetZ( *pOutScale ) * sAnim.z );
    }
    else
    {
        nn::util::VectorSetX( pOutScale, nn::util::VectorGetX( *pOutScale ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->scale8keyAnim.value[ 0 ].x );
        nn::util::VectorSetY( pOutScale, nn::util::VectorGetY( *pOutScale ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->scale8keyAnim.value[ 0 ].y );
        nn::util::VectorSetZ( pOutScale, nn::util::VectorGetZ( *pOutScale ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->scale8keyAnim.value[ 0 ].z );
    }

    // 揺らぎ
    // TODO: Z？
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplayScale )
    {
        float flucX = 1.0f;
        float flucY = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }
        flucY = flucX;

        if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplayScaleY )
        {
            //軸を個別に適用
            switch( index )
            {
            case ParticleFluctuationWaveType_Sin:
                flucY = CalculateFluctuationSineWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case ParticleFluctuationWaveType_SawTooth:
                flucY = CalculateFluctuationSawToothWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case ParticleFluctuationWaveType_Rectangle:
                flucY = CalculateFluctuationRectangleWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            default:
                break;
            }
        }

        // 揺らぎは乗算処理
        nn::util::VectorSetX( pOutScale, nn::util::VectorGetX( *pOutScale ) * flucX );
        nn::util::VectorSetY( pOutScale, nn::util::VectorGetY( *pOutScale ) * flucY );
    }

    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクルスケールを計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleScaleVecFromFrame(
    nn::util::Vector3fType*         pOutScale,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   initialScale,
    const nn::util::Vector4fType&   random,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    detail::Vector3fCopy( pOutScale, initialScale );

    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    //アニメーション
    if( pEmitterResource->m_ResEmitterStaticConstantBuffer->scaleAnimKeyNum > 1 )
    {
        float loopRate;
        if( pEmitterResource->m_pResEmitter->ptcl.scaleAnimIsLoop )
        {
            loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.scaleAnimLoopRate );
        }
        else
        {
            loopRate = 0.0f;
        }

        nn::util::Float3 sAnim;
        Calculate8KeyAnim( &sAnim,
            pEmitterResource->m_ResEmitterStaticConstantBuffer->scale8keyAnim,
            pEmitterResource->m_ResEmitterStaticConstantBuffer->scaleAnimKeyNum,
            nn::util::VectorGetX( random ), time, loopRate,
            static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.scaleAnimIsLoopInitRandom ),
            life );

        nn::util::VectorSetX( pOutScale, nn::util::VectorGetX( *pOutScale ) * sAnim.x );
        nn::util::VectorSetY( pOutScale, nn::util::VectorGetY( *pOutScale ) * sAnim.y );
        nn::util::VectorSetZ( pOutScale, nn::util::VectorGetZ( *pOutScale ) * sAnim.z );
    }

    // 揺らぎ
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplayScale )
    {
        float flucX = 1.0f;
        float flucY = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }
        flucY = flucX;

        if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplayScaleY )
        {
            //軸を個別に適用
            switch( index )
            {
            case ParticleFluctuationWaveType_Sin:
                flucY = CalculateFluctuationSineWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case ParticleFluctuationWaveType_SawTooth:
                flucY = CalculateFluctuationSawToothWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case ParticleFluctuationWaveType_Rectangle:
                flucY = CalculateFluctuationRectangleWave( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            default:
                break;
            }
        }

        // 揺らぎは乗算処理
        nn::util::VectorSetX( pOutScale, nn::util::VectorGetX( *pOutScale ) * flucX );
        nn::util::VectorSetY( pOutScale, nn::util::VectorGetY( *pOutScale ) * flucY );
    }
    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクルカラー0を計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleColor0VecFromTime(
    nn::util::Vector4fType*         pOutColor,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   random,
    const nn::util::Vector4fType&   emitterColor,
    const nn::util::Vector3fType&   emitterAnimColor,
    float                           emitterAnimAlpha,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    // 基本値
    nn::util::VectorSetX( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.x );
    nn::util::VectorSetY( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.y );
    nn::util::VectorSetZ( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.z );
    nn::util::VectorSetW( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.alpha0 );

    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    // ランダム
    if( pEmitterResource->m_pResEmitter->ptclColor.color0Type == ParticleColorCalculationMode_RandomColor )
    {
        int index = static_cast< int >( nn::util::VectorGetX( random ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum );
        nn::util::VectorSetX( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].x );
        nn::util::VectorSetY( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].y );
        nn::util::VectorSetZ( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].z );
    }

    // 8キーアニメーション
    if( pEmitterResource->m_pResEmitter->ptclColor.color0Type == ParticleColorCalculationMode_8KeysAnimation )
    {
        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum > 0 )
        {
            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.color0AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 c;
            Calculate8KeyAnim( &c,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color0AnimIsLoopInitRandom ),
                life );
            nn::util::VectorSetX( pOutColor, c.x );
            nn::util::VectorSetY( pOutColor, c.y );
            nn::util::VectorSetZ( pOutColor, c.z );
        }
    }
    if( pEmitterResource->m_pResEmitter->ptclColor.alpha0Type == ParticleColorCalculationMode_8KeysAnimation )
    {
        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0AnimKeyNum > 0 )
        {
            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 a;
            Calculate8KeyAnim( &a,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0AnimKeyNum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimIsLoopInitRandom ),
                life );
            nn::util::VectorSetW( pOutColor, a.x );
        }
    }

    // エミッタカラー / カラースケール
    const float colorScale = pEmitterResource->m_ResEmitterStaticConstantBuffer->colorScale;
    nn::util::VectorSetX( pOutColor, nn::util::VectorGetX( *pOutColor ) * ( nn::util::VectorGetX( emitterColor ) * nn::util::VectorGetX( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetY( pOutColor, nn::util::VectorGetY( *pOutColor ) * ( nn::util::VectorGetY( emitterColor ) * nn::util::VectorGetY( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetZ( pOutColor, nn::util::VectorGetZ( *pOutColor ) * ( nn::util::VectorGetZ( emitterColor ) * nn::util::VectorGetZ( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * ( nn::util::VectorGetW( emitterColor ) * emitterAnimAlpha ) );

    // アルファ揺らぎ
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplyAlpha )
    {
        float flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * flucX );
        if( nn::util::VectorGetW( *pOutColor )< 0 )
        {
            nn::util::VectorSetW( pOutColor, 0 );
        }
        if( nn::util::VectorGetW( *pOutColor ) > 1 )
        {
            nn::util::VectorSetW( pOutColor, 1 );
        }
    }
    return;
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
// 現在のパーティクルカラー0 の、
// エミッタカラー、エミッタセットカラー、カラースケールを除いた生の値を計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleColor0RawValue(
    nn::util::Vector4fType*         pOutColor,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   random,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    // 基本値
    nn::util::VectorSetX( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.x );
    nn::util::VectorSetY( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.y );
    nn::util::VectorSetZ( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color0.z );
    nn::util::VectorSetW( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.alpha0 );
    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    // ランダム
    if( pEmitterResource->m_pResEmitter->ptclColor.color0Type == ParticleColorCalculationMode_RandomColor )
    {
        int index = int( nn::util::VectorGetX( random ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum );
        nn::util::VectorSetX( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].x );
        nn::util::VectorSetY( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].y );
        nn::util::VectorSetZ( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim.value[ index ].z );
    }
    // 8キーアニメーション
    if( pEmitterResource->m_pResEmitter->ptclColor.color0Type == ParticleColorCalculationMode_8KeysAnimation )
    {

        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum > 0 )
        {
            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.color0AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 c;
            Calculate8KeyAnim(
                &c,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color0Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color0AnimKeynum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color0AnimIsLoopInitRandom ),
                life );
            nn::util::VectorSetX( pOutColor, c.x );
            nn::util::VectorSetY( pOutColor, c.y );
            nn::util::VectorSetZ( pOutColor, c.z );
        }
    }
    if( pEmitterResource->m_pResEmitter->ptclColor.alpha0Type == ParticleColorCalculationMode_8KeysAnimation )
    {
        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0AnimKeyNum > 0 )
        {

            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 a;
            Calculate8KeyAnim(
                &a,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha0AnimKeyNum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha0AnimIsLoopInitRandom ),
                life );

            nn::util::VectorSetW( pOutColor, a.x );
        }
    }

    // アルファ揺らぎ
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplyAlpha )
    {
        float flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * flucX );
        if( nn::util::VectorGetW( *pOutColor ) < 0 )
        {
            nn::util::VectorSetW( pOutColor, 0 );
        }
        if( nn::util::VectorGetW( *pOutColor ) > 1 )
        {
            nn::util::VectorSetW( pOutColor, 1 );
        }
    }
    return;
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
// 現在のパーティクルカラー１を計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleColor1VecFromTime(
    nn::util::Vector4fType*         pOutColor,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   random,
    const nn::util::Vector4fType&   emitterColor,
    const nn::util::Vector3fType&   emitterAnimColor,
    float                           emitterAnimAlpha,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    // 基本値
    nn::util::VectorSetX( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.x );
    nn::util::VectorSetY( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.y );
    nn::util::VectorSetZ( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.z );
    nn::util::VectorSetW( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.alpha1 );
    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    // ランダム
    if( pEmitterResource->m_pResEmitter->ptclColor.color1Type == ParticleColorCalculationMode_RandomColor )
    {
        int index = int( nn::util::VectorGetX( random ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum );
        nn::util::VectorSetX( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].x );
        nn::util::VectorSetY( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].y );
        nn::util::VectorSetZ( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].z );
    }
    // 8キーアニメーション
    if( pEmitterResource->m_pResEmitter->ptclColor.color1Type == ParticleColorCalculationMode_8KeysAnimation )
    {

        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum > 0 )
        {
            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.color1AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color1AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 c;
            Calculate8KeyAnim(
                &c,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color1AnimIsLoopInitRandom ),
                life );
            nn::util::VectorSetX( pOutColor, c.x );
            nn::util::VectorSetY( pOutColor, c.y );
            nn::util::VectorSetZ( pOutColor, c.z );
        }
    }
    if( pEmitterResource->m_pResEmitter->ptclColor.alpha1Type == ParticleColorCalculationMode_8KeysAnimation )
    {
        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1AnimKeyNum > 0 )
        {

            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 a;
            Calculate8KeyAnim(
                &a,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1AnimKeyNum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimIsLoopInitRandom ),
                life );

            nn::util::VectorSetW( pOutColor, a.x );
        }
    }

    // エミッタカラー / カラースケール
    const float colorScale = pEmitterResource->m_ResEmitterStaticConstantBuffer->colorScale;
    nn::util::VectorSetX( pOutColor, nn::util::VectorGetX( *pOutColor ) * ( nn::util::VectorGetX( emitterColor ) * nn::util::VectorGetX( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetY( pOutColor, nn::util::VectorGetY( *pOutColor ) * ( nn::util::VectorGetY( emitterColor ) * nn::util::VectorGetY( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetZ( pOutColor, nn::util::VectorGetZ( *pOutColor ) * ( nn::util::VectorGetZ( emitterColor ) * nn::util::VectorGetZ( emitterAnimColor ) * colorScale ) );
    nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * ( nn::util::VectorGetW( emitterColor ) * emitterAnimAlpha ) );

    // アルファ揺らぎ
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplyAlpha )
    {
        float flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * flucX );
        if( nn::util::VectorGetW( *pOutColor ) < 0 )
        {
            nn::util::VectorSetW( pOutColor, 0 );
        }
        if( nn::util::VectorGetW( *pOutColor ) > 1 )
        {
            nn::util::VectorSetW( pOutColor, 1 );
        }
    }
    return;
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
// 現在のパーティクルカラー1 の、
// エミッタカラー、エミッタセットカラー、カラースケールを除いた生の値を計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateParticleColor1RawValue(
    nn::util::Vector4fType*         pOutColor,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   random,
    float                           life,
    float                           time ) NN_NOEXCEPT
{
    // 基本値
    nn::util::VectorSetX( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.x );
    nn::util::VectorSetY( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.y );
    nn::util::VectorSetZ( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.color1.z );
    nn::util::VectorSetW( pOutColor, pEmitterResource->m_pResEmitter->ptclColor.alpha1 );
    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    // ランダム
    if( pEmitterResource->m_pResEmitter->ptclColor.color1Type == ParticleColorCalculationMode_RandomColor )
    {
        int index = int( nn::util::VectorGetX( random ) * pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum );
        nn::util::VectorSetX( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].x );
        nn::util::VectorSetY( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].y );
        nn::util::VectorSetZ( pOutColor, pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim.value[ index ].z );
    }
    // 8キーアニメーション
    if( pEmitterResource->m_pResEmitter->ptclColor.color1Type == ParticleColorCalculationMode_8KeysAnimation )
    {

        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum > 0 )
        {
            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.color1AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color1AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 c;
            Calculate8KeyAnim(
                &c,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color1Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->color1AnimKeynum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.color1AnimIsLoopInitRandom ),
                life );
            nn::util::VectorSetX( pOutColor, c.x );
            nn::util::VectorSetY( pOutColor, c.y );
            nn::util::VectorSetZ( pOutColor, c.z );
        }
    }
    if( pEmitterResource->m_pResEmitter->ptclColor.alpha1Type == ParticleColorCalculationMode_8KeysAnimation )
    {
        if( pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1AnimKeyNum > 0 )
        {

            float loopRate;
            if( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimIsLoop )
            {
                loopRate = static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nn::util::Float3 a;
            Calculate8KeyAnim(
                &a,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1Anim,
                pEmitterResource->m_ResEmitterStaticConstantBuffer->alpha1AnimKeyNum,
                nn::util::VectorGetX( random ), time, loopRate,
                static_cast< float >( pEmitterResource->m_pResEmitter->ptcl.alpha1AnimIsLoopInitRandom ),
                life );

            nn::util::VectorSetW( pOutColor, a.x );
        }
    }

    // アルファ揺らぎ
    if( pEmitterResource->m_pResEmitter->ptclFluctuation.isApplyAlpha )
    {
        float flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.x;
        fluc.cycle.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.x;
        fluc.phaseRnd.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.x;

        fluc.amplitude.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.amplitude.y;
        fluc.cycle.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.cycle.y;
        fluc.phaseRnd.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y = pEmitterResource->m_ResEmitterStaticConstantBuffer->fluctuationData.phaseInit.y;

        const uint32_t index = pEmitterResource->m_pResEmitter->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case ParticleFluctuationWaveType_Sin:
            flucX = CalculateFluctuationSineWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_SawTooth:
            flucX = CalculateFluctuationSawToothWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case ParticleFluctuationWaveType_Rectangle:
            flucX = CalculateFluctuationRectangleWave( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        nn::util::VectorSetW( pOutColor, nn::util::VectorGetW( *pOutColor ) * flucX );
        if( nn::util::VectorGetW( *pOutColor ) < 0 )
        {
            nn::util::VectorSetW( pOutColor, 0 );
        }
        if( nn::util::VectorGetW( *pOutColor ) > 1 )
        {
            nn::util::VectorSetW( pOutColor, 1 );
        }
    }
    return;
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
// 現在のパーティクル回転値を計算する
//------------------------------------------------------------------------------
void EmitterCalculator::CalculateRotationMatrix(
    nn::util::Vector3fType*         pOutRotate,
    const EmitterResource*          pEmitterResource,
    const nn::util::Vector4fType&   initialRotate,
    const nn::util::Vector4fType&   random,
    float                           time ) NN_NOEXCEPT
{
    detail::Vector3fCopy( pOutRotate, initialRotate );
    if( !pEmitterResource->m_ResEmitterStaticConstantBuffer )
    {
        return;
    }

    const nn::util::Float4& rInitRand = pEmitterResource->m_ResEmitterStaticConstantBuffer->rotateInitRand;
    const nn::util::Float3& rAdd = pEmitterResource->m_ResEmitterStaticConstantBuffer->rotateAdd;
    const nn::util::Float3& rAddRand = pEmitterResource->m_ResEmitterStaticConstantBuffer->rotateAddRand;
    const float rotVelRegist = pEmitterResource->m_ResEmitterStaticConstantBuffer->rotateRegist;

    float initialRotateX = nn::util::VectorGetX( initialRotate ); // 初期値 X
    float initialRotateY = nn::util::VectorGetY( initialRotate ); // 初期値 Y
    float initialRotateZ = nn::util::VectorGetZ( initialRotate ); // 初期値 Z
    float initialRotateRandomX = rInitRand.x;   // 初期値ランダム X
    float initialRotateRandomY = rInitRand.y;   // 初期値ランダム Y
    float initialRotateRandomZ = rInitRand.z;   // 初期値ランダム Z
    float rotVelX = rAdd.x;        // 加算値 X
    float rotVelY = rAdd.y;        // 加算値 Y
    float rotVelZ = rAdd.z;        // 加算値 Z
    float rotVelRandX = rAddRand.x;    // 加算値ランダム X
    float rotVelRandY = rAddRand.y;    // 加算値ランダム Y
    float rotVelRandZ = rAddRand.z;    // 加算値ランダム Z

    // 乱数
    float particleRandomX = nn::util::VectorGetX( random );
    float particleRandomY = nn::util::VectorGetY( random );
    float particleRandomZ = nn::util::VectorGetZ( random );

    // 乱数を増やす
    float tempRandX = ( particleRandomX + particleRandomY ) / 2;
    float tempRandY = ( particleRandomY + particleRandomZ ) / 2;
    float tempRandZ = ( particleRandomZ + particleRandomX ) / 2;

    // 回転速度（ランダム量込み）
    float rVelX = rotVelX + ( tempRandX * rotVelRandX );
    float rVelY = rotVelY + ( tempRandY * rotVelRandY );
    float rVelZ = rotVelZ + ( tempRandZ * rotVelRandZ );

    // 回転方向ランダム
    if( pEmitterResource->m_pResEmitter->ptcl.rotRevRandX && ( particleRandomY >= 0.5f ) )
    {
        rVelX    = -rVelX;
        initialRotateX = -initialRotateX;
    }
    if( pEmitterResource->m_pResEmitter->ptcl.rotRevRandY && ( particleRandomZ >= 0.5f ) )
    {
        rVelY    = -rVelY;
        initialRotateY = -initialRotateY;
    }
    if( pEmitterResource->m_pResEmitter->ptcl.rotRevRandZ && ( particleRandomX >= 0.5f ) )
    {
        rVelZ    = -rVelZ;
        initialRotateZ = -initialRotateZ;
    }

    // 回転初期値（ランダム量込み）
    float rotX = initialRotateX + ( particleRandomX * initialRotateRandomX );
    float rotY = initialRotateY + ( particleRandomY * initialRotateRandomY );
    float rotZ = initialRotateZ + ( particleRandomZ * initialRotateRandomZ );

    // 減衰率を計算
    float rotRegist = pow( rotVelRegist, time );
    float rotTime = ( rotVelRegist == 1.0 ) ? time : ( 1 - rotRegist ) / ( 1 - rotVelRegist );

    // 現在時刻の回転量を計算
    nn::util::VectorSetX( pOutRotate, ( rVelX * rotTime ) + rotX );
    nn::util::VectorSetY( pOutRotate, ( rVelY * rotTime ) + rotY );
    nn::util::VectorSetZ( pOutRotate, ( rVelZ * rotTime ) + rotZ );
    return;
}

//---------------------------------------------------
//! @brief  現在のパーティクル回転値を計算する
//---------------------------------------------------
void EmitterCalculator::CalculateRotationMatrix(
    nn::util::Vector3fType* pOutRotate,
    const EmitterResource* pEmitterResource,
    const nn::util::Vector4fType& random ) NN_NOEXCEPT
{
    NN_UNUSED( random );

    const nn::util::Float3& rAdd = pEmitterResource->m_ResEmitterStaticConstantBuffer->rotateAdd;
    //    const nn::util::Vector3fType& rAddRand  = emitterRes->resEmitterStaticConstantBuffer->rotateAddRand;
    //    const float rotVelRegist          = emitterRes->resEmitterStaticConstantBuffer->rotateRegist;

    float rotVelX = rAdd.x;        // 加算値 X
    float rotVelY = rAdd.y;        // 加算値 Y
    float rotVelZ = rAdd.z;        // 加算値 Z
    //    float rotVelRandX = rAddRand.x;    // 加算値ランダム X
    //    float rotVelRandY = rAddRand.y;    // 加算値ランダム Y
    //    float rotVelRandZ = rAddRand.z;    // 加算値ランダム Z

    nn::util::VectorSet( pOutRotate, rotVelX, rotVelY, rotVelZ );
}


//------------------------------------------------------------------------------
// XYZ 回転行列を生成します。
//------------------------------------------------------------------------------
void EmitterCalculator::MakeRotationMatrixXYZ( nn::util::Matrix4x4fType* pOutMatrix, const nn::util::Vector3fType& rotate ) NN_NOEXCEPT
{
    //回転の軸は常にWorld空間に固定されます

    const float sinX = nn::util::SinEst( nn::util::VectorGetX( rotate ) );
    const float cosX = nn::util::CosEst( nn::util::VectorGetX( rotate ) );

    const float sinY = nn::util::SinEst( nn::util::VectorGetY( rotate ) );
    const float cosY = nn::util::CosEst( nn::util::VectorGetY( rotate ) );

    const float sinZ = nn::util::SinEst( nn::util::VectorGetZ( rotate ) );
    const float cosZ = nn::util::CosEst( nn::util::VectorGetZ( rotate ) );

    const float opt1 = cosX * sinZ;
    const float opt2 = cosX * cosZ;
    const float opt3 = sinX * sinZ;
    const float opt4 = sinX * cosZ;

    const float m00 = cosY * cosZ;         // (cosY * cosZ)
    const float m01 = cosY * sinZ;         // (cosY * sinZ)
    const float m02 = -sinY;               //-(sinY)
    const float m03 = 0.0;
    const float m10 = opt4 * sinY - opt1;  // (sinX * sinY * cosZ) - (cosX * sinZ)
    const float m11 = opt3 * sinY + opt2;  // (sinX * sinY * sinZ) + (cosX * cosZ)
    const float m12 = sinX * cosY;         // (sinX * cosY)
    const float m13 = 0.0;
    const float m20 = opt2 * sinY + opt3;  // (cosX * sinY * cosZ) + (sinX * sinZ)
    const float m21 = opt1 * sinY - opt4;  // (cosX * sinY * sinZ) - (sinX * sinZ)
    const float m22 = cosX * cosY;         // (cosX * cosY)
    const float m23 = 0.0;
    const float m30 = 0.0;
    const float m31 = 0.0;
    const float m32 = 0.0;
    const float m33 = 1.0;

    nn::util::MatrixSet( pOutMatrix,
        m00, m01, m02, m03,
        m10, m11, m12, m13,
        m20, m21, m22, m23,
        m30, m31, m32, m33 );
}

#if 0
//------------------------------------------------------------------------------
// 回転行列生成 ( YZX順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixYZX( nn::util::Matrix4x4fType* dstMatrix, const nn::util::Vector3fType& rotate ) NN_NOEXCEPT
{
    //回転の軸は常にWorld空間に固定されます

    nn::util::Matrix4x4fType rotMat;

    float sinX = sin( -rotate.x );
    float cosX = cos( -rotate.x );

    float sinY = sin( -rotate.y );
    float cosY = cos( -rotate.y );

    float sinZ = sin( -rotate.z );
    float cosZ = cos( -rotate.z );

    float opt1 = sinX * sinY;
    float opt2 = cosX * sinY;
    float opt3 = sinX * cosY;
    float opt4 = cosX * cosY;

    dstMatrix->m[0][0] =  cosY * cosZ;        // (cosY * cosZ)
    dstMatrix->m[0][1] =  opt4 * sinZ + opt1; // (cosX * cosY * sinZ) + (sinX * sinY)
    dstMatrix->m[0][2] =  opt3 * sinZ - opt2; // (sinX * cosY * sinZ) - (cosX * sinY)
    dstMatrix->m[0][3] =  0.0;

    dstMatrix->m[1][0] = -sinZ;               //-(sinZ)
    dstMatrix->m[1][1] =  cosZ * cosX;        // (cosX * cosZ)
    dstMatrix->m[1][2] =  cosZ * sinX;        // (sinX * cosZ)
    dstMatrix->m[1][3] =  0.0;

    dstMatrix->m[2][0] =  sinY * cosZ;        // (sinY * cosZ)
    dstMatrix->m[2][1] =  opt2 * sinZ - opt3; // (cosX * sinY * sinZ) - (sinX * cosY)
    dstMatrix->m[2][2] =  opt1 * sinZ + opt4; //-(sinX * sinY * sinZ) + (cosX * cosY)
    dstMatrix->m[2][3] =  0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;

    return;
}

//------------------------------------------------------------------------------
// 回転行列生成 ( XYZ順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixXYZ( nn::util::Matrix4x4fType* dstMatrix, const nn::util::Vector3fType& rotate ) NN_NOEXCEPT
{
    //回転の軸は常にWorld空間に固定されます

    nn::util::Matrix4x4fType rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosX * sinZ;
    float opt2 = cosX * cosZ;
    float opt3 = sinX * sinZ;
    float opt4 = sinX * cosZ;

    dstMatrix->m[0][0] = cosY * cosZ;         // (cosY * cosZ)
    dstMatrix->m[0][1] = cosY * sinZ;         // (cosY * sinZ)
    dstMatrix->m[0][2] = -sinY;               //-(sinY)
    dstMatrix->m[0][3] = 0.0;

    dstMatrix->m[1][0] = opt4 * sinY - opt1;  // (sinX * sinY * cosZ) - (cosX * sinZ)
    dstMatrix->m[1][1] = opt3 * sinY + opt2;  // (sinX * sinY * sinZ) + (cosX * cosZ)
    dstMatrix->m[1][2] = sinX * cosY;         // (sinX * cosY)
    dstMatrix->m[1][3] = 0.0;

    dstMatrix->m[2][0] = opt2 * sinY + opt3;  // (cosX * sinY * cosZ) + (sinX * sinZ)
    dstMatrix->m[2][1] = opt1 * sinY - opt4;  // (cosX * sinY * sinZ) - (sinX * sinZ)
    dstMatrix->m[2][2] = cosX * cosY;         // (cosX * cosY)
    dstMatrix->m[2][3] = 0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;

    return;
}

//------------------------------------------------------------------------------
// 回転行列生成 ( ZXY順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixZXY( nn::util::Matrix4x4fType* dstMatrix, const nn::util::Vector3fType& rotate ) NN_NOEXCEPT
{
    //回転の軸は常にWorld空間に固定されます
    nn::util::Matrix4x4fType rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosY * cosZ;
    float opt2 = cosY * sinZ;
    float opt3 = sinY * cosZ;
    float opt4 = sinY * sinZ;

    dstMatrix->m[0][0] = sinX * opt4 + opt1;  // (sinX * sinY * sinZ) + (cosY * cosZ)
    dstMatrix->m[0][1] = cosX * sinZ;         // (cosX * sinZ)
    dstMatrix->m[0][2] = sinX * opt2 - opt3;  // (sinX * cosY * sinZ) - (sinY * cosZ)
    dstMatrix->m[0][3] = 0.0;

    dstMatrix->m[1][0] = sinX * opt3 - opt2;  // (sinX * sinY * cosZ) - (cosY * sinZ)
    dstMatrix->m[1][1] = cosX * cosZ;         // (cosX * cosZ)
    dstMatrix->m[1][2] = sinX * opt1 + opt4;  // (sinX * cosY * cosZ) + (sinY * sinZ)
    dstMatrix->m[1][3] = 0.0;

    dstMatrix->m[ 2 ][ 0 ] = cosX * sinY;         // (cosX * sinY)
    dstMatrix->m[ 2 ][ 1 ] = -sinX;               //-(sinX)
    dstMatrix->m[ 2 ][ 2 ] = cosX * cosY;         // (cosX * cosY)
    dstMatrix->m[ 2 ][ 3 ] = 0.0;

    dstMatrix->m[ 3 ][ 0 ] = 0.0;
    dstMatrix->m[ 3 ][ 1 ] = 0.0;
    dstMatrix->m[ 3 ][ 2 ] = 0.0;
    dstMatrix->m[ 3 ][ 3 ] = 1.0;

    return;
}
#endif

} // namespace detail
} // namespace vfx
} // namespace nn
