﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_KeyFrameAnim.h>
#include <nn/vfx/vfx_AreaLoop.h>
#include <nn/vfx/vfx_Stripe.h>
#include <nn/vfx/vfx_StripeConnection.h>
#include <nn/vfx/vfx_SuperStripe.h>

namespace nn {
namespace vfx {
namespace detail {

// NxAddon 3.0 で修正された チャイルドエミッタマトリクスの不具合修正をリバートしたい場合は、
// 以下をコメントアウトしてください。
#define _FIXED_CHILD_EMITTER_MATRIX_CALC

//---------------------------------------------------------------------------
//  コンストラクタ。
//---------------------------------------------------------------------------
EmitterCalculator::EmitterCalculator( System* pSystem ) NN_NOEXCEPT
    : m_pSystem( pSystem )
{
    // 共用利用のテクスチャサンプラ（カールノイズ用テクスチャ）
    m_TextureSamplerDataForCurlNoise.filter = TextureFilterMode_Linear;
    m_TextureSamplerDataForCurlNoise.wrapU = TextureWrapMode_Repeat;
    m_TextureSamplerDataForCurlNoise.wrapV = TextureWrapMode_Repeat;
    m_TextureSamplerDataForCurlNoise.isSphereMap = false;
    m_TextureSamplerDataForCurlNoise.mipLevel = 15.99f;
    m_TextureSamplerDataForCurlNoise.mipMapBias = 0.0f;
    m_pTextureSamplerForCurlNoise = TextureSampler::GetSamplerFromTable( &m_TextureSamplerDataForCurlNoise );

    // 共用利用のテクスチャサンプラ（カラー＆デプスバッファ）
    m_TextureSamplerDataForColorAndDepth.filter = TextureFilterMode_Linear;
    m_TextureSamplerDataForColorAndDepth.wrapU = TextureWrapMode_Clamp;
    m_TextureSamplerDataForColorAndDepth.wrapV = TextureWrapMode_Clamp;
    m_TextureSamplerDataForColorAndDepth.isSphereMap = false;
    m_TextureSamplerDataForColorAndDepth.mipLevel = 15.99f;
    m_TextureSamplerDataForColorAndDepth.mipMapBias = 0.0f;
    m_pTextureSamplerForColorAndDepth = TextureSampler::GetSamplerFromTable( &m_TextureSamplerDataForColorAndDepth );
}

//---------------------------------------------------------------------------
//  デストラクタ。
//---------------------------------------------------------------------------
EmitterCalculator::~EmitterCalculator() NN_NOEXCEPT{}


//---------------------------------------------------------------------------
//  エミッタアニメーションを適用
//---------------------------------------------------------------------------
void EmitterCalculator::ApplyEmitterAnimation( Emitter*  pEmitter, nn::util::Matrix4x3fType* pAnimMatrixSrt, nn::util::Matrix4x3fType* pAnimMatrixRt ) NN_NOEXCEPT
{
    // エミッタ時間アニメーションがある場合は、アニメーション * res * set の順で演算
    nn::util::MatrixIdentity( pAnimMatrixSrt );
    nn::util::MatrixIdentity( pAnimMatrixRt );

    nn::util::Vector3fType animScale  = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 1, 1 );
    nn::util::Vector3fType animRotate = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    nn::util::Vector3fType animTrans  = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );

    //if ( emitter->emitterRes->emitterAnimArray[ VFX_EMITTER_ANIM_SCALE ] )
    {
        nn::util::VectorLoad( &animScale, pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Scale ] );
    }
    if( pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ EmitterAnimationType_Rotate ] )
    {
        // Degree -> Radian 変換
        nn::util::VectorSetX( &animRotate, nn::util::DegreeToRadian( pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Rotate ].x ) );
        nn::util::VectorSetY( &animRotate, nn::util::DegreeToRadian( pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Rotate ].y ) );
        nn::util::VectorSetZ( &animRotate, nn::util::DegreeToRadian( pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Rotate ].z ) );
    }
    else
    {
        // 変換無しで受け取る
        nn::util::VectorLoad( &animRotate, pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Rotate ] );
    }
    //if ( emitter->emitterRes->emitterAnimArray[ VFX_EMITTER_ANIM_TRANS ] )
    {
        nn::util::VectorLoad( &animTrans, pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_Translate ] );
    }

    nn::util::Vector3fType tempScale = NN_UTIL_VECTOR_3F_INITIALIZER( 1, 1, 1 );

    MatrixCreateSrtXyz( pAnimMatrixRt, tempScale, animRotate, animTrans );
    MatrixCreateSrtXyz( pAnimMatrixSrt, animScale, animRotate, animTrans );
}

//---------------------------------------------------------------------------
//  エミッタの放出条件を満たしているならパーティクルを放出します。
//---------------------------------------------------------------------------
void EmitterCalculator::TryEmitParticle( float* pOutEmitCount, float* pOutEmitSaving, float* pOutEmitInterval, uint8_t* pOutEmitDone, Emitter* pEmitter, float frameRate, bool forceSearch ) NN_NOEXCEPT
{
    const ResEmitter* pResEmitter = pEmitter->m_pEmitterData;
    const EmitterSet* pEmitterSet = pEmitter->m_pEmitterSet;

    //-------------------------
    // 時間放出
    //-------------------------
    if( !pEmitter->m_pEmitterData->emission.isEmitDistEnabled )
    {
        // 初回時は、emitter->interval=0.0 で必ず放出される。
        if( *pOutEmitCount >= *pOutEmitInterval )
        {
            const float emitCountSub = *pOutEmitCount - *pOutEmitInterval;

            //放出レートランダム
            const float emitRateRandom = pEmitter->m_pEmitterData->emission.rateRandom / 100.0f * pResEmitter->emission.rate;
            const float emitReduction = emitRateRandom * pEmitter->m_Random.GetFloat();

            float emitRatio = pResEmitter->emission.rate;
            // 球等分割は放出レートが固定値なので、アニメーションの加味を行わない。
            if( pEmitter->m_pEmitterData->volume.volumeType != EmitterVolumeType_SphereEquallyDivided &&
                pEmitter->m_pEmitterData->volume.volumeType != EmitterVolumeType_SphereEqually64Divided )
            {
                emitRatio = pEmitter->m_EmitterAnimation.emissionRate;
            }

            *pOutEmitSaving += ( emitRatio - emitReduction ) * pEmitter->m_EmitRatio * pEmitterSet->GetEmissionRatioScale();
            if( *pOutEmitSaving < 0 )
            {
                *pOutEmitSaving = 0;
            }

            int count = static_cast< int >( *pOutEmitSaving );
            if( count != 0 )
            {
                Emit( pOutEmitDone, pEmitter, count, forceSearch );

                // 放出ごとに行う処理
                pEmitter->UpdateByEmit( pOutEmitInterval );
                {
                    // MEMO: 理想的には UpdateByEmit() 内で処理する方がキレイ
                    //       ここで常に Dirty フラグを立てておかないと、Calcの最後で機械的にフラグが降ろされて更新されないので暫定。
                    pEmitter->GetEmitterSet()->m_IsEmitterSrtDirty = 1;
                }

                *pOutEmitSaving -= count;
                *pOutEmitCount = frameRate + emitCountSub;   //放出間隔を通り過ぎた分（emitCntSub）も進める。

                // MEMO: m_EmitLastFrame はエミッタ削除の判定に使うので、寿命のない軽量版チャイルドでは使わない
                pEmitter->m_EmitLastFrame = pEmitter->m_Time;
            }
            else
            {
                *pOutEmitCount = 0;
            }
        }
        else
        {
            *pOutEmitCount += frameRate;
        }
    }
    else
    {
        //-------------------------
        // 距離放出
        //-------------------------
        if( pEmitterSet->m_EnableDistanceBasedEmission )
        {
            // 今回の移動距離
            float length = nn::util::VectorLength( pEmitter->m_EmitterLocalVec );

            // 貯金
            float vessel = *pOutEmitSaving;

            // 今の区間の距離スケーリング
            //float scale = 1.0f;

            // 仮想的な長さ
            float virtualLength = length;

            // 移動距離切り捨ての閾値
            if( virtualLength < pEmitter->m_pEmitterData->emission.emitDistMargin )
            {
                virtualLength = 0.0f;
            }

            if( virtualLength == 0.0f )
            {
                virtualLength = pEmitter->m_pEmitterData->emission.emitDistMin;
            }
            else if( virtualLength < pEmitter->m_pEmitterData->emission.emitDistMin )
            {
                // 最小距離にかかる場合
                virtualLength = length * pEmitter->m_pEmitterData->emission.emitDistMin / length;
            }
            else if( pEmitter->m_pEmitterData->emission.emitDistMax < virtualLength )
            {
                // 最大距離にかかる場合
                virtualLength = length * pEmitter->m_pEmitterData->emission.emitDistMax / length;
            }

            // 今回の区間の加算
            vessel += virtualLength;

            // 今回の放出個数を決定
            int count = 0;
            if( pEmitter->m_pEmitterData->emission.emitDistUnit != 0.0 )
            {
                count = static_cast< int >( vessel / pEmitter->m_pEmitterData->emission.emitDistUnit );
            }

            // 放出処理
            if( count > 0 )
            {
                const nn::util::Vector3fType prevPos = pEmitter->m_EmitterPrevPos;
                nn::util::Vector3fType currPos;
                nn::util::MatrixGetAxisW( &currPos, pEmitter->m_MatrixRt );

                for( int i = 0; i < count; ++i )
                {
                    vessel -= pEmitter->m_pEmitterData->emission.emitDistUnit;

                    // 位置の比率
                    float ratio = 0.0f;
                    if( virtualLength != 0.0f )
                    {
                        ratio = vessel / virtualLength;
                    }
                    // 位置
                    nn::util::Vector3fType pos;
                    nn::util::VectorLerp( &pos, currPos, prevPos, ratio );

                    nn::util::MatrixSetAxisW( &pEmitter->m_MatrixRt, pos );
                    nn::util::MatrixSetAxisW( &pEmitter->m_MatrixSrt, pos );

                    Emit( pOutEmitDone, pEmitter, 1, forceSearch );

                    nn::util::MatrixSetAxisW( &pEmitter->m_MatrixRt, currPos );
                    nn::util::MatrixSetAxisW( &pEmitter->m_MatrixSrt, currPos );
                }

                // MEMO: m_EmitLastFrame はエミッタ削除の判定に使うので、寿命のない軽量版チャイルドでは使わない
                pEmitter->m_EmitLastFrame = pEmitter->m_Time;
            }
            *pOutEmitSaving = vessel;
        }
    }
}

//---------------------------------------------------------------------------
//  EmitterLocalVec の更新処理（2か所で同じ処理をするので関数化）
//---------------------------------------------------------------------------
void EmitterCalculator::UpdateEmitterLocalVec( nn::util::Vector3fType* const pOutEmitterLocalVec, Emitter* const pEmitter ) const NN_NOEXCEPT
{
    nn::util::Vector3fType emitterPos;
    nn::util::Vector3fType emitterLocalPos;
    nn::util::Vector3fType emitterPrevLocalPos;
    nn::util::Matrix4x3fType invMatrix;

    nn::util::MatrixGetAxisW( &emitterPos, pEmitter->m_MatrixSrt );
    nn::util::MatrixInverse( &invMatrix, pEmitter->m_MatrixSrt );

    nn::util::VectorTransform( &emitterLocalPos, emitterPos, invMatrix );
    nn::util::VectorTransform( &emitterPrevLocalPos, pEmitter->m_EmitterPrevPos, invMatrix );

    nn::util::VectorSubtract( pOutEmitterLocalVec, emitterLocalPos, emitterPrevLocalPos );
}

void EmitterCalculator::UpdateCurrentParticleGpuBufferForCpuEmitter( Emitter* pEmitter ) NN_NOEXCEPT
{
    // CPUバッファからGPUバッファへコピー
    ResEmitter* pResEmitter = pEmitter->m_pEmitterData;
    if( pResEmitter->emitter.calcType == EmitterCalculationMode_Cpu )
    {
        detail::ParticleProperty*   pParticlePropertyFront = pEmitter->m_ParticlePropertyFront;
        detail::ParticleProperty*   pParticlePropertyCpu = &pEmitter->m_CpuParticleProperty;

        size_t bufferSize = sizeof( nn::util::Float4 ) * pEmitter->m_ParticleCount;

        memcpy( pParticlePropertyFront->localPos, pParticlePropertyCpu->localPos, bufferSize );
        memcpy( pParticlePropertyFront->localVec, pParticlePropertyCpu->localVec, bufferSize );
        memcpy( pParticlePropertyFront->localDiff, pParticlePropertyCpu->localDiff, bufferSize );
        memcpy( pParticlePropertyFront->random, pParticlePropertyCpu->random, bufferSize );
        memcpy( pParticlePropertyFront->scale, pParticlePropertyCpu->scale, bufferSize );
        memcpy( pParticlePropertyFront->initRotate, pParticlePropertyCpu->initRotate, bufferSize );

        if( pParticlePropertyFront->initColor0 )
        {
            memcpy( pParticlePropertyFront->initColor0, pParticlePropertyCpu->initColor0, bufferSize );
            memcpy( pParticlePropertyFront->initColor1, pParticlePropertyCpu->initColor1, bufferSize );
        }

        if( pParticlePropertyFront->emitterMatrixSrt0 )
        {
            memcpy( pParticlePropertyFront->emitterMatrixSrt0, pParticlePropertyCpu->emitterMatrixSrt0, bufferSize );
            memcpy( pParticlePropertyFront->emitterMatrixSrt1, pParticlePropertyCpu->emitterMatrixSrt1, bufferSize );
            memcpy( pParticlePropertyFront->emitterMatrixSrt2, pParticlePropertyCpu->emitterMatrixSrt2, bufferSize );
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタ行列の更新を反映させる
//---------------------------------------------------------------------------
void EmitterCalculator::UpdateEmitterMatrix( Emitter* pEmitter ) NN_NOEXCEPT
{
    EmitterSet* pEmitterSet = pEmitter->GetEmitterSet();
    if( !pEmitter->m_IsChild )
    {
        // エミッタマトリクス更新
        if( pEmitter->m_pEmitterRes->m_IsEmitterSrtAnim )
        {
            // エミッタ時間アニメーションがある場合は、アニメーション * res * set の順で演算
            nn::util::Matrix4x3fType animMatrixSrt;
            nn::util::Matrix4x3fType animMatrixRt;
            ApplyEmitterAnimation( pEmitter, &animMatrixSrt, &animMatrixRt );

            nn::util::MatrixMultiply( &pEmitter->m_MatrixSrt, animMatrixSrt, pEmitterSet->GetMatrixSrt() );
            nn::util::MatrixMultiply( &pEmitter->m_MatrixRt, animMatrixRt, pEmitterSet->GetMatrixRt() );
        }
        else
        {
            // エミッタアニメーションが一切なく、プログラム側でSRTが修正された場合、SRT行列を更新
            if( pEmitterSet->m_IsEmitterSrtDirty )
            {
                // アニメーション無しの場合、リソースの値を直接参照
                const nn::util::Matrix4x3fType& emitterSetMatrixSrt = pEmitterSet->GetMatrixSrt();
                const nn::util::Matrix4x3fType& emitterSetMatrixRt = pEmitterSet->GetMatrixRt();
                nn::util::MatrixMultiply( &pEmitter->m_MatrixSrt, pEmitter->m_ResMatrixSrt, emitterSetMatrixSrt );
                nn::util::MatrixMultiply( &pEmitter->m_MatrixRt, pEmitter->m_ResMatrixRt, emitterSetMatrixRt );
            }
        }
    }
    else
    {
#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
        /*
        修正版
        */
        // 親パーティクルにチャイルドエミッタ割り当て時
        if( pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
        {
            if( pEmitter->m_pEmitterRes->m_IsEmitterSrtAnim )
            {
                nn::util::Matrix4x3fType animMatrixSrt;
                nn::util::Matrix4x3fType animMatrixRt;
                ApplyEmitterAnimation( pEmitter, &animMatrixSrt, &animMatrixRt );
                pEmitter->m_MatrixSrt = animMatrixSrt;
                pEmitter->m_MatrixRt = animMatrixRt;
            }
            else
            {
                pEmitter->m_MatrixSrt = pEmitter->m_ResMatrixSrt;
                pEmitter->m_MatrixRt = pEmitter->m_ResMatrixRt;
            }

            // 親パーティクルの位置＋エミッタリソースのマトリクス
            nn::util::Vector3fType pos;
            nn::util::MatrixGetAxisW( &pos, pEmitter->m_MatrixSrt );
            nn::util::VectorAdd( &pos, pos, pEmitter->m_ParentParticleWorldPos );
            nn::util::MatrixSetAxisW( &pEmitter->m_MatrixSrt, pos );
            nn::util::MatrixSetAxisW( &pEmitter->m_MatrixRt, pos );

            // 親エミッタの追従タイプによって挙動が異なる
            if( pEmitter->m_pParentEmitter->GetFollowType() != ParticleFollowType_None )
            {
                nn::util::MatrixMultiply( &pEmitter->m_MatrixSrt, pEmitter->m_MatrixSrt, pEmitter->GetParentEmitter()->GetMatrixSrt() );
                nn::util::MatrixMultiply( &pEmitter->m_MatrixRt, pEmitter->m_MatrixRt, pEmitter->GetParentEmitter()->GetMatrixRt() );
            }
        }
        else
        {

            // 軽量版チャイルド の場合
            // MEMO: 共通で一つのエミッタしか持たないため、エミッタ位置調整はできない。
            // 親エミッタ( エミッタセットを含む ) * 子エミッタ
            if( pEmitter->m_pEmitterRes->m_IsEmitterSrtAnim )
            {
                nn::util::Matrix4x3fType animMatrixSrt;
                nn::util::Matrix4x3fType animMatrixRt;
                ApplyEmitterAnimation( pEmitter, &animMatrixSrt, &animMatrixRt );

                pEmitter->m_MatrixSrt = animMatrixSrt;
                pEmitter->m_MatrixRt = animMatrixRt;
            }
            else
            {
                pEmitter->m_MatrixSrt = pEmitter->m_ResMatrixSrt;
                pEmitter->m_MatrixRt = pEmitter->m_ResMatrixRt;
            }

            // 親エミッタの追従タイプによって挙動が異なる
            if( pEmitter->m_pParentEmitter->GetFollowType() != ParticleFollowType_None )
            {
                nn::util::MatrixMultiply( &pEmitter->m_MatrixSrt, pEmitter->m_MatrixSrt, pEmitter->GetParentEmitter()->GetMatrixSrt() );
                nn::util::MatrixMultiply( &pEmitter->m_MatrixRt, pEmitter->m_MatrixRt, pEmitter->GetParentEmitter()->GetMatrixRt() );
            }
        }
#else

        /*
        未修正版
        */
        if( pEmitter->m_pEmitterRes->m_IsEmitterSrtAnim )
        {
            nn::util::Matrix4x3fType animMatrixSrt;
            nn::util::Matrix4x3fType animMatrixRt;
            ApplyEmitterAnimation( pEmitter, &animMatrixSrt, &animMatrixRt );

            pEmitter->m_MatrixSrt = animMatrixSrt;
            pEmitter->m_MatrixRt = animMatrixRt;
        }
        else
        {
            pEmitter->m_MatrixSrt = pEmitter->m_ResMatrixSrt;
            pEmitter->m_MatrixRt = pEmitter->m_ResMatrixRt;
        }

        // 「軽量版チャイルド」の場合
        // MEMO: 共通で一つのエミッタしか持たないため、エミッタ位置調整はできない。
        if( pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
        {
            // 通常チャイルドの場合
            // 親パーティクルの位置＋エミッタリソースのマトリクス
            nn::util::Vector3fType pos;
            nn::util::MatrixGetAxisW( &pos, pEmitter->m_MatrixSrt );
            nn::util::VectorAdd( &pos, pos, pEmitter->m_ParentParticleWorldPos );
            nn::util::MatrixSetAxisW( &pEmitter->m_MatrixSrt, pos );
            nn::util::MatrixSetAxisW( &pEmitter->m_MatrixRt, pos );
        }
#endif
    }

    // エミッタマトリクスセットコールバック
    InvokeEmitterMatrixSetCallback( pEmitter );
}

//---------------------------------------------------------------------------
//  エミッタ計算処理をします。
//---------------------------------------------------------------------------
bool EmitterCalculator::Calculate( Emitter* pEmitter, float frameRate, BufferSwapMode bufferSwapMode, bool isFade, bool isEmission, bool isForceCalc ) NN_NOEXCEPT
{
    ResEmitter* pResEmitter = pEmitter->m_pEmitterData;
    EmitterSet* pEmitterSet = pEmitter->m_pEmitterSet;
    bool        isEmit = true;
    float       parentLife = 0.0f;
    bool        ret = true;

    // 放出開始フレーム
    float emitStart = static_cast< float >( pResEmitter->emission.start );

    // エミッタ時間
    float emitterTime = pEmitter->m_Time;

    // 放出終了フレーム
    float emitEnd = static_cast< float >( pResEmitter->emission.start + pResEmitter->emission.emitDuration );

    isEmit = isEmission;

    // バッファをスワップ
    bool bSwap = pEmitter->SwapBuffer( bufferSwapMode );
    if ( bSwap )
    {
        pEmitter->m_SavingFrameRate = frameRate;
    }
    else
    {
        // 未放出の場合は、SavingFrameRate に貯留しない。
        // 上記条件が無い場合、開始フレーム等で再生開始がずらされた際に、
        // 再生開始時に貯留分だけパーティクルの位置が進んでしまう問題がある。( コンピュートシェーダ時に発生 )
        if ( pEmitter->m_IsEmitDone )
        {
            pEmitter->m_SavingFrameRate += frameRate;
        }
    }

    // 削除予約されている場合は、何も処理せず削除フローに移る
    if ( pEmitter->m_IsKillReservation )
    {
        ret = false;
        goto exit;
    }

    // GPU に送るエミッタ時間を更新
    if( pEmitter->GetEmitterSet()->GetSystem()->IsEnabledPreciseGpuCounterMode() && emitterTime > 0.0f )
    {
        pEmitter->m_GpuTime += frameRate;
    }

    // エミッタ計算処理前コールバック
    {
        EmitterPreCalculateArg arg;
        arg.pEmitter     = pEmitter;
        arg.isBufferSwap = bSwap;
        InvokeEmitterPreCalculateCallback( arg );
    }

    // フレームレート
    pEmitter->m_FrameRate = frameRate;

    // チャイルドエミッタ時の補足処理
    if( pEmitter->m_IsChild )
    {
        NN_SDK_ASSERT_NOT_NULL( pEmitter->m_pParentEmitter->m_pEmitterCalculator );

        if( pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
        {
            // 通常チャイルド
            NN_SDK_ASSERT( pEmitter->m_pParentEmitter->m_EmitterCreateId == pEmitter->m_ParentEmitterCreateId );

            // 放出開始フレーム調整
            parentLife = pEmitter->m_ParentParticleLife;
            emitStart = static_cast< float >( parentLife * ( pResEmitter->emission.emitTiming / 100.f ) );

            // 放出終了フレームは、ワンタイムの場合即時もしくは親パーティクルの死亡時間
            emitEnd = ( pResEmitter->emission.isOneTime ) ?
                ( emitStart + pResEmitter->emission.emitDuration ) : static_cast< float >( pEmitter->m_ParentParticleLife );
        }
    }

    //-------------------------
    // エミッタ処理
    //-------------------------
    // エミッタ時間アニメーションの値を反映
    pEmitter->m_CalculatedEmitterAnimationCount = 0;
    if( pEmitter->m_pEmitterRes->m_IsUseEmitterAnim && !( pEmitter->m_IsChild && !pEmitter->m_pEmitterData->inherit.enableEmitterParticle ) )
    {
        // MEMO: 厳密なGPUカウンタモードを使用している場合は m_GpuTime を使用する
        const float emitterGpuTime =
            pEmitter->GetEmitterSet()->GetSystem()->IsEnabledPreciseGpuCounterMode() ? pEmitter->m_GpuTime : pEmitter->m_Time;

        for( int i = 0; i < EmitterAnimationType_MaxAnimationType; i++ )
        {
            if( pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ] &&
                pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->enable &&
                ( !pEmitter->m_EmitterAnimationEndFlag[ i ] || pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->loop ) )
            {
                // 現時間のエミッタ時間アニメーションの値を計算。
                CalculateEmitterKeyFrameAnimation( &pEmitter->m_EmitterAnimationVec[ i ],
                    &pEmitter->m_EmitterAnimationEndFlag[ i ],
                    pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ],
                    emitterGpuTime );
                pEmitter->m_CalculatedEmitterAnimationCount++;
            }
        }

        // 更に乗算が必要なエミッタアニメーションについて個別処理（※エミッタアニメーションの更新が入った時のみ）
        {
            int i = 0;
            i = EmitterAnimationType_AllDirectionalVelocity;
            if( pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ] &&
                pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->enable &&
                ( !pEmitter->m_EmitterAnimationEndFlag[ i ] || pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->loop ) )
            {
                // 全方向初速のスケールを乗算
                pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_AllDirectionalVelocity ].x *= pEmitterSet->GetAllDirectionalVel();
            }
        }
        {
            int i = 0;
            i = EmitterAnimationType_EmitterVolumeScale;
            if( pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ] &&
                pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->enable &&
                ( !pEmitter->m_EmitterAnimationEndFlag[ i ] || pEmitter->m_pEmitterRes->m_EmitterAnimationArray[ i ]->loop ) )
            {
                // エミッタ形状のスケールを乗算
                const nn::util::Vector3fType& runtimeVolumeScale = pEmitterSet->GetEmitterVolumeScale();
                pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].x *= nn::util::VectorGetX( runtimeVolumeScale );
                pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].y *= nn::util::VectorGetY( runtimeVolumeScale );
                pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].z *= nn::util::VectorGetZ( runtimeVolumeScale );
            }
        }
    }
    else
    {
        // エミッタ時間アニメーションなし

        // 全方向初速のスケールを乗算
        pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_AllDirectionalVelocity ].x = pEmitter->m_pEmitterRes->m_pResEmitter->ptclVel.allDirection *
            pEmitterSet->GetAllDirectionalVel();

        // エミッタ形状のスケールを乗算
        const nn::util::Float3& pVolumeScale = pEmitter->m_pEmitterRes->m_pResEmitter->volume.volumeFormScale;
        const nn::util::Vector3fType& runtimeVolumeScale = pEmitterSet->GetEmitterVolumeScale();
        pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].x = pVolumeScale.x * nn::util::VectorGetX( runtimeVolumeScale );
        pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].y = pVolumeScale.y * nn::util::VectorGetY( runtimeVolumeScale );
        pEmitter->m_EmitterAnimationVec[ EmitterAnimationType_EmitterVolumeScale ].z = pVolumeScale.z * nn::util::VectorGetZ( runtimeVolumeScale );
    }

    // エミッタアニメーションの値を補正
    if( pEmitter->m_CalculatedEmitterAnimationCount > 0 )
    {
        // 放出個数が変化する系は、データに埋め込まれた値以上に増えないように補正
        if( pEmitter->m_EmitterAnimation.particleLifeVec.x > static_cast< float >( pEmitter->m_pEmitterRes->m_pResEmitter->ptcl.life ) )
        {
            pEmitter->m_EmitterAnimation.particleLifeVec.x = static_cast< float >( pEmitter->m_pEmitterRes->m_pResEmitter->ptcl.life );
        }
    }

    // エミッタマトリクス更新
    UpdateEmitterMatrix( pEmitter );

    if( emitterTime == 0.0f )
    {
        // MEMO: Time == 0.0f のときだけ、今フレームのエミッタ位置を再取得する
        //       生成直後に SetPos した場合に値が反映されない問題の対応。
        //       このタイミングのため、エミッタ計算前コールバックでは正しい値が取れないのに注意。
        nn::util::MatrixGetAxisW( &pEmitter->m_EmitterPrevPos, pEmitter->m_MatrixSrt );
    }

    // エミッタ移動距離差分を計算
    if( pEmitter->m_Time != 0.0f )
    {
        UpdateEmitterLocalVec( &pEmitter->m_EmitterLocalVec, pEmitter );
    }

    // フェードイン処理
    // MEMO: 「フェードイン中」という状態は持たない
    {
        // αフェードインもしくはスケールフェードインの指定がある場合のみ処理
        if( pResEmitter->emitter.isAlphaFadeIn || pResEmitter->emitter.isScaleFadeIn )
        {
            if( pEmitter->m_FadeInAlphaValue < 1.0f )
            {
                const int fadeTime = pResEmitter->emitter.fadeInTime;
                if( fadeTime > 0 )
                {
                    // 初期値 0.0 なので、徐々に 1.0 まで加算していく
                    pEmitter->m_FadeInAlphaValue += frameRate / fadeTime;
                    if( pEmitter->m_FadeInAlphaValue > 1.0f )
                    {
                        pEmitter->m_FadeInAlphaValue = 1.0f;
                    }
                }
                else
                {
                    // フェード時間 0（データとしてはあまり好ましくない）
                    pEmitter->m_FadeInAlphaValue = 1.0f;
                }
            }
        }
    }

    // フェードアウト指定
    if( isFade || pEmitter->m_IsSoloFade )
    {
        // 「放出停止」の指定がある場合放出を止める
        if( pResEmitter->emitter.isFadeEmit )
        {
            isEmit = false;
        }

        // フェードアウト用のα値を計算する
        if( pResEmitter->emitter.isFadeAlphaFade || pResEmitter->emitter.isScaleFade )
        {
            const int fadeTime = pResEmitter->emitter.alphaFadeTime;
            if( fadeTime <= 0.0f )
            {
                pEmitter->m_FadeOutAlphaValue = 0;
                ret = false;
                goto exit;
            }

            // 初期値 1.0 なので、徐々に 0.0 まで減算していく
            pEmitter->m_FadeOutAlphaValue -= frameRate / fadeTime;
            if( pEmitter->m_FadeOutAlphaValue < 0.0f )
            {
                pEmitter->m_FadeOutAlphaValue = 0;
                ret = false;
                goto exit;
            }
        }
    }

    // 放出処理の頭にサーチ無効フラグをリセット
    pEmitter->m_AbortEmitSearch = false;

    // 手動放出分
    // MEMO: チャイルドでは使えない
    if( !pEmitter->m_IsChild && pEmitterSet->m_IsManualEmission && pEmitterSet->m_ManualEmissionCount > 0 )
    {
        pEmitter->m_EmitLastFrame = pEmitter->m_Time;

        // 一度の放出で何粒出るかの係数（マニュアル放出で、予約リストのインデックスを動かさないために使う）
        const float emitRateRandom = pEmitter->m_pEmitterData->emission.rateRandom / 100.0f * pEmitter->m_pEmitterData->emission.rate;
        const float emitReduction = emitRateRandom * pEmitter->m_Random.GetFloat();

        for( int i = 0; i < pEmitterSet->m_ManualEmissionCount; ++i )
        {
            const int emitNum = static_cast< int >(
                ( pEmitter->m_pEmitterData->emission.rate - emitReduction ) * pEmitter->m_EmitRatio * pEmitterSet->m_pEmitReservationListHead[ i ].emissionRatio );

            // m_ManualEmissionCount 回だけ放出のトリガを引く
            // 一度の放出トリガで何粒出るかは emitNum に記録。一度に出る粒は同じ予約リストの要素を使う。
            Emit( &pEmitter->m_IsEmitDone, pEmitter, emitNum, false, &pEmitterSet->m_pEmitReservationListHead[ i ] );
        }
    }

    // 放出処理
    if( isEmit && pEmitter->m_IsEnabledEmit )
    {
        if( pEmitter->m_IsChild && !pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
        {
            // 軽量版チャイルドの場合は親エミッタの CalcParticle() の中で放出処理を行うのでスキップ
        }
        else
        {
            if( emitStart <= emitterTime )
            {
                // ワンタイム or チャイルドの場合は放出区間がある
                if( ( pResEmitter->emission.isOneTime || pEmitter->m_IsChild ) && emitEnd <= emitterTime && pEmitter->m_IsEmitDone )
                {
                    goto exit_emit;
                }

                // 共通エミッタの持つ放出に関するデータ
                // 粒の持つ位置と時間の情報を集める
                float* pEmitCount    = &pEmitter->m_EmitCount;      // 放出カウンタ（[0, 放出間隔]）
                float* pEmitSaving   = &pEmitter->m_EmitSaving;     // 放出貯蓄
                float* pEmitInterval = &pEmitter->m_EmitInterval;   // 放出間隔
                uint8_t* pEmitDone   = &pEmitter->m_IsEmitDone;     // 放出終了フラグ

                // 放出処理を行う（※コードは通常の放出処理と揃える）
                TryEmitParticle( pEmitCount, pEmitSaving, pEmitInterval, pEmitDone, pEmitter, frameRate );
            }
        }
    }

exit_emit:

    // 子エミッタに軽量版チャイルドがいる場合の特殊な更新処理。
    // MEMO: CalculateParticle() を回す前に記憶する必要があるのでこの位置で実行する。
    if( pEmitter->m_pEmitterRes->m_ChildEmitterResCount > 0 )
    {
        // 「軽量版チャイルド」を持つ場合、配下の粒の時間も進めてやる
        // MEMO: 現状はチャイルドを持っていたら一律で更新
        for( int i = 0; i < pEmitter->m_ParticleCount; ++i )
        {
            // 親パーティクルの情報
            // MEMO: 現状親エミッタは必ず CPU エミッタなので、m_CpuParticleProperty を取ってOK
            detail::ParticleData* pParticleData = &pEmitter->m_pParticleData[ i ];
            detail::ParentParticleData* pParentParticleData = &pEmitter->m_pParentParticleData[ i ];
            if( pParticleData->createId == 0 )
            {
                // 粒は既に消えているので次の粒へ
                continue;
            }
            pParentParticleData->lightChildEmitterTime += frameRate;
        }
    }

    // CPUパーティクルパーティクル挙動計算を行う。
    if( pResEmitter->emitter.calcType == EmitterCalculationMode_Cpu || ( isForceCalc && pResEmitter->emitter.calcType == EmitterCalculationMode_GpuStreamOut ) )
    {
        pEmitter->m_pEmitterCalculator->CalculateParticle( pEmitter );
    }
    else
    {
        pEmitter->m_ParticleProcessingCount = pEmitter->m_ParticleCount;
    }

    // エミッタ計算処理後コールバック
    InvokeEmitterPostCalculateCallback( pEmitter );

    // emitter->particleAttrFill を調整
    // MEMO: 現状、無用な最適化になっているので除去
    //if( pEmitter->m_pEmitterData->ptcl.lifeRandom != 0 && pEmitter->m_ParticleAttrFillIndex > 0 )
    //{
    //    for( int j = pEmitter->m_ParticleAttrFillIndex - 1; j > 0; j-- )
    //    {
    //        float time = pEmitter->m_pParticleData[j].GetTime( pEmitter->m_Time );
    //        float life = pEmitter->m_pParticleData[j].GetLife();

    //        // 死亡チェック
    //        if( time <= life )
    //        {
    //            break;
    //        }

    //        pEmitter->m_ParticleAttrFillIndex = j;
    //        pEmitter->m_ParticleCount         = pEmitter->m_ParticleAttrFillIndex;
    //    }
    //}

    // エミッタ動的定数バッファ生成
    if( pEmitter->m_ParticleCount > 0 || pEmitter->m_pEmitterRes->m_pEmitterPluginData )
    {
        MakeDynamicConstantBuffer( pEmitter, pEmitter->m_FrameRate, pEmitter->m_SavingFrameRate );
    }

    // 今フレームのエミッタ位置を保存
    nn::util::MatrixGetAxisW( &pEmitter->m_EmitterPrevPos, pEmitter->m_MatrixSrt );

    // タイムを進める
    pEmitter->m_Time += frameRate;

    // エミッタ死亡判定
    // TODO : パーティクル放出時に最大エミッタ寿命を計算しておく。
    if( pEmitter->m_IsChild && !pEmitter->m_pEmitterData->inherit.enableEmitterParticle )
    {
        // 「軽量版チャイルド」の場合、親エミッタが死ぬまで死なない
        // 親エミッタが死亡＆子パーティクルがすべて死亡済みならここで false を返す
        if( pEmitter->m_LastCalculateResult == static_cast< uint8_t >( EmitterCalculationResult_Kill ) &&
            emitterTime > emitStart + pEmitter->m_EmitLastFrame + pResEmitter->ptcl.life )
        {
            ret = false;
            goto exit;
        }
    }
    else
    {
        // ワンタイム && マニュアル放出無し
        if( pResEmitter->emission.isOneTime && pEmitterSet->m_IsManualEmission == false )
        {
            // 寿命が有限（※寿命無限の場合はエミッタを終了させない）
            if( pResEmitter->ptcl.isLifeInfinity != 1 )
            {
                float endTime = emitEnd + pResEmitter->ptcl.life;

                if( pResEmitter->emitter.isFadeAlphaFade || pResEmitter->emitter.isScaleFade )
                {
                    endTime += pResEmitter->emitter.alphaFadeTime;
                }

                // エミッタプラグインを持つ場合、ワンタイム時に追加で待つ時間があるかどうか確認する
                if( pEmitter->m_pEmitterRes->m_EmitterPluginIndex > 0 )
                {
                    // MEMO: 最終的にはこの辺はインターフェースでまとめられると良い
                    switch( pEmitter->m_pEmitterRes->m_EmitterPluginIndex )
                    {
                        // MEMO:
                        // 連結式ストライプ・範囲内ループは何もしない
                    case detail::AreaLoopSystem::PluginId:
                    case detail::ConnectionStripeSystem::PluginId:
                        break;
                    case detail::StripeSystem::PluginId:
                        endTime += detail::StripeSystem::GetExtendedEndTimeForOneTimeEmitter( pEmitter );
                        break;
                    case detail::SuperStripeSystem::PluginId:
                        endTime += detail::SuperStripeSystem::GetExtendedEndTimeForOneTimeEmitter( pEmitter );
                        break;
                    default:
                        break;
                    }
                }

                if( emitterTime > endTime )
                {
                    // MEMO: ワンタイム x 通常チャイルドだと、粒は生きているがエミッタは死んでいるケースがここで発生する
                    ret = false;
                    goto exit;
                }
            }
        }
        else
        {
            if( isFade || pEmitter->m_IsSoloFade || pEmitter->m_IsChild )
            {
                // 通常チャイルド
                if( emitterTime > emitStart + pEmitter->m_EmitLastFrame + pResEmitter->ptcl.life )
                {
                    ret = false;
                    goto exit;
                }
            }

            if ( pEmitterSet->m_IsManualEmission )
            {
                // 非常駐型のマニュアル放出エミッタで、指定時間を経過して粒の数が0なら消える
                if ( pEmitter->IsManualEmitterReadyToExit() )
                {
                    return false;
                }
            }
        }
    }

exit:

    return ret;
}// NOLINT(readability/fn_size)



//---------------------------------------------------------------------------
//  パーティクル計算処理をします。
//---------------------------------------------------------------------------
void EmitterCalculator::CalculateParticle( Emitter* pEmitter ) NN_NOEXCEPT
{
    // パーティクルコールバック
    ParticleRemoveCallback particleRemoveCallbackEP = NULL;
    ParticleRemoveCallback particleRemoveCallbackCA = NULL;
    ParticleRemoveCallback particleRemoveCallbackCS = NULL;
    ParticleCalculateCallback   particleCalculateCallbackEP = NULL;
    ParticleCalculateCallback   particleCalculateCallbackCA = NULL;
    ParticleCalculateCallback   particleCalculateCallbackCS = NULL;
    bool                   callParticleRemoveCallback = false;
    bool                   callParticleCalculateCallback = false;

    if( pEmitter->m_pCallbackSet[ CallbackSetType_EmitterPlugin ] )
    {
        particleRemoveCallbackEP = pEmitter->m_pCallbackSet[ CallbackSetType_EmitterPlugin ]->particleRemove;
        particleCalculateCallbackEP = pEmitter->m_pCallbackSet[ CallbackSetType_EmitterPlugin ]->particleCalculate;
    }
    if( pEmitter->m_pCallbackSet[ CallbackSetType_CustomAction ] )
    {
        particleRemoveCallbackCA = pEmitter->m_pCallbackSet[ CallbackSetType_CustomAction ]->particleRemove;
        particleCalculateCallbackCA = pEmitter->m_pCallbackSet[ CallbackSetType_CustomAction ]->particleCalculate;
    }
    if( pEmitter->m_pCallbackSet[ CallbackSetType_CustomShader ] )
    {
        particleRemoveCallbackCS = pEmitter->m_pCallbackSet[ CallbackSetType_CustomShader ]->particleRemove;
        particleCalculateCallbackCS = pEmitter->m_pCallbackSet[ CallbackSetType_CustomShader ]->particleCalculate;
    }

    if( particleRemoveCallbackEP || particleRemoveCallbackCA || particleRemoveCallbackCS )
    {
        callParticleRemoveCallback = true;
    }
    if( particleCalculateCallbackEP || particleCalculateCallbackCA || particleCalculateCallbackCS )
    {
        callParticleCalculateCallback = true;
    }

    int processCount = 0;
    int processEnd = 0;
    NN_UNUSED( processEnd );

    for( int i = 0; i < pEmitter->m_ParticleCount; i++ )
    {
        detail::ParticleData*       pParticleData        = &pEmitter->m_pParticleData[ i ];
        detail::ParentParticleData* pParentParticleData  = &pEmitter->m_pParentParticleData[ i ];
        detail::ParticleProperty*   pCpuParticleProperty = &pEmitter->m_CpuParticleProperty;

        if( pParticleData->createId == 0 )
        {
            continue;
        }

        // ワンエミッタ向け対応。
        float time = pParticleData->GetTime( pEmitter->m_Time );
        float life = pParticleData->GetLife();

        // パーティクル削除チェック
        if( time >= life )
        {
            // パーティクル削除コールバック
            if( callParticleRemoveCallback )
            {
                InvokeParticleRemoveCallback( pEmitter, i, time, life, particleRemoveCallbackEP, particleRemoveCallbackCA, particleRemoveCallbackCS );
            }

            // 削除をマーキング
            pParticleData->createId = 0;
        }
        else
        {
            // 更新前の位置を保存
            nn::util::Vector3fType oldPos;
            nn::util::VectorLoad( &oldPos, pCpuParticleProperty->localPos[ i ].v );

            // 通常パーティクル挙動
            float birthTime     = pParticleData->createTime;
            float particleLife  = pParticleData->GetLife();

            nn::util::Vector3fType localPos;
            nn::util::VectorLoad( &localPos, pCpuParticleProperty->localPos[ i ].v );
            nn::util::Vector3fType localVec;
            nn::util::VectorLoad( &localVec, pCpuParticleProperty->localVec[ i ].v );

            CalculateParticleBehavior( &localPos, &localVec, &birthTime, &particleLife, pEmitter, i, time, localPos, localVec );

            nn::util::VectorStore( pCpuParticleProperty->localPos[ i ].v, localPos );
            nn::util::VectorStore( pCpuParticleProperty->localVec[ i ].v, localVec );

            pParticleData->createTime   = birthTime;
            pParticleData->life         = particleLife;

            // パーティクル計算コールバック
            if( callParticleCalculateCallback )
            {
                InvokeParticleCalculateCallback( pEmitter, i, time, life, particleCalculateCallbackEP, particleCalculateCallbackCA, particleCalculateCallbackCS );
            }

            pCpuParticleProperty->localPos[i].w = pParticleData->life;
            pCpuParticleProperty->localVec[i].w = pParticleData->createTime;

            // パーティクル移動差分を計算
            // NXAddon 5.4.Xにて 低フレームレート向けにPOSDIFF_MIN_VELの許容範囲を低く調整。
            const float POSDIFF_MIN_VEL = 0.0001f;
            nn::util::Vector3fType vel;
            nn::util::VectorSubtract( &vel, localPos, oldPos );

            if ( ::std::fabsf( nn::util::VectorGetX( vel ) ) > POSDIFF_MIN_VEL ||
                 ::std::fabsf( nn::util::VectorGetY( vel ) ) > POSDIFF_MIN_VEL ||
                 ::std::fabsf( nn::util::VectorGetZ( vel ) ) > POSDIFF_MIN_VEL )
            {
                nn::util::VectorStore( pCpuParticleProperty->localDiff[ i ].v, vel );
            }

            // 軽量版チャイルドの場合、ここで放出処理を行う
            // 親エミッタの粒の更新処理中なので、ParticleData にデータがあったら処理を行う
            for( int k = 0; k < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; ++k )
            {
                Emitter* pChildEmitter = pParentParticleData->pChildEmitter[ k ];
                if( pChildEmitter && !pChildEmitter->IsAlive() )
                {
                    // 通常チャイルド x ワンタイムのケースで、先にエミッタが死んでいるケースをここで弾く
                    continue;
                }

                const float childTime = pParentParticleData->lightChildEmitterTime;
                if( pChildEmitter == NULL )
                {
                    // チャイルドエミッタがもうないなら抜ける
                    break;
                }
                else
                {
                    detail::ResEmitter* pChildEmitterRes = pChildEmitter->m_pEmitterData;
                    if( !pChildEmitterRes->inherit.enableEmitterParticle )
                    {
                        if( pParticleData->createId == 0 )
                        {
                            // 粒は既に消えているので次の粒へ
                            // もしくは距離放出の初回処理は、まだ適切にデータが行きわたって無いのでスキップ
                            continue;
                        }

                        //---------------------------------------------------------------------------
                        // MEMO:「軽量版チャイルド」の場合、0フレーム目に初回放出を行う場合、この時点での処理の流れは、
                        //
                        //      - 親エミッタの Calc()         ← さっき済んだ
                        //      - 子パーティクルの Emit()     ← ★今ココ
                        //      - （共通）子エミッタの Calc() ← まだ処理されてない！
                        //
                        //      なので、「子エミッタの Calc() がまだ走っていない」ため pEmitter->m_MatrixSrt が初期値になっている。
                        //      その結果、「エミッタに追従しない」を選んだ場合に粒の初期位置が子エミッタの位置に揃わない。
                        //      これを解消するため、初回の場合はここで一度計算を回しておく。
                        //---------------------------------------------------------------------------
                        if( pChildEmitter->m_Time == 0 && pChildEmitterRes->inherit.enableEmitterParticle == 0 )
                        {
                            if( pChildEmitter->m_pEmitterRes->m_IsEmitterSrtAnim )
                            {
                                nn::util::Matrix4x3fType animMatrixSrt;
                                nn::util::Matrix4x3fType animMatrixRt;
                                ApplyEmitterAnimation( pChildEmitter, &animMatrixSrt, &animMatrixRt );

                                pChildEmitter->m_MatrixSrt = animMatrixSrt;
                                pChildEmitter->m_MatrixRt = animMatrixRt;
                            }
                            else
                            {
                                pChildEmitter->m_MatrixSrt = pChildEmitter->m_ResMatrixSrt;
                                pChildEmitter->m_MatrixRt = pChildEmitter->m_ResMatrixRt;
                            }
                        }

                        // 放出開始フレーム調整
                        float parentLife = pParticleData->GetLife();
                        float emitStart = static_cast< float >( parentLife * ( pChildEmitterRes->emission.emitTiming / 100.f ) );

                        if( emitStart <= childTime )
                        {
                            // この粒がワンタイム放出 or 放出時間を終えたか調査
                            uint8_t* pEmitDone = &pParentParticleData->lightChildEmitDone[ k ];      // 放出終了フラグ
                            float emitEnd = ( pChildEmitterRes->emission.isOneTime ) ? ( emitStart + pChildEmitterRes->emission.emitDuration ) : parentLife;

                            if( ( pChildEmitterRes->emission.isOneTime || pChildEmitter->m_IsChild ) && emitEnd <= childTime && *pEmitDone )
                            {
                                // 放出すべき期間を終えているので次の粒へ
                                continue;
                            }

                            // 粒の位置をエミッタSRTを加味して一時領域に保存
                            nn::util::Vector3fType parentParticlePos;
                            nn::util::Vector3fType parentParticleVec;
                            nn::util::VectorLoad( &parentParticlePos, pCpuParticleProperty->localPos[ i ].v );
                            nn::util::VectorLoad( &parentParticleVec, pCpuParticleProperty->localVec[ i ].v );
#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
                            // 親エミッタが追従無し設定の場合は、このタイミングで親パーティクルが持つ追従無しエミッタマトリクスを乗算する
                            if ( pEmitter->GetFollowType() == ParticleFollowType_None )
                            {
                                pEmitter->TransformToWorldPos( &parentParticlePos, parentParticlePos, i );
                                pEmitter->TransformToWorldVec( &parentParticleVec, parentParticleVec, i );
                            }
#else
                            pEmitter->TransformToWorldPos( &parentParticlePos, parentParticlePos, i );
                            pEmitter->TransformToWorldVec( &parentParticleVec, parentParticleVec, i );
#endif
                            pChildEmitter->m_pChildEmitterTempWorldPos = &parentParticlePos;
                            pChildEmitter->m_pChildEmitterTempWorldVec = &parentParticleVec;

                            // 子エミッタである粒のインデックスを一時的に記憶
                            pChildEmitter->m_ChildEmitterTempParentIndex = i;

                            // エミッタ移動距離差分を計算（距離放出を行う場合必要）
                            if( pChildEmitterRes->emission.isEmitDistEnabled && childTime != 0.0f )
                            {
                                // 親パーティクルのdiffが既に計算済み＋エミッタのdiffも計算済みなのでそれの足し算では？？？
                                nn::util::Vector3fType parentParticleDiff;
                                nn::util::VectorLoad( &parentParticleDiff, pCpuParticleProperty->localDiff[ i ].v );
                                pChildEmitter->m_EmitterLocalVec = parentParticleDiff;
                            }

                            // 放出処理を行う（※コードは通常の放出処理と一本化）
                            {
                                // 共通エミッタの持つ放出に関するデータ
                                float* pEmitCount       = &pParentParticleData->lightChildEmitCount[ k ];       // 放出カウンタ（[0, 放出間隔]）
                                float* pEmitSaving      = &pParentParticleData->lightChildEmitSaving[ k ];      // 放出貯蓄
                                float* pEmitInterval    = &pParentParticleData->lightChildEmitInterval[ k ];    // 放出間隔
                                const bool forceSearch  = false;

                                // MEMO: 本来は pChildEmitter->m_FrameRate から取得するべきだが、現状親子でフレームレートは同一であること、
                                //       また一時停止からの再開時（m_FrameRate=0.0f とされている）に、子エミッタの m_FrameRate の更新が来る前にこの処理に来るので、
                                //       今のところは親のフレームレートをそのまま流し込む。
                                float frameRate = pEmitter->m_FrameRate;
#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
                                // 放出前に pChildEmitter のエミッタマトリクスを更新する
                                // MEMO : このタイミングでは放出予約のみ行い、pChildEmitter の放出処理で放出すべき
                                if ( pChildEmitter->GetFollowType() == ParticleFollowType_None &&
                                     pEmitter->GetFollowType()      != ParticleFollowType_None )
                                {
                                    pChildEmitter->m_MatrixSrt = pChildEmitter->m_ResMatrixSrt;
                                    pChildEmitter->m_MatrixRt = pChildEmitter->m_ResMatrixRt;
                                    nn::util::MatrixMultiply( &pChildEmitter->m_MatrixSrt, pChildEmitter->m_MatrixSrt, pEmitter->GetMatrixSrt() );
                                    nn::util::MatrixMultiply( &pChildEmitter->m_MatrixRt, pChildEmitter->m_MatrixRt, pEmitter->GetMatrixRt() );
                                }
#endif
                                pChildEmitter->m_pEmitterCalculator->TryEmitParticle( pEmitCount, pEmitSaving, pEmitInterval, pEmitDone, pChildEmitter, frameRate, forceSearch );
                            }

                            // Temp領域の内容は、不要になった時点で消しておく
                            pChildEmitter->m_ChildEmitterTempParentIndex = -1;
                            pChildEmitter->m_pChildEmitterTempWorldPos = NULL;
                            pChildEmitter->m_pChildEmitterTempWorldVec = NULL;
                        }
                    }
                }
            }

            processEnd = i;
            processCount++;
        }

        // 子エミッタに対して 位置/速度 を伝える
        // この処理はもうPosAttrのフロントバッファが更新済みなので前から取る。
        // 親パーティクルにチャイルドエミッタ割り当て時に有効
        if( pEmitter->m_pChildEmitterResSet[ 0 ] && pParticleData->createId != 0 )
        {
            for( int j = 0; j < pEmitter->m_pEmitterRes->m_ChildEmitterResCount; j++ )
            {
                Emitter* pChildEmitter = pParentParticleData->pChildEmitter[ j ];
                if( !pChildEmitter )
                {
                    continue;
                }

#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
                if ( !pEmitter->m_pChildEmitterResSet[ j ]->m_pResEmitter->inherit.enableEmitterParticle )
                {
                    continue;
                }
#endif

                // 子エミッタが生存しているか
                if( pChildEmitter->IsAlive() && pChildEmitter->m_ParentEmitterCreateId == pEmitter->m_EmitterCreateId )
                {
                    // パーティクルの時間を子エミッタに伝達
                    pChildEmitter->m_ParentParticleTime = pParticleData->GetTime( pEmitter->m_Time );
                    pChildEmitter->m_ParentParticleLife = pParticleData->GetLife();

                    // パーティクルの位置を子エミッタに伝達
                    if( pEmitter->GetFollowType() == ParticleFollowType_EmitterFull )
                    {
                        nn::util::Vector3fType localPos;
                        nn::util::VectorLoad( &localPos, pEmitter->m_CpuParticleProperty.localPos[ i ].v );
#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
                        if ( pEmitter->GetFollowType() == ParticleFollowType_None )
                        {
                            nn::util::VectorTransform( &pChildEmitter->m_ParentParticleWorldPos, localPos, pEmitter->m_MatrixSrt );
                        }
                        else
                        {
                            nn::util::VectorLoad( &pChildEmitter->m_ParentParticleWorldPos, pEmitter->m_CpuParticleProperty.localPos[ i ].v );
                        }
#else
                        nn::util::VectorTransform( &pChildEmitter->m_ParentParticleWorldPos, localPos, pEmitter->m_MatrixSrt );
#endif
                    }
                    else
                    {
                        nn::util::Matrix4x3fType eMatrix;
                        nn::util::Vector3fType localPos;

                        pEmitter->GetParticleEmitterMatrixSrt( &eMatrix, i );
                        nn::util::VectorLoad( &localPos, pEmitter->m_CpuParticleProperty.localPos[ i ].v );
#ifdef _FIXED_CHILD_EMITTER_MATRIX_CALC
                        if ( pEmitter->GetFollowType() == ParticleFollowType_None )
                        {
                            nn::util::VectorTransform( &pChildEmitter->m_ParentParticleWorldPos, localPos, eMatrix );
                        }
                        else
                        {
                            nn::util::VectorLoad( &pChildEmitter->m_ParentParticleWorldPos, pEmitter->m_CpuParticleProperty.localPos[ i ].v );
                        }
#else
                        nn::util::VectorTransform( &pChildEmitter->m_ParentParticleWorldPos, localPos, eMatrix );
#endif
                    }

                    // パーティクルの速度を子エミッタに伝達
                    {
                        // Local系 -> World系に変換して渡す
                        nn::util::Vector3fType localVec;
                        nn::util::VectorLoad( &localVec, pEmitter->m_CpuParticleProperty.localVec[ i ].v );
                        nn::util::VectorTransformNormal( &pChildEmitter->m_ParentParticleWorldVec, localVec, pEmitter->m_MatrixRt );
                    }
                }
            }
        }
    }

    if( processCount == 0 )
    {
        pEmitter->m_ParticleCount           = 0;
        pEmitter->m_ParticleAttrFillIndex   = 0;
    }

    pEmitter->m_ParticleProcessingCount = processCount;

    // CPUバッファからGPUバッファへコピー
    UpdateCurrentParticleGpuBufferForCpuEmitter( pEmitter );
}// NOLINT(readability/fn_size)


//---------------------------------------------------------------------------
//  エミッタの動的定数バッファを生成します。
//---------------------------------------------------------------------------
void EmitterCalculator::MakeDynamicConstantBuffer( Emitter* pEmitter, float frameRate, float savingFrameRate ) NN_NOEXCEPT
{
    EmitterDynamicConstantBuffer* pDynamicConstantBuffer = reinterpret_cast<EmitterDynamicConstantBuffer*>(
                                                pEmitter->m_GfxObjects.m_DynamicConstantBufferAddr[pEmitter->m_BufferSide] );
    NN_SDK_ASSERT_NOT_NULL( pDynamicConstantBuffer );

    EmitterSet* pEmitterSet = pEmitter->m_pEmitterSet;

    const nn::util::Vector4fType& emitterSetColor = pEmitterSet->GetColor();

    nn::util::Vector4fType color0 = NN_UTIL_VECTOR_4F_INITIALIZER(
        nn::util::VectorGetX( pEmitter->m_Color0 ) * pEmitter->m_EmitterAnimation.color0.x * nn::util::VectorGetX( emitterSetColor ),
        nn::util::VectorGetY( pEmitter->m_Color0 ) * pEmitter->m_EmitterAnimation.color0.y * nn::util::VectorGetY( emitterSetColor ),
        nn::util::VectorGetZ( pEmitter->m_Color0 ) * pEmitter->m_EmitterAnimation.color0.z * nn::util::VectorGetZ( emitterSetColor ),
        nn::util::VectorGetW( pEmitter->m_Color0 ) * pEmitter->m_EmitterAnimation.alpha0.x );

    nn::util::Vector4fType color1 = NN_UTIL_VECTOR_4F_INITIALIZER(
        nn::util::VectorGetX( pEmitter->m_Color1 ) * pEmitter->m_EmitterAnimation.color1.x * nn::util::VectorGetX( emitterSetColor ),
        nn::util::VectorGetY( pEmitter->m_Color1 ) * pEmitter->m_EmitterAnimation.color1.y * nn::util::VectorGetY( emitterSetColor ),
        nn::util::VectorGetZ( pEmitter->m_Color1 ) * pEmitter->m_EmitterAnimation.color1.z * nn::util::VectorGetZ( emitterSetColor ),
        nn::util::VectorGetW( pEmitter->m_Color1 ) * pEmitter->m_EmitterAnimation.alpha1.x );

    // 子エミッタで親の値を毎フレーム継承する場合、ここで親エミッタのα値を乗算して継承。
    if( pEmitter->m_IsChild )
    {
        const Emitter* pParent = pEmitter->m_pParentEmitter;
        if( pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha0 && pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha0EachFrame )
        {
            const float parentAlpha = pParent->GetEmitterAnimValue().alpha0.x;
            nn::util::VectorSetW( &color0, nn::util::VectorGetW( color0 ) * parentAlpha );
        }
        if( pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha1 && pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha1EachFrame )
        {
            const float parentAlpha = pParent->GetEmitterAnimValue().alpha1.x;
            nn::util::VectorSetW( &color1, nn::util::VectorGetW( color1 ) * parentAlpha );
        }
    }
    nn::util::VectorStore( &pDynamicConstantBuffer->emitterColor0, color0 );
    nn::util::VectorStore( &pDynamicConstantBuffer->emitterColor1, color1 );

    // αフェードアウト・フェードイン
    float alpha = nn::util::VectorGetW( emitterSetColor );
    alpha *= pEmitter->CalculateCurrentAlphaFadeValue();

    // 子エミッタで親の値を毎フレーム継承する場合、ここで親エミッタのフェード（In/Out）値を乗算して継承。
    if( ( pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha0 && pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha0EachFrame ) ||
        ( pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha1 && pEmitter->m_pEmitterRes->m_pResEmitter->inherit.alpha1EachFrame ) )
    {
        const Emitter* pParent = pEmitter->m_pParentEmitter;
        alpha *= pParent->CalculateCurrentAlphaFadeValue();
    }

    // スケールフェードアウト・フェードイン
    const float fScale = pEmitter->CalculateCurrentScaleFadeValue();

    if( pEmitter->GetEmitterSet()->GetSystem()->IsEnabledPreciseGpuCounterMode() )
    {
        // MEMO: 厳密なGPUカウンタモードを使用している場合は m_GpuTime を使用する
        pDynamicConstantBuffer->emitterParam0.x = pEmitter->m_GpuTime;
    }
    else
    {
        pDynamicConstantBuffer->emitterParam0.x = pEmitter->m_Time;
    }
    pDynamicConstantBuffer->emitterParam0.y = static_cast< float >( pEmitter->m_ParticleCount );
    pDynamicConstantBuffer->emitterParam0.z = savingFrameRate;
    pDynamicConstantBuffer->emitterParam0.w = frameRate;
    if( frameRate == 0.0f && pEmitter->m_Time > 0.0f )
    {
        if( pEmitter->GetEmitterSet()->GetSystem()->IsEnabledPreciseGpuCounterMode() == false )
        {
            pDynamicConstantBuffer->emitterParam0.x = pEmitter->m_Time - pEmitter->m_FrameRate;
        }
        pDynamicConstantBuffer->emitterParam0.w = pEmitter->m_FrameRate;
    }
    pDynamicConstantBuffer->emitterParam1.x = alpha;
    pDynamicConstantBuffer->emitterParam1.y = nn::util::VectorGetX( pEmitterSet->m_ParticleScaleForCalculation ) * fScale;
    pDynamicConstantBuffer->emitterParam1.z = nn::util::VectorGetY( pEmitterSet->m_ParticleScaleForCalculation ) * fScale;
    pDynamicConstantBuffer->emitterParam1.w = nn::util::VectorGetZ( pEmitterSet->m_ParticleScaleForCalculation ) * fScale;

    {
        nn::util::Matrix4x4fType matrix;
        detail::Matrix4x3fTo4x4f( &matrix, pEmitter->m_MatrixSrt );
        nn::util::MatrixTranspose( &matrix, matrix );
        nn::util::MatrixStore( &pDynamicConstantBuffer->emitterMatrix, matrix );
    }

    {
        nn::util::Matrix4x4fType matrix;
        detail::Matrix4x3fTo4x4f( &matrix, pEmitter->m_MatrixRt );
        nn::util::MatrixTranspose( &matrix, matrix );
        nn::util::MatrixStore( &pDynamicConstantBuffer->emitterMatrixRT, matrix );
    }
}

} // namespace detail
} // namespace vfx
} // namespace nn
