﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>

#include <nn/util/util_Arithmetic.h>

#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/ui2d/ui2d_PaneEffect.h>
#include <nn/ui2d/detail/ui2d_Log.h>
#include <nn/font/font_DispStringBuffer.h>
#include <nn/util/util_Matrix.h>

#include <nn/perf.h>

namespace nn
{
namespace ui2d
{
namespace detail
{
namespace
{

/*!--------------------------------------------------------------------------*
  @brief        基点が中央であるときのカーソル位置の値を求めます。

  @param[in]    value   考慮する幅/高さ
  @param[in]    isCeil  半分の値が浮動小数点にならないように整数値に
                        切り上げる場合は真。

  @return       基点が中央であるときのカーソル位置補正の値を返します。
 *---------------------------------------------------------------------------*/
inline
float
AdjustCenterValue(
    float value,
    bool isCeil
)
{
    float ret = value / 2;
    return isCeil ? std::ceilf(ret): ret;
}

inline int StrLenUtf8(const char* pStr)
{
    int len = 0;
    char buffer[4];
    while (*pStr != 0)
    {
        nn::util::PickOutCharacterFromUtf8String(buffer, &pStr);
        len++;
    }
    return len;
}

} // namespace nn::ui2d::detail::{anonymous}

} // namespace nn::ui2d::detail

TextBox::TextBox()
{
    Initialize();
    InitializeMaterial();
}

TextBox::TextBox(bool isUtf8)
{
    Initialize();
    InitializeMaterial();
    m_IsUtf8 = isUtf8;
}

TextBox::TextBox(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    InitializeStringParam* pInitStringParam,
    const ResTextBox* pBaseBlock,
    const ResTextBox* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
    : Pane(pOutBuildResultInformation, pDevice, pBaseBlock, buildArgSet)
{
    const ResTextBox* pBlock;
    const ResTextBox* pBlockForText;
    const BuildResSet* pBuildResSet;
    const BuildResSet* pBuildResSetForText;
    if (pOverrideBlock)
    {
        if (buildArgSet.overrideUsageFlag == 0 && buildArgSet.overrideMaterialUsageFlag == 0)
        {
            // 部分上書きがない、丸ごと上書き
            pBlock = pOverrideBlock;
            pBuildResSet = buildArgSet.pOverrideBuildResSet;
            pBlockForText = pOverrideBlock;
            pBuildResSetForText = buildArgSet.pOverrideBuildResSet;
        }
        else
        {
            pBlock = pBaseBlock;
            pBuildResSet = buildArgSet.pCurrentBuildResSet;
            if (detail::TestBit(buildArgSet.overrideUsageFlag, TextBoxOverrideUsageFlag_TextEnabled))
            {
                // テキストを部分的に上書きする
                pBlockForText = pOverrideBlock;
                pBuildResSetForText = buildArgSet.pOverrideBuildResSet;
            }
            else
            {
                pBlockForText = pBaseBlock;
                pBuildResSetForText = buildArgSet.pCurrentBuildResSet;
            }
        }
    }
    else
    {
        // 上書きがない場合
        pBlock = pBaseBlock;
        pBlockForText = pBaseBlock;
        pBuildResSet = buildArgSet.pCurrentBuildResSet;
        pBuildResSetForText = buildArgSet.pCurrentBuildResSet;
    }

    Initialize();

    bool shadowEnabled = pBlock->textBoxFlag & (1 << TextBoxFlag_ShadowEnabled);

    uint32_t  allocStrBufLen =
        static_cast<uint32_t >(pBlock->textBufBytes / sizeof(uint16_t));

    if (allocStrBufLen > 0)
    {
        allocStrBufLen -= 1;
    }

    NN_DETAIL_UI2D_WARNING(!(buildArgSet.isUtf8 && pBlockForText->textStrBytes != 0),
        "In the utf-8 mode, reading texts of TextBox panes in bflyt is not supported yet.");

    uint32_t  resStrLen =
        static_cast<uint32_t >(pBlockForText->textStrBytes / sizeof(uint16_t));

    if (resStrLen > 0)
    {
        resStrLen -= 1;
    }

    for (int i = 0; i < TextColor_MaxTextColor; ++i)
    {
        m_TextColors[i] = pBlock->textCols[i];
    }

    m_TextPosition = pBlock->textPosition;
    m_Bits.textAlignment = pBlock->textAlignment;
    m_CharSpace = pBlock->charSpace;
    m_LineSpace = pBlock->lineSpace;
    m_ItalicRatio = pBlock->italicRatio;

    m_Bits.centerCeilingEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_CenterCeilingEnabled)) != 0;
    m_Bits.drawFromRightToLeft = (pBlock->textBoxFlag & (1 << TextBoxFlag_DrawFromRightToLeft)) != 0;
    const bool lineWidthOffsetEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_LineWidthOffsetEnabled)) != 0;
    const bool extendedTagEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_ExtendedTagEnabled)) != 0;

    m_Bits.shadowEnabled = shadowEnabled;
    m_ShadowOffset = pBlock->shadowOffset.operator const nn::util::Float2();
    m_ShadowScale = pBlock->shadowScale.operator const nn::util::Float2();
    m_ShadowTopColor = pBlock->shadowCols[0];
    m_ShadowBottomColor = pBlock->shadowCols[1];
    m_ShadowItalicRatio = pBlock->shadowItalicRatio;

    m_Bits.invisibleBorderEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_InvisibleBorderEnabled)) != 0;
    m_Bits.doubleDrawnBorderEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_DoubleDrawnBorderEnabled)) != 0;
    m_Bits.perCharacterTransformEnabled = (pBlock->textBoxFlag & (1 << TextBoxFlag_PerCharacterTransformEnabled)) != 0;
    m_Bits.perCharacterTransformSplitByCharWidth = (pBlock->textBoxFlag & (1 << TextBoxFlag_PerCharacterTransformSplitByCharWidth)) != 0;
    m_Bits.perCharacterTransformAutoShadowAlpha = (pBlock->textBoxFlag & (1 << TextBoxFlag_PerCharacterTransformAutoShadowAlpha)) != 0;
    m_Bits.perCharacterTransformOriginToCenter = (pBlock->textBoxFlag & (1 << TextBoxFlag_PerCharacterTransformOriginToCenter)) != 0;
    m_Bits.perCharacterTransformFixSpace = (pBlock->textBoxFlag & (1 << TextBoxFlag_PerCharacterTransformFixSpace)) != 0;
    m_Bits.widthLimitEnabled = true;

    if (lineWidthOffsetEnabled && pBlock->lineWidthOffsetOffset != 0)
    {
        const int lineWidthOffsetCount = static_cast<int>(*nn::util::ConstBytePtr(pBlock, pBlock->lineWidthOffsetOffset).Get<uint8_t>());
        const float* pLineWidthOffset = nn::util::ConstBytePtr(pBlock, pBlock->lineWidthOffsetOffset + sizeof(uint8_t)).Get<float>();
        {
            const size_t allocSize = sizeof(LineWidthOffset);
            m_pLineWidthOffset = static_cast<LineWidthOffset*>(Layout::AllocateMemory(allocSize));
        }
        {
            const size_t allocSize = sizeof(float) * TextBoxLineWidthOffsetCount;
            m_pLineWidthOffset->pLineOffset = static_cast<float*>(Layout::AllocateMemory(allocSize));
            m_pLineWidthOffset->pLineWidth = static_cast<float*>(Layout::AllocateMemory(allocSize));
        }
        const float* pPtr = pLineWidthOffset;
        for (int i = 0; i < TextBoxLineWidthOffsetCount; i++)
        {
            if (i >= lineWidthOffsetCount)
            {
                m_pLineWidthOffset->pLineOffset[i] = 0.0f;
            }
            else
            {
                m_pLineWidthOffset->pLineOffset[i] = *pPtr;
                pPtr++;
            }
        }
        for (int i = 0; i < TextBoxLineWidthOffsetCount; i++)
        {
            if (i >= lineWidthOffsetCount)
            {
                m_pLineWidthOffset->pLineWidth[i] = 0.0f;
            }
            else
            {
                m_pLineWidthOffset->pLineWidth[i] = *pPtr;
                pPtr++;
            }
        }
    }
    else
    {
        m_pLineWidthOffset = NULL;
    }

    if (extendedTagEnabled)
    {
        if (buildArgSet.isUtf8)
        {
            m_pTagProcessor.utf8 = nn::font::TextWriterBase<char>::GetExtendedTagProcessor();
        }
        else
        {
            m_pTagProcessor.utf16 = nn::font::TextWriterBase<uint16_t>::GetExtendedTagProcessor();
        }
    }

    if (m_Bits.perCharacterTransformEnabled && pBlock->perCharacterTransformOffset != 0)
    {
        const ResPerCharacterTransform* pPerCharacterTransform = nn::util::ConstBytePtr(pBlock, pBlock->perCharacterTransformOffset).Get<ResPerCharacterTransform>();
        const ResAnimationInfo* pAnimInfo = NULL;
        uint32_t  allocSize = sizeof(PerCharacterTransform);
        if (pPerCharacterTransform->hasAnimationInfo != 0) {
            if (buildArgSet.resourceVersion < NN_DETAIL_FONT_MAKE_VERSION(8, 1, 0, 0))
            {
                pAnimInfo = nn::util::ConstBytePtr(pPerCharacterTransform, sizeof(ResPerCharacterTransform)).Get<ResAnimationInfo>();
            }
            else
            {
                pAnimInfo = nn::util::ConstBytePtr(pPerCharacterTransform, sizeof(ResPerCharacterTransformExtended)).Get<ResAnimationInfo>();
            }
            if (pAnimInfo->count > 0) {
                allocSize += sizeof(CurveInfo) * (pAnimInfo->count - 1);
            }
        }
        m_pPerCharacterTransform = static_cast<PerCharacterTransform*>(Layout::AllocateMemory(allocSize));
        m_pPerCharacterTransform->Offset = pPerCharacterTransform->evalTimeOffset;
        m_pPerCharacterTransform->Width = pPerCharacterTransform->evalTimeWidth;
        m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        m_pPerCharacterTransform->LoopType = pPerCharacterTransform->loopType;
        m_pPerCharacterTransform->OriginV = pPerCharacterTransform->originV;
        if (buildArgSet.resourceVersion < NN_DETAIL_FONT_MAKE_VERSION(8, 1, 0, 0))
        {
            // 8.1.0.0 より古いバージョンでは 0 に初期化する
            m_pPerCharacterTransform->OriginVOffset = 0.0f;
            m_pPerCharacterTransform->FixSpaceWidth = 0.0f;
            m_pPerCharacterTransform->FixSpaceOrigin = HorizontalAlignment_Left;
        }
        else
        {
            const ResPerCharacterTransformExtended* pPerCharacterTransformExtended = static_cast<const ResPerCharacterTransformExtended*>(pPerCharacterTransform);
            m_pPerCharacterTransform->OriginVOffset = pPerCharacterTransformExtended->originVOffset;
            m_pPerCharacterTransform->FixSpaceWidth = pPerCharacterTransformExtended->fixSpaceWidth;
            m_pPerCharacterTransform->FixSpaceOrigin = static_cast<HorizontalAlignment>(pPerCharacterTransformExtended->fixSpaceOrigin);
        }
        if (pAnimInfo)
        {
            m_pPerCharacterTransform->CurveNumber = pAnimInfo->count;
            InitializePerCharacterTransformCurves(pAnimInfo);
        }
        else
        {
            m_pPerCharacterTransform->CurveNumber = 0;
        }
    }

    // Fontの構築
    NN_SDK_ASSERT(pBuildResSet->pFontList != NULL, "name[%s]", GetName());
    NN_SDK_ASSERT(pBlock->fontIdx < pBuildResSet->pFontList->fontCount, "name[%s]", GetName());
    const ResFont *const pFonts = nn::util::ConstBytePtr(pBuildResSet->pFontList, sizeof(*pBuildResSet->pFontList)).Get<ResFont>();
    const char *const pFontName = nn::util::ConstBytePtr(pFonts, pFonts[pBlock->fontIdx].nameStrOffset).Get<char>();

    m_pFont = pBuildResSet->pResAccessor->AcquireFont(pDevice, pFontName);
    if (m_pFont != NULL)
    {
        // 特殊フチ形式が設定可能でなければ、特殊フチ形式が設定されてはならない。
        NN_SDK_ASSERT(m_pFont->IsBorderAvailable() || (!m_Bits.invisibleBorderEnabled && !m_Bits.doubleDrawnBorderEnabled), "The border setting of the font cannot be set.[%s]", GetName());
    }

    // フォントサイズの設定
    if ((pBlock->textBoxFlag & (1 << TextBoxFlag_KeepingFontScaleEnabled)) != 0)
    {
        // pBlock->fontSize にはフォントサイズの代わりにフォントスケールが入っているため
        // スケールを維持するように幅と高さを設定する
        NN_SDK_ASSERT_NOT_NULL(m_pFont);
        m_FontSize.width = pBlock->fontSize.x * static_cast<float>(m_pFont->GetWidth());
        m_FontSize.height = pBlock->fontSize.y * static_cast<float>(m_pFont->GetHeight());
    }
    else
    {
        m_FontSize.width = pBlock->fontSize.x;
        m_FontSize.height = pBlock->fontSize.y;
    }

    // マテリアルの作成
    {
        const ResMaterial *const pResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseBlock->materialIdx);
        const ResMaterial *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            pOverrideResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideBlock->materialIdx);
        }
        m_pMaterial = Layout::AllocateAndConstruct<Material, BuildResultInformation*, nn::gfx::Device*, const ResMaterial*, const ResMaterial*, const BuildArgSet&>(pOutBuildResultInformation, pDevice, pResMaterial, pOverrideResMaterial, buildArgSet);
    }

    // テキストIDの設定
    if (pBlockForText->textIdOffset != 0)
    {
        // Initメソッドの中でNULLに初期化されているので、このif文の中に来ない場合はNULLになる
        m_pTextId = nn::util::ConstBytePtr(pBlockForText, pBlockForText->textIdOffset).Get<char>();
    }

    pInitStringParam->textBoxFlag = pBlock->textBoxFlag;
    pInitStringParam->pResourceLayout = pBuildResSetForText->pLayout;
    pInitStringParam->pBlockText = nn::util::ConstBytePtr(pBlockForText, pBlockForText->textStrOffset).Get<uint16_t>();
    pInitStringParam->allocStrBufLen = allocStrBufLen;
    pInitStringParam->resStrLen = resStrLen;

    m_IsUtf8 = buildArgSet.isUtf8;
}// NOLINT(impl/function_size)

void TextBox::InitializeString(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const BuildArgSet& buildArgSet,
    const InitializeStringParam& initStringParam)
{
    bool isTextSetByTextSearcher = false; // TextSearcherによってテキストが設定されたか否か
    if (buildArgSet.pTextSearcher)
    {
        if (buildArgSet.isUtf8)
        {
            TextSearcher::TextInfoUtf8 textInfo;
            textInfo.SetDefault();

            // 「文字列の長さを強制的に指定する」のチェックが入っている場合、TextInfoに設定する
            if (initStringParam.textBoxFlag & (1 << TextBoxFlag_ForceAssignTextLength))
            {
                textInfo.forceAssignedTextLength = static_cast<int32_t>(initStringParam.allocStrBufLen);
            }
            // テキストIDから設定するテキストの情報を取得する
            buildArgSet.pTextSearcher->SearchTextUtf8(&textInfo, m_pTextId, buildArgSet.pLayout, this, initStringParam.pResourceLayout);

            NN_SDK_ASSERT(textInfo.textLength <= 0xffff, "pText length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.textLength, GetName());
            NN_SDK_ASSERT(textInfo.bufferLength <= 0xffff, "buffer length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.bufferLength, GetName());

            isTextSetByTextSearcher = InitializeStringWithTextSearcherInfoUtf8(pDevice, buildArgSet, textInfo);
        }
        else
        {
            TextSearcher::TextInfo textInfo;
            textInfo.SetDefault();

            // 「文字列の長さを強制的に指定する」のチェックが入っている場合、TextInfoに設定する
            if (initStringParam.textBoxFlag & (1 << TextBoxFlag_ForceAssignTextLength))
            {
                textInfo.forceAssignedTextLength = static_cast<int32_t>(initStringParam.allocStrBufLen);
            }
            // テキストIDから設定するテキストの情報を取得する
            buildArgSet.pTextSearcher->SearchText(&textInfo, m_pTextId, buildArgSet.pLayout, this, initStringParam.pResourceLayout);

            NN_SDK_ASSERT(textInfo.textLength <= 0xffff, "pText length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.textLength, GetName());
            NN_SDK_ASSERT(textInfo.bufferLength <= 0xffff, "buffer length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.bufferLength, GetName());

            isTextSetByTextSearcher = InitializeStringWithTextSearcherInfo(pDevice, buildArgSet, textInfo);
        }

    }

    // TextSearcherによって設定されなかった場合、バッファを確保する
    if (!isTextSetByTextSearcher) {
        // レイアウトエディタで指定された値とテキストの長さの大きい方を取る
        AllocateStringBuffer(pDevice, static_cast<uint16_t>(std::max(initStringParam.allocStrBufLen, static_cast<size_t>(initStringParam.resStrLen))));
    }

    // TextSearcherで文字列が設定されなかった場合、初期文字列のコピー
    if (initStringParam.resStrLen >= 1 && m_TextBuf.neutral && !isTextSetByTextSearcher)
    {
        if (buildArgSet.isUtf8)
        {
            const char* pBlockText = reinterpret_cast<const char*>(initStringParam.pBlockText);
            SetStringUtf8(pBlockText, 0, static_cast<uint16_t>(initStringParam.resStrLen));
        }
        else
        {
            const uint16_t *const pBlockText = initStringParam.pBlockText;
#if defined( NN_BUILD_CONFIG_ENDIAN_BIG )
            // 現在リソースのエンディアンは、リトルエンディアン。
            // リソースのエンディアンが逆転している場合は、スワップしながら文字列を設定しなければならない。SetStringを使わずに特別処理する。
            m_TextLen = std::min(static_cast<uint16_t>(initStringParam.resStrLen), GetStringBufferLength());
            for (int i = 0; i < m_TextLen; i++)
            {
                m_TextBuf.utf16[i] = nn::font::detail::ByteSwap(static_cast<uint16_t>(pBlockText[i]));
            }
            m_TextBuf.utf16[m_TextLen] = 0;
            UpdatePTDirty(true);
#else
            SetString(pBlockText, 0, static_cast<uint16_t>(initStringParam.resStrLen));
#endif
        }
    }

    // フォントの描画に必要なバッファサイズを集計する。
    nn::font::DispStringBuffer::InitializeArg arg;
    arg.SetCharCountMax(m_pDispStringBuf == NULL ? 0 : m_pDispStringBuf->GetCharCountMax());
    arg.SetShadowEnabled(m_Bits.shadowEnabled);
    arg.SetDoubleDrawnBorder(m_Bits.doubleDrawnBorderEnabled);
    arg.SetPerCharacterTransformEnabled(m_Bits.perCharacterTransformEnabled);
    arg.SetPerCharacterTransformAutoShadowAlpha(m_Bits.perCharacterTransformAutoShadowAlpha);
    size_t constantBufferSize = nn::font::DispStringBuffer::GetRequiredConstantBufferSize(pDevice, arg);
    if (pOutBuildResultInformation != NULL)
    {
        pOutBuildResultInformation->requiredFontConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, constantBufferSize);
    }
}

bool TextBox::InitializeStringWithTextSearcherInfo(nn::gfx::Device* pDevice, const BuildArgSet& buildArgSet, const TextSearcher::TextInfo& textInfo)
{
    NN_UNUSED(buildArgSet);

    if (textInfo.pText) {
        //---- テキストが指定されたとき
        // テキストの長さを取得する、0が指定された場合は指定されたテキストの長さを数える
        uint16_t  textLength = static_cast<uint16_t >(textInfo.textLength);
        if (textLength == 0) {
            textLength = static_cast<uint16_t >(nn::font::CalculateWideCharString16Length(textInfo.pText));
        }
        if (textInfo.bufferLength == 0) {
            // バッファの長さの指定がないときは、テキストの長さで確保する
            AllocateStringBuffer(pDevice, textLength);
        } else {
            // バッファの長さが指定されたらそれを使用する
            AllocateStringBuffer(pDevice, static_cast<uint16_t >(textInfo.bufferLength));
        }
        if (GetStringBuffer()) {
            // AllocStringBufferの結果バッファが設定されたら文字列をセット。
            // バッファが設定されないのは、textInfo.textに空文字が設定され、textLengthとbufferLengthが両方0の場合のみ
            SetString(textInfo.pText, 0, textLength);
        }
        // trueを返すと、レイアウトエディタで指定した文字列は設定されない
        return true;
    } else {
        return false;
    }
}

bool TextBox::InitializeStringWithTextSearcherInfoUtf8(nn::gfx::Device* pDevice, const BuildArgSet& buildArgSet, const TextSearcher::TextInfoUtf8& textInfo)
{
    NN_UNUSED(buildArgSet);

    if (textInfo.pText) {
        //---- テキストが指定されたとき
        // テキストの長さを取得する、0が指定された場合は指定されたテキストの長さを数える
        uint16_t  textLength = static_cast<uint16_t >(textInfo.textLength);
        if (textLength == 0) {
            textLength = static_cast<uint16_t >(detail::StrLenUtf8(textInfo.pText));
        }
        if (textInfo.bufferLength == 0) {
            // バッファの長さの指定がないときは、テキストの長さで確保する
            AllocateStringBuffer(pDevice, textLength);
        } else {
            // バッファの長さが指定されたらそれを使用する
            AllocateStringBuffer(pDevice, static_cast<uint16_t >(textInfo.bufferLength));
        }
        if (GetStringBuffer()) {
            // AllocStringBufferの結果バッファが設定されたら文字列をセット。
            // バッファが設定されないのは、textInfo.textに空文字が設定され、textLengthとbufferLengthが両方0の場合のみ
            SetStringUtf8(textInfo.pText, 0, textLength);
        }
        // trueを返すと、レイアウトエディタで指定した文字列は設定されない
        return true;
    } else {
        return false;
    }
}

void TextBox::InitializePerCharacterTransformCurves(const ResAnimationInfo* pAnimInfo)
{
    const uint32_t* pAnimTargetOffsets = nn::util::ConstBytePtr(pAnimInfo, sizeof(*pAnimInfo)).Get<uint32_t>();
    for (int k = 0; k < pAnimInfo->count; k++)
    {
        const ResAnimationTarget* pAnimTarget = nn::util::ConstBytePtr(pAnimInfo, pAnimTargetOffsets[k]).Get<ResAnimationTarget>();
        m_pPerCharacterTransform->CurveInfos[k].pCurve = pAnimTarget;
        m_pPerCharacterTransform->CurveInfos[k].CurveType = static_cast<AnimTargetPerCharacterTransformCurve>(pAnimTarget->target);
    }
}

void TextBox::CopyLineWidthOffset(const TextBox& textBox)
{
    if (textBox.m_pLineWidthOffset != NULL)
    {
        {
            const size_t allocSize = sizeof(LineWidthOffset);
            m_pLineWidthOffset = static_cast<LineWidthOffset*>(Layout::AllocateMemory(allocSize));
        }
        {
            const size_t allocSize = sizeof(float) * TextBoxLineWidthOffsetCount;
            m_pLineWidthOffset->pLineOffset = static_cast<float*>(Layout::AllocateMemory(allocSize));
            m_pLineWidthOffset->pLineWidth = static_cast<float*>(Layout::AllocateMemory(allocSize));
        }
        for (int i = 0; i < TextBoxLineWidthOffsetCount; i++)
        {
            m_pLineWidthOffset->pLineOffset[i] = textBox.m_pLineWidthOffset->pLineOffset[i];
            m_pLineWidthOffset->pLineWidth[i] = textBox.m_pLineWidthOffset->pLineWidth[i];
        }
    }
}

void TextBox::CopyPerCharacterTransform(const TextBox& textBox)
{
    if (textBox.m_pPerCharacterTransform != NULL)
    {
        uint32_t  allocSize = sizeof(PerCharacterTransform);
        if (textBox.m_pPerCharacterTransform->CurveNumber > 1)
        {
            allocSize += sizeof(CurveInfo) * (textBox.m_pPerCharacterTransform->CurveNumber - 1);
        }
        m_pPerCharacterTransform = static_cast<PerCharacterTransform*>(Layout::AllocateMemory(allocSize));
        // カーブ情報も含めてコピーする。
        memcpy(m_pPerCharacterTransform, textBox.m_pPerCharacterTransform, allocSize);
        m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
    }
}

void TextBox::CopyCommonImpl(const TextBox& textBox)
{
    m_pTextId = textBox.m_pTextId;
    m_pFont = textBox.m_pFont;
    m_FontSize = textBox.m_FontSize;
    m_LineSpace = textBox.m_LineSpace;
    m_CharSpace = textBox.m_CharSpace;
    m_pTagProcessor = textBox.m_pTagProcessor;
    m_TextBufLen = 0;
    m_TextLen = 0;
    m_Bits = textBox.m_Bits;
    m_TextPosition = textBox.m_TextPosition;
    m_IsUtf8 = textBox.m_IsUtf8;
    m_ItalicRatio = textBox.m_ItalicRatio;
    m_ShadowOffset = textBox.m_ShadowOffset;
    m_ShadowScale = textBox.m_ShadowScale;
    m_ShadowTopColor = textBox.m_ShadowTopColor;
    m_ShadowBottomColor = textBox.m_ShadowBottomColor;
    m_ShadowItalicRatio = textBox.m_ShadowItalicRatio;
    m_pLineWidthOffset = NULL;
    m_pMaterial = NULL;
    m_pDispStringBuf = NULL;
    m_pPerCharacterTransform = NULL;

    m_TextBuf.neutral = NULL;

    for (int i = 0; i < TextColor_MaxTextColor; ++i)
    {
        m_TextColors[i] = textBox.m_TextColors[i];
    }

    // LineWidthOffset 情報をコピー
    CopyLineWidthOffset(textBox);
    // PerCharacterTransform 情報をコピー
    CopyPerCharacterTransform(textBox);
}

void TextBox::CopyImpl(const TextBox& textBox, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    NN_UNUSED(pResAccessor);
    NN_UNUSED(pNewRootName);

    CopyCommonImpl(textBox);

    if (textBox.GetStringBufferLength() > 0)
    {
        // 文字列バッファを確保
        AllocateStringBuffer(pDevice, textBox.GetStringBufferLength(), textBox.GetDrawStringBufferLength());
        if (m_IsUtf8)
        {
            // 文字をセット
            SetStringUtf8(textBox.GetStringUtf8(), 0, textBox.GetStringLength());
        }
        else
        {
            // 文字をセット
            SetString(textBox.GetString(), 0, textBox.GetStringLength());
        }
    }
    // マテリアルの複製
    m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, nn::gfx::Device*>(*textBox.m_pMaterial, pDevice);
}

void TextBox::CopyImpl(const TextBox& textBox, nn::gfx::Device* pDevice, uint16_t bufferStrlen, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    NN_UNUSED(pResAccessor);
    NN_UNUSED(pNewRootName);

    CopyCommonImpl(textBox);

    if (bufferStrlen > 0)
    {
        // 文字列バッファを確保、文字列のセットは行わない
        AllocateStringBuffer(pDevice, bufferStrlen);
    }
    // マテリアルの複製
    m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, nn::gfx::Device*>(*textBox.m_pMaterial, pDevice);
}

void
TextBox::Initialize()
{
    m_TextBuf.neutral = NULL;
    m_pTextId = 0;
    for (int i = 0; i < TextColor_MaxTextColor; i++)
    {
        nn::util::Unorm8x4 textColor = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        m_TextColors[i] = textColor;
    }
    m_pFont = 0;
    m_FontSize.Set(0.f, 0.f);
    m_LineSpace = 0;
    m_CharSpace = 0;
    m_pTagProcessor.neutral = NULL;
    m_TextBufLen = 0;
    m_TextLen = 0;
    std::memset(&m_Bits, 0, sizeof(m_Bits));
    m_TextPosition = 0;
    m_IsUtf8 = false;
    m_ItalicRatio = 0.0f;
    m_ShadowOffset.v[0] = 0.0f;
    m_ShadowOffset.v[1] = 0.0f;
    m_ShadowScale.v[0] = 1.0f;
    m_ShadowScale.v[1] = 1.0f;
    nn::util::Unorm8x4 shadowTopColor = { { 0, 0, 0, std::numeric_limits<uint8_t>::max() } };
    m_ShadowTopColor = shadowTopColor;
    nn::util::Unorm8x4 shadowBottomColor = { { 0, 0, 0, std::numeric_limits<uint8_t>::max() } };
    m_ShadowBottomColor = shadowBottomColor;
    m_ShadowItalicRatio = 0.0f;
    m_pLineWidthOffset = NULL;
    m_pMaterial = NULL;
    m_pDispStringBuf = 0;
    m_pPerCharacterTransform = NULL;

    SetTextPositionX(HorizontalPosition_Center);
    SetTextPositionY(VerticalPosition_Center);
}

void
TextBox::InitializeMaterial()
{
    // マテリアルの作成
    m_pMaterial = Layout::AllocateAndConstruct<Material>();
    if (m_pMaterial)
    {
        m_pMaterial->SetShaderId(ShaderId_NullTexture);
        m_pMaterial->ReserveMem(0, 0, 0);
    }
}

TextBox::~TextBox()
{
    NN_SDK_ASSERT(
        m_pMaterial == 0 && m_pPerCharacterTransform == NULL,
        "Textbox must be finalized before destruction for name[%s]", GetName());
}

void
TextBox::Finalize(nn::gfx::Device* pDevice)
{
    Pane::Finalize(pDevice);

    if (m_pMaterial && !m_pMaterial->IsUserAllocated())
    {
        m_pMaterial->Finalize(pDevice);
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    FreeStringBuffer(pDevice);

    if (m_pLineWidthOffset != NULL)
    {
        Layout::FreeMemory(m_pLineWidthOffset->pLineOffset);
        Layout::FreeMemory(m_pLineWidthOffset->pLineWidth);
        Layout::FreeMemory(m_pLineWidthOffset);
        m_pLineWidthOffset = NULL;
    }

    if (m_pPerCharacterTransform != NULL)
    {
        Layout::FreeMemory(m_pPerCharacterTransform);
        m_pPerCharacterTransform = NULL;
    }
}

uint8_t
TextBox::GetMaterialCount() const
{
    return static_cast<uint8_t >(m_pMaterial ? 1 : 0);
}

Material*
TextBox::GetMaterial(int  idx) const
{
    NN_DETAIL_UI2D_WARNING(idx < GetMaterialCount(), "idx >= GetMaterialCount() : %d >= %d", idx, GetMaterialCount());

    return (idx == 0) ? m_pMaterial : NULL;
}

void TextBox::SetMaterial(Material* pMaterial)
{
    if (m_pMaterial && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }
    m_pMaterial = pMaterial;
}

void TextBox::UnsetMaterial()
{
    m_pMaterial = NULL;
}

float TextBox::GetAngleFromItalicRatio() const
{
    float angle = std::atanf(m_ItalicRatio * m_FontSize.width / m_FontSize.height);
    return angle;
}

void TextBox::SetAngleToItalicRatio(float angle)
{
    NN_SDK_ASSERT(-nn::util::FloatPi / 2.0f < angle && angle < nn::util::FloatPi / 2.0f, "Out of range for textbox[%s].", GetName());
    float ratio = std::tanf(angle) / m_FontSize.width * m_FontSize.height;
    if (UpdatePTDirty(m_ItalicRatio != ratio))
    {
        m_ItalicRatio = ratio;
    }
}

float TextBox::GetAngleFromShadowItalicRatio() const
{
    float angle = std::atanf(m_ShadowItalicRatio * m_FontSize.width / m_FontSize.height);
    return angle;
}

void TextBox::SetAngleToShadowItalicRatio(float angle)
{
    NN_SDK_ASSERT(-nn::util::FloatPi / 2.0f < angle && angle < nn::util::FloatPi / 2.0f, "Out of range for textbox[%s].", GetName());
    float ratio = std::tanf(angle) / m_FontSize.width * m_FontSize.height;
    if (UpdatePTDirty(m_ShadowItalicRatio != ratio))
    {
        m_ShadowItalicRatio = ratio;
    }
}

const nn::util::Unorm8x4
TextBox::GetVertexColor(int  idx) const
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] < VertexColor_MaxVertexColor for TextBox[%s]", idx, GetName());

    return GetTextColor(idx / 2);
}

void
TextBox::SetVertexColor(
    int          idx,
    const nn::util::Unorm8x4& value
)
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] < VertexColor_MaxVertexColor for TextBox[%s]", idx, GetName());

    SetTextColor(idx / 2, value);
}

uint8_t
TextBox::GetVertexColorElement(int  idx) const
{
    NN_SDK_ASSERT(idx < AnimTargetPaneColor_MaxVertex, "out of bounds: idx[%u] < AnimTargetPaneColor_MaxVertex for TextBox[%s]", idx, GetName());

    const int elementSize = 4; // rgba の計4つ
    return m_TextColors[idx / (2 * elementSize)].v[idx % elementSize];
}

void
TextBox::SetVertexColorElement(int  idx, uint8_t  value)
{
    NN_SDK_ASSERT(idx < AnimTargetPaneColor_MaxVertex, "out of bounds: idx[%u] < AnimTargetPaneColor_MaxVertex for TextBox[%s]", idx, GetName());

    const int elementSize = 4; // rgba の計4つ
    uint8_t & elm = m_TextColors[idx / (2 * elementSize)].v[idx % elementSize];

    UpdatePTDirty(elm != value);
    elm = value;
}

const nn::font::Rectangle
TextBox::GetTextDrawRect() const
{
    if (m_pFont == NULL)
    {
        NN_DETAIL_UI2D_ERROR("m_pFont is NULL");
        return nn::font::Rectangle();
    }

    nn::font::Rectangle textRect;
    Size textSize;

    if (m_IsUtf8)
    {
        nn::font::TextWriter writer;
        writer.SetCenterCeilingEnabled(m_Bits.centerCeilingEnabled);
        writer.SetCursor(0, 0);
        writer.SetItalicRatio(m_ItalicRatio);
        SetFontInfoUtf8(&writer);

        writer.CalculateStringRect(&textRect, m_TextBuf.utf8, m_TextLen);

        textSize = Size::Create(textRect.GetWidth(), textRect.GetHeight());
    }
    else
    {
        nn::font::WideTextWriter writer;
        writer.SetCenterCeilingEnabled(m_Bits.centerCeilingEnabled);
        writer.SetCursor(0, 0);
        writer.SetItalicRatio(m_ItalicRatio);
        SetFontInfo(&writer);

        writer.CalculateStringRect(&textRect, m_TextBuf.utf16, m_TextLen);

        textSize = Size::Create(textRect.GetWidth(), textRect.GetHeight());
    }

    nn::util::Float2 ltPos = GetVertexPos();

    const nn::util::Float2& curPos  = AdjustTextPos(GetSize(), false);
    const nn::util::Float2& textPos = AdjustTextPos(textSize, m_Bits.centerCeilingEnabled);

    ltPos.v[0] += curPos.v[0] - textPos.v[0];
    ltPos.v[1] -= curPos.v[1] - textPos.v[1];

    textRect.left   = ltPos.v[0];
    textRect.top    = ltPos.v[1];
    textRect.right  = ltPos.v[0] + textSize.width;
    textRect.bottom = ltPos.v[1] - textSize.height;

    return textRect;
}

void
TextBox::SetupConstantBufferAdditionalContent(nn::font::ConstantBufferAdditionalContent& drawContent) const
{
    // 定数バッファの更新
    if (m_Bits.invisibleBorderEnabled)
    {
        drawContent.SetBorderVisibility(false);
    }

    {
        nn::util::Float4  whiteAsFloat;
        nn::util::Float4  blackAsFloat;
        m_pMaterial->GetColorWithFloatConversion(whiteAsFloat, InterpolateColor_White);
        m_pMaterial->GetColorWithFloatConversion(blackAsFloat, InterpolateColor_Black);

        drawContent.SetInterpolateBlack(blackAsFloat);
        drawContent.SetInterpolateWhite(whiteAsFloat);
        drawContent.SetInterpolateAlpha(GetGlobalAlpha());
    }

    if (m_pMaterial->IsFontShadowParameterCap())
    {
        const ResFontShadowParameter& color = m_pMaterial->GetFontShadowParameter();
        const nn::util::Unorm8x4 black = { { color.blackInterporateColor[0], color.blackInterporateColor[1], color.blackInterporateColor[2], 0 } };
        drawContent.SetShadowInterpolateBlack(black);
        const nn::util::Unorm8x4 white = { { color.whiteInterporateColor[0], color.whiteInterporateColor[1], color.whiteInterporateColor[2], color.whiteInterporateColor[3] } };
        drawContent.SetShadowInterpolateWhite(white);
        drawContent.SetShadowInterpolateAlpha(GetGlobalAlpha());
    }
    else
    {
        uint8_t  alpha = GetGlobalAlpha();
        const nn::util::Unorm8x4 black = { { 0, 0, 0, std::numeric_limits<uint8_t>::max() } };
        drawContent.SetShadowInterpolateBlack(black);
        const nn::util::Unorm8x4 white = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        drawContent.SetShadowInterpolateWhite(white);
        drawContent.SetShadowInterpolateAlpha(alpha);
    }

    if (m_Bits.perCharacterTransformEnabled)
    {
        drawContent.SetPerCharacterTransformInfos(m_pPerCharacterTransform->pPerCharacterTransformInfos);
        drawContent.SetPerCharacterTransformCenter(static_cast<nn::font::ConstantBufferAdditionalContent::PerCharacterTransformCenter>(m_pPerCharacterTransform->OriginV));
        drawContent.SetPerCharacterTransformCenterOffset(m_pPerCharacterTransform->OriginVOffset);
    }
}

void
TextBox::BuildConstantBuffer(DrawInfo& drawInfo) const
{
    // 定数バッファの更新
    nn::font::ConstantBufferAdditionalContent drawContent;

    // 以下の設定はローカル変数のポインタを渡しているため drawContent と同じスコープにおく必要がある。
    nn::util::MatrixT4x3fType mtx;
    GetTextGlobalMtx(&mtx, drawInfo);

    drawContent.SetViewMatrix(&mtx);
    nn::util::MatrixT4x3fType identity;
    nn::util::MatrixIdentity(&identity);
    drawContent.SetLocalMatrix(&identity);

    nn::font::ShadowParameter param;
    if (m_Bits.shadowEnabled)
    {
        param.shadowUpperColor = m_ShadowTopColor;
        param.shadowLowerColor = m_ShadowBottomColor;
        param.shadowOffset = m_ShadowOffset;
        param.shadowScale = m_ShadowScale;
        param.shadowItalicOffset = m_ShadowItalicRatio * m_FontSize.width;
        drawContent.SetShadowParam(&param);
    }

    // ConstantBufferAdditionalContent の内容を設定する。
    SetupConstantBufferAdditionalContent(drawContent);

    const nn::util::MatrixT4x4fType*  pProjMtx = &drawInfo.GetProjectionMtx();
    nn::util::MatrixT4x4fType   captureProjMtx;

    // TextBox の姿勢行列は Material ではなく、フォント描画のコンスタントバッファのものが使用されます。
    // ペインエフェクト機能のキャプチャ時の描画位置を調整するため、Pane::UpdateMaterialConstantBufferForEffectCapture() 相当の処理を別途実装しています。
    if (IsPaneEffectEnabled())
    {
        CalculateCaptureProjectionMatrix(captureProjMtx);
        pProjMtx = &captureProjMtx;

        // ドロップシャドウ静的レンダリング時に透明度の適用を最終キャプチャ結果描画時に遅延させるため
        // キャプチャ時の描画では透明度を 255 でレンダリングする。
        if (IsDropShadowEnabled())
        {
            const detail::PaneEffect* pPaneEffect = GetPaneEffectInstance();
            if (pPaneEffect != NULL &&
                pPaneEffect->IsDropShadowCacheRenderingNeeded())
            {
                drawContent.SetInterpolateAlpha(255);
            }
        }
    }

    if (m_Bits.perCharacterTransformOriginToCenter)
    {
        m_pDispStringBuf->BuildConstantBuffer(*pProjMtx, &drawContent, m_Bits.drawFromRightToLeft, true);
    }
    else
    {
        m_pDispStringBuf->BuildConstantBuffer(*pProjMtx, &drawContent, m_Bits.drawFromRightToLeft);
    }
}

void
TextBox::Calculate(DrawInfo& drawInfo, Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::TextBox::CalculateMtx");

    Pane::Calculate(drawInfo, context, isDirtyParentMtx);

    if (m_TextLen <= 0 || !m_pFont || !m_pMaterial)
    {
        SetConstantBufferReadySelf(false);
        return;
    }

#if defined(NN_UI2D_FIX_INFLUENCE_ALPHA_BUG)
    if (CheckInvisibleAndUpdateConstantBufferReady())
    {
        return;
    }
#else
    // 非表示、もしくはアルファ値が 0 になって見えない状態のペインは
    // コンスタントバッファの確保と更新処理は不要なので行わない。
    if (!IsVisible() || GetGlobalAlpha() == 0)
    {
        if (context.isInfluenceAlpha)
        {
            SetConstantBufferReady(false);
        }
        else
        {
            SetConstantBufferReadySelf(false);
        }
        return;
    }
#endif

    // 描画に使用するバッファを設定する
    m_pDispStringBuf->SetConstantBuffer(drawInfo.GetFontConstantBuffer());

    m_pMaterial->SetupBlendState(&drawInfo);

    // 頂点バッファの更新の必要がなければ抜ける
    if (m_Bits.isPTDirty == 0)
    {
        BuildConstantBuffer(drawInfo);
        return;
    }
    m_Bits.isPTDirty = 0;

    // 描画文字列バッファの作成
    if (m_IsUtf8)
    {
        nn::font::TextWriter writer;
        writer.SetCenterCeilingEnabled(m_Bits.centerCeilingEnabled);
        SetupTextWriterUtf8(&writer);
        writer.SetDispStringBuffer(this->m_pDispStringBuf);

        writer.StartPrint();
        if (m_pLineWidthOffset != NULL)
        {
            writer.Print(m_TextBuf.utf8, m_TextLen, TextBoxLineWidthOffsetCount, m_pLineWidthOffset->pLineOffset, m_pLineWidthOffset->pLineWidth);
        }
        else
        {
            writer.Print(m_TextBuf.utf8, m_TextLen);
        }
        writer.EndPrint();

        if (m_Bits.perCharacterTransformEnabled)
        {
            UpdatePerCharacterTransform<char>(&writer.GetTagProcessor());
        }
    }
    else
    {
        nn::font::WideTextWriter writer;
        writer.SetCenterCeilingEnabled(m_Bits.centerCeilingEnabled);
        SetupTextWriter(&writer);
        writer.SetDispStringBuffer(this->m_pDispStringBuf);

        writer.StartPrint();
        if (m_pLineWidthOffset != NULL)
        {
            writer.Print(m_TextBuf.utf16, m_TextLen, TextBoxLineWidthOffsetCount, m_pLineWidthOffset->pLineOffset, m_pLineWidthOffset->pLineWidth);
        }
        else
        {
            writer.Print(m_TextBuf.utf16, m_TextLen);
        }
        writer.EndPrint();

        if (m_Bits.perCharacterTransformEnabled)
        {
            UpdatePerCharacterTransform<uint16_t>(&writer.GetTagProcessor());
        }
    }

    BuildConstantBuffer(drawInfo);
}// NOLINT(impl/function_size)

void
TextBox::SetupPaneEffectSourceImageRenderState(nn::gfx::CommandBuffer& commandBuffer) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pMaterial);

    m_pMaterial->SetCommandBufferOnlyBlend(commandBuffer);
}

void
TextBox::DrawSelf(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::TextBox::DrawSelf");

    if (m_TextLen <= 0 || !m_pFont || !m_pMaterial)
    {
        return;
    }

    m_pMaterial->SetCommandBufferOnlyBlend(commandBuffer);
    if (IsPaneEffectEnabled())
    {
        commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(PresetBlendStateId_OpaqueOrAlphaTest));
    }
    nn::font::RectDrawer& drawer = drawInfo.GetGraphicsResource()->GetFontDrawer();

    // RecrDrawer::Draw() 内で font の SetShader が行われるため、使用中シェーダの情報をリセットする
    drawInfo.ResetCurrentShader();
    drawer.Draw(
        commandBuffer,
        *m_pDispStringBuf);

    drawInfo.ResetVertexBufferState();
}

bool
TextBox::CompareCopiedInstanceTest(const TextBox& target) const
{
    if (m_IsUtf8)
    {
        if (memcmp(m_TextBuf.utf8, target.m_TextBuf.utf8, sizeof(uint8_t) * m_TextBufLen) != 0)
        {
            return false;
        }
    }
    else
    {
        if (memcmp(m_TextBuf.utf16, target.m_TextBuf.utf16, sizeof(uint16_t) * m_TextBufLen) != 0)
        {
            return false;
        }
    }

    if (m_pTagProcessor.neutral != target.m_pTagProcessor.neutral ||
        m_pTextId != target.m_pTextId ||
        m_pFont != target.m_pFont ||
        m_LineSpace != target.m_LineSpace ||
        m_CharSpace != target.m_CharSpace ||
        m_TextBufLen != target.m_TextBufLen ||
        m_TextLen != target.m_TextLen ||
        m_TextPosition != target.m_TextPosition ||
        m_IsUtf8 != target.m_IsUtf8 ||
        m_ItalicRatio != target.m_ItalicRatio ||
        m_ShadowItalicRatio != target.m_ShadowItalicRatio)
    {
        return false;
    }

    // 以下のメンバ変数はパディングが入らない前提で memcmp で比較しています。
    if (memcmp(m_TextColors, target.m_TextColors, sizeof(nn::util::Unorm8x4) * TextColor_MaxTextColor) != 0 ||
        memcmp(&m_FontSize, &target.m_FontSize, sizeof(Size)) != 0 ||
        memcmp(&m_Bits, &target.m_Bits, sizeof(Bits)) != 0 ||
        memcmp(&m_ShadowOffset, &target.m_ShadowOffset, sizeof(nn::util::Float2)) != 0 ||
        memcmp(&m_ShadowScale, &target.m_ShadowScale, sizeof(nn::util::Float2)) != 0 ||
        memcmp(&m_ShadowTopColor, &target.m_ShadowTopColor, sizeof(nn::util::Unorm8x4)) != 0 ||
        memcmp(&m_ShadowBottomColor, &target.m_ShadowBottomColor, sizeof(nn::util::Unorm8x4)) != 0)
    {
        return false;
    }

    if (target.m_pMaterial != NULL)
    {
        if (m_pMaterial == NULL ||
            m_pMaterial->CompareCopiedInstanceTest(*target.m_pMaterial) == false)
        {
            return false;
        }
    }

    if (target.m_pLineWidthOffset != NULL)
    {
        if (m_pLineWidthOffset == NULL)
        {
            return false;
        }

        // 以下のメンバ変数はパディングが入らない前提で memcmp で比較しています。
        if (memcmp(m_pLineWidthOffset->pLineOffset, target.m_pLineWidthOffset->pLineOffset,
            sizeof(float) * TextBoxLineWidthOffsetCount) != 0 ||
            memcmp(m_pLineWidthOffset->pLineWidth, target.m_pLineWidthOffset->pLineWidth,
                sizeof(float) * TextBoxLineWidthOffsetCount) != 0)
        {
            return false;
        }
    }

    if (target.m_pPerCharacterTransform != NULL)
    {
        if (m_pPerCharacterTransform == NULL)
        {
            return false;
        }

        // PerCharacterTransformInfos はインスタンスごとに異なるため、その部分を除いて比較する。
        if (m_pPerCharacterTransform->Offset != target.m_pPerCharacterTransform->Offset ||
            m_pPerCharacterTransform->Width != target.m_pPerCharacterTransform->Width ||
            m_pPerCharacterTransform->LoopType != target.m_pPerCharacterTransform->LoopType ||
            m_pPerCharacterTransform->OriginV != target.m_pPerCharacterTransform->OriginV ||
            m_pPerCharacterTransform->OriginVOffset != target.m_pPerCharacterTransform->OriginVOffset ||
            m_pPerCharacterTransform->CurveNumber != target.m_pPerCharacterTransform->CurveNumber)
        {
            return false;
        }

        for (int i = 0; i < m_pPerCharacterTransform->CurveNumber; ++i)
        {
            if (m_pPerCharacterTransform->CurveInfos[i].pCurve != target.m_pPerCharacterTransform->CurveInfos[i].pCurve ||
                m_pPerCharacterTransform->CurveInfos[i].CurveType != target.m_pPerCharacterTransform->CurveInfos[i].CurveType)
            {
                return false;
            }
        }
    }

    if (target.m_pDispStringBuf != NULL)
    {
        if (m_pDispStringBuf == NULL)
        {
            return false;
        }

        if (m_pDispStringBuf->CompareCopiedInstanceTest(*target.m_pDispStringBuf) == false)
        {
            return false;
        }
    }

    return true;
}

uint16_t
TextBox::GetStringBufferLength() const
{
    if (m_TextBufLen == 0)
    {
        return 0;
    }

    return static_cast<uint16_t >(m_TextBufLen - 1);
}

uint16_t
TextBox::GetDrawStringBufferLength() const
{
    if (m_pDispStringBuf)
    {
        return static_cast<uint16_t >(m_pDispStringBuf->GetCharCountMax());
    }
    else
    {
        return 0;
    }
}

void
TextBox::AllocateStringBuffer(nn::gfx::Device* pDevice, uint16_t  minLen)
{
    AllocateStringBuffer(pDevice, minLen, minLen);
}

void
TextBox::AllocateStringBuffer(nn::gfx::Device* pDevice, uint16_t  minStrLen, uint16_t  minDrawStrLen)
{
#if !defined(NN_SDK_BUILD_RELEASE)
    if (IsCalculationFinishedSelf())
    {
        // Calculate が完了して Draw が行われるまでの状態にあるため、
        // そのタイミングで AllocateStringBuffer が呼ばれることは想定外とし、警告を出す。
        NN_DETAIL_UI2D_ERROR("AllocateStringBuffer must not be called between calculation and drawing.\n");
    }
#endif

    if (minStrLen == 0 || minDrawStrLen == 0)
    {
        return;
    }

    int32_t  textBufLen = minStrLen;
    int32_t  dispStringCharAttrLen = minDrawStrLen;
    int32_t  dispStringVertexBufLen = minDrawStrLen;
    if (m_Bits.shadowEnabled)
    {
        dispStringVertexBufLen *= 2;
    }
    ++textBufLen; // 終端文字分インクリメント

    // 要求する長さのバッファを既に確保している場合は何もしない
    if (textBufLen <= m_TextBufLen && (m_pDispStringBuf != NULL && dispStringCharAttrLen <= m_pDispStringBuf->GetCharCountMax()))
    {
        return;
    }

    FreeStringBuffer(pDevice);

    // 指定した文字数分VBOバッファを確保
    nn::font::DispStringBuffer::InitializeArg   drawBufferSizeCalculateArg;

    // DrawBuffer のサイズ判断条件を arg に設定する。
    drawBufferSizeCalculateArg.SetCharCountMax(dispStringCharAttrLen);

    const size_t  drawBufSize = sizeof(nn::font::DispStringBuffer) + nn::font::DispStringBuffer::GetRequiredDrawBufferSize(drawBufferSizeCalculateArg);

    // UTF-8 は 1 文字あたりのサイズが可変なので、最大の 4byte 分ずつ確保する
    void* pTextBuf = Layout::AllocateMemory((m_IsUtf8 ? (sizeof(char) * 4) : sizeof(uint16_t)) * textBufLen);
    void* pDispStringBuf = Layout::AllocateMemory(drawBufSize);
    if (NULL != m_pPerCharacterTransform)
    {
        m_pPerCharacterTransform->pPerCharacterTransformInfos = Layout::NewArray<nn::font::PerCharacterTransformInfo>(minDrawStrLen);
    }
    if (NULL == pTextBuf || NULL == pDispStringBuf || (NULL != m_pPerCharacterTransform && NULL == m_pPerCharacterTransform->pPerCharacterTransformInfos))
    {
        if (NULL != pTextBuf)
        {
            Layout::FreeMemory(pTextBuf);
        }
        if (NULL != pDispStringBuf)
        {
            Layout::FreeMemory(pDispStringBuf);
        }
        if (NULL != m_pPerCharacterTransform && NULL != m_pPerCharacterTransform->pPerCharacterTransformInfos)
        {
            Layout::DeletePrimArray(m_pPerCharacterTransform->pPerCharacterTransformInfos);
            m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        }
        return;
    }

    m_TextBuf.neutral = pTextBuf;
    m_TextBufLen = static_cast<uint16_t >(textBufLen);

    // DispStringBuffer を初期化する。
    m_pDispStringBuf = new(pDispStringBuf) nn::font::DispStringBuffer();

    nn::font::DispStringBuffer::InitializeArg arg;
    arg.SetCharCountMax(dispStringCharAttrLen);
    arg.SetDrawBuffer(nn::util::BytePtr(pDispStringBuf).Advance(sizeof(nn::font::DispStringBuffer)).Get());
    arg.SetShadowEnabled(m_Bits.shadowEnabled);
    arg.SetDoubleDrawnBorder(m_Bits.doubleDrawnBorderEnabled);
    arg.SetPerCharacterTransformEnabled(m_Bits.perCharacterTransformEnabled);
    arg.SetPerCharacterTransformAutoShadowAlpha(m_Bits.perCharacterTransformAutoShadowAlpha);

    m_pDispStringBuf->Initialize(pDevice, arg);
}

void
TextBox::FreeStringBuffer(nn::gfx::Device* pDevice)
{
    if (m_TextBuf.neutral)
    {
        if (m_pDispStringBuf != NULL)
        {
            m_pDispStringBuf->Finalize(pDevice);

            m_pDispStringBuf->~DispStringBuffer();
            Layout::FreeMemory(m_pDispStringBuf);
            m_pDispStringBuf = NULL;
        }
        Layout::FreeMemory(m_TextBuf.neutral);

        m_TextBuf.neutral = NULL;
        m_TextBufLen = 0;
        m_TextLen = 0;
    }
    if (m_pPerCharacterTransform)
    {
        if (m_pPerCharacterTransform->pPerCharacterTransformInfos != NULL)
        {
            Layout::DeletePrimArray(m_pPerCharacterTransform->pPerCharacterTransformInfos);
            m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        }
    }
}

uint16_t
TextBox::SetString(
    const uint16_t* pStr,
    uint16_t  dstIdx
)
{
    return SetStringImpl(pStr, dstIdx, nn::font::CalculateWideCharString16Length(pStr));
}

uint16_t
TextBox::SetStringUtf8(
    const char* pStr,
    uint16_t  dstIdx
)
{
    return SetStringImplUtf8(pStr, dstIdx, static_cast<int>(strlen(pStr)));
}

uint16_t
TextBox::SetString(
    const uint16_t* pStr,
    uint16_t  dstIdx,
    uint16_t  strLen
)
{
    return SetStringImpl(pStr, dstIdx, strLen);
}

uint16_t
TextBox::SetStringUtf8(
    const char* pStr,
    uint16_t  dstIdx,
    uint16_t  strLen
)
{
    return SetStringImplUtf8(pStr, dstIdx, strLen);
}

uint16_t
TextBox::SetStringImpl(
    const uint16_t*  pStr,
    uint16_t       dstIdx,
    int            strLen
)
{
    NN_SDK_ASSERT(!m_IsUtf8, "Invalid function for Utf8 Textbox! Use SetStringImplUtf8() instead. name[%s]", GetName());

    if (m_pFont == 0)
    {
        NN_DETAIL_UI2D_ERROR("m_pFont is NULL.\n");
        return 0;
    }

    if (m_TextBuf.utf16 == NULL)
    {
        NN_DETAIL_UI2D_ERROR("m_TextBuf is NULL.\n");
        return 0;
    }

    const int  bufLen = GetStringBufferLength();

    if (dstIdx >= bufLen)   // バッファを超えているためコピーしない
    {
        NN_DETAIL_UI2D_ERROR("dstIdx is out of range.\n");
        return 0;
    }

    int  cpLen = bufLen;
    cpLen -= dstIdx;        // コピー可能な文字数

    cpLen = std::min(strLen, cpLen);
    NN_DETAIL_UI2D_WARNING(cpLen >= strLen, "%d character(s) droped.\n", strLen - cpLen);

    std::memcpy(m_TextBuf.utf16 + dstIdx, pStr, cpLen * sizeof(uint16_t));

    // m_TextLen にセットする値は 16bitの bufLen 以下の値。
    if (m_TextBuf.utf16[0] == 0) {
        // ライブラリユーザが指定したstrLenが誤っていた場合、0か否かは致命的なので、
        // pStrに空文字列が指定された場合だけは正しい値(0)をm_TextLenに入れる
        m_TextLen = 0;
    } else {
        m_TextLen = static_cast<uint16_t >(dstIdx + cpLen);
    }
    m_TextBuf.utf16[m_TextLen] = 0;

    UpdatePTDirty(true);
    SetConstantBufferReadySelf(false);

    return static_cast<uint16_t >(cpLen);
}

uint16_t
TextBox::SetStringImplUtf8(
    const char*  pStr,
    uint16_t       dstIdx,
    int            strLen
)
{
    NN_SDK_ASSERT(m_IsUtf8, "Invalid function for Utf16 Textbox! Use SetStringImpl() instead. name[%s]", GetName());

    if (m_pFont == 0)
    {
        NN_DETAIL_UI2D_ERROR("m_pFont is NULL.\n");
        return 0;
    }

    if (m_TextBuf.utf8 == NULL)
    {
        NN_DETAIL_UI2D_ERROR("m_TextBuf is NULL.\n");
        return 0;
    }

    const int  bufLen = GetStringBufferLength();

    if (dstIdx >= bufLen)   // バッファを超えているためコピーしない
    {
        NN_DETAIL_UI2D_ERROR("dstIdx is out of range.\n");
        return 0;
    }

    int  cpLen = bufLen;
    cpLen -= dstIdx;        // コピー可能な文字数

    cpLen = std::min(strLen, cpLen);
    NN_DETAIL_UI2D_WARNING(cpLen >= strLen, "%d character(s) droped.\n", strLen - cpLen);

    std::memcpy(m_TextBuf.utf8 + dstIdx, pStr, cpLen);

    // m_TextLen にセットする値は 16bitの bufLen 以下の値。
    if (*m_TextBuf.utf8 == 0) {
        // ライブラリユーザが指定したstrLenが誤っていた場合、0か否かは致命的なので、
        // pStrに空文字列が指定された場合だけは正しい値(0)をm_TextLenに入れる
        m_TextLen = 0;
    } else {
        m_TextLen = static_cast<uint16_t >(dstIdx + cpLen);
    }
    m_TextBuf.utf8[m_TextLen] = 0;

    UpdatePTDirty(true);
    SetConstantBufferReadySelf(false);

    return static_cast<uint16_t >(cpLen);
}

const nn::font::Font*
TextBox::GetFont() const
{
    return m_pFont;
}

void
TextBox::SetFont(const nn::font::Font* pFont)
{
    if (pFont && pFont->GetCharacterCode() != nn::font::CharacterCode_Unicode)
    {
        NN_DETAIL_UI2D_ERROR("pFont is not uincode encoding.");
        return;
    }

    if (UpdatePTDirty(m_pFont != pFont))
    {
        m_pFont = pFont;

        if (m_pFont)
        {
            SetFontSize(Size::Create(static_cast<float>(m_pFont->GetWidth()), static_cast<float>(m_pFont->GetHeight())));
        }
        else
        {
            SetFontSize(Size::Create(0.f, 0.f));
        }
    }
}

void TextBox::SetFontSize(const Size& fontSize)
{
    if (UpdatePTDirty(!(m_FontSize == fontSize)))
    {
        if (m_FontSize.height == 0.0f)
        {
            // m_FontSize が初期化時の状態だった場合は斜体率を維持しない
            m_FontSize = fontSize;
        }
        else
        {
            // フォントサイズ変更前の斜体の角度θに対する
            // tanθ の値を求めます。
            const float widthPerHeight = m_FontSize.width / m_FontSize.height;
            float tanTheta = m_ItalicRatio * widthPerHeight;
            float shadowTanTheta = m_ShadowItalicRatio * widthPerHeight;

            m_FontSize = fontSize;

            // tanθ の値から、フォントサイズ変更後の斜体率に変換します。
            const float heightPerWidth = m_FontSize.height / m_FontSize.width;
            m_ItalicRatio = tanTheta * heightPerWidth;
            m_ShadowItalicRatio = shadowTanTheta * heightPerWidth;
        }
    }
}

void
TextBox::LoadMtx(DrawInfo& drawInfo)
{
    NN_UNUSED(drawInfo);
}

void
TextBox::SetFontInfo(nn::font::WideTextWriter* pWriter) const
{
    NN_SDK_ASSERT(!m_IsUtf8, "Invalid function for Utf8 Textbox! Use SetFontInfoUtf8() instead. name[%s]", GetName());

    pWriter->SetFont(m_pFont);
    if (m_pFont != NULL)
    {
        pWriter->SetFontSize(m_FontSize.width, m_FontSize.height);
        pWriter->SetLineSpace(m_LineSpace);
        pWriter->SetCharSpace(m_CharSpace);
        if (IsWidthLimitEnabled())
        {
            pWriter->SetWidthLimit(GetSize().width);
        }
        else
        {
            pWriter->ResetWidthLimit();
        }
    }

    if (m_pTagProcessor.utf16)
    {
        pWriter->SetTagProcessor(m_pTagProcessor.utf16);
    }
}

void
TextBox::SetFontInfoUtf8(nn::font::TextWriter* pWriter) const
{
    NN_SDK_ASSERT(m_IsUtf8, "Invalid function for Utf16 Textbox! Use SetFontInfo() instead. name[%s]", GetName());

    pWriter->SetFont(m_pFont);
    if (m_pFont != NULL)
    {
        pWriter->SetFontSize(m_FontSize.width, m_FontSize.height);
        pWriter->SetLineSpace(m_LineSpace);
        pWriter->SetCharSpace(m_CharSpace);
        if (IsWidthLimitEnabled())
        {
            pWriter->SetWidthLimit(GetSize().width);
        }
        else
        {
            pWriter->ResetWidthLimit();
        }
    }

    if (m_pTagProcessor.utf8)
    {
        pWriter->SetTagProcessor(m_pTagProcessor.utf8);
    }
}

void
TextBox::SetTextPos(nn::font::WideTextWriter* pWriter) const
{
    NN_SDK_ASSERT(!m_IsUtf8, "Invalid function for Utf8 Textbox! Use SetTextPosUtf8() instead. name[%s]", GetName());

    uint32_t  value = 0;

    if (m_pLineWidthOffset != NULL)
    {
        // 行ごとの幅とオフセットが指定されているときは左上揃えにする
        value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignLeft;
    }
    else
    {
        switch (GetTextAlignment())
        {
        case TextAlignment_Synchronous:
        default:
            switch (GetTextPositionX())
            {
            case HorizontalPosition_Left:
            default:                        value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignLeft;   break;
            case HorizontalPosition_Center: value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignCenter; break;
            case HorizontalPosition_Right:  value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignRight;  break;
            }
            break;

        case TextAlignment_Left:        value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignLeft;    break;
        case TextAlignment_Center:      value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignCenter;  break;
        case TextAlignment_Right:       value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignRight;   break;
        }

        switch (GetTextPositionX())
        {
        case HorizontalPosition_Left:
        default:
            // NOTE: HORIZONTAL_ORIGIN_LEFT は 0 なので何もしなくていい。
            //value |= font::WideTextWriter::HORIZONTAL_ORIGIN_LEFT;
            break;
        case HorizontalPosition_Center:
            value |= nn::font::WideTextWriter::PositionFlag_HorizontalOriginCenter;
            break;
        case HorizontalPosition_Right:
            value |= nn::font::WideTextWriter::PositionFlag_HorizontalOriginRight;
            break;
        }

        switch (GetTextPositionY())
        {
        case VerticalPosition_Top:
        default:
            // NOTE: VERTICAL_ORIGIN_TOP は 0 なので何もしなくていい。
            //value |= font::WideTextWriter::VERTICAL_ORIGIN_TOP;
            break;
        case VerticalPosition_Center:
            value |= nn::font::WideTextWriter::PositionFlag_VerticalOriginMiddle;
            break;
        case VerticalPosition_Bottom:
            value |= nn::font::WideTextWriter::PositionFlag_VerticalOriginBottom;
            break;
        }
    }

    pWriter->SetDrawFlag(value);
}

void
TextBox::SetTextPosUtf8(nn::font::TextWriter* pWriter) const
{
    NN_SDK_ASSERT(m_IsUtf8, "Invalid function for Utf16 Textbox! Use SetTextPos() instead. name[%s]", GetName());

    uint32_t  value = 0;

    if (m_pLineWidthOffset != NULL)
    {
        // 行ごとの幅とオフセットが指定されているときは左上揃えにする
        value = nn::font::WideTextWriter::PositionFlag_HorizontalAlignLeft;
    }
    else
    {
        switch (GetTextAlignment())
        {
        case TextAlignment_Synchronous:
        default:
            switch (GetTextPositionX())
            {
            case HorizontalPosition_Left:
            default:                        value = nn::font::TextWriter::PositionFlag_HorizontalAlignLeft;   break;
            case HorizontalPosition_Center: value = nn::font::TextWriter::PositionFlag_HorizontalAlignCenter; break;
            case HorizontalPosition_Right:  value = nn::font::TextWriter::PositionFlag_HorizontalAlignRight;  break;
            }
            break;

        case TextAlignment_Left:        value = nn::font::TextWriter::PositionFlag_HorizontalAlignLeft;    break;
        case TextAlignment_Center:      value = nn::font::TextWriter::PositionFlag_HorizontalAlignCenter;  break;
        case TextAlignment_Right:       value = nn::font::TextWriter::PositionFlag_HorizontalAlignRight;   break;
        }

        switch (GetTextPositionX())
        {
        case HorizontalPosition_Left:
        default:
            // NOTE: HORIZONTAL_ORIGIN_LEFT は 0 なので何もしなくていい。
            //value |= font::TextWriter::HORIZONTAL_ORIGIN_LEFT;
            break;
        case HorizontalPosition_Center:
            value |= nn::font::TextWriter::PositionFlag_HorizontalOriginCenter;
            break;
        case HorizontalPosition_Right:
            value |= nn::font::TextWriter::PositionFlag_HorizontalOriginRight;
            break;
        }

        switch (GetTextPositionY())
        {
        case VerticalPosition_Top:
        default:
            // NOTE: VERTICAL_ORIGIN_TOP は 0 なので何もしなくていい。
            //value |= font::TextWriter::VERTICAL_ORIGIN_TOP;
            break;
        case VerticalPosition_Center:
            value |= nn::font::TextWriter::PositionFlag_VerticalOriginMiddle;
            break;
        case VerticalPosition_Bottom:
            value |= nn::font::TextWriter::PositionFlag_VerticalOriginBottom;
            break;
        }
    }

    pWriter->SetDrawFlag(value);
}

nn::util::Float2
TextBox::AdjustTextPos(
    const Size&     size,
    bool            isCeil
) const
{
    nn::util::Float2 pos;

    if (m_pLineWidthOffset != NULL)
    {
        // 行ごとの幅とオフセットが指定されているときは左上揃えにする
        pos.v[0] = 0.f;
        pos.v[1] = 0.f;
    }
    else
    {
        switch (GetTextPositionX())
        {
        case HorizontalPosition_Left:
        default:
            pos.v[0] = 0.f;
            break;
        case HorizontalPosition_Center:
            pos.v[0] = detail::AdjustCenterValue(size.width, isCeil);
            break;
        case HorizontalPosition_Right:
            pos.v[0] = size.width;
            break;
        }

        switch (GetTextPositionY())
        {
        case VerticalPosition_Top:
        default:
            pos.v[1] = 0.f;
            break;
        case VerticalPosition_Center:
            pos.v[1] = detail::AdjustCenterValue(size.height, isCeil);
            break;
        case VerticalPosition_Bottom:
            pos.v[1] = size.height;
            break;
        }
    }

    return pos;
}

void
TextBox::GetTextGlobalMtx(nn::util::MatrixT4x3fType* pMtx, const DrawInfo& drawInfo) const
{
    // TextBox の姿勢行列は Material ではなく、フォント描画のコンスタントバッファのものが使用されます。
    // ペインエフェクト機能のキャプチャ時の描画位置を調整するため、Pane::UpdateMaterialConstantBufferForEffectCapture() 相当の処理を別途実装しています。
    if (IsPaneEffectEnabled())
    {
        detail::CalculateCaptureRootMatrix(*pMtx, drawInfo);
    }
    else
    {
        *pMtx = GetGlobalMtx();
    }

    // テキストの描画開始位置の調整をViewModel行列で行う。
    nn::util::Float2 pos = GetVertexPos();
    const nn::util::Float2& txtPos = AdjustTextPos(GetSize(), false);

    pos.v[0] += txtPos.v[0];
    pos.v[1] -= txtPos.v[1];

    nn::util::FloatT4x3 mtx;
    nn::util::MatrixStore(&mtx, *pMtx);

    mtx.m[0][3] += mtx.m[0][0] * pos.v[0] + mtx.m[0][1] * pos.v[1];
    mtx.m[1][3] += mtx.m[1][0] * pos.v[0] + mtx.m[1][1] * pos.v[1];
    mtx.m[2][3] += mtx.m[2][0] * pos.v[0] + mtx.m[2][1] * pos.v[1];

    // TextWriterはY軸正を下にしているので、YとZを反転させる。
    //     Yの反転。
    mtx.m[0][1] = - mtx.m[0][1];
    mtx.m[1][1] = - mtx.m[1][1];
    mtx.m[2][1] = - mtx.m[2][1];

    //     Zの反転。
    //     Zは影響が無いので計算を省略。
    /*
    mtx.m[0][2] = - mtx.m[0][2];
    mtx.m[1][2] = - mtx.m[1][2];
    mtx.m[2][2] = - mtx.m[2][2];
    */

    nn::util::MatrixLoad(pMtx, mtx);
}

void
TextBox::SetupTextWriter(nn::font::WideTextWriter* pWriter)
{
    pWriter->SetCursor(0, 0);
    SetFontInfo(pWriter);
    SetTextPos(pWriter);

    pWriter->SetTextColor(m_TextColors[TextColor_Top], m_TextColors[TextColor_Bottom]);
    pWriter->SetItalicRatio(m_ItalicRatio);
}

void
TextBox::SetupTextWriterUtf8(nn::font::TextWriter* pWriter)
{
    pWriter->SetCursor(0, 0);
    SetFontInfoUtf8(pWriter);
    SetTextPosUtf8(pWriter);

    pWriter->SetTextColor(m_TextColors[TextColor_Top], m_TextColors[TextColor_Bottom]);
    pWriter->SetItalicRatio(m_ItalicRatio);
}

void TextBox::SetPerCharacterTransform(int idx, float value)
{
    NN_SDK_ASSERT_NOT_NULL(m_pPerCharacterTransform);
    switch (idx)
    {
    case AnimTargetPerCharacterTransform_EvalTypeOffset:
        m_pPerCharacterTransform->Offset = value;
        break;
    case AnimTargetPerCharacterTransform_EvalTypeWidth:
        m_pPerCharacterTransform->Width = value;
        break;
    default:
        NN_SDK_ASSERT(false);
        break;
    }
    UpdatePTDirty(true);
}

float TextBox::GetPerCharacterTransform(int idx) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pPerCharacterTransform);
    switch (idx)
    {
    case AnimTargetPerCharacterTransform_EvalTypeOffset:
        return m_pPerCharacterTransform->Offset;
    case AnimTargetPerCharacterTransform_EvalTypeWidth:
        return m_pPerCharacterTransform->Width;
    default:
        NN_SDK_ASSERT(false);
        return 0.0f;
    }
}

template <typename T>
void TextBox::UpdatePerCharacterTransform(nn::font::TagProcessorBase<T>* pTagProcessor)
{
    NN_SDK_ASSERT(m_pPerCharacterTransform != NULL, "name[%s]", GetName());
    const uint32_t  charCount = m_pDispStringBuf->GetCharCount();
    if (charCount == 0)
    {
        return;
    }

    // 1 文字ごとに掛ける係数を求める
    const float step = CalculateStep(pTagProcessor);

    // 文字毎にアニメーションを設定する
    float frame = m_pPerCharacterTransform->Offset;
    const void* pStr = m_TextBuf.neutral;
    float width = 0.0f;

    if (IsPerCharacterTransformFixSpaceEnabled())
    {
        // 文字揃えにより開始位置を変える
        switch (m_pPerCharacterTransform->FixSpaceOrigin)
        {
        case HorizontalAlignment_Left:
            // 何もしない
            break;
        case HorizontalAlignment_Center:
            frame = m_pPerCharacterTransform->Offset + (m_pPerCharacterTransform->Width - m_pPerCharacterTransform->FixSpaceWidth * charCount) / 2.0f;
            break;
        case HorizontalAlignment_Right:
            frame = m_pPerCharacterTransform->Offset + m_pPerCharacterTransform->Width - m_pPerCharacterTransform->FixSpaceWidth * charCount;
            break;
        default:
            NN_SDK_ASSERT(false);
            break;
        }
    }

    for (uint32_t i = 0; i < charCount; i++)
    {
        if (IsPerCharacterTransformFixSpaceEnabled())
        {
            frame += step / 2.0f;
        }
        else if (m_Bits.perCharacterTransformSplitByCharWidth)
        {
            uint32_t c;
            pStr = AcquireNextPrintableChar(&c, pTagProcessor, pStr);
            if (pStr == NULL)
            {
                break;
            }
            width = static_cast<float>(m_pFont->GetCharWidth(c)) + GetCharSpace();
            if (m_Bits.perCharacterTransformOriginToCenter)
            {
                // 文字の中心を基準にするために、文字幅の半分を 2 回に分けて足す(前半)
                frame += width * step / 2.0f;
            }
            else if (i > 0)
            {
                frame += width * step;
            }
        }
        else
        {
            // frame += step; とすると計算誤差が累積して互換性が失われるためその都度乗算で求める。
            frame = m_pPerCharacterTransform->Offset + step * static_cast<float>(i);
        }
        nn::font::PerCharacterTransformInfo* pPerCharacterTransformInfo = &m_pPerCharacterTransform->pPerCharacterTransformInfos[i];
        ApplyPerCharacterTransformCurve(pPerCharacterTransformInfo, frame);

        if (IsPerCharacterTransformFixSpaceEnabled())
        {
            frame += step / 2.0f;
        }
        else if (m_Bits.perCharacterTransformOriginToCenter && m_Bits.perCharacterTransformSplitByCharWidth)
        {
            // 文字の中心を基準にするために、文字幅の半分を 2 回に分けて足す(後半)
            frame += width * step / 2.0f;
        }
    }
}

template <typename T>
float TextBox::CalculateStep(nn::font::TagProcessorBase<T>* pTagProcessor)
{
    const uint32_t  charCount = m_pDispStringBuf->GetCharCount();
    float step = 0.0f;
    if (IsPerCharacterTransformFixSpaceEnabled())
    {
        step = m_pPerCharacterTransform->FixSpaceWidth;
    }
    else if (m_Bits.perCharacterTransformSplitByCharWidth)
    {
        NN_SDK_ASSERT(m_pFont != NULL, "Font was not set to the TextBox.");
        {
            float totalWidth = 0.0f;
            const void* pStr = m_TextBuf.neutral;
            for (uint32_t i = 0; i < charCount; i++)
            {
                uint32_t c;
                pStr = AcquireNextPrintableChar(&c, pTagProcessor, pStr);
                if (pStr == NULL)
                {
                    break;
                }
                if (m_Bits.perCharacterTransformOriginToCenter)
                {
                    totalWidth += static_cast<float>(m_pFont->GetCharWidth(c)) + GetCharSpace();
                }
                else if (i > 0)
                {
                    totalWidth += static_cast<float>(m_pFont->GetCharWidth(c)) + GetCharSpace();
                }
            }
            step = totalWidth == 0.0f ? 0.0f : m_pPerCharacterTransform->Width / totalWidth;
        }
    }
    else
    {
        step = charCount == 1 ? 0.0f : m_pPerCharacterTransform->Width / static_cast<float>(charCount - 1);
    }
    return step;
}

template <typename T>
const void* TextBox::AcquireNextPrintableChar(uint32_t* pChar, nn::font::TagProcessorBase<T>* pTagProcessor, const void* pStr)
{
    bool isPrintable;
    const T* pNext;

    // 描画可能文字の位置が得られるまで繰り返す
    for (; ; )
    {
        pNext = pTagProcessor->AcquireNextPrintableChar(&isPrintable, static_cast<const T*>(pStr));
        NN_SDK_ASSERT(pNext != NULL, "nn::font::TagProcessorBase::AcquireNextPrintableChar must not return NULL.");
        if (isPrintable)
        {
            break;
        }
        if (!ValidateNextPrintableChar(static_cast<const T*>(pStr), pNext))
        {
            *pChar = 0;
            return NULL;
        }
        pStr = pNext;
    }

    // 描画可能文字の位置から 1 文字を取得
    *pChar = GetCharFromPointer(static_cast<const T*>(pStr));
    return pNext;
}

bool TextBox::ValidateNextPrintableChar(const char* pOld, const char* pNew)
{
    if (pNew > m_TextBuf.utf8 + m_TextLen)
    {
        NN_SDK_ASSERT(false, "nn::font::TagProcessorBase::AcquireNextPrintableChar returned a pointer over the length of the string.");
        return false;
    }
    if (pNew <= pOld)
    {
        NN_SDK_ASSERT(false, "nn::font::TagProcessorBase::AcquireNextPrintableChar must return a pointer ahead of the previous one.");
        return false;
    }
    return true;
}

bool TextBox::ValidateNextPrintableChar(const uint16_t* pOld, const uint16_t* pNew)
{
    if (pNew > m_TextBuf.utf16 + m_TextLen)
    {
        NN_SDK_ASSERT(false, "nn::font::TagProcessorBase::AcquireNextPrintableChar returned a pointer over the length of the string.");
        return false;
    }
    if (pNew <= pOld)
    {
        NN_SDK_ASSERT(false, "nn::font::TagProcessorBase::AcquireNextPrintableChar must return a pointer ahead of the previous one.");
        return false;
    }
    return true;
}

uint32_t TextBox::GetCharFromPointer(const char* pStr)
{
    return nn::font::ConvertCharUtf8ToUtf32AndIncrement(&pStr);
}

uint32_t TextBox::GetCharFromPointer(const uint16_t* pStr)
{
    return *pStr;
}

void TextBox::ApplyPerCharacterTransformCurve(nn::font::PerCharacterTransformInfo* pPerCharacterTransformInfo, float frame)
{
    const uint32_t  curveNumber = m_pPerCharacterTransform->CurveNumber;
    for (uint32_t  j = 0; j < curveNumber; j++)
    {
        const nn::ui2d::ResAnimationTarget* pAnimTarget = m_pPerCharacterTransform->CurveInfos[j].pCurve;
        AnimTargetPerCharacterTransformCurve curveType = m_pPerCharacterTransform->CurveInfos[j].CurveType;
        const ResHermiteKey* keys = nn::util::ConstBytePtr(pAnimTarget, pAnimTarget->keysOffset).Get<ResHermiteKey>();
        const uint16_t keyCountMin = 2;
        const float loopFrame = (m_pPerCharacterTransform->LoopType == static_cast<uint8_t>(PerCharacterTransformLoopType_Loop)
            && pAnimTarget->keyCount >= keyCountMin) ? GetLoopFrame(frame, keys[0].frame, keys[pAnimTarget->keyCount - 1].frame) : frame;
        const float value = nn::ui2d::GetHermiteCurveValue(loopFrame, keys, pAnimTarget->keyCount);
        if (curveType <= AnimTargetPerCharacterTransformCurve_TranslateZ)
        {
            // AnimTargetPerCharacterTransformCurve_TranslateX から ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Zまで
            uint32_t  index = curveType - AnimTargetPerCharacterTransformCurve_TranslateX;
            pPerCharacterTransformInfo->Translation[index] = (curveType == AnimTargetPerCharacterTransformCurve_TranslateY ? - value : value);
        }
        else if (curveType <= AnimTargetPerCharacterTransformCurve_RotateZ)
        {
            // AnimTargetPerCharacterTransformCurve_RotateX から ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Zまで
            const uint32_t index = curveType - AnimTargetPerCharacterTransformCurve_RotateX;
            const float degree = (curveType == AnimTargetPerCharacterTransformCurve_RotateY ? value : - value);
            nn::util::SinCosTable(&pPerCharacterTransformInfo->RotationSin[index], &pPerCharacterTransformInfo->RotationCos[index], nn::util::DegreeToAngleIndex(degree));
        }
        else if (curveType <= AnimTargetPerCharacterTransformCurve_LeftTopA)
        {
            // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_RからANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_Aまで
            const uint32_t index = curveType - AnimTargetPerCharacterTransformCurve_LeftTopR;
            pPerCharacterTransformInfo->LT[index] = static_cast<uint8_t >(value);
        }
        else if (curveType <= AnimTargetPerCharacterTransformCurve_LeftBottomA)
        {
            // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_RからANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_Aまで
            const uint32_t index = curveType - AnimTargetPerCharacterTransformCurve_LeftBottomR;
            pPerCharacterTransformInfo->LB[index] = static_cast<uint8_t >(value);
        }
        else/* if (curveType <= AnimTargetPerCharacterTransformCurve_ScaleY)*/
        {
            // AnimTargetPerCharacterTransformCurve_ScaleX から AnimTargetPerCharacterTransformCurve_ScaleY まで
            const uint32_t index = curveType - AnimTargetPerCharacterTransformCurve_ScaleX;
            pPerCharacterTransformInfo->Scale[index] = value;
        }
    }
}

} // namespace nn::ui2d
} // namespace nn
