﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Picture.h>

#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/detail/ui2d_Log.h>

namespace nn
{
namespace ui2d
{

Picture::Picture(int  texNum)
{
    NN_SDK_ASSERT(texNum <= TexMapMax, "out of bounds: texNum[%u] <= TexMapMax for Picture[%s]", texNum, GetName());

    Initialize(texNum);
    InitializeMaterial(texNum);
}

Picture::Picture(const TextureInfo& textureInfo)
{
    const int texNum = 1;
    Initialize(texNum);
    InitializeMaterial(texNum);
    if (m_pMaterial)
    {
        Append(textureInfo);
    }
}

Picture::Picture(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResPicture* pBaseBlock,
    const ResPicture* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
    : Base(pOutBuildResultInformation, pDevice, pBaseBlock, buildArgSet)
{
    const ResPicture* pBlock;
    if (pOverrideBlock && buildArgSet.overrideMaterialUsageFlag == 0)
    {
        // 丸ごと上書きする場合
        pBlock = pOverrideBlock;
    }
    else
    {
        // 上書きがない、もしくは部分上書きの場合
        pBlock = pBaseBlock;
    }

    bool shapeEnabled = CheckAdditionalInfoFlag(pBlock, PicturePaneFlag_ShapeBinaryIndex);
    const uint8_t  texCoordNum = static_cast<uint8_t >(std::min(static_cast<int>(pBlock->texCoordCount), TexMapMax));

    // Shape 機能追加時に、使用しないユーザーへの影響を極力少なくするために
    // TexCoord と Shape 情報は同じメモリ領域を共用しています。
    // そのため、Shape 機能が有効の場合は TexCoord 用のメモリを確保しないように
    // Initialize 呼び出し時の引数を強制的に設定します。
    if (shapeEnabled)
    {
        Initialize(0);
    }
    else
    {
        Initialize(texCoordNum);
    }

    // 頂点カラー
    for (int i = 0; i < VertexColor_MaxVertexColor; ++i)
    {
        m_VertexColors[i] = pBlock->vtxCols[i];
    }

    // 追加情報へアクセスする際のオフセット
    size_t  additionalDataOffset = sizeof(*pBlock);

    // テクスチャ座標
    if (texCoordNum > 0)
    {
        if (!shapeEnabled &&
            !m_SharedMemory.m_TexCoordArray.IsEmpty())
        {
            m_SharedMemory.m_TexCoordArray.Copy(reinterpret_cast<const char*>(pBlock) + additionalDataOffset, texCoordNum);
        }
        additionalDataOffset += sizeof(nn::util::Float2) * PaneVertex_MaxPaneVertex * texCoordNum;
    }

    // シェイプバイナリインデックス
    if (shapeEnabled)
    {
        uint32_t    shapeBinaryIndex = *(reinterpret_cast<const uint32_t*>((reinterpret_cast<const char*>(pBlock) + additionalDataOffset)));
        additionalDataOffset += sizeof(uint32_t);

        InitializeShape(pOutBuildResultInformation, pDevice, pBlock, buildArgSet, shapeBinaryIndex);
    }

    // マテリアルの作成
    {
        const ResMaterial *const pResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseBlock->materialIdx);
        const ResMaterial *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            pOverrideResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideBlock->materialIdx);
        }

        const size_t systemDataSize = sizeof(SystemDataCopyCaptureTextureInfo) + sizeof(CaptureTextureCopyInfo);
        void* pSystemData = alloca(systemDataSize);
        NN_SDK_ASSERT_NOT_NULL(pSystemData);
        memset(pSystemData, 0, systemDataSize);

        CaptureTextureCopyInfo* pCopyInfo = nn::util::BytePtr(pSystemData, sizeof(SystemDataCopyCaptureTextureInfo)).Get<CaptureTextureCopyInfo>();

        m_pMaterial = Layout::AllocateAndConstruct<Material, BuildResultInformation*, nn::gfx::Device*, const ResMaterial*, const ResMaterial*, const BuildArgSet&, CaptureTextureCopyInfo*>(pOutBuildResultInformation, pDevice, pResMaterial, pOverrideResMaterial, buildArgSet, pCopyInfo);
        if (pCopyInfo->useFlags != 0)
        {
            SystemDataCopyCaptureTextureInfo* pCopyInfoHeader = nn::util::BytePtr(pSystemData, 0).Get<SystemDataCopyCaptureTextureInfo>();
            pCopyInfoHeader->type = PaneSystemDataType_CaptureTextureRuntimeCopyInfo;
            pCopyInfoHeader->count = 1;
            AddDynamicSystemExtUserData(PaneSystemDataType_CaptureTextureRuntimeCopyInfo, pCopyInfoHeader, systemDataSize);
        }
    }

    // 角丸機能の初期化処理
    InitializeProceduralShape(pOutBuildResultInformation, pDevice);
}

void
Picture::CopyImpl(const Picture& picture, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    // 頂点カラー
    for (int i = 0; i < VertexColor_MaxVertexColor; ++i)
    {
        m_VertexColors[i] = picture.m_VertexColors[i];
    }
    // テクスチャ座標
    if (picture.IsTexCoordArrayEnabled())
    {
        const int  texCoordNum = picture.m_SharedMemory.m_TexCoordArray.GetSize();
        if (texCoordNum > 0)
        {
            m_SharedMemory.m_TexCoordArray.Initialize();

            ReserveTexCoord(texCoordNum);
            m_SharedMemory.m_TexCoordArray.SetSize(texCoordNum);
            for (int i = 0; i < texCoordNum; i++)
            {
                m_SharedMemory.m_TexCoordArray.SetCoord(i, picture.m_SharedMemory.m_TexCoordArray.GetArray()[i]);
            }
        }
    }
    else
    {
        CopyShapeInfo(picture);
    }

    if (detail::TestBit(picture.GetSystemExtDataFlag(), PaneSystemDataType_CaptureTextureRuntimeCopyInfo))
    {
        MaterialCopyContext copyContext;
        copyContext.pCaptureTextureCopyInfo = nn::util::ConstBytePtr(picture.FindSystemExtDataByType(PaneSystemDataType_CaptureTextureRuntimeCopyInfo), sizeof(SystemDataCopyCaptureTextureInfo)).Get<CaptureTextureCopyInfo>();
        copyContext.pDevice = pDevice;
        copyContext.pResAccessor = pResAccessor;
        copyContext.pNewRootName = pNewRootName;

        // マテリアルの複製
        m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, MaterialCopyContext&>(*picture.m_pMaterial, copyContext);
    }
    else
    {
        // マテリアルの複製
        m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, nn::gfx::Device*>(*picture.m_pMaterial, pDevice);
    }


    // 角丸機能の初期化処理
    InitializeProceduralShape(NULL, pDevice);
}

void
Picture::CopyShapeInfo(const Picture& picture)
{
    ShapeDrawInfo* pOriginalInfo = picture.m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo;

    switch (pOriginalInfo->m_ShapeType)
    {
    case ResShapeInfo::ShapeInfo_GfxPrimitiveRoundRect:
        {
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = Layout::AllocateAndConstruct<ShapeDrawInfo>();
            RoundRectShape* pShape = Layout::AllocateAndConstruct<
                RoundRectShape,
                nn::gfx::util::PrimitiveShapeFormat,
                nn::gfx::PrimitiveTopology,
                float, float, uint32_t>(VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, 0.0f, 0.0f, 1);
            RoundRectShape* pOriginalShape = static_cast<RoundRectShape*>(pOriginalInfo->pShape);
            // パラメータのコピー
            // 頂点バッファ、インデックスは中でメモリを確保してコピーしている
            pShape->CopyParams(*pOriginalShape);
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape = pShape;
        }
        break;

    case ResShapeInfo::ShapeInfo_GfxPrimitiveCircle:
        {
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = Layout::AllocateAndConstruct<ShapeDrawInfo>();
            Ui2dCircleShape* pOriginalShape = static_cast<Ui2dCircleShape*>(pOriginalInfo->pShape);

            // 分割数は CircleShape から取得するインターフェイスがないため頂点数から逆算して設定している。
            Ui2dCircleShape* pShape = Layout::AllocateAndConstruct<
                Ui2dCircleShape,
                nn::gfx::util::PrimitiveShapeFormat,
                nn::gfx::PrimitiveTopology,
                int>(VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, pOriginalShape->GetVertexCount() - 1);
            // パラメータのコピー
            // 頂点バッファ、インデックスは中でメモリを確保してコピーしている
            pShape->CopyParams(*pOriginalShape);
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape = pShape;
        }
        break;
    default:
        m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = NULL;
        break;
    }

    // PrimitiveShape で頂点バッファをインデックスバッファを作成する。
    if (m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo != NULL)
    {
        m_SharedMemory.m_UnionTypeFlag = SharedMemory::UnionType_ShapeDrawInfo;
        m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->m_ShapeType = pOriginalInfo->m_ShapeType;
    }
}

void
Picture::Initialize(int  texNum)
{
    m_pMaterial = NULL;

    for (int i = 0; i < VertexColor_MaxVertexColor; i++)
    {
        nn::util::Unorm8x4 vertexColor = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        m_VertexColors[i] = vertexColor;
    }

    if (texNum > 0)
    {
        m_SharedMemory.m_TexCoordArray.Initialize();
        ReserveTexCoord(texNum);
    }
}

void
Picture::InitializeMaterial(int texCount)
{
    // マテリアルの作成
    m_pMaterial = Layout::AllocateAndConstruct<Material>();
    if (m_pMaterial)
    {
        SetDefaultShaderId(m_pMaterial, texCount);

        m_pMaterial->ReserveMem(texCount, texCount, texCount);
    }
}

void
Picture::InitializeShape(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResPicture* pBlock,
    const BuildArgSet& buildArgSet,
    uint32_t shapeBinaryIndex)
{
    const BuildResSet*  pBuildRes = buildArgSet.pCurrentBuildResSet;
    if (buildArgSet.pOverrideBuildResSet != NULL &&
        buildArgSet.pOverrideBuildResSet->pShapeInfoList != NULL)
    {
        pBuildRes = buildArgSet.pOverrideBuildResSet;
    }

    const ResShapeInfo *const pResShapeInfo = nn::ui2d::detail::GetResShapeInfo(pBuildRes, shapeBinaryIndex);

    const char* pDetailInfoTop = reinterpret_cast<const char*>(pResShapeInfo) + sizeof(ResShapeInfo);

    // シェイプの形状によって PrimitiveShape インスタンスを作成する。
    switch (pResShapeInfo->type)
    {
    case ResShapeInfo::ShapeInfo_GfxPrimitiveRoundRect:
        {
            const ResShapeInfoGfxPrimitiveRoundRect* pResRoundRect = reinterpret_cast<const ResShapeInfoGfxPrimitiveRoundRect*>(pDetailInfoTop);

            float   radiusH = 0.0f;
            float   radiusV = 0.0f;
            float   aspectRatio = 1.0f;
            if (pBlock->size.x > pBlock->size.y)
            {
                aspectRatio = fabs(pBlock->size.x / pBlock->size.y);
                radiusV = fabs((float)pResRoundRect->radius / pBlock->size.y);
                radiusH = radiusV / aspectRatio;
            }
            else
            {
                aspectRatio = fabs(pBlock->size.y / pBlock->size.x);
                radiusH = fabs((float)pResRoundRect->radius / pBlock->size.x);
                radiusV = radiusH / aspectRatio;
            }

            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = Layout::AllocateAndConstruct<ShapeDrawInfo>();
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape = Layout::AllocateAndConstruct<
                RoundRectShape,
                nn::gfx::util::PrimitiveShapeFormat,
                nn::gfx::PrimitiveTopology,
                float, float, uint32_t>(VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, radiusH, radiusV, pResRoundRect->slice);
        }
        break;

    case ResShapeInfo::ShapeInfo_GfxPrimitiveCircle:
        {
            const ResShapeInfoGfxPrimitiveCircle* pResCircle = reinterpret_cast<const ResShapeInfoGfxPrimitiveCircle*>(pDetailInfoTop);
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = Layout::AllocateAndConstruct<ShapeDrawInfo>();
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape = Layout::AllocateAndConstruct<
                Ui2dCircleShape,
                nn::gfx::util::PrimitiveShapeFormat,
                nn::gfx::PrimitiveTopology,
                int>(VertexFormat_PUv, nn::gfx::PrimitiveTopology_TriangleList, pResCircle->slice);
        }
        break;
    default:
        m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = NULL;
        break;
    }

    // PrimitiveShape で頂点バッファをインデックスバッファを作成する。
    if (m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo != NULL)
    {
        m_SharedMemory.m_UnionTypeFlag = SharedMemory::UnionType_ShapeDrawInfo;
        m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->m_ShapeType = pResShapeInfo->type;
        size_t  vertexBufferSize = m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBufferSize();
        size_t  indexBufferSize = m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBufferSize();

        // バッファの計算
        m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->Calculate(
            Layout::AllocateMemory(vertexBufferSize),
            vertexBufferSize,
            Layout::AllocateMemory(indexBufferSize),
            indexBufferSize);

        if (pOutBuildResultInformation != NULL)
        {
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_VertexBuffer, vertexBufferSize);
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_IndexBuffer, indexBufferSize);
        }
    }
}


Picture::~Picture()
{
    NN_SDK_ASSERT(m_pMaterial == 0, "m_pMaterial must be freed before destruction for Picture[%s]", GetName());
}

void
Picture::Finalize(nn::gfx::Device* pDevice)
{
    Pane::Finalize(pDevice);

    if (m_pMaterial && ! m_pMaterial->IsUserAllocated())
    {
        m_pMaterial->Finalize(pDevice);
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    if (IsTexCoordArrayEnabled())
    {
        m_SharedMemory.m_TexCoordArray.Free();
    }
    else
    {
        if (m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo != NULL)
        {
            void*   pVertexBuffer = m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBuffer();
            if (pVertexBuffer != NULL)
            {
                Layout::FreeMemory(pVertexBuffer);
            }
            void*   pIndexBuffer = m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBuffer();
            if (pIndexBuffer != NULL)
            {
                Layout::FreeMemory(pIndexBuffer);
            }
            Layout::DeleteObj(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape);
            Layout::DeleteObj(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo);
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo = NULL;
        }
    }
}

uint8_t
Picture::GetMaterialCount() const
{
    return static_cast<uint8_t >(m_pMaterial ? 1 : 0);
}

Material*
Picture::GetMaterial(int  idx) const
{
    NN_DETAIL_UI2D_WARNING(idx < GetMaterialCount(), "idx >= GetMaterialCount() : %d >= %d", idx, GetMaterialCount());

    return idx == 0 ? m_pMaterial : 0;
}

void Picture::SetMaterial(Material* pMaterial)
{
    if (m_pMaterial == pMaterial)
    {
        return;
    }

    if (m_pMaterial != NULL && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }

    m_pMaterial = pMaterial;
}

void
Picture::Append(const TextureInfo& textureInfo)
{
    if (m_pMaterial->GetTexMapCount() >= m_pMaterial->GetTexMapCap() || m_pMaterial->GetTexCoordGenCount() >= m_pMaterial->GetTexCoordGenCap())
    {
        NN_DETAIL_UI2D_ERROR("m_pMaterial->GetTexMapCount(%d) is large. m_pMaterial->GetTexMapCap(%d), m_pMaterial->GetTexCoordGenCap(%d)\n", m_pMaterial->GetTexMapCount(), m_pMaterial->GetTexMapCap(), m_pMaterial->GetTexCoordGenCap());
        return;
    }

    const uint8_t  texIdx = m_pMaterial->GetTexMapCount();
    m_pMaterial->AppendTexMap(textureInfo);
    m_pMaterial->SetTexCoordGenCount(m_pMaterial->GetTexMapCount());
    m_pMaterial->SetTexCoordGen(texIdx, ResTexCoordGen());

    SetTexCoordCount(m_pMaterial->GetTexMapCount());

    // サイズが設定されていないときは、サイズを設定する。
    if (GetSize() == Size::Create(0.f, 0.f) && m_pMaterial->GetTexMapCount() == 1)
    {
        const TextureInfo* pTextureInfo = m_pMaterial->GetTexMap(0).GetTextureInfo();
        if (pTextureInfo != NULL)
        {
            const TexSize& texSize = pTextureInfo->GetSize();
            SetSize(Size::Create(static_cast<float>(texSize.width), static_cast<float>(texSize.height)));
        }
        else
        {
            SetSize(Size::Create(0.0f, 0.0f));
        }
    }
}

void
Picture::ReserveTexCoord(int  num)
{
    // m_TexCoordArray が使用されている場合のみ使用可能。
    NN_SDK_ASSERT(IsTexCoordArrayEnabled());

    // TODO:
    // * Cap値を取得する方法がない。
    // * 広げることはできるがちじめることができない。
    m_SharedMemory.m_TexCoordArray.Reserve(num);
}

int
Picture::GetTexCoordCount() const
{
    // m_TexCoordArray が使用されている場合のみ使用可能。
    NN_SDK_ASSERT(IsTexCoordArrayEnabled());

    return m_SharedMemory.m_TexCoordArray.GetSize();
}

void
Picture::SetTexCoordCount(int  num)
{
    // m_TexCoordArray が使用されている場合のみ使用可能。
    NN_SDK_ASSERT(IsTexCoordArrayEnabled());

    m_SharedMemory.m_TexCoordArray.SetSize(num);
}

void
Picture::GetTexCoord(
    TexCoordQuad coords,
    int  idx
) const
{
    // m_TexCoordArray が使用されている場合のみ使用可能。
    NN_SDK_ASSERT(IsTexCoordArrayEnabled());

    return m_SharedMemory.m_TexCoordArray.GetCoord(coords, idx);
}

void
Picture::SetTexCoord(
    int  idx,
    const TexCoordQuad coords
)
{
    // m_TexCoordArray が使用されている場合のみ使用可能。
    NN_SDK_ASSERT(IsTexCoordArrayEnabled());

    m_SharedMemory.m_TexCoordArray.SetCoord(idx, coords);
}

const nn::util::Unorm8x4
Picture::GetVertexColor(int  idx) const
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] <= VertexColor_MaxVertexColor for Picture[%s]", idx, GetName());

    return m_VertexColors[idx];
}

void
Picture::SetVertexColor(
    int  idx,
    const nn::util::Unorm8x4& value
)
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] <= VertexColor_MaxVertexColor for Picture[%s]", idx, GetName());

    m_VertexColors[idx] = value;
}

uint8_t
Picture::GetVertexColorElement(int  idx) const
{
    return detail::GetVertexColorElement(m_VertexColors, idx);
}

void
Picture::SetVertexColorElement(int  idx, uint8_t  value)
{
    detail::SetVertexColorElement(m_VertexColors, idx, value);
}

void
Picture::LoadMtx(DrawInfo& drawInfo)
{
    if (detail::TestBit(GetSystemExtDataFlag(), PaneSystemDataType_SimpleOBBMesh))
    {
        nn::ui2d::Size  size = GetSize();

        // 最適化メッシュ情報をを検索します。
        const SystemDataSimpleOBB* pMeshData = static_cast<const SystemDataSimpleOBB*>(FindSystemExtDataByType(PaneSystemDataType_SimpleOBBMesh));
        NN_SDK_ASSERT_NOT_NULL(pMeshData);

        float offsetX = pMeshData->offsetX * size.width;
        float offsetY = pMeshData->offsetY * size.height;
        float optimizedWidth = size.width * pMeshData->width;
        float optimizedHeight = size.height * pMeshData->height;

        nn::util::MatrixT4x3fType    mTrans, mInvTrans, mRotZ;
        nn::util::MatrixT4x3fType    mModelView;
        nn::util::MatrixT4x3fType    mTemp;

        nn::util::Vector3f          vTrans, vInvTrans, vRot;

        nn::util::VectorSet(&vInvTrans, -offsetX - optimizedWidth * 0.5f + size.width * 0.5f, offsetY + optimizedHeight * 0.5f - size.height * 0.5f, 0.0f);
        nn::util::VectorSet(&vTrans, -vInvTrans.GetX(), -vInvTrans.GetY(), 0.0f);
        nn::util::VectorSet(&vRot, 0.0f, 0.0f, pMeshData->rotateZ);

        nn::util::MatrixIdentity(&mTrans);
        nn::util::MatrixIdentity(&mInvTrans);
        nn::util::MatrixIdentity(&mRotZ);

        // オフセットを戻して、傾けて元の位置へ戻す。
        nn::util::MatrixSetTranslate(&mTrans, vTrans);
        nn::util::MatrixSetTranslate(&mInvTrans, vInvTrans);
        nn::util::MatrixSetRotateXyz(&mRotZ, vRot);

        nn::util::MatrixMultiply(&mModelView, mInvTrans, mRotZ);
        nn::util::MatrixMultiply(&mTemp, mModelView, mTrans);

        nn::util::MatrixMultiply(&mModelView, mTemp, GetGlobalMtx());

        drawInfo.SetModelViewMtx(mModelView);
    }
    else
    {
        drawInfo.SetModelViewMtx(GetGlobalMtx());
    }
}

void
Picture::Calculate(DrawInfo& drawInfo, Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    Pane::Calculate(drawInfo, context, isDirtyParentMtx);

    if (!m_pMaterial)
    {
        SetConstantBufferReady(false);
        return;
    }

    LoadMtx(drawInfo);

#if defined(NN_UI2D_FIX_INFLUENCE_ALPHA_BUG)
    if (CheckInvisibleAndUpdateConstantBufferReady())
    {
        return;
    }
#else
    // 非表示、もしくはアルファ値が 0 になって見えない状態のペインは
    // コンスタントバッファの確保と更新処理は不要なので行わない。
    if (!IsVisible() || GetGlobalAlpha() == 0)
    {
        if (context.isInfluenceAlpha)
        {
            SetConstantBufferReady(false);
        }
        else
        {
            SetConstantBufferReadySelf(false);
        }
        return;
    }
#endif

    m_pMaterial->AllocateConstantBuffer(drawInfo);
    m_pMaterial->SetupBlendState(&drawInfo);

    const bool isProceduralShapeEnabled = IsProceduralShapeEnabled();

    // 角丸が有効な場合は角丸描画用のコンスタントバッファを確保する。
    if (isProceduralShapeEnabled)
    {
        SystemDataProceduralShapeRuntimeInfo* pProceduralShapeInfo = static_cast<SystemDataProceduralShapeRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShapeRuntimeInfo)));
        if (pProceduralShapeInfo != NULL)
        {
            size_t  constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();
            pProceduralShapeInfo->constantBufferOffset = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(ProceduralShapeConstantBuffer), constantBufferAlignment)));
        }
    }

    // 使用メモリ量計測モード時に GpuBuffer からメモリが確保できない状態になることがある。
    // メモリが確保できない場合、ここ以下の処理は継続できないため途中で抜けます。
    // なお、計測モード時は実際の GpuBuffer::Allocate 呼び出しによって使用量を計測しているため
    // m_pMaterial->AllocateConstantBuffer 呼び出しの後で抜ける必要があります。
    if (m_pMaterial->GetConstantBufferForVertexShader(drawInfo) == NULL ||
        m_pMaterial->GetConstantBufferForPixelShader(drawInfo) == NULL)
    {
        return;
    }

    if (IsTexCoordArrayEnabled())
    {
        bool vertexColorEnabled = detail::TestVertexColorEnabled(m_VertexColors);

        if (vertexColorEnabled || GetGlobalAlpha() != 255)
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), ShaderVariation_Standard, true, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

            detail::SetupVertexColor(drawInfo, *m_pMaterial, m_VertexColors);
        }
        else
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), ShaderVariation_WithoutVertexColor, true, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
        }

        if (detail::TestBit(GetSystemExtDataFlag(), PaneSystemDataType_SimpleAABBMesh) ||
            detail::TestBit(GetSystemExtDataFlag(), PaneSystemDataType_SimpleOBBMesh))
        {
            // 最適化メッシュ情報をを検索します。
            // SystemDataSimpleAABB か SystemDataSimpleOBB の可能性があるが
            // ここで参照する範囲では AABB も OBB も同じバイナリ構造のため AABB でアクセスする。
            const SystemDataSimpleAABB* pMeshData = static_cast<const SystemDataSimpleAABB*>(FindSystemExtDataByType(PaneSystemDataType_SimpleAABBMesh));
            if (pMeshData == nullptr)
            {
                pMeshData = static_cast<const SystemDataSimpleAABB*>(FindSystemExtDataByType(PaneSystemDataType_SimpleOBBMesh));
            }
            NN_SDK_ASSERT_NOT_NULL(pMeshData);

            nn::util::Float2 pos = GetVertexPos();
            nn::ui2d::Size  size = GetSize();

            pos.x += (pMeshData->offsetX * size.width);
            pos.y -= (pMeshData->offsetY * size.height);
            size.width *= pMeshData->width;
            size.height *= pMeshData->height;

            detail::CalculateQuadWithTexCoords(
                drawInfo,
                m_pMaterial->GetConstantBufferForVertexShader(drawInfo),
                pos,
                size,
                m_SharedMemory.m_TexCoordArray.GetSize(),
                m_SharedMemory.m_TexCoordArray.GetArray());
        }
        else
        {
            detail::CalculateQuadWithTexCoords(
                drawInfo,
                m_pMaterial->GetConstantBufferForVertexShader(drawInfo),
                GetVertexPos(),
                GetSize(),
                m_SharedMemory.m_TexCoordArray.GetSize(),
                m_SharedMemory.m_TexCoordArray.GetArray());
        }
    }
    else
    {
        m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), ShaderVariation_GfxPrimitive, true, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        // シェイプで使用する頂点バッファとインデックスバッファを GpuMemory へ領域を確保してコピーします。
        CopyShapeVertexData(drawInfo);

        // コンスタントバッファは必要なもののみパラメターを調整して設定します。
        Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = m_pMaterial->GetConstantBufferForVertexShader(drawInfo);

        // ローカル座標変換設定
        pVertexShaderConstantBuffer->transform[0] = GetSize().width * 0.5f;
        pVertexShaderConstantBuffer->transform[1] = GetSize().height * 0.5f;
        // 矩形ではないので頂点の開始位置オフセットは指定しなくてもよい。
        pVertexShaderConstantBuffer->transform[2] = 0.0f;
        pVertexShaderConstantBuffer->transform[3] = 0.0f;
        drawInfo.LoadMtxModelView(pVertexShaderConstantBuffer->modelView);
    }

    // ペインエフェクト機能が有効の場合はキャプチャに備えてマテリアルのコンスタントバッファ内の行列を書き換えます。
    // ペインエフェクト機能は Pane 側の機能ですが、派生先で自分自身を描画する際のコンスタントバッファの作成には直接関与することができないため
    // キャプチャ用の ModelView 行列や Projection 行列の設定は派生クラス側で行う必要があります。
    // また、コンスタントバッファにせていされる前の行列を操作すると階層ツリーの姿勢に影響してしまうためここで直接書き換えています。
    if (IsPaneEffectEnabled())
    {
        UpdateMaterialConstantBufferForEffectCapture(drawInfo);
    }

    // 角丸コンスタントバッファの更新処理。
    if (isProceduralShapeEnabled)
    {
        CalculateProceduralShape(drawInfo);
    }

    // ユーザーシェーダー更新コールバックが設定されていたら呼び出す。
    UpdateUserShaderCallback pCallback = drawInfo.GetUpdateUserShaderCallback();
    void*   pUserData = drawInfo.GetUpdateUserShaderCallbackUserData();

    if (pCallback != NULL)
    {
        (*pCallback)(drawInfo, this, pUserData);
    }
}// NOLINT(impl/function_size)

void
Picture::SetupPaneEffectSourceImageRenderState(nn::gfx::CommandBuffer& commandBuffer) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pMaterial);

    m_pMaterial->SetCommandBufferOnlyBlend(commandBuffer);
}

void
Picture::CopyShapeVertexData(DrawInfo& drawInfo)
{
    size_t  vertexBufferAlignment = drawInfo.GetGraphicsResource()->GetVertexBufferAlignment();
    size_t  indexBufferAlignment = drawInfo.GetGraphicsResource()->GetIndexBufferAlignment();

    // 頂点バッファ、インデックスバッファ用のメモリを GpuBuffer から確保してデータをコピーします。
    // GpuBuffer は先頭からコンスタントバッファ用の領域を確保しているため、アラインメントの違うバッファを同じ領域に混ぜることができない。
    // そのため、GpuBuffer の後ろからアラインメントが同じ下記のバッファを確保してアラインメントの問題が発生しないように実装している。
    size_t  offset = drawInfo.GetUi2dConstantBuffer()->AllocateFromTail(
        nn::util::align_up(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBufferSize(), vertexBufferAlignment));
    NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for the shape vertex buffer[%s]", GetName());
    m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->vertexBufferGpuMemoryOffset = offset;

    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   ptr(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        void*   pVertexBuffer = ptr.Advance(offset).Get();
        memcpy(
            pVertexBuffer,
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBuffer(),
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBufferSize());
    }

    offset = drawInfo.GetUi2dConstantBuffer()->AllocateFromTail(
        nn::util::align_up(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBufferSize(), indexBufferAlignment));
    NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for the shape index buffer[%s]", GetName());
    m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->indexBufferGpuMemoryOffset = offset;

    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   ptr(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        void*   pIndexBuffer = ptr.Advance(offset).Get();
        memcpy(
            pIndexBuffer,
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBuffer(),
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBufferSize());
    }
}

void
Picture::InitializeProceduralShape(BuildResultInformation* pOutBuildResultInformation, nn::gfx::Device* pDevice)
{
    SystemDataProceduralShapeRuntimeInfo* pProceduralShapeInfo = static_cast<SystemDataProceduralShapeRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShapeRuntimeInfo)));
    if (pProceduralShapeInfo != NULL)
    {
        const ShaderInfo* pShaderInfo = GetMaterial()->GetShaderInfo();

        pProceduralShapeInfo->constantBufferSlot = pShaderInfo->GetPixelShader(GetMaterial()->GetShaderVariation())->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "uProceduralShape");

        // ペインエフェクトを初期化する
        if (pOutBuildResultInformation != NULL)
        {
            pOutBuildResultInformation->requiredUi2dConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(ProceduralShapeConstantBuffer));
        }
    }
}

void
Picture::CalculateProceduralShape(DrawInfo& drawInfo)
{
    const SystemDataProceduralShape* pProceduralShapeData = static_cast<const SystemDataProceduralShape*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShape));
    SystemDataProceduralShapeRuntimeInfo* pProceduralShapeInfo = static_cast<SystemDataProceduralShapeRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShapeRuntimeInfo)));

    if (pProceduralShapeInfo != NULL)
    {
        if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
        {
            nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
            ProceduralShapeConstantBuffer* pCB = static_cast<ProceduralShapeConstantBuffer*>(pConstantBufferForVertexShader.Advance(pProceduralShapeInfo->constantBufferOffset).Get());

            float halfWidth = GetSize().width * 0.5f;
            float halfHeight = GetSize().height * 0.5f;
            float radius = pProceduralShapeData->radius;

            // 角丸のサイズはペインのサイズを超えて大きくならないようにする
            detail::ClampValue(radius, 0.0, halfWidth);
            detail::ClampValue(radius, 0.0, halfHeight);

            pCB->paneWidth = GetSize().width;
            pCB->paneHeight = GetSize().height;
            pCB->cornerSize = radius;

            pCB->exp = pProceduralShapeData->exp;

            // 形状の値をクランプします。
            detail::ClampValue(pCB->exp, 0.5f, 3.0f);

            float cornerWidth = radius / halfWidth;
            if (cornerWidth == 0.0f)
            {
                cornerWidth = 0.00001f;
            }
            float cornerHeight = radius / halfHeight;
            if (cornerHeight == 0.0f)
            {
                cornerHeight = 0.00001f;
            }

            pCB->cornerOffset[0] = 1.0f - cornerWidth;
            detail::ClampValue(pCB->cornerOffset[0], 0.0f, 1.0f);
            pCB->cornerOffset[1] = 1.0f - cornerHeight;
            detail::ClampValue(pCB->cornerOffset[1], 0.0f, 1.0f);

            pCB->cornerRatio = GetSize().width / radius;

            // 境界線(内側)
            if (detail::TestBit(pProceduralShapeData->flags, SystemDataProceduralShape::ProceduralShapeFlag_InnerStrokeEnabled))
            {
                pCB->innerStrokeSize = pProceduralShapeData->innerStrokeSize;
                pCB->innerStrokeBlendType = pProceduralShapeData->innerStrokeBlendType;
                for (int i = 0; i < 4; ++i)
                {
                    pCB->innerStrokeColor[i] = pProceduralShapeData->innerStrokeColor[i] / 255.0f;
                }
            }
            else
            {
                pCB->innerStrokeColor[3] = 0.0f;
            }

            // シャドウ(内側)
            CalculateProceduralShapeInnerShadow(pCB, pProceduralShapeData, radius);

            // カラーオーバーレイ
            if (detail::TestBit(pProceduralShapeData->flags, SystemDataProceduralShape::ProceduralShapeFlag_ColorOverlayEnabled))
            {
                pCB->colorOverlayBlendType = pProceduralShapeData->colorOverlayBlendType;
                for (int i = 0; i < 4; ++i)
                {
                    pCB->colorOverlayColor[i] = pProceduralShapeData->colorOverlayColor[i] / 255.0f;
                }
            }
            else
            {
                pCB->colorOverlayColor[3] = 0.0f;
            }

            // グラデーションオーバーレイ
            CalculateProceduralShapeGradationOverlay(pCB, pProceduralShapeData);
        }
    }
}

void
Picture::CalculateProceduralShapeInnerShadow(ProceduralShapeConstantBuffer* pConstantBuffer, const SystemDataProceduralShape* pData, float radius)
{
    if (detail::TestBit(pData->flags, SystemDataProceduralShape::ProceduralShapeFlag_InnerShadowEnabled))
    {
        pConstantBuffer->innerShadowSize = (pData->innerShadowSize / radius) + 0.00001f;
        pConstantBuffer->innerShadowBlendType = pData->innerShadowBlendType;

        float radAngle = nn::util::DegreeToRadian(pData->innerShadowAngle);
        pConstantBuffer->innerShadowOffsetX = cosf(radAngle) * (pData->innerShadowDistance / GetSize().width);
        pConstantBuffer->innerShadowOffsetY = -sinf(radAngle) * (pData->innerShadowDistance / GetSize().height);

        for (int i = 0; i < 4; ++i)
        {
            pConstantBuffer->innerShadowColor[i] = pData->innerShadowColor[i] / 255.0f;
            pConstantBuffer->innerShadowAttenuation[i] = 0.0;
        }

        pConstantBuffer->innerShadowAttenuation[pData->innerShadowAttenuationType] = 1.0f;
    }
    else
    {
        pConstantBuffer->innerShadowColor[3] = 0.0f;
    }
}

void
Picture::CalculateProceduralShapeGradationOverlay(ProceduralShapeConstantBuffer* pConstantBuffer, const SystemDataProceduralShape* pData)
{
    // また、シェーダー側で末尾の使用しないコントロールポイントは 1.0 で埋まっていることを期待している。
    // そのためコントロールポイントの初期状態としてすべて 1.0 で埋めておく。
    if (detail::TestBit(pData->flags, SystemDataProceduralShape::ProceduralShapeFlag_GradationOverlayEnabled))
    {
        pConstantBuffer->gradationOverlayBlendType = pData->gradationOverlayBlendType;
        float prevControlPointValue = 0.0f;
        int finishColorIndex = -1;
        for (int i = 0; i <SystemDataProceduralShape::GradationOverlayControlPointCount; ++i)
        {
            float value = pData->gradationOverlayControlPoint[i];
            detail::ClampValue(value, 0.0f, 1.0f);

            // コントロールポイントの値が 1.0 を超えた場合は、最初に 1.0 を超えたコントロールポイントのカラーを使用する
            if (finishColorIndex >= 0)
            {
                pConstantBuffer->gradationOverlayControlPoints[i] = 1.0f;
                for (int j = 0; j < 4; ++j)
                {
                    pConstantBuffer->gradationOverlayColors[i][j] = pData->gradationOverlayColor[finishColorIndex][j] / 255.0f;
                }
            }
            else
            {
                // ひとつ前のコントロールポイントよりも小さい値が設定されないように値をクリップします。
                if (value < prevControlPointValue)
                {
                    pConstantBuffer->gradationOverlayControlPoints[i] = prevControlPointValue;
                }
                else
                {
                    pConstantBuffer->gradationOverlayControlPoints[i] = value;
                }

                prevControlPointValue = value;

                for (int j = 0; j < 4; ++j)
                {
                    pConstantBuffer->gradationOverlayColors[i][j] = pData->gradationOverlayColor[i][j] / 255.0f;
                }

                // コントロールポイントが 1.0 を超えていないか判定
                // 超えている場合はこれ以降、このコントロールポイントの値が使用されるようにする。
                if (value >= 1.0f)
                {
                    finishColorIndex = i;
                }
            }
        }
        pConstantBuffer->gradationOverlayAngle = nn::util::DegreeToRadian(pData->gradationOverlayAngle);
    }
    else
    {
        // コントロールポイントの 0 番目に 0.0 以上の値が設定されてた場合にグラデーションオーバーレイが有効と判断する。
        for (int i = 0; i < 4; ++i)
        {
            pConstantBuffer->gradationOverlayControlPoints[i] = -1.0;
        }
    }
}

void
Picture::DrawSelf(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    if (!m_pMaterial)
    {
        return;
    }

    // 使用メモリ計測モード時は描画は行わない。
    if (drawInfo.GetUi2dConstantBuffer()->IsAnalyseMode())
    {
        return;
    }

    m_pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);

    SystemDataProceduralShapeRuntimeInfo* pProceduralShapeInfo = static_cast<SystemDataProceduralShapeRuntimeInfo*>(const_cast<void*>(FindSystemExtDataByType(PaneSystemDataType_ProceduralShapeRuntimeInfo)));
    if (pProceduralShapeInfo != NULL)
    {
        nn::gfx::GpuAddress fsConstantBufferAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
        fsConstantBufferAddr.Offset(pProceduralShapeInfo->constantBufferOffset);
        size_t cbSize = sizeof(ProceduralShapeConstantBuffer);
        if (pProceduralShapeInfo->constantBufferSlot >= 0)
        {
            commandBuffer.SetConstantBuffer(pProceduralShapeInfo->constantBufferSlot, nn::gfx::ShaderStage_Pixel, fsConstantBufferAddr, cbSize);
        }
    }

    if (IsTexCoordArrayEnabled())
    {
        // 通常の矩形描画処理
        detail::SetupMaterialRenderState(commandBuffer, drawInfo, *m_pMaterial);
        if (IsPaneEffectEnabled())
        {
            UpdateRenderStateForPaneEffectCapture(commandBuffer, drawInfo);
        }
        detail::DrawQuad(commandBuffer, drawInfo);
    }
    else
    {
        // 通常矩形以外のメッシュ描画処理

        // シェーダの設定
        if (drawInfo.RecordCurrentShader(m_pMaterial->GetShaderInfo(), m_pMaterial->GetShaderVariation()))
        {
            m_pMaterial->SetShader(commandBuffer);
        }


        nn::gfx::GpuAddress gpuAddr;
        gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
        gpuAddr.Offset(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->vertexBufferGpuMemoryOffset);

        commandBuffer.SetVertexBuffer(
            0,
            gpuAddr,
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetStride(),
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetVertexBufferSize());
        // 頂点バッファを変更するため、次の矩形の描画のために頂点バッファを再設定するように設定する。
        drawInfo.ResetVertexBufferState();

        m_pMaterial->SetCommandBuffer(commandBuffer, drawInfo);

        if (IsPaneEffectEnabled())
        {
            UpdateRenderStateForPaneEffectCapture(commandBuffer, drawInfo);
        }

        gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
        gpuAddr.Offset(m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->indexBufferGpuMemoryOffset);

        commandBuffer.DrawIndexed(
            nn::gfx::PrimitiveTopology_TriangleList,
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexBufferFormat(),
            gpuAddr,
            m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo->pShape->GetIndexCount(), 0);
    }
}

bool
Picture::CompareCopiedInstanceTest(const Picture& target) const
{
    if (Pane::CompareCopiedInstanceTest(target) == false)
    {
        return false;
    }

    if (m_pMaterial->CompareCopiedInstanceTest(*(target.m_pMaterial)) == false)
    {
        return false;
    }

    // 下記構造体はパディングが入らない前提で memcmp で比較している。
    if (memcmp(&m_VertexColors, &m_VertexColors, sizeof(nn::util::Unorm8x4) * VertexColor_MaxVertexColor) != 0)
    {
        return false;
    }

    if (IsTexCoordArrayEnabled())
    {
        if (m_SharedMemory.m_TexCoordArray.GetSize() != target.m_SharedMemory.m_TexCoordArray.GetSize())
        {
            return false;
        }

        if (m_SharedMemory.m_TexCoordArray.CompareCopiedInstanceTest(target.m_SharedMemory.m_TexCoordArray) == false)
        {
            return false;
        }
    }
    else
    {
        if (m_SharedMemory.m_UnionTypeFlag != target.m_SharedMemory.m_UnionTypeFlag)
        {
            return false;
        }

        ShapeDrawInfo*  pInfo = m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo;
        ShapeDrawInfo*  pTargetInfo = target.m_SharedMemory.m_ShapeDrawInfoAccessor.pShapeDrawInfo;

        // 形状タイプ
        if (pInfo->m_ShapeType != pTargetInfo->m_ShapeType)
        {
            return false;
        }
        // トポロジー
        if (pInfo->pShape->GetPrimitiveTopology() != pTargetInfo->pShape->GetPrimitiveTopology())
        {
            return false;
        }
        // 頂点フォーマット
        if (pInfo->pShape->GetVertexFormat() != pTargetInfo->pShape->GetVertexFormat())
        {
            return false;
        }
        // 頂点数
        if (pInfo->pShape->GetVertexCount() != pTargetInfo->pShape->GetVertexCount())
        {
            return false;
        }
        // 頂点バッファサイズ
        if (pInfo->pShape->GetVertexBufferSize() != pTargetInfo->pShape->GetVertexBufferSize())
        {
            return false;
        }
        // 頂点バッファの内容
        if (memcpy(pInfo->pShape->GetVertexBuffer(), pTargetInfo->pShape->GetVertexBuffer(), pInfo->pShape->GetVertexBufferSize()) != 0)
        {
            return false;
        }
        // インデックス数
        if (pInfo->pShape->GetIndexCount() != pTargetInfo->pShape->GetIndexCount())
        {
            return false;
        }
        // インデックスフォーマット
        if (pInfo->pShape->GetIndexBufferFormat() != pTargetInfo->pShape->GetIndexBufferFormat())
        {
            return false;
        }
        // インデックスバッファサイズ
        if (pInfo->pShape->GetIndexBufferSize() != pTargetInfo->pShape->GetIndexBufferSize())
        {
            return false;
        }
        // インデックスバッファの内容
        if (memcpy(pInfo->pShape->GetIndexBuffer(), pTargetInfo->pShape->GetIndexBuffer(), pInfo->pShape->GetIndexBufferSize()) != 0)
        {
            return false;
        }
    }

    return true;
}

} // namespace nn::ui2d
} // namespace nn
