﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ui2d/ui2d_Layout.h>

#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_Bounding.h>
#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Group.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Picture.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_Resources.h>
#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_Window.h>
#include <nn/ui2d/ui2d_Capture.h>
#include <nn/ui2d/ui2d_ControlCreator.h>
#include <nn/ui2d/ui2d_Alignment.h>
#include <nn/ui2d/ui2d_Scissor.h>
#include <nn/ui2d/ui2d_PaneEffect.h>
#include <nn/ui2d/detail/ui2d_Log.h>
#include <nn/ui2d/ui2d_StateMachine.h>

#include <nn/perf.h>

namespace nn
{
namespace ui2d
{

nn::AlignedAllocateFunctionWithUserData  Layout::g_pAllocateFunction     = NULL;
nn::FreeFunctionWithUserData             Layout::g_pFreeFunction         = NULL;
void*                                    Layout::g_pUserDataForAllocator = NULL;

CreateRenderTargetTextureCallback        Layout::g_pCreateRenderTargetTextureCallback   = NULL;
DestroyRenderTargetTextureCallback       Layout::g_pDestroyRenderTargetTextureCallback  = NULL;
CreateRenderTargetTextureResourceCallback        Layout::g_pCreateRenderTargetTextureResourceCallback   = NULL;
DestroyRenderTargetTextureResourceCallback       Layout::g_pDestroyRenderTargetTextureResourceCallback  = NULL;
void*                                    Layout::g_pRenderTargetTextureCallbackUserData = NULL;




namespace
{

/*---------------------------------------------------------------------------*
  @brief 再帰的にペインにタグプロセッサ設定します。
 *---------------------------------------------------------------------------*/
void
SetTagProcessorImpl(
    Pane*                           pPane,
    nn::font::TagProcessorBase<uint16_t>*  pTagProcessor
)
{
    // TextBox ペインか
    if (TextBox* pTextBox = DynamicCast<TextBox*>(pPane))
    {
        pTextBox->SetTagProcessor(pTagProcessor);
    }

    // 再帰的にセット
    PaneList::iterator endIter = pPane->GetChildList().end();
    for (PaneList::iterator iter = pPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        SetTagProcessorImpl(&(*iter), pTagProcessor);
    }
}

bool
IsIncludeAnimationGroupRef(
    GroupContainer*                 pGroupContainer,
    const ResAnimationGroupRef *const  pGroupRefs,
    uint16_t                        bindGroupNum,
    bool                            bDescendingBind,
    Pane*                           pTargetPane
)
{
    for (uint16_t  groupIndex = 0; groupIndex < bindGroupNum; ++groupIndex)
    {
        // 共有対象のペインがバインド指定のグループが持つペインリストに含まれるか調べます。
        Group *const pGroup = pGroupContainer->FindGroupByName(pGroupRefs[groupIndex].GetName());
        PaneLinkList& paneList = pGroup->GetPaneList();
        PaneLinkList::iterator endIter = paneList.end();
        for (PaneLinkList::iterator iter = paneList.begin(); iter != endIter; ++iter)
        {
            if (iter->pTarget == pTargetPane)
            {
                return true;
            }

            if (bDescendingBind)
            {
                // pTargetPaneの親方向で一致するものが無いか調べる。
                for (Pane* pParentPane = pTargetPane->GetParent(); pParentPane; pParentPane = pParentPane->GetParent())
                {
                    if (iter->pTarget == pParentPane)
                    {
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

//----------------------------------------------------------------------
// 次のバイナリブロックから、拡張ユーザー情報を取得しあれば返す。
const ResExtUserDataList*
TryReadNextBlockUserDataIfExist(
    const nn::font::detail::BinaryFileHeader *const pFileHead,
    int* pIndex,
    const void** ppDataPtr,
    const nn::font::detail::BinaryBlockHeader** ppDataBlockHead)
{
    // これが最後のブロックでなければ、次のデータブロックを覗く、もしDATABLOCKKIND_USERDATALISTがあれば、取得して、ブロックポインタを進める。
    if (*pIndex + 1 < pFileHead->dataBlocks)
    {
        const nn::font::detail::BinaryBlockHeader* pNextDataBlockHead = nn::util::ConstBytePtr(*ppDataPtr, (*ppDataBlockHead)->size).Get<nn::font::detail::BinaryBlockHeader>();
        if (pNextDataBlockHead->kind == DataBlockKindUserDataList) {

            // 次のブロックへと読み進める。
            *pIndex += 1;
            *ppDataPtr = reinterpret_cast<const uint8_t *>(pNextDataBlockHead);
            *ppDataBlockHead = pNextDataBlockHead;

            return reinterpret_cast<const ResExtUserDataList*>(pNextDataBlockHead);
        }
    }

    return NULL;
}

} // namespace nn::ui2d::{no-name}

void
Layout::SetAllocator(nn::AlignedAllocateFunctionWithUserData pAllocateFunction, nn::FreeFunctionWithUserData pFreeFunction, void* pUserData)
{
    g_pAllocateFunction      = pAllocateFunction;
    g_pFreeFunction          = pFreeFunction;
    g_pUserDataForAllocator  = pUserData;
}

void*
Layout::AllocateMemory(size_t  size, size_t  alignment)
{
    NN_SDK_ASSERT(g_pAllocateFunction != NULL, "You must call SetAllocator() before use this function.");

    void *const pMem = g_pAllocateFunction(size, alignment, g_pUserDataForAllocator);
    NN_DETAIL_UI2D_WARNING(pMem, "can't alloc memory.");
    return pMem;
}

void*
Layout::AllocateMemory(size_t  size)
{
    NN_SDK_ASSERT_NOT_NULL(g_pAllocateFunction);

    void *const pMem = g_pAllocateFunction(size, DefaultAligment, g_pUserDataForAllocator);
    NN_DETAIL_UI2D_WARNING(pMem, "can't alloc memory.");
    return pMem;
}



void
Layout::FreeMemory(void* pMemory)
{
    NN_SDK_ASSERT(g_pFreeFunction != NULL, "You must call SetAllocator() before use this function.");

    g_pFreeFunction(pMemory, g_pUserDataForAllocator);
}

Layout::Layout()
: m_pRootPane(NULL),
m_pGroupContainer(NULL),
m_Name(NULL),
m_pResExtUserDataList(NULL),
m_pResourceAccessor(NULL),
m_pCaptureTextureList(NULL)
{
    m_LayoutSize.Set(0.f,0.f);
}

Layout::~Layout()
{
    NN_SDK_ASSERT(m_pRootPane == NULL);
}

void
Layout::Finalize(nn::gfx::Device* pDevice)
{
    if (m_pCaptureTextureList != NULL)
    {
        if (m_pCaptureTextureList->pCaptureTextures != NULL)
        {
            for (int i = 0; i < m_pCaptureTextureList->count; ++i)
            {
                m_pCaptureTextureList->pCaptureTextures[i].Finalize(pDevice, m_pResourceAccessor);
            }
            Layout::DeleteArray(m_pCaptureTextureList->pCaptureTextures, m_pCaptureTextureList->count);
        }
        Layout::DeleteObj(m_pCaptureTextureList);
        m_pCaptureTextureList = NULL;
    }

    // 部品ペインの実態（リンクリストのノードの実体）を破棄する前にリストを解放しておきます。
    m_PartsPaneList.clear();

    DeleteObj(m_pGroupContainer);
    m_pGroupContainer = NULL;

    if (m_pRootPane && !m_pRootPane->IsUserAllocated())
    {
        m_pRootPane->Finalize(pDevice);
        DeleteObj(m_pRootPane);
        m_pRootPane = NULL;
    }

    for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != m_AnimTransList.end();)
    {
        AnimTransformList::iterator currIter = iter++;
        m_AnimTransList.erase(currIter);
        DeleteObj(&(*currIter));
    }

    m_Name = NULL;
    m_pResExtUserDataList = NULL;
    m_pResourceAccessor = NULL;
    m_LayoutSize.Set(0.f,0.f);
}

bool Layout::Build(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    ResourceAccessor* pResAcsr,
    ControlCreator* pControlCreator /* = NULL */,
    TextSearcher* pTextSearcher /* = NULL */,
    const void* pLayoutResource,
    const BuildOption& buildOption /* = BuildOption() */,
    bool utf8 /* = false */)
{
    BuildArgSet buildArgSet;
    buildArgSet.magnify.v[0] = 1.f;
    buildArgSet.magnify.v[1] = 1.f;
    buildArgSet.partsSize.v[0] = 0.f;
    buildArgSet.partsSize.v[1] = 0.f;
    buildArgSet.pControlCreator = pControlCreator;
    buildArgSet.pTextSearcher = pTextSearcher;
    buildArgSet.pLayout = NULL;
    buildArgSet.pBodyLayout = this;
    memset(buildArgSet.pCaptureTexturePrefixStack, 0, sizeof(char*) * 8);
    buildArgSet.captureTexturePrefixStackPosition = 0;
    buildArgSet.captureTextureOverridePosition = -1;

    buildArgSet.isRootPaneParts = buildOption.isRootPaneParts;
    buildArgSet.isUtf8 = utf8;
    buildArgSet.pGetUserShaderInformationFromUserDataCallback = buildOption.pGetUserShaderInformationFromUserDataCallback;
    buildArgSet.pGetUserShaderInformationFromUserDataCallbackUserData = buildOption.pGetUserShaderInformationFromUserDataCallbackUserData;
    buildArgSet.resourceVersion = static_cast<const nn::font::detail::BinaryFileHeader*>(pLayoutResource)->version;

#if ! defined(NN_SDK_BUILD_RELEASE)
    buildArgSet.isViewerInvisibleByParent = false;
#endif
    return BuildImpl(pOutBuildResultInformation, pDevice, pLayoutResource, pResAcsr, buildArgSet, NULL);
}

bool Layout::BuildWithName(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    ResourceAccessor* pResAcsr,
    ControlCreator* pControlCreator /* = NULL */,
    TextSearcher* pTextSearcher /* = NULL */,
    const BuildOption& buildOption /* = BuildOption() */,
    const char* pLayoutFileName,
    bool utf8 /* = false */)
{
    const void* pLayoutResource = pResAcsr->FindResourceByName(nn::ui2d::ResourceTypeLayout, pLayoutFileName);
    if (pLayoutResource)
    {
        return Build(pOutBuildResultInformation, pDevice, pResAcsr, pControlCreator, pTextSearcher, pLayoutResource, buildOption, utf8);
    }
    else
    {
        return false;
    }
}

//-------------------------------------------------------------------------------------------------

int
Layout::AcquireAnimTagNameCount() const
{
    auto* pTagName = static_cast<const SystemDataLayoutTagNames*>(FindSystemExtDataByType(LayoutSystemDataType_AnimTagName));
    return pTagName != NULL ? pTagName->count : 0;
}

//-------------------------------------------------------------------------------------------------

const char*
Layout::AcquireAnimTagNameByIndex(int index) const
{
    auto* pTagName = static_cast<const SystemDataLayoutTagNames*>(FindSystemExtDataByType(LayoutSystemDataType_AnimTagName));
    NN_SDK_ASSERT_NOT_NULL(pTagName);
    NN_SDK_ASSERT_RANGE(index, 0, (int)pTagName->count);

    const uint32_t* pNameOffsetArray = reinterpret_cast<const uint32_t*>(pTagName + 1);
    const char* pNameBuffer = nn::util::ConstBytePtr(pTagName, static_cast<ptrdiff_t>(pNameOffsetArray[index])).Get<char>();
    return pNameBuffer;
}

//-------------------------------------------------------------------------------------------------

Pane* Layout::BuildPartsImpl
(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const void* pData,
    const PartsBuildDataSet* pPartsBuildDataSet,
    BuildArgSet& buildArgSet,
    BuildResSet& buildResSet,
    const uint32_t kind
)
{
    Pane* pPane = NULL;
    const void* pResBlock = pData;
    const void* pOverrideResBlock = NULL;

    buildArgSet.overrideUsageFlag = 0;
    buildArgSet.overrideBasicUsageFlag = 0;
    buildArgSet.overrideMaterialUsageFlag = 0;
    buildArgSet.pOverridePartsPaneBasicInfo = NULL;

    if (pPartsBuildDataSet) {
        // 部品の場合
        if (this->GetRootPane() == 0) {
            // ルートペインとして既に作られている部品ペインを使用する
            pPane = pPartsBuildDataSet->GetPartsPane();
        }
        else {
            const ResPane* pResPane = static_cast<const ResPane*>(pResBlock);
            const ResPartsProperty* pResPartsProperty = pPartsBuildDataSet->FindPartsPropertyFromName(pResPane->name);
            if (pResPartsProperty) {
                // 上書きがある。
                pOverrideResBlock = pPartsBuildDataSet->GetPropertyResBlock(pResPartsProperty);
                NN_SDK_ASSERT(pOverrideResBlock == NULL || (static_cast<const nn::font::detail::BinaryBlockHeader*>(pOverrideResBlock))->kind == kind);
                buildArgSet.overrideUsageFlag = pResPartsProperty->usageFlag;
                buildArgSet.overrideBasicUsageFlag = pResPartsProperty->basicUsageFlag;
                buildArgSet.overrideMaterialUsageFlag = pResPartsProperty->materialUsageFlag;
                buildArgSet.pOverridePartsPaneBasicInfo = pPartsBuildDataSet->GetPartsPaneBasicInfoResBlock(pResPartsProperty);
                bool isOverrideExtUserData = false;
                const ResExtUserDataList* pUserDataList = pPartsBuildDataSet->GetExtUserDataListResBlock(&isOverrideExtUserData, pResPartsProperty);
                if (isOverrideExtUserData) {
                    buildArgSet.pExtUserDataList = pUserDataList;
                }
            }
        }
    }
    else {
        if (this->GetRootPane() == 0 && buildArgSet.isRootPaneParts)
        {
            // 部品ではなく、まだルートペインがなく、かつルートペインをPartsにする設定の場合
            // ResPartsをその場で作成して構築
            ResParts resParts;
            *static_cast<ResPane*>(&resParts) = *static_cast<const ResPane*>(pResBlock);
            resParts.blockHeader.kind = DataBlockKindParts;
            resParts.propertyCount = 0;
            resParts.magnify.x = 1.f;
            resParts.magnify.y = 1.f;
            pPane = BuildPaneObj(pOutBuildResultInformation, pDevice, DataBlockKindParts, &resParts, pOverrideResBlock, buildArgSet);
            static_cast<Parts*>(pPane)->SetLayout(this);
        }
    }

    if (pPane == NULL) {
        // ここまででペインが構築されなかった場合は通常の方法でペインを作成する
        pPane = BuildPaneObj(pOutBuildResultInformation, pDevice, kind, pResBlock, pOverrideResBlock, buildArgSet);
    }

    if (pPane)
    {
        // ルートペインがまだ設定されていないときはルートペインに設定
        if (this->GetRootPane() == 0)
        {
            this->SetRootPane(pPane);
        }

        // 親がいたら親に登録
        if (buildArgSet.pParentPane)
        {
            buildArgSet.pParentPane->AppendChild(pPane);
        }

        if (kind == DataBlockKindParts) {
            // 部品ペインの場合に必要な処理
            Parts* pPartsPane = static_cast<Parts*>(pPane);
            const void* pLocalResBlock;
            const BuildResSet* pLocalBuildResSet;

            if (pOverrideResBlock)
            {
                // 上書きが発生している場合は、リソースブロックのPaneの部分は正しい値でなくなるが、
                // 以後は使用しないので問題ない。(もし必要になったときも、PartsBuildDataSet内のパーツ
                // ペインから同等の情報を取得できる。)
                pLocalResBlock = pOverrideResBlock;
                pLocalBuildResSet = buildArgSet.pOverrideBuildResSet;
            }
            else
            {
                pLocalResBlock = pResBlock;
                pLocalBuildResSet = &buildResSet;
            }

            const ResParts* pResParts = static_cast<const ResParts*>(pLocalResBlock);
            PartsBuildDataSet localPartsBuildDataSet(pPartsPane, pResParts, pLocalBuildResSet, &(static_cast<const ResParts*>(pResBlock)->size));

            // キャプチャテクスチャのパス関連処理

            // キャプチャテクスチャで上書きされたパーツの名前ルートを決定します。
            //
            // パーツの上書きデータはパーツ階層中で最上位で実際に上書きしているレイアウトにまとめて保持されています。
            // A -> B -> C -> D というパーツ階層があり、C で D のペインを上書きして、さらに B で C のペインを上書きした場合
            // C で D を上書きした情報もまとめて B の上書きデータとして B に保存されます。
            //
            // 一方、キャプチャテクスチャでは各パーツ事にテクスチャを区別するため、ボディ中に配置されたパーツペイン名の階層を連結して区別しています。
            // A -> B -> C というパーツ階層があった場合、例えば B の Capture00 というテクスチャと C の Capture00 を区別するため
            // B%Capture00 と B%C%Capture00 のようになります。
            // このときの名前は上書きを設定したレイアウトを基準として保存されるため、このパーツをさらに上位のレイアウトで配置して上書きしなかった場合
            // パーツの上書きデータの仕様から、不完全なパーツ階層パスとなってしまいテクスチャが見つからなくなります。
            //
            // そのため、最上位で上書きしたレイアウトまでのパーツ階層パスと、そのデータに保存されている相対パスをつなげることで最終的なパスとして使用します。
            //
            if (buildArgSet.captureTextureOverridePosition < 0 &&
                pLocalBuildResSet->pCaptureTextureList != NULL &&
                localPartsBuildDataSet.IsOverwriting())
            {
                // パーツを上書きしたデータの階層名を動的なパスとは別に保存しておく。
                buildArgSet.captureTextureOverridePosition = buildArgSet.captureTexturePrefixStackPosition;
            }

            // キャプチャペインの名前生成のためにパーツ名で階層を作成する。
            buildArgSet.pCaptureTexturePrefixStack[buildArgSet.captureTexturePrefixStackPosition] = pPartsPane->GetName();
            ++buildArgSet.captureTexturePrefixStackPosition;

            NN_SDK_ASSERT(buildArgSet.captureTexturePrefixStackPosition < CaptureTexturePartsLayerMax,
                "The maximum number of capture texture parts nests has been exceeded.");

            // 部品のレイアウトを構築
            Layout* pLayout = BuildPartsLayout(
                pOutBuildResultInformation,
                pDevice,
                nn::util::ConstBytePtr(pLocalResBlock, sizeof(ResParts) + sizeof(ResPartsProperty) * pResParts->propertyCount).Get<char>(),
                localPartsBuildDataSet, buildArgSet);
            if (pLayout) {
                // 作成したレイアウトをパーツペインにセット
                pPartsPane->SetLayout(pLayout);
                // 部品のレイアウトのペインリストに登録
                m_PartsPaneList.push_back(*pPartsPane);
            }

            --buildArgSet.captureTexturePrefixStackPosition;

            // 最上位のパーツの処理が終了した段階で次のパーツの処理に備えてキャプチャテクスチャ修飾名上書きポイントをリセットしておく。
            if (buildArgSet.captureTextureOverridePosition == 0)
            {
                buildArgSet.captureTextureOverridePosition = -1;
            }

            // BuildPartsLayoutがNULLを返したときは、部品としてこのレイアウトにくっつける必要がなかったということになる
        }

        // 構築したペインを返す。
        return pPane;
    }

    return NULL;

}// NOLINT(impl/function_size)

//-------------------------------------------------------------------------------------------------

bool IsFlytOlderThanBinary8200(const nn::font::detail::BinaryFileHeader *const pFileHead)
{
    // 0.12.x と 1.6.x と 3.0.x で使われているバイナリバージョン。
    const uint32_t    BinaryFileFormatVersion8000 = NN_DETAIL_FONT_MAKE_VERSION(8, 0, 0, 0);
    const uint32_t    BinaryFileFormatVersion8100 = NN_DETAIL_FONT_MAKE_VERSION(8, 1, 0, 0);

    return pFileHead->version == BinaryFileFormatVersion8000 || pFileHead->version == BinaryFileFormatVersion8100;
}

//-------------------------------------------------------------------------------------------------

bool IsFlytHigherThanBinary8600(const nn::font::detail::BinaryFileHeader *const pFileHead)
{
    // 8.6.0.0 での変更点
    // - ResCaptureTexture のバイナリフォーマットが新しい形式に変更された
    return pFileHead->version >= NN_DETAIL_FONT_MAKE_VERSION(8, 6, 0, 0);
}


//-------------------------------------------------------------------------------------------------

bool IsDetailedCombinerWithVariationVariable(const nn::font::detail::BinaryFileHeader *const pFileHead)
{
    // 詳細コンバイナモードにバリエーション変数対応が行われたバージョン
    const uint32_t availableVersion = NN_DETAIL_FONT_MAKE_VERSION(8, 3, 0, 0);

    if (NN_DETAIL_VERSION_MAJOR(pFileHead->version) > NN_DETAIL_VERSION_MAJOR(availableVersion))
    {
        return true;
    }
    else if (NN_DETAIL_VERSION_MAJOR(pFileHead->version) == NN_DETAIL_VERSION_MAJOR(availableVersion) &&
             NN_DETAIL_VERSION_MINOR(pFileHead->version) >= NN_DETAIL_VERSION_MINOR(availableVersion))
    {
        return true;
    }
    return false;
}

//-------------------------------------------------------------------------------------------------
void SetBuildArgSetFromParent(BuildArgSet &buildArgSet, const BuildArgSet &parentBuildArgSet)
{
    buildArgSet.pControlCreator = parentBuildArgSet.pControlCreator;
    buildArgSet.pTextSearcher   = parentBuildArgSet.pTextSearcher;
    buildArgSet.pBodyLayout     = parentBuildArgSet.pBodyLayout;
    buildArgSet.isRootPaneParts = parentBuildArgSet.isRootPaneParts;
    buildArgSet.isUtf8          = parentBuildArgSet.isUtf8;
    memcpy(buildArgSet.pCaptureTexturePrefixStack, parentBuildArgSet.pCaptureTexturePrefixStack, sizeof(char*) * CaptureTexturePartsLayerMax);
    buildArgSet.captureTexturePrefixStackPosition = parentBuildArgSet.captureTexturePrefixStackPosition;
    buildArgSet.captureTextureOverridePosition = parentBuildArgSet.captureTextureOverridePosition;
    buildArgSet.pGetUserShaderInformationFromUserDataCallback = parentBuildArgSet.pGetUserShaderInformationFromUserDataCallback;
    buildArgSet.pGetUserShaderInformationFromUserDataCallbackUserData = parentBuildArgSet.pGetUserShaderInformationFromUserDataCallbackUserData;
}

//-------------------------------------------------------------------------------------------------
void ReadBlock(Layout::LayoutBuildContext& buildContext, const void* pLayoutResource)
{
    auto pFileHead = static_cast<const nn::font::detail::BinaryFileHeader*>(pLayoutResource);

    auto pDataBlockHead = static_cast<const nn::font::detail::BinaryBlockHeader*>(buildContext.pData);
    buildContext.kind = pDataBlockHead->kind;

    const void* pData = buildContext.pData;
    int         index = buildContext.index;

    switch (buildContext.kind)
    {
    case DataBlockKindLayout:
    {
        const ResLayout* pResLyt = static_cast<const ResLayout*>(pData);
        buildContext.buildArgSet.partsSize.v[0] = pResLyt->partsSize.x;
        buildContext.buildArgSet.partsSize.v[1] = pResLyt->partsSize.y;
        buildContext.pResExtUserDataList = TryReadNextBlockUserDataIfExist(pFileHead, &index, &pData, &pDataBlockHead);
    }
    break;
    case DataBlockKindControl:
        buildContext.pResExtUserDataList = TryReadNextBlockUserDataIfExist(pFileHead, &index, &pData, &pDataBlockHead);
        break;
    case DataBlockKindTextureList:
        buildContext.buildResSet.pTextureList = static_cast<const ResTextureList*>(pData);
        break;
    case DataBlockKindFontList:
        buildContext.buildResSet.pFontList = static_cast<const ResFontList*>(pData);
        break;
    case DataBlockKindMaterialList:
        buildContext.buildResSet.pMaterialList = static_cast<const ResMaterialList*>(pData);
        break;
    case DataBlockKindShapeInfoList:
        buildContext.buildResSet.pShapeInfoList = static_cast<const ResShapeInfoList*>(pData);
        break;
    case DataBlockKindCaptureTextureList:
        buildContext.buildResSet.pCaptureTextureList = static_cast<const ResCaptureTextureList*>(pData);
        break;
    case DataBlockKindPane:
    case DataBlockKindPicture:
    case DataBlockKindTextBox:
    case DataBlockKindWindow:
    case DataBlockKindBounding:
    case DataBlockKindCapture:
    case DataBlockKindParts:
    case DataBlockKindAlignment:
    case DataBlockKindScissor:
        buildContext.pResExtUserDataList = TryReadNextBlockUserDataIfExist(pFileHead, &index, &pData, &pDataBlockHead);
    break;
    case DataBlockKindPaneBegin:
        buildContext.buildArgSet.pParentPane = buildContext.pLastBuiltPane;    // 最後に作成したペインを親とする
        break;
    case DataBlockKindPaneEnd:
        buildContext.pLastBuiltPane = buildContext.buildArgSet.pParentPane;    // 親ペインを最後に作成したペインとする
        buildContext.buildArgSet.pParentPane = buildContext.pLastBuiltPane != NULL ? buildContext.pLastBuiltPane->GetParent() : NULL;
        break;
    case DataBlockKindGroupBegin:
        buildContext.groupNestLevel++;
        break;
    case DataBlockKindGroupEnd:
        buildContext.groupNestLevel--;
        break;
    default:
        break;
    }

    // 次の要素があるかどうか
    buildContext.indexNext = index + 1;
    if (buildContext.indexNext >= pFileHead->dataBlocks)
    {
        buildContext.pDataNext = NULL;
    }
    else
    {
        buildContext.pDataNext = static_cast<const uint8_t *>(pData) + pDataBlockHead->size;    // 次のブロック位置へ
    }
}

//-------------------------------------------------------------------------------------------------
void ReadFirstBlock(Layout::LayoutBuildContext& buildContext, const void* pLayoutResource)
{
    auto pFileHead = static_cast<const nn::font::detail::BinaryFileHeader*>(pLayoutResource);
    NN_SDK_ASSERT(pFileHead->dataBlocks != 0);
    buildContext.pData = static_cast<const uint8_t *>(pLayoutResource) + pFileHead->headerSize;

    ReadBlock(buildContext, pLayoutResource);
}

//-------------------------------------------------------------------------------------------------
bool ReadNextBlock(Layout::LayoutBuildContext& buildContext, const void* pLayoutResource)
{
    if (buildContext.pDataNext == NULL)
    {
        return false;
    }

    buildContext.pData = buildContext.pDataNext;
    buildContext.index = buildContext.indexNext;

    ReadBlock(buildContext, pLayoutResource);

    return true;
}

//-------------------------------------------------------------------------------------------------
void Layout::FindResPaneByName(const ResPane** ppOutResPane, const ResExtUserDataList** ppOutExtUserDataList, const void* pLayoutResource, const char* pName, const ResPane* pSearchStartPane)
{
    *ppOutResPane = NULL;
    *ppOutExtUserDataList = NULL;

    NN_SDK_ASSERT_NOT_NULL(pLayoutResource);

    auto pFileHead = static_cast<const nn::font::detail::BinaryFileHeader*>(pLayoutResource);
    NN_SDK_ASSERT(nn::font::detail::IsValidBinaryFile(pFileHead, FileSignatureFlyt, BinaryFileFormatVersion));

    LayoutBuildContext buildContext = {};

    // buildResSet
    buildContext.buildResSet.isFlytOlderThanBinary8200 = IsFlytOlderThanBinary8200(pFileHead);

    // buildArgSet
    buildContext.buildArgSet.resourceVersion = pFileHead->version;
    buildContext.buildArgSet.pCurrentBuildResSet = &buildContext.buildResSet;

    ReadFirstBlock(buildContext, pLayoutResource);
    do
    {
        switch (buildContext.kind)
        {
        case DataBlockKindPane:
        case DataBlockKindPicture:
        case DataBlockKindTextBox:
        case DataBlockKindWindow:
        case DataBlockKindBounding:
        case DataBlockKindCapture:
        case DataBlockKindParts:
        case DataBlockKindAlignment:
        case DataBlockKindScissor:
        {
            const ResPane* pResPane = static_cast<const ResPane*>(buildContext.pData);

            // SearchStartPane が指定されている場合は、当該ペインまでスキップします。
            if (pSearchStartPane != NULL)
            {
                if (pSearchStartPane == pResPane)
                {
                    pSearchStartPane = NULL;
                }

                continue;
            }

            if (strcmp(pResPane->name, pName) == 0)
            {
                *ppOutExtUserDataList = buildContext.pResExtUserDataList;
                *ppOutResPane = pResPane;

                return;
            }
        }
        break;
        default:
            break;
        }

    } while (ReadNextBlock(buildContext, pLayoutResource));
}

//-------------------------------------------------------------------------------------------------
void Layout::PrepareBuildArgSet(BuildArgSet& buildArgSet, const BuildArgSet& parentBuildArgSet, const PartsBuildDataSet* pPartsBuildDataSet)
{
    // 親レイアウトの情報を元に、自分のBuildArgSetを作成する
    if (pPartsBuildDataSet)
    {
        buildArgSet.magnify = pPartsBuildDataSet->GetMagnify();
        buildArgSet.pOverrideBuildResSet = pPartsBuildDataSet->GetPropertyBuildResSet();
    }
    else
    {
        // 部品レイアウトでない場合は1.f
        buildArgSet.magnify.v[0] = 1.f;
        buildArgSet.magnify.v[1] = 1.f;
        buildArgSet.pOverrideBuildResSet = NULL;
    }

    // 一部の設定は、親buildArgSetの内容を継承します。
    SetBuildArgSetFromParent(buildArgSet, parentBuildArgSet);
    // 一部の設定は、初期値を設定します。
    buildArgSet.overrideUsageFlag = 0;
    buildArgSet.overrideMaterialUsageFlag = 0;
    buildArgSet.pParentPane = NULL;
    buildArgSet.partsSize.v[0] = 0.f;
    buildArgSet.partsSize.v[1] = 0.f;

#if ! defined(NN_SDK_BUILD_RELEASE)
    buildArgSet.isViewerInvisibleByParent = parentBuildArgSet.isViewerInvisibleByParent
        || (pPartsBuildDataSet && pPartsBuildDataSet->GetPartsPane()->IsViewerInvisible());
#endif
}

//-------------------------------------------------------------------------------------------------
void Layout::SetByResLayout(LayoutBuildContext& buildContext)
{
    const ResLayout* pResLyt = static_cast<const ResLayout*>(buildContext.pData);

    this->m_LayoutSize.width = pResLyt->layoutSize.x;
    this->m_LayoutSize.height = pResLyt->layoutSize.y;

    this->SetName(nn::util::ConstBytePtr(pResLyt, sizeof(ResLayout)).Get<char>());
    this->m_pResExtUserDataList = buildContext.pResExtUserDataList;
}

//-------------------------------------------------------------------------------------------------
void Layout::BuildControl(LayoutBuildContext& buildContext, nn::gfx::Device* pDevice)
{
    if (buildContext.buildArgSet.pControlCreator)
    {
        nn::ui2d::ControlSrc controlSrc(buildContext.pData, buildContext.pResExtUserDataList);
        buildContext.buildArgSet.pControlCreator->CreateControl(pDevice, this, controlSrc);
    }
}

//-------------------------------------------------------------------------------------------------
void Layout::BuildPaneByResPane(
    LayoutBuildContext& buildContext,
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const PartsBuildDataSet* pPartsBuildDataSet)
{
    // ペインを構築する前にユーザーデータを buildArgSet にセットする
    buildContext.buildArgSet.pExtUserDataList = buildContext.pResExtUserDataList;

    Pane* pPane = this->BuildPartsImpl(
        pOutBuildResultInformation,
        pDevice,
        buildContext.pData,
        pPartsBuildDataSet,
        buildContext.buildArgSet,
        buildContext.buildResSet,
        buildContext.kind);

    if (pPane != NULL)
    {
        buildContext.pLastBuiltPane = pPane;
    }
}

//-------------------------------------------------------------------------------------------------
void Layout::BuildStateMachine(LayoutBuildContext& buildContext, nn::gfx::Device* pDevice)
{
    if (this->GetStateMachine() != nullptr)
    {
        return;
    }

    StateMachine*            pStateMachine = AllocateAndConstruct<StateMachine>();
    pStateMachine->Initialize(this);

    StateMachineFactory factory(pDevice, this);
    factory.Build(pStateMachine, const_cast<void*>(buildContext.pData));

    RegisteredStateMachine(pStateMachine);
}

//-------------------------------------------------------------------------------------------------
void Layout::BuildGroup(LayoutBuildContext& buildContext)
{
    if (this->GetGroupContainer() == NULL)   // グループのルートに初めて到達
    {
        auto pNewGroupContainer = AllocateAndConstruct<GroupContainer>();
        this->SetGroupContainer(pNewGroupContainer);
    }
    else
    {
        NN_SDK_ASSERT_NOT_NULL(this->GetGroupContainer());
        if (buildContext.groupNestLevel == 1)
        {
            auto pResGroup = reinterpret_cast<const ResGroup*>(buildContext.pData);
            this->GetGroupContainer()->AppendGroup(AllocateAndConstruct<Group>(pResGroup, this->GetRootPane()));
        }
    }
}

//-------------------------------------------------------------------------------------------------
bool
Layout::BuildImpl(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const void* pLayoutResource,
    ResourceAccessor* pResAcsr,
    const BuildArgSet& parentBuildArgSet,
    const PartsBuildDataSet* pPartsBuildDataSet
)
{
    NN_SDK_ASSERT_NOT_NULL(g_pAllocateFunction);
    NN_SDK_ASSERT_NOT_NULL(pLayoutResource);

    auto pFileHead = static_cast<const nn::font::detail::BinaryFileHeader*>(pLayoutResource);
    NN_SDK_ASSERT(nn::font::detail::IsValidBinaryFile(pFileHead, FileSignatureFlyt, BinaryFileFormatVersion), "not valid layout file.");

    m_pResourceAccessor = pResAcsr;

    LayoutBuildContext buildContext = { };

    // buildResSet
    buildContext.buildResSet.pResAccessor = pResAcsr;
    buildContext.buildResSet.pLayout = this;
    buildContext.buildResSet.isFlytOlderThanBinary8200 = IsFlytOlderThanBinary8200(pFileHead);
    buildContext.buildResSet.isDetailedCombinerWithVariationVariable = IsDetailedCombinerWithVariationVariable(pFileHead);

    // buildArgSet
    PrepareBuildArgSet(buildContext.buildArgSet, parentBuildArgSet, pPartsBuildDataSet);
    buildContext.buildArgSet.pLayout = this;
    buildContext.buildArgSet.resourceVersion = pFileHead->version;
    buildContext.buildArgSet.pCurrentBuildResSet = &buildContext.buildResSet;

    bool isFlytHigerThanBinary8600 = IsFlytHigherThanBinary8600(pFileHead);

    ReadFirstBlock(buildContext, pLayoutResource);
    do
    {
        switch (buildContext.kind)
        {
        case DataBlockKindLayout:
        {
            SetByResLayout(buildContext);
        }
        break;
        case DataBlockKindCaptureTextureList:
        {
            // ペインの初期化以前に Material で参照される TextureInfo のインスタンスを作成するため、このタイミングでメモリの確保だけ行っています。
            // レイアウトのデータ構造上、ペインデータより先にこのデータブロックが配置されていることに期待した処理です。
            const char* pPrefix = NULL;

            // キャプチャテクスチャの名前修飾用のプリフィックス文字列を作成します。
            // 一時的なもののためメモリはスタックから確保します。
            int concatenatedStackPosition = buildContext.buildArgSet.captureTexturePrefixStackPosition;

            if (concatenatedStackPosition > 0)
            {
                size_t  fullTextureNameLength = CalcCaptureTexturePrefixLength(buildContext.buildArgSet, concatenatedStackPosition);

                char* pDynamicPrefix = static_cast<char*>(alloca(fullTextureNameLength));
                pPrefix = pDynamicPrefix;

                ConcatCaptureTexturePrefixString(pDynamicPrefix, fullTextureNameLength, buildContext.buildArgSet, concatenatedStackPosition);
            }
            if (isFlytHigerThanBinary8600)
            {
                buildContext.buildResSet.pLayout->AllocateCaptureTexture<ResCaptureTexture>(buildContext.buildResSet.pCaptureTextureList, pResAcsr, pPrefix);
            }
            else
            {
                buildContext.buildResSet.pLayout->AllocateCaptureTexture<ResCaptureTextureOld>(buildContext.buildResSet.pCaptureTextureList, pResAcsr, pPrefix);
            }
        }
        break;

        case DataBlockKindControl:
        {
            BuildControl(buildContext, pDevice);
        }
        break;
        case DataBlockKindPane:
        case DataBlockKindPicture:
        case DataBlockKindTextBox:
        case DataBlockKindWindow:
        case DataBlockKindBounding:
        case DataBlockKindCapture:
        case DataBlockKindParts:
        case DataBlockKindAlignment:
        case DataBlockKindScissor:
        {
            BuildPaneByResPane(buildContext, pOutBuildResultInformation, pDevice, pPartsBuildDataSet);
        }
        break;
        case DataBlockKindStateMachine:
        {
            BuildStateMachine(buildContext, pDevice);
        }
        break;
        case DataBlockKindGroup:
        {
            BuildGroup(buildContext);
        }
        break;
        default:
            break;
        }

    } while (ReadNextBlock(buildContext, pLayoutResource));

    // キャプチャテクスチャ関連のリソースを初期化する。
    if (buildContext.buildResSet.pCaptureTextureList != NULL)
    {
        if (isFlytHigerThanBinary8600)
        {
            InitializeCaptureTextureList<ResCaptureTexture>(pDevice, buildContext.buildResSet.pCaptureTextureList, pPartsBuildDataSet);
        }
        else
        {
            InitializeCaptureTextureList<ResCaptureTextureOld>(pDevice, buildContext.buildResSet.pCaptureTextureList, pPartsBuildDataSet);
        }
    }

    return true;
}// NOLINT(impl/function_size)

//-------------------------------------------------------------------------------------------------
template<typename T>
void
Layout::AllocateCaptureTexture(const ResCaptureTextureList* pResCaptureTextureList, ResourceAccessor* pResAcsr, const char* pNamePrefix)
{
    NN_SDK_ASSERT_NOT_NULL(pResCaptureTextureList);
    NN_SDK_ASSERT_NOT_NULL(pResAcsr);

    m_pCaptureTextureList = Layout::AllocateAndConstruct<CaptureTextureList>();
    m_pCaptureTextureList->count = pResCaptureTextureList->texCount;
    m_pCaptureTextureList->pCaptureTextures = Layout::NewArray<CaptureTexture>(m_pCaptureTextureList->count);

    uint32_t resOffset = sizeof(*pResCaptureTextureList);
    for (int i = 0; i < m_pCaptureTextureList->count; ++i)
    {
        const T *const pResCaptureTexture = nn::util::ConstBytePtr(pResCaptureTextureList, resOffset).Get<T>();

        const char* pTextureName = GetCaptureTextureName(pResCaptureTextureList, pResCaptureTexture);
        m_pCaptureTextureList->pCaptureTextures[i].RegisterMaterialReferencedTextureInfo(pResAcsr, pTextureName, pNamePrefix);

        resOffset += (sizeof(T) + sizeof(ResCaptureTextureFilter));
    }
}

//-------------------------------------------------------------------------------------------------
template<typename T>
void
Layout::InitializeCaptureTextureList(nn::gfx::Device* pDevice, const nn::ui2d::ResCaptureTextureList* pResCaptureTextureList, const PartsBuildDataSet* pPartsBuildDataSet)
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT_NOT_NULL(pResCaptureTextureList);

    // キャプチャテクスチャ関連の初期化処理
    m_pCaptureTextureList->count = pResCaptureTextureList->texCount;

    size_t resOffset = sizeof(*pResCaptureTextureList);
    for (int i = 0; i < m_pCaptureTextureList->count; ++i)
    {
        const T *const pResCaptureTexture = nn::util::ConstBytePtr(pResCaptureTextureList, resOffset).Get<T>();
        const char* pPaneName = GetCaptureTextureReferencedPaneName(pResCaptureTextureList, pResCaptureTexture);

        // キャプチャテクスチャの上書き処理
        bool overriding = false;
        if (pPartsBuildDataSet != NULL)
        {
            const ResPartsProperty* pResPartsProperty = pPartsBuildDataSet->FindPartsPropertyFromName(pPaneName);
            if (pResPartsProperty != NULL)
            {
                const void* pOverrideResBlock = pPartsBuildDataSet->GetPropertyResBlock(pResPartsProperty);
                if (pOverrideResBlock != NULL &&
                    (static_cast<const nn::font::detail::BinaryBlockHeader*>(pOverrideResBlock))->kind == DataBlockKindCapture)
                {
                    const T * pResOverrideCaptureTexture = nn::util::ConstBytePtr(pOverrideResBlock, sizeof(ResCapture)).Get<T>();

                    // キャプチャテクスチャ上書きデータの nameStrOffset には 0 が固定で入っているが、名前はベースのキャプチャテクスチャデータと同じため使用しない。
                    m_pCaptureTextureList->pCaptureTextures[i].Initialize(pDevice, this, pResOverrideCaptureTexture, GetRootPane()->FindPaneByName(pPaneName));
                    overriding = true;
                }
            }
        }

        if (!overriding)
        {
            m_pCaptureTextureList->pCaptureTextures[i].Initialize(pDevice, this, pResCaptureTexture, GetRootPane()->FindPaneByName(pPaneName));
        }
        resOffset += (sizeof(T) + sizeof(ResCaptureTextureFilter));
    }
}

//-------------------------------------------------------------------------------------------------
void Layout::CopyLayoutInstanceImpl(
    nn::gfx::Device* pDevice,
    const Layout& src,
    Layout* pParentLayout,
    const char* pRootPaneName,
    const char* pNewCaptureTextureRootName)
{
    m_pRootPane = NULL;
    m_pGroupContainer = NULL;
    m_LayoutSize = src.m_LayoutSize;
    m_Name = src.m_Name;
    m_pResExtUserDataList = NULL;
    m_pResourceAccessor = src.m_pResourceAccessor;
    m_pCaptureTextureList = NULL;

    // キャプチャテクスチャのコピー処理１段階目
    // ペインのコピーの際にキャプチャテクスチャで作成される TextureInfo が参照されるためペインツリーのクローンの前に登録だけ行う。
    // コピーコンストラクタで実行される。
    if (src.m_pCaptureTextureList != NULL &&
        src.m_pCaptureTextureList->count > 0)
    {
        m_pCaptureTextureList = Layout::AllocateAndConstruct<CaptureTextureList>();
        m_pCaptureTextureList->count = src.m_pCaptureTextureList->count;

        m_pCaptureTextureList->pCaptureTextures = static_cast<CaptureTexture*>(Layout::AllocateMemory(sizeof(CaptureTexture) * m_pCaptureTextureList->count));
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
            // コピーコンストラクタの呼び出し
            new(&m_pCaptureTextureList->pCaptureTextures[i]) CaptureTexture(src.m_pCaptureTextureList->pCaptureTextures[i], src.m_pResourceAccessor, pNewCaptureTextureRootName);
        }
    }

    // ペインツリーのコピー
    m_pRootPane = ClonePaneTreeWithPartsLayout(src.GetRootPane(), src.IsPartsLayout() ? this : NULL, pDevice, this, m_pResourceAccessor, pNewCaptureTextureRootName, this);
    if (pRootPaneName) {
        m_pRootPane->SetName(pRootPaneName);
    }

    // グループのコピー
    m_pGroupContainer = AllocateAndConstruct<GroupContainer>();
    const nn::ui2d::GroupList& srcGroups = src.GetGroupContainer()->GetGroupList();
    for (nn::ui2d::GroupList::const_iterator iter = srcGroups.begin(), endIter = srcGroups.end(); iter != endIter; ++iter) {
        m_pGroupContainer->AppendGroup(AllocateAndConstruct<Group, const Group&, Pane*>(static_cast<const Group&>(*iter), m_pRootPane));
    }

    // 親のレイアウトがある場合は、RootPaneが部品ペインのはずなので、親のレイアウトの部品ペインリストに登録する
    if (pParentLayout) {
        Parts* pParts = DynamicCast<Parts*>(m_pRootPane);
        if (pParts) {
            pParentLayout->GetPartsPaneList().push_back(*pParts);
        }
    }

    // キャプチャテクスチャのコピー処理２段階目
    // キャプチャテクスチャのパラメータをコピーする。
    // パラメータには参照するペインのポインタが必要なためペインツリーのクローンが完了してから処理を行う。
    if (src.m_pCaptureTextureList != NULL&&
        src.m_pCaptureTextureList->count > 0)
    {
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
            const Pane* pPane = src.m_pCaptureTextureList->pCaptureTextures[i].GetTargetPane();
            if (pPane != NULL)
            {
                m_pCaptureTextureList->pCaptureTextures[i].Initialize(pDevice, this, src.m_pCaptureTextureList->pCaptureTextures[i], m_pRootPane->FindPaneByName(pPane->GetName()));
            }
        }
    }
}

void Layout::CalculateGlobalMatrix(DrawInfo& drawInfo, bool forceGlobalMatrixDirty)
{
    if (this->GetRootPane() == NULL)
    {
        return;
    }

    Pane::CalculateContext context;
    context.Set(drawInfo, this);
    this->GetRootPane()->CalculateGlobalMatrix(context, forceGlobalMatrixDirty);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic()
{
    return CreateAnimTransform<AnimTransformBasic>();
}

void
Layout::DeleteAnimTransform(AnimTransform *pAnimTransform)
{
    NN_SDK_ASSERT(pAnimTransform, "pAnimTransform must not be NULL for Layout[%s]", GetName());

    this->GetAnimTransformList().erase(this->GetAnimTransformList().iterator_to(*pAnimTransform));
    DeleteObj(pAnimTransform);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    nn::gfx::Device* pDevice,
    const void* pAnimResource)
{
    return CreateAnimTransform<AnimTransformBasic>(pDevice, pAnimResource);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    nn::gfx::Device* pDevice,
    const AnimResource& animRes)
{
    return CreateAnimTransform<AnimTransformBasic>(pDevice, animRes);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    nn::gfx::Device* pDevice,
    const char* pTagName)
{
    return CreateAnimTransform<AnimTransformBasic>(pDevice, pTagName);
}

PaneAnimator* Layout::CreatePaneAnimator(nn::gfx::Device* pDevice, const char* pTagName, Pane* pPane, bool enabled /* = true */)
{
    const void* pResData = GetAnimResourceData(pTagName);
    NN_SDK_ASSERT(pResData, "cannot find resource of pTagName[%s] pane[%s] Layout[%s]", pTagName, pPane->GetName(), GetName());
    PaneAnimator* pAnimator = CreateAnimTransform<PaneAnimator>(pDevice, pResData);
    pAnimator->Setup(pPane, enabled);
    return pAnimator;
}

GroupAnimator* Layout::CreateGroupAnimator(nn::gfx::Device* pDevice, const char* pTagName, Group* pGroup, bool enabled /* = true */)
{
    const void* pResData = GetAnimResourceData(pTagName);
    NN_SDK_ASSERT(pResData, "cannot find resource of pTagName[%s] group[%s] Layout[%s]", pTagName, pGroup->GetName(), GetName());
    GroupAnimator* pAnimator = CreateAnimTransform<GroupAnimator>(pDevice, pResData);
    pAnimator->Setup(pGroup, enabled);
    return pAnimator;
}

GroupAnimator* Layout::CreateGroupAnimatorWithIndex(nn::gfx::Device* pDevice, const char* pTagName, int  groupIndex, bool enabled /* = true */)
{
    const void* pResData = GetAnimResourceData(pTagName);
    NN_SDK_ASSERT(pResData, "cannot find resource of pTagName[%s] groupIndex[%d] Layout[%s]", pTagName, groupIndex, GetName());
    AnimResource animRes(pResData);
    GroupAnimator* pAnimator = CreateAnimTransform<GroupAnimator>(pDevice, animRes);
    pAnimator->Setup(animRes, m_pGroupContainer, groupIndex, enabled);
    return pAnimator;
}

GroupArrayAnimator* Layout::CreateGroupArrayAnimator(nn::gfx::Device* pDevice, const AnimResource& animRes, bool enabled)
{
    // newの回数を減らすためにCreateAnimTransformをばらして使う
    NN_SDK_ASSERT(g_pAllocateFunction != NULL, "AllocateFunction must not be NULL for Layout[%s]", GetName());

    const ResAnimationBlock *const pAnimBlock = animRes.GetResourceBlock();

    if (! pAnimBlock)
    {
        return NULL;
    }
    void* pPtr = Layout::AllocateMemory(sizeof(GroupArrayAnimator) + animRes.GetGroupCount() * sizeof(Group*));
    if (pPtr == NULL) {
        return NULL;
    }
    GroupArrayAnimator* pAnimator = new(pPtr) GroupArrayAnimator();
    pAnimator->SetResource(pDevice, m_pResourceAccessor, pAnimBlock);
    this->GetAnimTransformList().push_back(*pAnimator);

    pAnimator->Setup(animRes, m_pGroupContainer, nn::util::BytePtr(pPtr, sizeof(GroupArrayAnimator)).Get<Group*>(), enabled);
    return pAnimator;
}

GroupArrayAnimator* Layout::CreateGroupArrayAnimator(nn::gfx::Device* pDevice, const char* pTagName, bool enabled /* = true */)
{
    const void* pResData = GetAnimResourceData(pTagName);
    NN_SDK_ASSERT(pResData, "cannot find resource of pTagName[%s] Layout[%s]", pTagName, GetName());
    AnimResource animRes(pResData);
    return CreateGroupArrayAnimator(pDevice, animRes, enabled);
}

Animator* Layout::CreateGroupAnimatorAuto(nn::gfx::Device* pDevice, const char* pTagName, bool enabled /* = true */)
{
    const void* pResData = GetAnimResourceData(pTagName);
    NN_SDK_ASSERT(pResData, "cannot find resource of pTagName[%s] Layout[%s]", pTagName, GetName());
    AnimResource animRes(pResData);

    if (animRes.GetGroupCount() > 1)
    {
        return CreateGroupArrayAnimator(pDevice, animRes, enabled);
    }
    else
    {
        GroupAnimator* pAnimator = CreateAnimTransform<GroupAnimator>(pDevice, animRes);
        pAnimator->Setup(animRes, m_pGroupContainer, 0, enabled);
        return pAnimator;
    }
}

void
Layout::BindAnimation(AnimTransform* pAnimTrans)
{
    if (this->GetRootPane())
    {
        pAnimTrans->BindPane(this->GetRootPane(), true);
    }
}

void
Layout::UnbindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->UnbindAll();
}

void
Layout::UnbindAnimation(Pane* pPane)
{
    AnimTransformList::iterator endIter = m_AnimTransList.end();
    for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != endIter; ++iter)
    {
        iter->UnbindPane(pPane);
    }
}

void
Layout::UnbindAllAnimation()
{
    AnimTransformList::iterator endIter = m_AnimTransList.end();
    for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != endIter; ++iter)
    {
        iter->UnbindAll();
    }
}

bool
Layout::BindAnimationAuto(
    nn::gfx::Device* pDevice,
    const AnimResource& animRes
)
{
    if (! this->GetRootPane())
    {
        return false;
    }

    if (! animRes.GetResourceBlock())
    {
        return false;
    }

    AnimTransform *const pAnimTrans = CreateAnimTransformBasic();  // AnimTransform オブジェクトの作成
    if (pAnimTrans == NULL)
    {
        NN_DETAIL_UI2D_ERROR("Create AnimTransform failed.");
        return false;
    }

    bool bResult = true;

    // 最初に名前によるバインドを一通り行う
    const uint16_t  bindGroupNum = animRes.GetGroupCount();

    uint16_t  animNum = 0;
    if (bindGroupNum == 0)  // バインドする対象を限定していない
    {
        animNum = animRes.GetResourceBlock()->animContCount;

        // バインドするアニメーション数を明示的に指定してリソースをセットします。
        pAnimTrans->SetResource(pDevice, m_pResourceAccessor, animRes.GetResourceBlock(), animNum);

        const bool bRecursive = true;
        this->GetRootPane()->BindAnimation(pAnimTrans, bRecursive, false/* bEnable */);
    }
    else    // グループを指定してのバインド
    {
        const ResAnimationGroupRef *const pGroupRefs = animRes.GetGroupArray();
        for (int groupIndex = 0; groupIndex < bindGroupNum; ++groupIndex)
        {
            // グループにあるペイン全てで必要になるアニメーションの個数を数えます。
            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(pGroupRefs[groupIndex].GetName());
            if (pGroup == NULL)
            {
                NN_DETAIL_UI2D_ERROR("Group not found: %s", pGroupRefs[groupIndex].GetName());
                bResult = false;
                continue;
            }

            animNum += animRes.CalculateAnimationCount(pGroup, animRes.IsDescendingBind());
        }

        // バインドするアニメーション数を明示的に指定してリソースをセットします。
        pAnimTrans->SetResource(pDevice, m_pResourceAccessor, animRes.GetResourceBlock(), animNum);

        for (int groupIndex = 0; groupIndex < bindGroupNum; ++groupIndex)
        {
            // グループにあるペイン全てで必要になるアニメーションの個数を数えます。
            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(pGroupRefs[groupIndex].GetName());
            if (pGroup == NULL)
            {
                continue;
            }

            // アニメーションをバインドします。
            nn::ui2d::BindAnimation(pAnimTrans, pGroup, false/* bEnable */);
        }
    }

    const uint16_t  animShareInfoNum = animRes.GetAnimationShareInfoCount();
    if (animShareInfoNum > 0)   // アニメーション共有によるバインド
    {
        const ResAnimationShareInfo *const pAnimShareInfos = animRes.GetAnimationShareInfoArray();
        NN_SDK_ASSERT_NOT_NULL(pAnimShareInfos);

        for (int i = 0; i < animShareInfoNum; ++i)
        {
            Pane *const pSrcPane = this->GetRootPane()->FindPaneByName(pAnimShareInfos[i].GetSrcPaneName());
            if (pSrcPane == NULL)
            {
                NN_DETAIL_UI2D_ERROR("Source pane of animation-share is not found: %s", pAnimShareInfos[i].GetSrcPaneName());
                bResult = false;
                continue;
            }

            detail::AnimPaneTree animPaneTree(pSrcPane, animRes);
            if (! animPaneTree.IsEnabled()) // 共有元ペインにアニメーションが無い?
            {
                continue;
            }

            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(pAnimShareInfos[i].GetTargetGroupName());
            if (pGroup == NULL)
            {
                NN_DETAIL_UI2D_ERROR("Target group of animation-share is not found: %s", pAnimShareInfos[i].GetTargetGroupName());
                bResult = false;
                continue;
            }

            PaneLinkList& paneList = pGroup->GetPaneList();
            uint32_t  animIdx = 0;
            for (PaneLinkList::iterator iter = paneList.begin(); iter != paneList.end(); ++iter, ++animIdx)
            {
                if (iter->pTarget != pSrcPane)
                {
                    if (bindGroupNum > 0)   // バインド対象の指定があり
                    {
                        // 関連グループに含まれてない場合は共有しない
                        const bool bInclude = IsIncludeAnimationGroupRef(
                            this->GetGroupContainer(),
                            animRes.GetGroupArray(),
                            bindGroupNum,
                            animRes.IsDescendingBind(),
                            iter->pTarget);

                        if (! bInclude)
                        {
                            continue;
                        }
                    }

                    // srcPaneNameと異なるペインに対してアニメーションをバインド
                    animPaneTree.Bind(pDevice, this, iter->pTarget, m_pResourceAccessor);
                }
            }
        }
    }

    return bResult;
}// NOLINT(impl/function_size)

void
Layout::CalculateImpl(DrawInfo& drawInfo, bool forceGlbMtxDirty)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Layout::CalculateMtx");

    if (! this->GetRootPane())
    {
        return;
    }

    Pane::CalculateContext context;
    context.Set(drawInfo, this);

    drawInfo.SetLayout(this);

    this->GetRootPane()->Calculate(drawInfo, context, forceGlbMtxDirty);

    drawInfo.SetLayout(0);
}

void
Layout::DrawCaptureTexture(nn::gfx::Device* pDevice, DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    if (! this->GetRootPane())
    {
        return;
    }

    drawInfo.ConfigureBeforeDrawing(this);

    // 子パーツのキャプチャテクスチャを先に更新する。
    PartsPaneList::iterator endIter = m_PartsPaneList.end();
    for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
    {
        iter->GetLayout()->DrawCaptureTexture(pDevice, drawInfo, commandBuffer);
    }

    if (m_pCaptureTextureList != NULL)
    {
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
            m_pCaptureTextureList->pCaptureTextures[i].Draw(pDevice, drawInfo, commandBuffer);
        }
    }

    commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    drawInfo.ConfigureAfterDrawing();
}

void
Layout::DrawDropShadowStaticCache(nn::gfx::Device* pDevice, DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    if (! this->GetRootPane())
    {
        return;
    }

    drawInfo.ConfigureBeforeDrawing(this);

    // 子パーツのドロップシャドウキャッシュを更新する。
    PartsPaneList::iterator endIter = m_PartsPaneList.end();
    for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
    {
        iter->GetLayout()->DrawDropShadowStaticCache(pDevice, drawInfo, commandBuffer);
    }

    // ドロップシャドウはキャプチャテクスチャに依存した機能のため、キャプチャテクスチャのリストから逆にペインを引いてくることで検索しないようにする。
    if (m_pCaptureTextureList != NULL)
    {
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
            const Pane* pPane = m_pCaptureTextureList->pCaptureTextures[i].GetTargetPane();
            detail::PaneEffect* pPaneEffect = pPane->GetPaneEffectInstance();

            if (pPaneEffect == NULL)
            {
                continue;
            }

            // ドロップシャドウの静的レンダリングが有効で、かつまだキャッシュが作成されていない場合のみレンダリングする。
            if (pPaneEffect->IsDropShadowStaticRenderingEnabled() &&
                !pPaneEffect->IsDropShadowStaticRenderingReady())
            {
                pPaneEffect->DrawDropShadow(drawInfo, commandBuffer);
            }
        }
    }

    commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    drawInfo.ConfigureAfterDrawing();
}


void
Layout::Draw(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Layout::Draw");

    if (! this->GetRootPane())
    {
        return;
    }

    drawInfo.ConfigureBeforeDrawing(this);

    this->GetRootPane()->Draw(drawInfo, commandBuffer);

    drawInfo.ConfigureAfterDrawing();
}

void
Layout::Animate()
{
    {
        AnimTransformList::iterator endIter = m_AnimTransList.end();
        for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != endIter; ++iter)
        {
            iter->Animate();
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::iterator endIter = m_PartsPaneList.end();
        for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
        {
            iter->GetLayout()->Animate();
        }
    }
}

void
Layout::UpdateAnimFrame(float progressFrame)
{
    {
        AnimTransformList::iterator endIter = m_AnimTransList.end();
        for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != endIter; ++iter)
        {
            iter->UpdateFrame(progressFrame);
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::iterator endIter = m_PartsPaneList.end();
        for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
        {
            iter->GetLayout()->UpdateAnimFrame(progressFrame);
        }
    }
}

void
Layout::AnimateAndUpdateAnimFrame(float progressFrame)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Layout::AnimateAndUpdateAnimFrame");

    {
        AnimTransformList::iterator endIter = m_AnimTransList.end();
        for (AnimTransformList::iterator iter = m_AnimTransList.begin(); iter != endIter; ++iter)
        {
            iter->Animate();
            iter->UpdateFrame(progressFrame);
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::iterator endIter = m_PartsPaneList.end();
        for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
        {
            iter->GetLayout()->AnimateAndUpdateAnimFrame(progressFrame);
        }
    }
}

const nn::font::Rectangle
Layout::GetLayoutRect() const
{
    return nn::font::Rectangle(- m_LayoutSize.width / 2, m_LayoutSize.height / 2, m_LayoutSize.width / 2, - m_LayoutSize.height / 2);
}

void
Layout::SetTagProcessor(nn::font::TagProcessorBase<uint16_t>* pTagProcessor)
{
    SetTagProcessorImpl(this->GetRootPane(), pTagProcessor);
}

Parts*
Layout::FindPartsPaneByName(const char* pFindName)
{
    PartsPaneList::iterator endIter = m_PartsPaneList.end();
    for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
    {
        if (detail::EqualsResName(pFindName, iter->GetName()))
        {
            return &(*iter);
        }
    }

    return NULL;
}

const CaptureTexture*
Layout::FindCaptureTextureByPanePtr(const Pane* pTargetPane) const
{
    NN_SDK_ASSERT_NOT_NULL(pTargetPane);

    // 自分の持っているリストから探す。
    if (m_pCaptureTextureList != NULL)
    {
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
             const Pane* pPane = m_pCaptureTextureList->pCaptureTextures[i].GetTargetPane();
             if (pPane == pTargetPane)
             {
                 return &(m_pCaptureTextureList->pCaptureTextures[i]);
             }
        }
    }

    // 子パーツから再帰的に探す。
    PartsPaneList::const_iterator endIter = m_PartsPaneList.end();
    for (PartsPaneList::const_iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
    {
        const Layout* pLayout = iter->GetLayout();
        if (pLayout != NULL)
        {
            const CaptureTexture* pTexture = pLayout->FindCaptureTextureByPanePtr(pTargetPane);
            if (pTexture != NULL)
            {
                return pTexture;
            }
        }
    }

    return NULL;
}

void
Layout::ResetFirstFrameCaptureUpdatdFlag()
{
    if (! this->GetRootPane())
    {
        return;
    }

    // 子パーツを処理する
    PartsPaneList::iterator endIter = m_PartsPaneList.end();
    for (PartsPaneList::iterator iter = m_PartsPaneList.begin(); iter != endIter; ++iter)
    {
        iter->GetLayout()->ResetFirstFrameCaptureUpdatdFlag();
    }

    if (m_pCaptureTextureList != NULL)
    {
        for (int i = 0; i < m_pCaptureTextureList->count; ++i)
        {
            m_pCaptureTextureList->pCaptureTextures[i].ResetFirstFrameCaptureUpdatdFlag();
        }
    }
}

const Parts*
Layout::FindPartsPaneByName(const char* pFindName) const
{
    return const_cast<Layout*>(this)->FindPartsPaneByName(pFindName);
}

bool
Layout::CompareCopiedInstanceTest(const Layout& target) const
{
    /*
    以下のメンバー変数はコピーコンストラクタではコピーされない。

    AnimTransformList m_AnimTransList;
    const ResExtUserDataList* m_pResExtUserDataList;
    PartsPaneList m_PartsPaneList;
    */

    if (m_LayoutSize.width != target.m_LayoutSize.width ||
        m_LayoutSize.height != target.m_LayoutSize.height ||
        m_Name != target.m_Name ||
        m_pResourceAccessor != target.m_pResourceAccessor)
    {
        return false;
    }

    // ペインツリーをすべて比較。
    if (ComparePaneTreeTest(m_pRootPane, target.m_pRootPane) == false)
    {
        return false;
    }

    if (m_pGroupContainer->GetGroupList().size() != target.m_pGroupContainer->GetGroupList().size())
    {
        return false;
    }

    const nn::ui2d::GroupList& groupsLhs = m_pGroupContainer->GetGroupList();
    const nn::ui2d::GroupList& groupsRhs = target.m_pGroupContainer->GetGroupList();

    nn::ui2d::GroupList::const_iterator iterLhs = groupsLhs.begin(), endIter = groupsLhs.end();
    nn::ui2d::GroupList::const_iterator iterRhs = groupsRhs.begin();

    for (; iterLhs != endIter; ++iterLhs, ++iterRhs)
    {
        if ((*iterLhs).CompareCopiedInstanceTest(*iterRhs) == false)
        {
            return false;
        }
    }

    return true;
}

const ShaderInfo* Layout::AcquireArchiveShader(nn::gfx::Device* pDevice, const char* pName) const
{
    return m_pResourceAccessor->AcquireShader(pDevice, pName);
}

Pane*
Layout::BuildPaneObj(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    uint32_t        kind,
    const void*        pBlock,
    const void*        pOverrideBlock,
    const BuildArgSet& buildArgSet
)
{
    switch (kind)
    {
    case DataBlockKindPane:
    {
        const ResPane* pResPane = static_cast<const ResPane*>(pBlock);
        return AllocateAndConstruct<Pane, BuildResultInformation*, nn::gfx::Device*, const ResPane*, const BuildArgSet&>(pOutBuildResultInformation, pDevice, pResPane, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindPicture:
    {
        const ResPicture* pResPic = static_cast<const ResPicture*>(pBlock);
        const ResPicture* pResPicOverride = static_cast<const ResPicture*>(pOverrideBlock);
        return AllocateAndConstruct<Picture, BuildResultInformation*, nn::gfx::Device*, const ResPicture*, const ResPicture*, const BuildArgSet&>(pOutBuildResultInformation, pDevice, pResPic, pResPicOverride, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindTextBox:
    {
        const ResTextBox* pResTextBox = static_cast<const ResTextBox*>(pBlock);
        const ResTextBox* pResTextBoxOverride = static_cast<const ResTextBox*>(pOverrideBlock);
        TextBox::InitializeStringParam initStringParam;
        TextBox* textBox = AllocateAndConstruct<TextBox, BuildResultInformation*, nn::gfx::Device*, TextBox::InitializeStringParam*, const ResTextBox*, const ResTextBox*, const BuildArgSet&>(pOutBuildResultInformation, pDevice, &initStringParam, pResTextBox, pResTextBoxOverride, buildArgSet, Layout::AlignmentForPaneInstance);
        textBox->InitializeString(pOutBuildResultInformation, pDevice, buildArgSet, initStringParam);
        return textBox;
    }
    case DataBlockKindWindow:
    {
        const ResWindow* pResWindow = static_cast<const ResWindow*>(pBlock);
        const ResWindow* pResWindowOverride = static_cast<const ResWindow*>(pOverrideBlock);
        return AllocateAndConstruct<Window, BuildResultInformation*, nn::gfx::Device*, const ResWindow*, const ResWindow*, const BuildArgSet&>(pOutBuildResultInformation, pDevice, pResWindow, pResWindowOverride, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindBounding:
    {
        const ResBounding* pResBounding = static_cast<const ResBounding*>(pBlock);
        const ResBounding* pResBoundingOverride = static_cast<const ResBounding*>(pOverrideBlock);
        return AllocateAndConstruct<Bounding, const ResBounding*, const ResBounding*, const BuildArgSet&>(pResBounding, pResBoundingOverride, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindCapture:
    {
        const ResCapture* pResCapture = static_cast<const ResCapture*>(pBlock);
        const ResCapture* pResCaptureOverride = static_cast<const ResCapture*>(pOverrideBlock);
        return AllocateAndConstruct<Capture, const ResCapture*, const ResCapture*, const BuildArgSet&>(pResCapture, pResCaptureOverride, buildArgSet, Layout::AlignmentForPaneInstance);
    }

    case DataBlockKindParts:
    {
        const ResParts* pResParts = static_cast<const ResParts*>(pBlock);
        const ResParts* pResPartsOverride = static_cast<const ResParts*>(pOverrideBlock);
        return AllocateAndConstruct<Parts, const ResParts*, const ResParts*, const BuildArgSet&>(pResParts, pResPartsOverride, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindAlignment:
    {
        const ResAlignment* pResAlignment = static_cast<const ResAlignment*>(pBlock);
        return AllocateAndConstruct<Alignment, const ResAlignment*, const BuildArgSet&>(pResAlignment, buildArgSet, Layout::AlignmentForPaneInstance);
    }
    case DataBlockKindScissor:
    {
        const ResScissor* pResScissor = static_cast<const ResScissor*>(pBlock);
        return AllocateAndConstruct<Scissor, const ResScissor*, const BuildArgSet&>(pResScissor, buildArgSet, Layout::AlignmentForPaneInstance);
    }

    default:
        NN_SDK_ASSERT(false, "unknown data type for Layout[%s]", GetName());
        break;
    }

    return 0;
}

Layout* Layout::BuildPartsLayout(BuildResultInformation* pOutBuildResultInformation, nn::gfx::Device* pDevice, const char* pName, const PartsBuildDataSet& partsBuildDataSet, const BuildArgSet& buildArgSet)
{
    const void* pResPtr = GetLayoutResourceData(pName);
    NN_SDK_ASSERT(pResPtr, "cannot find resource of parts Layout[%s] in body Layout[%s]", pName, GetName());

    // レイアウトオブジェクトの作成
    Layout* pLayout = AllocateAndConstruct<Layout>();

    // レイアウトのビルド
    pLayout->BuildImpl(pOutBuildResultInformation, pDevice, pResPtr, m_pResourceAccessor, buildArgSet, &partsBuildDataSet);

    return pLayout;
}

const void* Layout::GetAnimResourceData(const char* pTagName) const
{
    char brlanPath[LayoutNameLengthMax + AnimTagNameStrMax + 8];
    nn::util::SNPrintf(brlanPath, sizeof(brlanPath), "%s_%s.bflan", m_Name, pTagName);

    return m_pResourceAccessor->FindResourceByName(ResourceTypeAnimation, brlanPath);
}

const void* Layout::GetLayoutResourceData(const char* pLayoutName) const
{
    char bclytPath[LayoutNameLengthMax + 8];
    nn::util::SNPrintf(bclytPath, sizeof(bclytPath), "%s.bflyt", pLayoutName);

    return m_pResourceAccessor->FindResourceByName(ResourceTypeLayout, bclytPath);
}

Layout::PartsBuildDataAccessor::PartsBuildDataAccessor(const ResParts* pResParts)
    : m_PropertyCount(pResParts->propertyCount)
    , m_pPropertyTable(nn::util::ConstBytePtr(pResParts, sizeof(ResParts)).Get<ResPartsProperty>())
    , m_pResParts(pResParts)
{

}

const ResPartsProperty* Layout::PartsBuildDataAccessor::FindPartsPropertyFromName(const char* pName) const
{
    for (int32_t i = 0; i < m_PropertyCount; i++) {
        if (detail::EqualsResName(pName, m_pPropertyTable[i].name))
        {
            return &m_pPropertyTable[i];
        }
    }
    return NULL;
}

bool Layout::PartsBuildDataAccessor::IsOverwriting() const
{
    // レイアウト中にパーツが配置されている場合は、上書きしていなくても propertyCount がペインの数分設定されているため
    // 実際に上書きしているかどうかは ResPartsProperty の内容が 0 かどうかで判定します。
    // レイアウトコンバーターでバイナリデータを書き出す際に上書きしていない ResPartsProperty は
    // その領域すべてが 0 で埋められる仕様となっておりこの挙動を利用した判定です。
    for (int32_t i = 0; i < m_PropertyCount; i++) {
        if (m_pPropertyTable[i].name[0] != 0)
        {
            return true;
        }
    }

    return false;
}

const void* Layout::PartsBuildDataAccessor::GetPropertyResBlock(const ResPartsProperty* pResPartsProperty) const
{
    if (pResPartsProperty->propertyOffset == 0)
    {
        return NULL;
    }
    else
    {
        return nn::util::ConstBytePtr(m_pResParts, pResPartsProperty->propertyOffset).Get<void>();
    }
}

const ResExtUserDataList* Layout::PartsBuildDataAccessor::GetExtUserDataListResBlock(bool* pIsOverride, const ResPartsProperty* pResPartsProperty) const
{
    if (pResPartsProperty->extUserDataOffset == ExtUserDataOverrideOffset_NoOverride)
    {
        // 上書きしない
        *pIsOverride = false;
        return NULL;
    }
    else if (pResPartsProperty->extUserDataOffset == ExtUserDataOverrideOffset_NoData)
    {
        // 上書きするが、データがない
        *pIsOverride = true;
        return NULL;
    }
    else
    {
        // 上書きする。データのリストを返す
        *pIsOverride = true;
        return static_cast<const ResExtUserDataList*>(nn::util::ConstBytePtr(m_pResParts, pResPartsProperty->extUserDataOffset).Get<void>());
    }
}

const ResPartsPaneBasicInfo* Layout::PartsBuildDataAccessor::GetPartsPaneBasicInfoResBlock(const ResPartsProperty* pResPartsProperty) const
{
    if (pResPartsProperty->paneBasicInfoOffset == 0)
    {
        return NULL;
    }
    else
    {
        return static_cast<const ResPartsPaneBasicInfo*>(nn::util::ConstBytePtr(m_pResParts, pResPartsProperty->paneBasicInfoOffset).Get<void>());
    }
}

Layout::PartsBuildDataSet::PartsBuildDataSet(Parts* pPartsPane, const ResParts* pResParts, const BuildResSet* pBuildResSet, const ResVec2* pOriginalSize)
: m_PartsBuildDataAccessor(pResParts)
, m_pPartsPane(pPartsPane)
, m_pPropertyBuildResSet(pBuildResSet)
{
    m_Magnify.v[0] = (pPartsPane->GetSize().width / pOriginalSize->x) * pResParts->magnify.x;
    m_Magnify.v[1] = (pPartsPane->GetSize().height / pOriginalSize->y) * pResParts->magnify.y;
}

} // namespace nn::ui2d
} // namespace nn
