﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>

#include <nn/ui2d/ui2d_FontContainer.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/font/font_ResFont.h>

namespace nn
{
namespace ui2d
{

FontRefLink::FontRefLink()
:   m_pFont(0),
    m_IsOwned(false)
{
}

FontRefLink::~FontRefLink()
{
    NN_SDK_ASSERT(!(m_IsOwned && m_pFont != NULL));
}

void FontRefLink::Finalize(nn::gfx::Device* pDevice)
{
    if (m_IsOwned && m_pFont != NULL)
    {
        m_pFont->Finalize(pDevice);
        Layout::DeleteObj(m_pFont);
        m_pFont = NULL;
    }
}

void
FontRefLink::Set(
    const char* pName,
    nn::font::Font* pFont,
    bool own
)
{
    nn::util::Strlcpy(m_FontName, pName, sizeof(m_FontName));
    m_pFont = pFont;
    m_IsOwned = own;
}

FontContainer::~FontContainer()
{
    NN_SDK_ASSERT(this->empty());
}

void
FontContainer::Finalize(nn::gfx::Device* pDevice)
{
    while (!this->empty())
    {
        FontRefLink* pLink = &this->front();
        pLink->Finalize(pDevice);
        this->erase(this->iterator_to(*pLink));
        Layout::DeleteObj(pLink);
    }
}

nn::font::Font*
FontContainer::FindFontByName(
    const char* pName
) const
{
    const_iterator endIter = this->end();
    for (const_iterator iter = this->begin(); iter != endIter; ++iter)
    {
        if (detail::EqualsFontName(pName, iter->GetFontName()))
        {
            return iter->GetFont();
        }
    }

    return NULL;
}

FontKey
FontContainer::RegisterFont(const char* pName, nn::font::Font* pFont, bool own)
{
    FontRefLink* pLink = Layout::AllocateAndConstruct<FontRefLink>();

    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->Set(pName, pFont, own);

    this->push_back(*pLink);

    return reinterpret_cast<FontKey>(pLink);
}

void
FontContainer::UnregisterFont(FontKey key)
{
    NN_SDK_ASSERT_NOT_NULL(key);

    FontRefLink* pLink = (FontRefLink*)(key);
    this->erase(this->iterator_to(*pLink));
    Layout::DeleteObj(pLink);
}

void
FontContainer::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot pRegisterTextureViewSlot, void* pUserData)
{
    iterator endIter = this->end();
    for (iterator iter = this->begin(); iter != endIter; ++iter)
    {
        if (iter->IsOwned())
        {
            reinterpret_cast<nn::font::ResFontBase*>(iter->GetFont())->RegisterTextureViewToDescriptorPool(pRegisterTextureViewSlot, pUserData);
        }
    }
}

void
FontContainer::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot pUnregisterTextureViewSlot, void* pUserData)
{
    iterator endIter = this->end();
    for (iterator iter = this->begin(); iter != endIter; ++iter)
    {
        if (iter->IsOwned())
        {
            reinterpret_cast<nn::font::ResFontBase*>(iter->GetFont())->UnregisterTextureViewFromDescriptorPool(pUnregisterTextureViewSlot, pUserData);
        }
    }
}

}   // namespace ui2d
}   // namespace nn
