﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_DrawInfo.h>

#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/util/util_Matrix.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
DrawInfo::DrawInfo()
: m_pGraphicsResource(NULL)
, m_pLayout(NULL)
, m_pUi2dConstantBuffer(NULL)
, m_pFontConstantBuffer(NULL)
, m_pFramebufferTexture(NULL)
, m_FramebufferWidth(0)
, m_FramebufferHeight(0)
, m_pFramebufferTextureDescriptorSlot(NULL)
, m_pFramebufferSamplerDescriptorSlot(NULL)
, m_pDefaultColorTargetView(NULL)
, m_pUpdateUserShaderCallback(NULL)
, m_pUpdateUserShaderCallbackUserData(NULL)
, m_pApplyCaptureTextureFilterCallback(NULL)
, m_pApplyCaptureTextureFilterCallbackUserData(NULL)
, m_ModelViewMtxLoaded(false)
, m_FirstDraw(false)
, m_AllowableTextureQuantity(0)
, m_pCurrentShaderInfo(NULL)
, m_CurrentShaderVariation(0)
{
    m_LocationAdjustScale.v[0] = 1.0f;
    m_LocationAdjustScale.v[1] = 1.0f;

    m_DefaultViewportInfo.SetDefault();
    m_DefaultScissorInfo.SetDefault();

    std::memset(&m_Flag, 0, sizeof(m_Flag));

    nn::util::MatrixIdentity(&m_ProjMtx);
    nn::util::MatrixIdentity(&m_ViewMtx);
    nn::util::MatrixIdentity(&m_ModelViewMtx);
}

//----------------------------------------
DrawInfo::~DrawInfo()
{
}

//----------------------------------------
void DrawInfo::ConfigureBeforeDrawing(Layout* pLayout)
{
    ResetDrawState();
    SetLayout(pLayout);
}

//----------------------------------------
void DrawInfo::ConfigureAfterDrawing()
{
    SetLayout(NULL);
}

//----------------------------------------
void DrawInfo::ResetDrawState()
{
    ResetVertexBufferState();
    ResetCurrentShader();
}

//----------------------------------------
void DrawInfo::Map(int bufferIndex)
{
    if (m_pUi2dConstantBuffer != NULL)
    {
        m_pUi2dConstantBuffer->Map(bufferIndex);
    }

    if (m_pFontConstantBuffer != NULL)
    {
        m_pFontConstantBuffer->Map(bufferIndex);
    }
}

//----------------------------------------
void DrawInfo::Unmap()
{
    if (m_pUi2dConstantBuffer != NULL)
    {
        m_pUi2dConstantBuffer->Unmap();
    }

    if (m_pFontConstantBuffer != NULL)
    {
        m_pFontConstantBuffer->Unmap();
    }
}

//----------------------------------------
void DrawInfo::SetGpuAccessBufferIndex(int bufferIndex)
{
    if (m_pUi2dConstantBuffer != NULL)
    {
        m_pUi2dConstantBuffer->SetGpuAccessBufferIndex(bufferIndex);
    }

    if (m_pFontConstantBuffer != NULL)
    {
        m_pFontConstantBuffer->SetGpuAccessBufferIndex(bufferIndex);
    }
}

//----------------------------------------
void DrawInfo::SetFramebufferTexture(nn::gfx::Texture* pTexture, int width, int height)
{
    m_pFramebufferTexture = pTexture;
    m_FramebufferWidth = width;
    m_FramebufferHeight = height;
}


//----------------------------------------
void
DrawInfo::SetProjectionMtx(const nn::util::MatrixT4x4fType& mtx)
{
    // 直ちにシェーダに反映させると複数の DrawInfo がある場合に
    // 誤作動するので、ここでは値を保持するのみ。
    // 実際の設定は Layout::Draw() で行う。
    m_ProjMtx = mtx;
}

//----------------------------------------
void
DrawInfo::SetupProgram(nn::gfx::CommandBuffer* pCommandBuffer)
{
    if (m_FirstDraw)
    {
        m_FirstDraw = false;
        m_pGraphicsResource->ActivateVertexBuffer(pCommandBuffer);
    }
}

//----------------------------------------
bool DrawInfo::RecordCurrentShader(const ShaderInfo* pShaderInfo, uint8_t shaderVariation)
{
    if (m_pCurrentShaderInfo == pShaderInfo && m_CurrentShaderVariation == shaderVariation)
    {
        return false;
    }
    m_pCurrentShaderInfo = pShaderInfo;
    m_CurrentShaderVariation = shaderVariation;

    return true;
}

//----------------------------------------
void DrawInfo::ResetCurrentShader()
{
    m_pCurrentShaderInfo = NULL;
    m_CurrentShaderVariation = 0;
}

//----------------------------------------
void DrawInfo::ResetRenderTarget(nn::gfx::CommandBuffer& commandBuffer) const
{
    commandBuffer.SetRenderTargets(1, &m_pDefaultColorTargetView, NULL);
    commandBuffer.SetScissors(0, 1, &m_DefaultScissorInfo);
    commandBuffer.SetViewports(0, 1, &m_DefaultViewportInfo);
}

//----------------------------------------
void
DrawInfo::LoadProjectionMtx(float pProjectionMtx[4][4])
{
    NN_SDK_ASSERT(m_pGraphicsResource->Initialized());

    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pProjectionMtx), m_ProjMtx);
}

//----------------------------------------
void
DrawInfo::LoadMtxModelView(float pModelView[3][4])
{
    if (!m_ModelViewMtxLoaded)
    {
        m_ModelViewMtxLoaded = true;
        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pModelView), m_ModelViewMtx);
    }
}

//----------------------------------------
void
DrawInfo::SetFramebufferTextureDescriptorSlot(nn::gfx::DescriptorSlot* pTextureDescriptorSlot)
{
    // コンバイナユーザーシェーダでフレームバッファテクスチャを描画として利用する場合に使われるテクスチャデスクリプタを設定します。
    // この関数は、未公開のコンバイナユーザーシェーダで使われる為、 Documents に載らないようにヘッダのコメントを省いております。
    m_pFramebufferTextureDescriptorSlot = pTextureDescriptorSlot;
}

void
DrawInfo::SetFramebufferSamplerDescriptorSlot(nn::gfx::DescriptorSlot* pSamplerDescriptorSlot)
{
    // コンバイナユーザーシェーダでフレームバッファテクスチャを描画として利用する場合に使われるサンプラデスクリプタを設定します。
    // この関数は、未公開のコンバイナユーザーシェーダで使われる為、 Documents に載らないようにヘッダのコメントを省いております。
    m_pFramebufferSamplerDescriptorSlot = pSamplerDescriptorSlot;
}

const nn::gfx::DescriptorSlot*
DrawInfo::GetFramebufferTextureDescriptorSlot() const
{
    return m_pFramebufferTextureDescriptorSlot;
}

const nn::gfx::DescriptorSlot*
DrawInfo::GetFramebufferSamplerDescriptorSlot() const
{
    return m_pFramebufferSamplerDescriptorSlot;
}


} // namespace nn::ui2d
} // namespace nn
