﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_BinaryFormat.h>
#include <nn/gfx.h>

#include <nn/font/font_ResourceFormat.h>
#include <nn/font/font_ResFont.h>
#include <nn/ui2d/ui2d_ArcExtractor.h>
#include <nn/ui2d/detail/ui2d_Log.h>

namespace
{
    //------------------------------------------------------------------------------

    inline uint32_t  calcHash32( const char* pStr, uint32_t  key )
    {
        uint32_t  ret = 0;
        for( int i = 0; pStr[ i ] != '\0'; ++i )
        {
            ret = ret * key + pStr[ i ];
        }
        return ret;
    }

    //------------------------------------------------------------------------------

    template<typename TValue>
    inline
    TValue EndianToHost(nn::ui2d::EndianTypes endian, const TValue& value)
    {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        return (endian == nn::ui2d::EndianTypes_Big)
            ? nn::font::detail::ByteSwap(value)
            : value;
#else
        return (endian == nn::ui2d::EndianTypes_Little)
            ? nn::font::detail::ByteSwap(value)
            : value;
#endif
    }

    //------------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    inline int32_t binarySearch_( uint32_t  hash, const nn::ui2d::ArcExtractor::FATEntry* pEntries, int32_t start, int32_t end, nn::ui2d::EndianTypes endianType )
#else
    inline int32_t binarySearch_( uint32_t  hash, const nn::ui2d::ArcExtractor::FATEntry* pEntries, int32_t start, int32_t end )
#endif
    {
        // 中点を取る.
        int32_t middle;
        for(;;)
        {
            middle = ( start + end ) / 2;
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
            uint32_t  bufHash =EndianToHost( endianType, pEntries[ middle ].hash );
#else
            uint32_t  bufHash = pEntries[ middle ].hash;
#endif
            if ( bufHash == hash )
            {
                return middle;
            }
            else if ( bufHash < hash )
            {
                if ( start == middle )
                {
                    return -1;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    return -1;
                }
                end = middle;
            }
        }
    }

    //------------------------------------------------------------------------------

    /**
     *  エンディアンマーク(Byte Order Mark)からエンディアンを判定します.
     *
     *  @param[in]  mark    エンディアンマーク(BOM). 0xff 0xfeか0xfe 0xff以外を渡した場合、エラー.
     *  @return             判定結果.
     */
    inline nn::ui2d::EndianTypes MarkToEndian(uint16_t  mark)
    {
        uint8_t * pMark8 = reinterpret_cast<uint8_t *>( &mark );
        if ( ( *pMark8 == 0xff ) && ( *(pMark8 + 1 ) == 0xfe ) )
        {
            return nn::ui2d::EndianTypes_Little;
        }
        else if ( ( *pMark8 == 0xfe ) && ( *(pMark8 + 1 ) == 0xff ) )
        {
            return nn::ui2d::EndianTypes_Big;
        }
        else
        {
            NN_SDK_ASSERT(false, "Undefined endian mark(0x%02x 0x%02x).", *((uint8_t *)&mark),  *((uint8_t *)&mark + 1) );
            return nn::ui2d::EndianTypes_Little;
        }
    }

}

namespace nn
{
namespace ui2d
{

#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
#define SHARC_ENDIAN_TO_HOST(val) (EndianToHost(m_EndianType, (val)))
#else
#define SHARC_ENDIAN_TO_HOST(val) (val)
#endif

//------------------------------------------------------------------------------

ArcExtractor::ArcExtractor(const void* pArchive)
: m_pArchiveBlockHeader( NULL )
, m_pFATBlockHeader( NULL )
, m_pFNTBlock( NULL )
, m_pFATEntries( NULL )
, m_FATEntryCount(0)
, m_pDataBlock( NULL )
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
, m_EndianType(EndianTypes_Little)
#else
, m_EndianType(EndianTypes_Big)
#endif
{
    PrepareArchive(pArchive);
}

//------------------------------------------------------------------------------

ArcExtractor::ArcExtractor()
: m_pArchiveBlockHeader( NULL )
, m_pFATBlockHeader( NULL )
, m_pFNTBlock( NULL )
, m_pFATEntries( NULL )
, m_FATEntryCount(0)
, m_pDataBlock( NULL )
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
, m_EndianType(EndianTypes_Little)
#else
, m_EndianType(EndianTypes_Big)
#endif
{
}

//------------------------------------------------------------------------------

ArcExtractor::~ArcExtractor()
{
}

//------------------------------------------------------------------------------

void ArcExtractor::Relocate(const void * pArchive)
{
    NN_UNUSED(pArchive);
}

//------------------------------------------------------------------------------

void ArcExtractor::Unrelocate(const void * pArchive)
{
    ArcExtractor    arc(pArchive);

    int count = arc.GetFileCount();

    for (int i = 0; i < count; ++i)
    {
        void*   pFile = arc.GetFileFast(i);
        nn::util::BinaryFileHeader* pBinaryFileHeader = static_cast< nn::util::BinaryFileHeader* >(pFile);

        // gfx のテクスチャとシェーダー。
        if (pBinaryFileHeader->signature.IsValid(nn::gfx::ResTextureFile::Signature) ||
            pBinaryFileHeader->signature.IsValid(nn::gfx::ResShaderFile::Signature))
        {
            if (pBinaryFileHeader->IsRelocated())
            {
                pBinaryFileHeader->GetRelocationTable()->Unrelocate();
                pBinaryFileHeader->SetRelocated(false);
            }
        }

        // nn::font のフォントリソース。
        if (pBinaryFileHeader->signature.IsValid(nn::font::BinFileSignatureFont))
        {
            nn::font::ResFont::Unrelocate(pFile);
        }
    }
}

//------------------------------------------------------------------------------

bool ArcExtractor::PrepareArchive( const void *pArchive )
{
    if ( !pArchive )
    {
        NN_SDK_ASSERT(false, "archive must not be NULL." );
        return false;
    }
    const uint8_t * pArchive8 = reinterpret_cast<const uint8_t *>( pArchive );
    m_pArchiveBlockHeader = reinterpret_cast<const ArchiveBlockHeader*>( pArchive8 );
    if ( std::strncmp( m_pArchiveBlockHeader->signature, "SARC", 4 ) != 0 )
    {
        NN_SDK_ASSERT(false, "Invalid ArchiveBlockHeader" );
        return false;
    }

    m_EndianType = MarkToEndian( m_pArchiveBlockHeader->byteOrder );

#if !defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    // Win以外であればエンディアンが異なるのは許容できない
    if( m_EndianType != EndianTypes_HostEndian )
    {
        NN_SDK_ASSERT(false, "Invalid Endian Type" );
        return false;
    }
#endif

    if ( SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->version) != ArchiveVersion )
    {
        NN_SDK_ASSERT(false, "unmatching version ( expect: %x, actual: %x )", ArchiveVersion, m_pArchiveBlockHeader->version );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->headerSize) != sizeof( ArchiveBlockHeader ) )
    {
        NN_SDK_ASSERT(false, "Invalid ArchiveBlockHeader" );
        return false;
    }
    m_pFATBlockHeader = reinterpret_cast<const FATBlockHeader*>( pArchive8 + SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->headerSize) );
    if ( std::strncmp( m_pFATBlockHeader->signature, "SFAT", 4 ) != 0 )
    {
        NN_SDK_ASSERT(false, "Invalid FATBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->headerSize) != sizeof( FATBlockHeader ) )
    {
        NN_SDK_ASSERT(false, "Invalid FATBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->fileCount) > ArchiveEntryMax)
    {
        NN_SDK_ASSERT(false, "Invalid FATBlockHeader" );
        return false;
    }

    m_FATEntryCount = SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->fileCount);
    m_pFATEntries = const_cast<FATEntry*>(
        reinterpret_cast<const FATEntry*>(pArchive8 + SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->headerSize) +
        SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->headerSize ) ) );

    const FNTBlockHeader* pFontBlockHeader = reinterpret_cast<const FNTBlockHeader*>( pArchive8 +
                                                      SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->headerSize) +
                                                      SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->headerSize) +
                                                      SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->fileCount ) * sizeof( FATEntry ) );
    if ( std::strncmp( pFontBlockHeader->signature, "SFNT", 4 ) != 0 )
    {
        NN_SDK_ASSERT(false, "Invalid FNTBlockHeader" );
        return false;
    }
    if( SHARC_ENDIAN_TO_HOST(pFontBlockHeader->headerSize) != sizeof(FNTBlockHeader) )
    {
        NN_SDK_ASSERT(false, "Invalid FNTBlockHeader" );
        return false;
    }
    m_pFNTBlock = reinterpret_cast<const char*>( reinterpret_cast<const uint8_t *>( pFontBlockHeader ) + SHARC_ENDIAN_TO_HOST(pFontBlockHeader->headerSize) );
    if( SHARC_ENDIAN_TO_HOST(static_cast<int32_t>(m_pArchiveBlockHeader->dataBlockOffset)) < reinterpret_cast<intptr_t>(m_pFNTBlock) - reinterpret_cast<intptr_t>(m_pArchiveBlockHeader))
    {
        NN_SDK_ASSERT(false, "Invalid data block offset" );
        return false;
    }
    m_pDataBlock = pArchive8 + SHARC_ENDIAN_TO_HOST(m_pArchiveBlockHeader->dataBlockOffset);
    return true;
}

//------------------------------------------------------------------------------

int ArcExtractor::GetFileCount() const
{
    if (m_FATEntryCount < 0)
    {
        return 0;
    }

    return m_FATEntryCount;
}

//------------------------------------------------------------------------------

void* ArcExtractor::GetFileFast( ArcFileInfo* pArcFileInfo, int entryId )
{
    if( entryId < 0 || entryId >= m_FATEntryCount )
    {
        return NULL;
    }
    uint32_t  startOffset = SHARC_ENDIAN_TO_HOST(m_pFATEntries[entryId].dataStartOffset);

    if ( pArcFileInfo )
    {
        uint32_t  endOffset = SHARC_ENDIAN_TO_HOST(m_pFATEntries[entryId].dataEndOffset);
        if( startOffset > endOffset )
        {
            return NULL;
        }
        uint32_t  length = endOffset - startOffset;
        SetArcFileInfo( pArcFileInfo, startOffset, length );
    }

    return nn::util::BytePtr(const_cast<uint8_t*>(m_pDataBlock)).Advance(startOffset).Get();
}

//------------------------------------------------------------------------------

int ArcExtractor::ConvertPathToEntryId( const char* pFilePath ) const
{
    // ハッシュ計算.
    uint32_t  hash = calcHash32( pFilePath, SHARC_ENDIAN_TO_HOST( m_pFATBlockHeader->hashKey ) );

    // テーブルを引く.
    int32_t start = 0;
    int32_t end = m_FATEntryCount;
    // 二分探索.
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    int32_t id = binarySearch_( hash, m_pFATEntries, start, end, m_EndianType );
#else
    int32_t id = binarySearch_( hash, m_pFATEntries, start, end );
#endif
    if ( id == -1 )
    {
        return -1;
    }

    // 名前引き.
    uint32_t  offset = SHARC_ENDIAN_TO_HOST(m_pFATEntries[id].nameOffset);
    if ( offset != 0 )
    {
        id = id - ( ( offset >> 24 ) - 1 );
        while( id < end )
        {
            const FATEntry* entry = &m_pFATEntries[id];
            if ( static_cast<uint32_t>(SHARC_ENDIAN_TO_HOST(entry->hash)) != hash )
            {
                return -1;
            }
            else {
                uint32_t  nameOffset = SHARC_ENDIAN_TO_HOST(entry->nameOffset);

                if( nn::util::ConstBytePtr(m_pFNTBlock).Advance(nameOffset & 0x00ffffff).Get() > m_pDataBlock )
                {
                    NN_DETAIL_UI2D_ERROR("Invalid data start offset" );
                    return -1;
                }
                //if ( pFilePath.isEqual( m_pFNTBlock + ( nameOffset & 0x00ffffff ) * FileNameTableAlign ) )
                if ( std::strcmp(pFilePath, m_pFNTBlock + ( nameOffset & 0x00ffffff ) * FileNameTableAlign ) == 0 )
                {
                    return id;
                }
            }
            ++id;
        }
    }

    return id;
}

//------------------------------------------------------------------------------

int  ArcExtractor::ReadEntry( int * pHandle, ArcEntry pEntries[], int  num ) const
{
    // sharcでは、HandleBufferにuint32_t の数値を入れて、現在位置を記憶します.
    int32_t  count = 0;

    while( *pHandle + count < SHARC_ENDIAN_TO_HOST(m_pFATBlockHeader->fileCount) && count < num )
    {
        int32_t  id = *pHandle + count;
        // オーバーフローを監視するアサート
        NN_SDK_ASSERT( id >= (*pHandle) );
        uint32_t  offset = SHARC_ENDIAN_TO_HOST(m_pFATEntries[id].nameOffset);
        if ( offset == 0 )
        {
            // 名前は記録されていないのでハッシュ値を出力.
            nn::util::SNPrintf(
                pEntries[ count ].name,
                ArcEntry::NameLengthMax,
                "%08x",
                SHARC_ENDIAN_TO_HOST(m_pFATEntries[id].hash));
        }
        else
        {
            // 記録されている名前を出力.
            if( nn::util::ConstBytePtr(m_pFNTBlock).Advance(offset & 0x00ffffff).Get() > m_pDataBlock )
            {
                NN_DETAIL_UI2D_ERROR("Invalid data start offset" );
                pEntries[ count ].name[0] = '\0';
            }
            else
            {
                nn::util::Strlcpy( pEntries[ count ].name, m_pFNTBlock + ( offset & 0x00ffffff ) * FileNameTableAlign, ArcEntry::NameLengthMax );
            }
        }
        count++;
    }
    *pHandle += count;
    return count;
}

} // namespace ui2d
} // namespace nn

