﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#include <nn/spy/detail/fnd/os/spyfnd_Thread.h>

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT(const int Thread::OsThreadAlignment);

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(int intArg, void* ptrArg) NN_NOEXCEPT
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NN_SDK_ASSERT_NOT_NULL(owner);
        NN_SDK_ASSERT_NOT_NULL(owner->m_Handler);

        owner->OnRun();

        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
    }
};

//---------------------------------------------------------------------------
Thread::Thread() NN_NOEXCEPT
    : m_State(State_NotRun)
    , m_Priority(DefaultThreadPriority)
    , m_Name("")
    , m_Handler(NULL)
{
    m_Handle = RoundUpTo<Handle>(m_HandleBuffer, OsThreadAlignment);
}

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const NN_NOEXCEPT
{
    if(stack == NULL)
    {
        return false;
    }

    if(stackSize == 0)
    {
        return false;
    }

    if(priority > MaxThreadPriority)
    {
        return false;
    }

    if(handler == NULL)
    {
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(int value) NN_NOEXCEPT
{
    OSSetThreadPriority(m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan) NN_NOEXCEPT
{
    OSSleepTicks(timeSpan.GetTick());
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, const RunArgs& args) NN_NOEXCEPT
{
    if(!OSCreateThread(
        handle,
        ThreadMain::Run,
        1,
        this,
        AddOffsetToPtr(args.stack, args.stackSize),
        args.stackSize,
        args.priority,
        0))
    {
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach() NN_NOEXCEPT
{
    if(IsTerminated())
    {
        return;
    }

    OSDetachThread(m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name) NN_NOEXCEPT
{
    OSSetThreadName(m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AffinityMask value) NN_NOEXCEPT
{
    // OS_THREAD_ATTR_AFFINITY_CORE0 ～ OS_THREAD_ATTR_AFFINITY_CORE2 と同じ値
    OSSetThreadAffinity(m_Handle, static_cast<uint16_t>(value));
}

//---------------------------------------------------------------------------
void
Thread::Resume() NN_NOEXCEPT
{
    OSResumeThread(m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::Join() NN_NOEXCEPT
{
    // TODO : ★Cafe版で結果を返せるか検討
    OSJoinThread(m_Handle, NULL);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const NN_NOEXCEPT
{
    return OSIsThreadTerminated(m_Handle);
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn
