﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <climits>

#include <nn/spy/detail/fnd/hio/spyfnd_HioChannel.h>

namespace nn {
namespace spy {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT(const int HioChannel::MaxPortLength);

namespace {

    const int InvalidSocket = -1;

}

//----------------------------------------------------------
HioChannel::HioChannel() NN_NOEXCEPT
    : m_Socket(InvalidSocket)
    , m_ClientSocket(InvalidSocket)
{
    m_Port[0] = '\0';
}

//----------------------------------------------------------
bool
HioChannel::Open(const char* port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(port);
    NN_SDK_REQUIRES_LESS(std::strlen(port), static_cast<size_t>(MaxPortLength));

    int result;

    m_ClientSocket = InvalidSocket;

    m_Socket = nn::ige::HtcsHelperSocket();
    if (m_Socket == InvalidSocket)
    {
        return false;
    }

    String::Copy(m_Port, port);

    // ポートにバインド
    {
        nn::htcs::SockAddrHtcs addr;
        std::memset(&addr, 0, sizeof(addr));
        addr.family = nn::htcs::HTCS_AF_HTCS;
        addr.peerName = nn::htcs::GetPeerNameAny();
        String::Copy(addr.portName.name, port);

        result = nn::ige::HtcsHelperBind(m_Socket, &addr);
        NN_SDK_ASSERT(result != -1);

        if (result == -1)
        {
            Close();
            return false;
        }
    }

    // リッスン
    {
        const int BacklogCount = 1;
        result = nn::ige::HtcsHelperListen(m_Socket, BacklogCount);
        NN_SDK_ASSERT(result != -1);

        if (result == -1)
        {
            Close();
            return false;
        }
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Close() NN_NOEXCEPT
{
    if(m_ClientSocket != InvalidSocket)
    {
        nn::ige::HtcsHelperClose(m_ClientSocket);
        m_ClientSocket = InvalidSocket;
    }

    if(m_Socket != InvalidSocket)
    {
        nn::ige::HtcsHelperClose(m_Socket);
        m_Socket = InvalidSocket;
    }

    m_Port[0] = '\0';
}

//----------------------------------------------------------
bool
HioChannel::Connect() NN_NOEXCEPT
{
    if(!IsOpened())
    {
        NN_SDK_ASSERT(false, "HioChannel is not opened.");
        return false;
    }

    if(IsConnected())
    {
        return true;
    }

    nn::htcs::SockAddrHtcs client;
    m_ClientSocket = nn::ige::HtcsHelperAccept(m_Socket, &client);

    if(m_ClientSocket == InvalidSocket)
    {
        Disconnect();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Disconnect() NN_NOEXCEPT
{
    if(m_ClientSocket != InvalidSocket)
    {
        nn::ige::HtcsHelperClose(m_ClientSocket);
        m_ClientSocket = InvalidSocket;
    }
}

//----------------------------------------------------------
bool
HioChannel::IsOpened() const NN_NOEXCEPT
{
    return m_Port[0] != '\0';
}

//----------------------------------------------------------
bool
HioChannel::IsConnected() const NN_NOEXCEPT
{
    return IsOpened() && m_ClientSocket != InvalidSocket;
}

//----------------------------------------------------------
const char*
HioChannel::GetPort() const NN_NOEXCEPT
{
    return m_Port;
}

//----------------------------------------------------------
size_t
HioChannel::Read(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_LESS(length, static_cast<size_t>(std::numeric_limits<int>::max()));

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    char* current = reinterpret_cast<char*>(buffer);
    size_t readLength = 0;

    while(readLength < length)
    {
        auto recvLength = nn::ige::HtcsHelperRecv(m_ClientSocket, current, static_cast<int>(length - readLength), 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(recvLength == 0)
        {
            Disconnect();
            break;
        }

        if(recvLength < 0)
        {
            auto errCode = nn::ige::HtcsHelperGetLastError();
            if(errCode == nn::htcs::HTCS_EAGAIN)
            {
                continue;
            }

            Disconnect();
            break;
        }

        current += recvLength;
        readLength += recvLength;
    }

    return readLength;
}

//----------------------------------------------------------
size_t
HioChannel::Write(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_LESS(length, static_cast<size_t>(std::numeric_limits<int>::max()));

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    const char* current = reinterpret_cast<const char*>(buffer);
    size_t writtenLength = 0;

    while(writtenLength < length)
    {
        auto sendLength = nn::ige::HtcsHelperSend(m_ClientSocket, current, static_cast<int>(length - writtenLength), 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(sendLength == 0)
        {
            Disconnect();
            break;
        }

        if(sendLength < 0)
        {
            auto errCode = nn::ige::HtcsHelperGetLastError();
            if(errCode == nn::htcs::HTCS_EAGAIN)
            {
                continue;
            }

            Disconnect();
            break;
        }

        current += sendLength;
        writtenLength += sendLength;
    }

    return writtenLength;
}

} // namespace nn::spy::detail::fnd
} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
