﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ige/ige_BufferWriter.h>
#include <nn/ige/ige_ValueNodeTraits.h>
#include "detail/util/ige_BufferReader.h"
#include <nn/util/util_StringUtil.h>

namespace nn { namespace ige {

namespace {

template<typename T>
void SerializeValue(BufferWriter* pWriter, T* pValue, int count) NN_NOEXCEPT
{
    if (T* ptr = pValue)
    {
        for (int i = 0; i < count; ++i, ++ptr)
        {
            pWriter->Write(*ptr);
        }
    }
}

template<typename T>
void DeserializeValue(const BufferView& data, T* pValue, int count) NN_NOEXCEPT
{
    if (T* ptr = pValue)
    {
        detail::BufferReader reader(data);
        for (int i = 0; i < count; ++i, ++ptr)
        {
            if (reader.GetRemainingSize() < sizeof(T))
            {
                return;
            }
            reader.Read(ptr);
        }
    }
}

} // anonymous namespace

template<> util::string_view ValueNodeTraits<bool>    ::GetTypeName() NN_NOEXCEPT { return util::string_view("bool",     4); }
template<> util::string_view ValueNodeTraits<int8_t>  ::GetTypeName() NN_NOEXCEPT { return util::string_view("int8_t",   6); }
template<> util::string_view ValueNodeTraits<uint8_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("uint8_t",  7); }
template<> util::string_view ValueNodeTraits<int16_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int16_t",  7); }
template<> util::string_view ValueNodeTraits<uint16_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint16_t", 8); }
template<> util::string_view ValueNodeTraits<int32_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int32_t",  7); }
template<> util::string_view ValueNodeTraits<uint32_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint32_t", 8); }
template<> util::string_view ValueNodeTraits<int64_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int64_t",  7); }
template<> util::string_view ValueNodeTraits<uint64_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint64_t", 8); }
template<> util::string_view ValueNodeTraits<float>   ::GetTypeName() NN_NOEXCEPT { return util::string_view("float",    5); }
template<> util::string_view ValueNodeTraits<double>  ::GetTypeName() NN_NOEXCEPT { return util::string_view("double",   6); }

template<> void ValueNodeTraits<bool>    ::Serialize(BufferWriter* pWriter, bool*     pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<int8_t>  ::Serialize(BufferWriter* pWriter, int8_t*   pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<uint8_t> ::Serialize(BufferWriter* pWriter, uint8_t*  pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<int16_t> ::Serialize(BufferWriter* pWriter, int16_t*  pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<uint16_t>::Serialize(BufferWriter* pWriter, uint16_t* pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<int32_t> ::Serialize(BufferWriter* pWriter, int32_t*  pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<uint32_t>::Serialize(BufferWriter* pWriter, uint32_t* pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<int64_t> ::Serialize(BufferWriter* pWriter, int64_t*  pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<uint64_t>::Serialize(BufferWriter* pWriter, uint64_t* pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<float>   ::Serialize(BufferWriter* pWriter, float*    pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }
template<> void ValueNodeTraits<double>  ::Serialize(BufferWriter* pWriter, double*   pValue, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, 1); }

template<> void ValueNodeTraits<bool>    ::Deserialize(const BufferView& buffer, bool*     pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<int8_t>  ::Deserialize(const BufferView& buffer, int8_t*   pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<uint8_t> ::Deserialize(const BufferView& buffer, uint8_t*  pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<int16_t> ::Deserialize(const BufferView& buffer, int16_t*  pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<uint16_t>::Deserialize(const BufferView& buffer, uint16_t* pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<int32_t> ::Deserialize(const BufferView& buffer, int32_t*  pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<uint32_t>::Deserialize(const BufferView& buffer, uint32_t* pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<int64_t> ::Deserialize(const BufferView& buffer, int64_t*  pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<uint64_t>::Deserialize(const BufferView& buffer, uint64_t* pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<float>   ::Deserialize(const BufferView& buffer, float*    pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }
template<> void ValueNodeTraits<double>  ::Deserialize(const BufferView& buffer, double*   pValue, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, 1); }

template<> util::string_view ValueArrayNodeTraits<int8_t>  ::GetTypeName() NN_NOEXCEPT { return util::string_view("int8_t[]",   8); }
template<> util::string_view ValueArrayNodeTraits<uint8_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("uint8_t[]",  9); }
template<> util::string_view ValueArrayNodeTraits<int16_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int16_t[]",  9); }
template<> util::string_view ValueArrayNodeTraits<uint16_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint16_t[]", 10); }
template<> util::string_view ValueArrayNodeTraits<int32_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int32_t[]",  9); }
template<> util::string_view ValueArrayNodeTraits<uint32_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint32_t[]", 10); }
template<> util::string_view ValueArrayNodeTraits<int64_t> ::GetTypeName() NN_NOEXCEPT { return util::string_view("int64_t[]",  9); }
template<> util::string_view ValueArrayNodeTraits<uint64_t>::GetTypeName() NN_NOEXCEPT { return util::string_view("uint64_t[]", 10); }
template<> util::string_view ValueArrayNodeTraits<float>   ::GetTypeName() NN_NOEXCEPT { return util::string_view("float[]",    7); }
template<> util::string_view ValueArrayNodeTraits<double>  ::GetTypeName() NN_NOEXCEPT { return util::string_view("double[]",   8); }

template<> void ValueArrayNodeTraits<int8_t>  ::Serialize(BufferWriter* pWriter, int8_t*   pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<uint8_t> ::Serialize(BufferWriter* pWriter, uint8_t*  pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<int16_t> ::Serialize(BufferWriter* pWriter, int16_t*  pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<uint16_t>::Serialize(BufferWriter* pWriter, uint16_t* pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<int32_t> ::Serialize(BufferWriter* pWriter, int32_t*  pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<uint32_t>::Serialize(BufferWriter* pWriter, uint32_t* pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<int64_t> ::Serialize(BufferWriter* pWriter, int64_t*  pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<uint64_t>::Serialize(BufferWriter* pWriter, uint64_t* pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<float>   ::Serialize(BufferWriter* pWriter, float*    pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }
template<> void ValueArrayNodeTraits<double>  ::Serialize(BufferWriter* pWriter, double*   pValue, int count, const TNode&) NN_NOEXCEPT { SerializeValue(pWriter, pValue, count); }

template<> void ValueArrayNodeTraits<int8_t>  ::Deserialize(const BufferView& buffer, int8_t*   pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<uint8_t> ::Deserialize(const BufferView& buffer, uint8_t*  pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<int16_t> ::Deserialize(const BufferView& buffer, int16_t*  pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<uint16_t>::Deserialize(const BufferView& buffer, uint16_t* pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<int32_t> ::Deserialize(const BufferView& buffer, int32_t*  pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<uint32_t>::Deserialize(const BufferView& buffer, uint32_t* pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<int64_t> ::Deserialize(const BufferView& buffer, int64_t*  pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<uint64_t>::Deserialize(const BufferView& buffer, uint64_t* pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<float>   ::Deserialize(const BufferView& buffer, float*    pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }
template<> void ValueArrayNodeTraits<double>  ::Deserialize(const BufferView& buffer, double*   pValue, int count, const TNode&) NN_NOEXCEPT { DeserializeValue(buffer, pValue, count); }

util::string_view StringNodeTraits::GetTypeName() NN_NOEXCEPT { return util::string_view("string", 6);}

void StringNodeTraits::Serialize(BufferWriter* pWriter, char* pValue, int count, const TNode&) NN_NOEXCEPT
{
    if (pValue)
    {
        int length = util::Strnlen(pValue, count);
        pWriter->Write(util::string_view(pValue, length));
    }
}

void StringNodeTraits::Deserialize(const BufferView& buffer, char* pValue, int count, const TNode&) NN_NOEXCEPT
{
    if (const char* pSrc = static_cast<const char*>(buffer.GetPtr()))
    {
        if (pValue)
        {
            util::Strlcpy(pValue, pSrc, count);
        }
    }
}

}} // namespace nn::ige
