﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ige/ige_EditService.h>
#include <nn/ige/ige_NodeHolder.h>
#include "detail/ige_Allocator.h"
#include "detail/messages/ige_UserCommandMessage.h"
#include "detail/messages/ige_NodeMessage.h"

namespace nn { namespace ige {

NodeHolder::NodeHolder() NN_NOEXCEPT
    : m_pService(NULL)
    , m_pChild(NULL)
    , m_Id(detail::s_InvalidId)
    , m_IsEnabledAutoNotify(true)
{
}

NodeHolder::NodeHolder(EditService* pService) NN_NOEXCEPT
    : m_pService(pService)
    , m_pChild(NULL)
    , m_Id(detail::s_RootId)
    , m_IsEnabledAutoNotify(true)
{
}

NodeHolder::~NodeHolder() NN_NOEXCEPT
{
    m_pService = NULL;
    m_pChild = NULL;
    m_Id = detail::s_InvalidId;
    m_IsEnabledAutoNotify = true;
    UnregisterAll();
}

void NodeHolder::Register(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pService != NULL);
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NN_SDK_REQUIRES(!pNode->IsRegisterd());

    pNode->m_pParent = this;
    pNode->m_pService = m_pService;

    if (m_pChild)
    {
        pNode->m_pNext = m_pChild;
    }
    m_pChild = pNode;

    m_pService->RegisterToService(pNode);

    if (m_IsEnabledAutoNotify)
    {
        pNode->PushInfo();
    }
}

void NodeHolder::Unregister(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pService != NULL);
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    for (Node* pPrev = NULL, * pCurrent = m_pChild; pCurrent != NULL; pPrev = pCurrent, pCurrent = pCurrent->m_pNext)
    {
        if (pCurrent != pNode)
        {
            continue;
        }

        pNode->UnregisterAll();
        m_pService->UnregisterFromService(pNode);

        if (pNode == m_pChild)
        {
            m_pChild = pNode->m_pNext;
        }
        else if (pPrev)
        {
            pPrev->m_pNext = pNode->m_pNext;
        }

        pNode->m_pNext = NULL;
        pNode->m_pParent = NULL;
        pNode->m_pService = NULL;

        return;
    }
}

void NodeHolder::UnregisterAll() NN_NOEXCEPT
{
    while (m_pChild)
    {
        Unregister(m_pChild);
    }
}

void NodeHolder::PushChildInfo() NN_NOEXCEPT
{
    for (Node* pCurrent = m_pChild; pCurrent != NULL; pCurrent = pCurrent->m_pNext)
    {
        pCurrent->PushInfo();
    }
}

}} // namespace nn::ige
