﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_StringView.h>

namespace nn { namespace ige { namespace detail {

class ServerSocket
{
    NN_DISALLOW_COPY(ServerSocket);
public:
    struct SetupArg
    {
        util::string_view channelName; // チャンネル名を指定します。最大31文字まで指定が可能です。
    };

public:
    virtual ~ServerSocket() NN_NOEXCEPT
    {
    }

    bool Initialize(const SetupArg& arg) NN_NOEXCEPT
    {
        ResetReadFlag();
        ResetWriteFlag();
        return InitializeInternal(arg);
    }

    void RequestOpen() NN_NOEXCEPT
    {
        m_IsOpenRequested = true;
    }

    bool Open() NN_NOEXCEPT
    {
        m_IsOpened = OpenInternal();
        return m_IsOpened;
    }

    bool IsOpened() const NN_NOEXCEPT
    {
        return m_IsOpened;
    }

    bool IsOpenRequested() const NN_NOEXCEPT
    {
        return m_IsOpenRequested;
    }

    void Close() NN_NOEXCEPT
    {
        CloseInternal();
        m_IsOpenRequested = false;
        m_IsOpened = false;
    }

    bool WriteAsync(const void* buf, size_t size) NN_NOEXCEPT;
    bool ReadAsync(void* buf, size_t size) NN_NOEXCEPT;
    bool Write(const void* buf, size_t size) NN_NOEXCEPT;
    bool Read(void* buf, size_t size) NN_NOEXCEPT;

    bool IsWriting() const NN_NOEXCEPT
    {
        return m_WriteSizeRemained != 0;
    }

    bool IsReading() const NN_NOEXCEPT
    {
        return m_ReadSizeRemained != 0;
    }

    void ResetReadFlag() NN_NOEXCEPT
    {
        m_ReadSizeRemained = 0;
        m_ReadBufferSize = 0;
        m_pReadBuffer = NULL;
    }

    void ResetWriteFlag() NN_NOEXCEPT
    {
        m_WriteBufferSize = 0;
        m_WriteSizeRemained = 0;
        m_pWriteBuffer = NULL;
    }

    void Poll() NN_NOEXCEPT
    {
        if (!IsOpened() && m_IsOpenRequested)
        {
            Open();
        }

        PollInternal();
    }

    void SetWriteLogEnabled() NN_NOEXCEPT
    {
        m_IsWriteLogEnabled = true;
    }

    void SetWriteLogDisabled() NN_NOEXCEPT
    {
        m_IsWriteLogEnabled = false;
    }

    virtual bool IsConnected() const NN_NOEXCEPT = 0;

protected:
    ServerSocket() NN_NOEXCEPT;

    virtual bool InitializeInternal(const SetupArg& arg) = 0;

    virtual bool OpenInternal() = 0;
    virtual void CloseInternal() = 0;

    virtual bool WriteAsyncInternal(const void* buf, size_t size) = 0;
    virtual bool ReadAsyncInternal(void* buf, size_t size) = 0;

    virtual bool ReadAsyncForPoll() = 0;
    virtual bool WriteAsyncForPoll() = 0;

    virtual void PollInternal() = 0;

    const uint8_t* m_pWriteBuffer; // 送信データバッファ
    uint8_t* m_pReadBuffer;  // 受信データバッファ

    volatile size_t m_WriteBufferSize;      // 送信データのバッファサイズ == 送信データの総サイズ
    volatile size_t m_WriteSizeRemained;    // 未送信データのサイズ

    volatile size_t m_ReadBufferSize;       // 受信データのバッファサイズ == 受信データの総サイズ
    volatile size_t m_ReadSizeRemained;     // 未受信データのサイズ

    volatile bool m_IsOpenRequested;
    volatile bool m_IsOpened;
    volatile bool m_IsWriteLogEnabled;
};

}}} // namespace nn::ige::detail
