﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ige_NodeIdManager.h"
#include <nn/ige/ige_Node.h>

namespace nn { namespace ige {

// 宣言は ige_NodeIdType.h 内
NN_DEFINE_STATIC_CONSTANT(const NodeIdType s_InvalidId());
NN_DEFINE_STATIC_CONSTANT(const NodeIdType s_RootId());

namespace detail {

NodeIdManager::NodeIdManager(IgeAllocator* pAllocator) NN_NOEXCEPT
    : m_Impl(*pAllocator)
    , m_LastGeneratedId(s_RootId)
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
}

NodeIdType NodeIdManager::GenerateId() NN_NOEXCEPT
{
    // IDサーチの負荷軽減と m_Impl 内のソート保証のため
    // ノードIDを決め打ちでインクリメントする
    // アサートに失敗する事態になれば他のノード管理方法を考えること

    ++m_LastGeneratedId;
    NN_SDK_ASSERT(m_LastGeneratedId != s_InvalidId, "ige NodeId limits");
    return m_LastGeneratedId;
}

NodeIdType NodeIdManager::Register(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NodeIdType id = GenerateId();
    m_Impl.push_back(pNode);
    return id;
}

bool NodeIdManager::Unregister(NodeIdType id) NN_NOEXCEPT
{
    int found = FindImpl(id);
    if (found < 0) return false;

    m_Impl.erase(m_Impl.begin() + found);
    return true;
}

bool NodeIdManager::IsRegistered(NodeIdType id) const NN_NOEXCEPT
{
    return Find(id) != NULL;
}

void NodeIdManager::Clear() NN_NOEXCEPT
{
    m_Impl.clear();
    m_LastGeneratedId = s_RootId;
}

Node* NodeIdManager::Find(NodeIdType id) NN_NOEXCEPT
{
    int found = FindImpl(id);
    return found < 0 ? NULL : m_Impl[found];
}

const Node* NodeIdManager::Find(NodeIdType id) const NN_NOEXCEPT
{
    int found = FindImpl(id);
    return found < 0 ? NULL : m_Impl[found];
}

int NodeIdManager::FindImpl(NodeIdType id) const NN_NOEXCEPT
{
    int size = static_cast<int>(m_Impl.size());
    int lo = 0;
    int hi = size - 1;

    while (lo <= hi)
    {
        int i = lo + ((hi - lo) >> 1);
        int order = m_Impl[i]->GetId() - id;

        if (order == 0) return i;

        if (order < 0) lo = i + 1;
        else           hi = i - 1;
    }

    return ~lo;
}

}}} // namespace nn::ige::detail
