﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../g3d_ViewerDetailDefine.h"
#include "../g3d_Allocator.h"

namespace nn { namespace g3d { namespace viewer {

namespace detail {

template<typename T>
class DynamicRingBuffer
{
    NN_DISALLOW_COPY(DynamicRingBuffer<T>);

public:
    typedef int(*CompareCallback)(const void* pValue, const void* pKey);

    explicit DynamicRingBuffer(nn::g3d::viewer::detail::IAllocator* pAllocator) NN_NOEXCEPT
        : m_pAllocator(pAllocator)
        , m_BufferCount(0)
        , m_BufferCountMax(DefaultArraySize)
        , m_pBuffer(nullptr)
        , m_Start(0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pAllocator);

        m_pBuffer = static_cast<T*>(m_pAllocator->Allocate(sizeof(T) * m_BufferCountMax, Alignment, AllocateType_DynamicBuffer));
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pBuffer);
    }

    ~DynamicRingBuffer() NN_NOEXCEPT
    {
        Destroy();
    }



    int GetCount() const NN_NOEXCEPT
    {
        return m_BufferCount;
    }

    int GetCountMax() const NN_NOEXCEPT
    {
        return m_BufferCountMax;
    }

    void Clear() NN_NOEXCEPT
    {
        m_BufferCount = 0;
    }

    T* UnsafeGet(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(index, m_BufferCount);
        return &m_pBuffer[GetIndex(index)];
    }

    const T* UnsafeGet(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(index, m_BufferCount);
        return &m_pBuffer[GetIndex(index)];
    }

    T& operator[](int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_BufferCount);
        return m_pBuffer[GetIndex(index)];
    }

    const T& operator[](int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_BufferCount);
        return m_pBuffer[GetIndex(index)];
    }

    bool PushBack(const T& value) NN_NOEXCEPT
    {
        if (m_BufferCount < m_BufferCountMax)
        {
            int index = m_BufferCount;
            ++m_BufferCount;
            m_pBuffer[GetIndex(index)] = value;
            return true;
        }

        if (Resize(m_BufferCountMax + 1))
        {
            int index = m_BufferCount;
            ++m_BufferCount;
            m_pBuffer[GetIndex(index)] = value;
            return true;
        }

        return false;
    }

    bool PushFront(const T& value) NN_NOEXCEPT
    {
        if (m_BufferCount < m_BufferCountMax)
        {
            int index = -1;
            ++m_BufferCount;
            m_Start = GetIndex(index);
            m_pBuffer[GetIndex(0)] = value;
            return true;
        }

        if (Resize(m_BufferCountMax + 1))
        {
            int index = -1;
            ++m_BufferCount;
            m_Start = GetIndex(index);
            m_pBuffer[GetIndex(0)] = value;
            return true;
        }

        return false;
    }

    T PopFront() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_BufferCount, 0);
        T value = *UnsafeGet(0);
        EraseAt(0);
        return value;
    }

    T PopBack() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_BufferCount, 0);
        int index = m_BufferCount - 1;
        T value = *UnsafeGet(index);
        EraseAt(index);
        return value;
    }

    void EraseAt(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_BufferCount, 0);
        NN_SDK_REQUIRES_RANGE(index, 0, m_BufferCount);

        --m_BufferCount;

        if (index == 0)
        {
            m_Start = GetIndex(1);
            return;
        }

        // 要素を削除するので、移動数は元の要素数 - 1
        for (int idx = index; idx < m_BufferCount; ++idx)
        {
            int destIndex = GetIndex(idx);
            int srcIndex = GetIndex(idx + 1);
            m_pBuffer[destIndex] = m_pBuffer[srcIndex];
        }
    }

    bool Resize(int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);

        if (count <= m_BufferCountMax)
        {
            return true;
        }

        int bufferCount = m_BufferCountMax << 1;

        T* pNewBuffer = static_cast<T*>(m_pAllocator->Allocate(sizeof(T) * bufferCount, Alignment, AllocateType_DynamicBuffer));
        if (!pNewBuffer)
        {
            return false;
        }

        // 値を前から積み、スタートはリセットする
        for (int index = 0; index < m_BufferCount; ++index)
        {
            pNewBuffer[index] = m_pBuffer[GetIndex(index)];
        }
        m_Start = 0;
        m_BufferCountMax = bufferCount;

        m_pAllocator->Free(m_pBuffer);
        m_pBuffer = pNewBuffer;
        return true;
    }

    const T* Find(const T* ptr, CompareCallback compareCallback) const NN_NOEXCEPT
    {
        return static_cast<const T*>(FindImpl(ptr, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

    T* Find(const T* ptr, CompareCallback compareCallback) NN_NOEXCEPT
    {
        return static_cast<const T*>(FindImpl(ptr, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

    template<typename TKey>
    const T* Find(const TKey* key, int(*compareCallback)(const T* a, const TKey* b)) const NN_NOEXCEPT
    {
        return static_cast<T*>(FindImpl(key, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

    template<typename TKey>
    T* Find(const TKey* key, int(*compareCallback)(const T* a, const TKey* b)) NN_NOEXCEPT
    {
        return static_cast<T*>(FindImpl(key, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

private:

    int GetIndex(int index) const NN_NOEXCEPT
    {
        int idx = m_Start + index;
        if (idx < 0)
        {
            idx = m_BufferCountMax + (idx % m_BufferCountMax);
            return idx;
        }

        if (idx >= m_BufferCountMax)
        {
            idx %= m_BufferCountMax;
        }

        return idx;
    }

    void Destroy() NN_NOEXCEPT
    {
        m_pAllocator->Free(m_pBuffer);
        m_pBuffer = nullptr;
        m_BufferCount = 0;
        m_pAllocator = nullptr;
    }

    typedef int(*CompareCallbackImpl)(const void* a, const void* b);
    void* FindImpl(const void* ptr, CompareCallbackImpl compareCallback) const NN_NOEXCEPT
    {
        for (int index = 0; index < m_BufferCount; ++index)
        {
            void* target = UnsafeGet(index);
            if (compareCallback(target, ptr) == 0)
            {
                return target;
            }
        }
        return nullptr;
    }

private:
    static const int DefaultArraySize = 8;
    static const size_t Alignment = NN_ALIGNOF(T);

    IAllocator* m_pAllocator;
    int         m_BufferCount;
    int         m_BufferCountMax;
    T*          m_pBuffer;

    int         m_Start;
};

} } } } // namespace nn::g3d::viewer::detail
