﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include "../g3d_CallbackCaller.h"
#include "../g3d_IEditTargetObj.h"
#include "g3d_EditMaterialObj.h"
#include "g3d_EditShapeObj.h"
#include "g3d_EditSkeletonObj.h"
#include "../util/g3d_DynamicArray.h"
#include "../util/g3d_DynamicPtrArray.h"
#include "../util/g3d_ViewerUtility.h"
#include "../util/g3d_EditWorkBuffer.h"
#include "../util/g3d_FixedSizeArray.h"
#include "../g3d_DeviceDependentObj.h"
#include "../g3d_ViewerKeyManager.h"

#include <nn/g3d/g3d_ModelObj.h>

#include <nn/g3d/viewer/g3d_ViewerCallback.h>

#include <stdint.h>

namespace nn { namespace g3d {

class ModelObj;
class ResFile;
class ResModel;
class ResShaderArchive;

namespace viewer {

class ViewerServer;

namespace detail {

class EditCommandManager;
class EditAnimObj;
class Allocator;

/**
    @briefprivate 編集対象モデルクラスです。
 */
class EditModelObj : public DeviceDependentObj, IEditTargetObj
{
public:
    explicit EditModelObj(
        nn::gfx::Device* pDevice,
        Allocator* pAllocator,
        ResModel* pOriginalResModel,
        CallbackCaller* pCallbackCaller) NN_NOEXCEPT;
    virtual ~EditModelObj() NN_NOEXCEPT;

    const char* GetName() const NN_NOEXCEPT
    {
        return GetTargetModelObj(0)->GetResource()->GetName();
    }

    /**
        @brief 指定されたModelObj に対してアタッチ処理を行います。
     */
    RuntimeErrorCode AttachModelObjs(
        nn::g3d::ResFile* pAttachModelResFile,
        nn::g3d::ModelObj** pAttachTargetModelObjs, int modelObjCount) NN_NOEXCEPT;

    RuntimeErrorCode ReloadResFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;

    //! @brief テクスチャパターンアニメーションによって書き換えられたテクスチャ参照を元に戻します。
    RuntimeErrorCode ResetTextureRefToOriginal() NN_NOEXCEPT;

    /**
        @brief 初期状態に戻します。
     */
    RuntimeErrorCode ResetToOriginal() NN_NOEXCEPT;

    //! @brief アタッチされている ModelObj を取得します。
    ModelObj* GetTargetModelObj(int index) NN_NOEXCEPT
    {
        NN_G3D_REQUIRES_RANGE(index, 0, m_EditModelDataSets.GetCount(), m_pOriginalResModel->GetName());
        return m_EditModelDataSets[index].pOriginalModelObj;
    }

    //! @brief アタッチされている ModelObj を取得します。
    const ModelObj* GetTargetModelObj(int index) const NN_NOEXCEPT
    {
        NN_G3D_REQUIRES_RANGE(index, 0, m_EditModelDataSets.GetCount(), m_pOriginalResModel->GetName());
        return m_EditModelDataSets[index].pOriginalModelObj;
    }

    EditMaterialObj* GetEditMaterialObj(int modelIndex, int materialIndex) NN_NOEXCEPT
    {
        NN_G3D_REQUIRES_RANGE(modelIndex, 0, m_EditModelDataSets.GetCount(), m_pOriginalResModel->GetName());
        NN_G3D_REQUIRES_RANGE(materialIndex, 0, m_EditModelDataSets[modelIndex].materialArray.GetCount(), m_pOriginalResModel->GetName());
        return m_EditModelDataSets[modelIndex].materialArray.UnsafeAt(materialIndex);
    }

    const EditMaterialObj* GetEditMaterialObj(int modelIndex, int materialIndex) const NN_NOEXCEPT
    {
        NN_G3D_REQUIRES_RANGE(modelIndex, 0, m_EditModelDataSets.GetCount(), m_pOriginalResModel->GetName());
        NN_G3D_REQUIRES_RANGE(materialIndex, 0, m_EditModelDataSets[modelIndex].materialArray.GetCount(), m_pOriginalResModel->GetName());
        return m_EditModelDataSets[modelIndex].materialArray.UnsafeAt(materialIndex);
    }

    //! @brief アタッチされている ModelObj の数を取得します。
    int GetAttachedModelObjCount() const NN_NOEXCEPT
    {
        return m_EditModelDataSets.GetCount();
    }

    ViewerKeyType GetKey() const NN_NOEXCEPT
    {
        if (m_EditModelDataSets.GetCount() == 0)
        {
            return InvalidKey;
        }

        return m_EditModelDataSets[0].modelObjKey;
    }

    //! @brief アタッチされている ModelObj のキーを取得します。
    ViewerKeyType GetModelObjKey(int index) const NN_NOEXCEPT
    {
        NN_G3D_REQUIRES_RANGE(index, 0, m_EditModelDataSets.GetCount(), m_pOriginalResModel->GetName());
        return m_EditModelDataSets[index].modelObjKey;
    }

    //! @brief このクラスにアタッチされたユーザが管理しているオリジナルの ResFile を設定します。
    //! ユーザにオリジナルモデルを解放させるためのコールバックを呼ぶために使います。
    //! 設定する必要があるのはロードモデルのときのみです。
    void SetOriginalResFileManagedByUser(ResFile* pResFile) NN_NOEXCEPT
    {
        m_pOriginalResFileManagedByUser = pResFile;
    }

    //! @brief このクラスにアタッチされたユーザが管理しているオリジナルの ResFile キーを取得します。
    ResFile* GetOriginalResFileManagedByUser() const NN_NOEXCEPT
    {
        return m_pOriginalResFileManagedByUser;
    }

    ViewerKeyType GetResModelKey() const NN_NOEXCEPT
    {
        return ViewerKeyManager::GetInstance().FindKey(m_pOriginalResModel);
    }

    void UpdateRenderInfo(
        int materialIndex,
        const void* pRenderInfoUpdateCommandData, size_t renderInfoUpdateCommandDataSize,
        ptrdiff_t renderInfoArrayOffset) NN_NOEXCEPT;

    void SetLodLevel(int level) NN_NOEXCEPT;
    void ResetLodLevel() NN_NOEXCEPT;

    virtual bool IsResFileBound(ResFile* pResFile) const NN_NOEXCEPT
    {
        return m_pCurrentAttachedResFile == pResFile;
    }

    int GetMaterialCount() const NN_NOEXCEPT
    {
        return m_EditModelDataSets[0].materialArray.GetCount();
    }

    int GetShaderCount() const NN_NOEXCEPT;

    const ResModel* GetResModel() const NN_NOEXCEPT;

    int GetBoundAnimCount() const NN_NOEXCEPT;

    bool AddBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT;

    bool RemoveBoundAnim(int index) NN_NOEXCEPT;
    bool RemoveBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT;

    EditAnimObj* GetBoundAnimAt(int index) NN_NOEXCEPT;

    int GetIndexOfBoundAnim(EditAnimObj* pEditAnimObj) NN_NOEXCEPT;

    //! @brief オリジナルモデルが 3DEditor からロードされたモデルであれば true、
    //! アプリケーションからアタッチされたモデルであれば false を返します。
    bool IsOriginalModelFrom3dEditor() const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pOriginalResModel);
        return m_pOriginalResFileManagedByUser != nullptr;
    }

    //! @brief マテリアル選択時の点滅を開始します。
    void StartMaterialBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT;

    //! @brief マテリアル選択時の点滅を終了します。
    void BreakMaterialBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT;

    //! @brief マテリアルの点滅状態を更新します。
    void UpdateMaterialBlinking() NN_NOEXCEPT;

    //! @brief シェイプ選択時の点滅を開始します。
    void StartShapeBlinking(const TargetSelectedArg& arg) NN_NOEXCEPT;

    //! @brief モデル選択時の点滅を開始します。
    void StartModelBlinking() NN_NOEXCEPT;

    //! @brief シェイプの点滅状態を更新します。
    void UpdateShapeBlinking() NN_NOEXCEPT;

    //! @biref 現在編集対象となっているリソースを取得します。
    ResModel* GetResource() NN_NOEXCEPT
    {
        if (m_pCurrentAttachedResModel)
        {
            return m_pCurrentAttachedResModel;
        }

        return nullptr;
    }

    //! @biref 現在編集対象となっているリソースを取得します。
    const ResModel* GetResource() const NN_NOEXCEPT
    {
        if (m_pCurrentAttachedResModel)
        {
            return m_pCurrentAttachedResModel;
        }

        return nullptr;
    }

    void ExecuteFunctionForeachInstance(void(*func)(ModelObj*, void*), void* pUserData = nullptr) NN_NOEXCEPT
    {
        for (int modelIndex = 0, modelCount = GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            func(GetTargetModelObj(modelIndex), pUserData);
        }
    }

    void PrintBoundAnimInfo() NN_NOEXCEPT;

    // ----------------------------------------------
    // EditServer::ExecuteMultiFileLoadCommand だけでしか使わない
    void SetUpdateResFile(ResFile* pResFile) NN_NOEXCEPT
    {
        m_pUpdateResFile = pResFile;
    }
    RuntimeErrorCode UpdateShaders(bool useShaders) NN_NOEXCEPT;
    void ResizeAndResetMaterialShaderInfoArray(int size) NN_NOEXCEPT;

    // 外部からセットされたシェーダアーカイブの破棄責任は持たない
    void SetMaterialShaderInfo(int index, const ShaderAssignUpdatedArg::MaterialShaderInfo& info) NN_NOEXCEPT;

    void DestroyShaders() NN_NOEXCEPT;
    // ----------------------------------------------

    bool IsResShaderArchiveUsed(const ResShaderArchive* pTargetResShaderArchive) const NN_NOEXCEPT
    {
        for(int i = 0, end = m_MaterialShaderInfoArray.GetCount(); i < end; ++i)
        {
            ShaderAssignUpdatedArg::MaterialShaderInfo const * pInfo = m_MaterialShaderInfoArray.UnsafeAt(i);
            if (pInfo->pResShaderArchive == pTargetResShaderArchive)

            {
                return true;
            }
        }

        return false;
    }

    bool ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT
    {
        for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            EditModelDataSet model = m_EditModelDataSets[modelIndex];
            for (int matIndex = 0, matCount = model.materialArray.GetCount(); matIndex < matCount; ++matIndex)
            {
                EditMaterialObj* pEditMaterialObj = model.materialArray.UnsafeAt(matIndex);
                if (pEditMaterialObj->ExaminesTextureFileUsed(pResTextureFile))
                {
                    return true;
                }
            }
        }

        return false;
    }

    void ReloadTexture(const nn::gfx::ResTextureFile* pOldResTextureFile, const nn::gfx::ResTextureFile* pNewResTextureFile, CallbackCaller& callback) NN_NOEXCEPT;
    void ForceBindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT;

    //! @brief テクスチャバインドコールバックを呼びつつバインドを解除します。
    void UnbindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT;

    //! @brief テクスチャのバインド情報を更新します。
    void UpdateTextureBindings(TextureBindingBlock* pBlock) NN_NOEXCEPT;

    bool ExaminesTextureBindable(ViewerKeyType textureKey) const NN_NOEXCEPT
    {
        return m_BoundTextureKeys.Contains(textureKey);
    }

    void SetMaterialVisible(int modelIndex, int materialIndex, bool visible) NN_NOEXCEPT
    {
        EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
        EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
        pEditMaterialObj->SetMaterialVisible(visible);
    }

    void UpdateMaterialVisible() NN_NOEXCEPT
    {
        for (int modelIndex = 0, modelCount = m_EditModelDataSets.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            EditModelDataSet* pEditData = &m_EditModelDataSets[modelIndex];
            for (int materialIndex = 0, materialCount = pEditData->materialArray.GetCount(); materialIndex < materialCount; ++materialIndex)
            {
                EditMaterialObj* pEditMaterialObj = pEditData->materialArray.UnsafeAt(materialIndex);
                pEditMaterialObj->UpdateMaterialVisible();
            }
        }
    }

private:
    void ResetModelsToOriginal() NN_NOEXCEPT;
    void ResetMaterialsToOriginal() NN_NOEXCEPT;
    void ResetShapesToOriginal() NN_NOEXCEPT;
    void ResetSkeletonsToOriginal() NN_NOEXCEPT;
    void CallShaderAssignUpdatedCallback(
        nn::g3d::ResModel* oldResModel,
        nn::g3d::ResModel* newResModel,
        bool useShaders,
        CallbackCaller* pCallback) NN_NOEXCEPT;
    void CallUpdateMaterialsCallback(
        nn::g3d::ResModel* oldResModel,
        nn::g3d::ResModel* newResModel,
        CallbackCaller* pCallback) NN_NOEXCEPT;
    void CallModelAttachedCallback(
        CallbackCaller* pCallback) NN_NOEXCEPT;
    void CallShapeUpdatedCallback(CallbackCaller* pCallback) NN_NOEXCEPT;

    void DestroyEditModelObj() NN_NOEXCEPT;

    size_t CalculateBlockBufferSize(const nn::g3d::ResModel* pNewResModel) const NN_NOEXCEPT;
    size_t CalculateShapeBlockBufferSize(const nn::g3d::ResModel* pNewResModel) const NN_NOEXCEPT;

    RuntimeErrorCode SwapResModel(
        nn::g3d::ResModel* pOldResModel,
        nn::g3d::ResModel* pNewResModel,
        bool useShaders,
        CallbackCaller* pCallbackCaller) NN_NOEXCEPT;

    //! @brief バインドされているテクスチャパターンアニメーション、シェーダーパラメータアニメーションを再バインドします。
    //!        モデルリソースが更新された際にテクスチャパターンアニメーションやシェーダーパラメータアニメーションの再初期化が必要になるので、
    //!        この関数をコールする必要があります。
    bool RebindTexPatternOrShaderParamAnims() NN_NOEXCEPT;

    //! @brief テクスチャバインドコールバックなしで強制的にバインドを解除します。
    void UnbindAllTexturesWithoutCallback() NN_NOEXCEPT;

    //! @brief 新しいモデルリソースでインスタンスを再構築します。
    void RebuildModelObj(nn::g3d::ResModel* pNewResModel) NN_NOEXCEPT;

    //! @brief モデル内のブロックバッファーを切り替えます。
    //! 切り替えた先にフロックバッファーが存在する場合には、破棄します。
    void SwitchWorkBlockBuffers() NN_NOEXCEPT;

private:
    static const int s_CheckByteSize = 4;
    struct EditModelDataSet
    {
        ViewerKeyType modelObjKey;

        // アタッチ時: ユーザがアタッチした ModelObj
        // ロード時: ユーザが LoadFile コールバックでアタッチした ModelObj
        nn::g3d::ModelObj* pOriginalModelObj;

        EditSkeletonObj* pEditSkeletonObj;
        FixedSizeArray<EditMaterialObj>   materialArray;
        FixedSizeArray<EditShapeObj>      shapeArray;

        // 元のモデルに差し戻すときに元のモデルが破棄されていないかをチェックするため
        uint8_t destroyCheckData[s_CheckByteSize];
    };

    DynamicArray<EditModelDataSet> m_EditModelDataSets;

    // アタッチ時: ユーザがアタッチした ModelObj の ResModel
    // ロード時: ユーザのコールバックで確保された ResModel
    ResModel* m_pOriginalResModel;

    // アタッチ時: nullptr
    // ロード時：ユーザのコールバックで確保された ResFile
    ResFile* m_pOriginalResFileManagedByUser;

    // 現在アタッチされているリソース
    ResFile* m_pCurrentAttachedResFile;
    ResModel* m_pCurrentAttachedResModel;

    FixedSizeArray<ShaderAssignUpdatedArg::MaterialShaderInfo> m_MaterialShaderInfoArray;

    DynamicPtrArray<EditAnimObj>      m_BoundAnimPtrArray;

    bool m_HasBlinkingMaterials;
    bool m_HasBlinkingShapes;
    int m_ShapeLodLevel;

    // UpdateShaders で一時的に使うためのリソース
    ResFile* m_pUpdateResFile;


    // 元のモデルが破棄されてしまったときに元モデル名を表示できるように名前を保持しておく
    char* m_pModelName;

    CallbackCaller* m_pCallbackCaller;

    // モデルにバインドを行うテクスチャのキー配列
    DynamicArray<ViewerKeyType> m_BoundTextureKeys;

    // モデル(シェイプ、スケルトン、マテリアル)更新用のメモリプール管理用
    MemoryPoolManager* m_MemoryPoolManager;

private:
    NN_DISALLOW_COPY( EditModelObj );
};

}}
}}


