﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerServer.h>

#include "g3d_PingMonitor.h"
#include "g3d_CommandMonitor.h"
#include "g3d_EditCommandExecutor.h"
#include "g3d_Allocator.h"
#include "g3d_HostFileDevice.h"
#include "g3d_EditCommandManager.h"
#include "model/g3d_EditRenderInfo.h"
#include "anim/g3d_EditAnimControl.h"
#include "g3d_EditPickup.h"
#include "g3d_CallbackCaller.h"

namespace nn { namespace g3d { namespace viewer {
namespace detail {
class ResourceManager;
}

class ViewerServer::ViewerServerProxy::Impl
{
public:
    Impl(nn::gfx::Device* pDevice,
        detail::Allocator* pAllocator,
        const ViewerCallback callbackFunc,
        void* pCallbackUserData,
        const ViewerTextureBindCallback textureBindCallback,
        void* pTextureBindCallbackUserData) NN_NOEXCEPT;
    ~Impl() NN_NOEXCEPT;

    ViewerResult Initialize(const InitializeArg& arg) NN_NOEXCEPT;

    detail::Allocator& GetAllocator() NN_NOEXCEPT
    {
        return *m_pAllocator;
    }

    bool IsConnected() const NN_NOEXCEPT
    {
        return m_CommandMonitor.IsConnected();
    }

    bool IsOpened() const NN_NOEXCEPT
    {
        return m_PingMonitor.IsConnected();
    }

    bool IsOpenRequested() const NN_NOEXCEPT
    {
        return m_PingMonitor.IsOpenRequested();
    }

    nn::g3d::viewer::detail::PingMonitor& GetPingMonitor() NN_NOEXCEPT
    {
        return m_PingMonitor;
    }

    nn::g3d::viewer::detail::CommandMonitor& GetCommandMonitor() NN_NOEXCEPT
    {
        return m_CommandMonitor;
    }

    nn::g3d::viewer::detail::EditCommandManager& GetEditCommandManager() NN_NOEXCEPT
    {
        return m_EditCommandManager;
    }

    nn::g3d::viewer::detail::ResourceManager& GetResourceManager() NN_NOEXCEPT
    {
        return *m_EditCommandManager.GetResourceManager();
    }

    //! @brief 現在設定対象となっている描画情報定義を取得します。
    nn::g3d::viewer::detail::EditRenderInfo* GetCurrentEditTargetRenderInfoDefinition() NN_NOEXCEPT
    {
        return m_EditCommandExecutor.GetCurrentEditTargetRenderInfoDefinition();
    }

    //! @brief 3DEditor がトランザクションコマンドの実行中かどうかを取得します。
    bool IsFreezing() const NN_NOEXCEPT
    {
        return m_PingMonitor.IsFreezing();
    }

    //! @biref 通信処理をポーリングします。
    void PollDataCommunication() NN_NOEXCEPT;

    //! @brief 編集処理をポーリングします。
    void PollDataEdit() NN_NOEXCEPT;

    void ClearState() NN_NOEXCEPT;

    void* GetFileBuffer() NN_NOEXCEPT
    {
        return m_FileBuffer;
    }

    void** GetFileBufferRef() NN_NOEXCEPT
    {
        return &m_FileBuffer;
    }

    size_t GetFileBufferSize() const NN_NOEXCEPT
    {
        return m_FileBufferSize;
    }

    void ClearFileBuffer() NN_NOEXCEPT
    {
        if (m_FileBuffer != nullptr)
        {
            m_pAllocator->Free(m_FileBuffer);
            m_FileBuffer = nullptr;
            m_FileBufferSize = 0;
        }
    }

    detail::EditPickup& GetEditPickup() NN_NOEXCEPT
    {
        return m_EditPickup;
    }

    //! @brief クローズ時に 3DEditor にクローズ要求(通信処理)を行うかどうかを取得します。
    bool IsCloseRequestEnabledOnClose() const NN_NOEXCEPT
    {
        return m_IsCloseRequestEnabledOnClose;
    }

    //! @brief クローズ時に 3DEditor へのクローズ要求(通信処理)を無効化します。
    void SetCloseRequestDisabledOnClose() NN_NOEXCEPT
    {
        m_IsCloseRequestEnabledOnClose = false;
    }

    ViewerResult Open() NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;
    void CloseForce() NN_NOEXCEPT;

private:
    struct ExecuteModelFileLoadedArg
    {
        ModelFileLoadedArg loadFileArg;
        int key;
    };

    // 複数ファイル処理用状態の列挙型
    enum MultiFileState
    {
        MULTI_FILE_START = 0,   // 処理始め
        MULTI_FILE_INIT,        // 初期化処理
        MULTI_FILE_LOOPING,     // 複数ファイルループ処理中
        MULTI_FILE_LOADING,     // ファイルロード中
        MULTI_FILE_LOADED_ALL,  // 全ての複数ファイルロード終了
        MULTI_FILE_END,         // 処理終了
    };

    static nn::g3d::viewer::detail::CommandMonitor::AnalyzeCommandResult
        AnalyzeCommand(nn::g3d::viewer::detail::CommandMonitor::AnalyzeCommandArg& arg) NN_NOEXCEPT
    {
        return nn::g3d::viewer::ViewerServer::GetInstance().m_Proxy.m_pImpl->AnalyzeCommandImpl(arg);
    }

    static nn::g3d::viewer::detail::CommandMonitor::ProcessCommandResult
        ProcessCommand(nn::g3d::viewer::detail::CommandMonitor::ProcessCommandArg& arg) NN_NOEXCEPT
    {
        return nn::g3d::viewer::ViewerServer::GetInstance().m_Proxy.m_pImpl->ProcessCommandImpl(arg);
    }

    // @brief コマンド解析をして通信スレッド、メインスレッドで排他に扱う必要ない処理はここで実行します。
    nn::g3d::viewer::detail::CommandMonitor::AnalyzeCommandResult
        AnalyzeCommandImpl(nn::g3d::viewer::detail::CommandMonitor::AnalyzeCommandArg& arg) NN_NOEXCEPT;

    // @brief 通信スレッド、メインスレッドで排他で扱う必要がある処理を実行します。
    nn::g3d::viewer::detail::CommandMonitor::ProcessCommandResult
        ProcessCommandImpl(nn::g3d::viewer::detail::CommandMonitor::ProcessCommandArg& arg) NN_NOEXCEPT;

    // ファイルロード周りの処理は諸々EditCommandExecutorに処理を移したいが、
    // 通信処理で使っているファイルバッファ周りの取り回しが面倒なので保留
    bool ExecuteFileLoadCommand(detail::CommandMonitor::ProcessCommandArg& processCommandArg) NN_NOEXCEPT;
    bool ExecuteMultiFileLoadCommand(detail::CommandMonitor::ProcessCommandArg& processCommandArg) NN_NOEXCEPT;
    void SwapShaderArchiveAlignSizeFromFileBuffer(size_t* outAlignSize) const NN_NOEXCEPT;
    bool ExecuteLoadFile(const ExecuteModelFileLoadedArg& arg, const detail::FileDataBlock* block) NN_NOEXCEPT;
    bool ExecuteReloadFile(const detail::FileDataBlock* block) NN_NOEXCEPT;
    bool ExecuteResFileLoadCommand(detail::CommandFlag command, const detail::FileDataBlock* block, void* pFileBuffer, size_t fileBufferSize) NN_NOEXCEPT;
    bool ExecuteShaderLoadCommand(detail::CommandFlag command, const detail::FileDataBlock* block) NN_NOEXCEPT;

    const char* GetLogIndent(detail::CommandFlag command) const NN_NOEXCEPT
    {
        if (command == detail::SYSTEM_BEGIN_FREEZE_COMMAND_FLAG
            || command == detail::SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG
            || command == detail::SYSTEM_END_FREEZE_COMMAND_FLAG)
        {
            return "";
        }

        if (m_PingMonitor.IsFreezing())
        {
            return "    ";
        }
        else
        {
            return "";
        }
    }

private:
    detail::Allocator* m_pAllocator;
    detail::CallbackCaller m_CallbackCaller;
    detail::PingMonitor m_PingMonitor;
    detail::CommandMonitor m_CommandMonitor;
    detail::EditCommandExecutor m_EditCommandExecutor;
    detail::HostFileDeviceBase* m_pFileDevice;
    detail::EditCommandManager m_EditCommandManager;
    detail::EditPickup m_EditPickup;
    detail::EditAnimControl m_EditAnimControl;

    nn::os::Mutex m_Mutex;
    void* m_FileBuffer;
    size_t m_FileBufferSize;
    volatile MultiFileState m_MultiFileState;
    volatile bool m_IsFileLoading;
    volatile bool m_IsCloseRequestEnabledOnClose;
};

}}} // namespace nn::g3d::viewer


