﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_PingMonitor.h"
#include <nn/g3d/detail/g3d_Inlines.h>
#include "util\g3d_ScopedLock.h"

using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;

namespace nn {
namespace g3d {
namespace viewer {
namespace detail {

PingMonitor::PingMonitor(Allocator* pAllocator) NN_NOEXCEPT
    : m_pAllocator(pAllocator)
    , m_pPingSocket(nullptr)
    , m_IsBeginFreezeEnabled(false)
    , m_IsCommSyncEnabled(true)
    , m_IsReadStarted(false)
    , m_IsWriteStarted(false)
    , m_IsWritePingSucceeded(true)
    , m_IsCloseRequested(false)
    , m_IsClosing(false)
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);

    m_SendPingPacket.header.magic = NN_G3D_EDIT_MAGIC;
    m_SendPingPacket.header.verWord = NN_G3D_EDIT_VERSION;
    m_SendPingPacket.header.dataSize = sizeof(PingBlock);
    m_SendPingPacket.header.command = SYSTEM_PING_SEND_COMMAND_FLAG;

    void* buffer = m_pAllocator->Allocate(sizeof(EditSocketGeneric), Alignment_Default, AllocateType_Communication);
    m_pPingSocket = new (buffer) EditSocketGeneric();
}

ViewerResult PingMonitor::Initialize(int codePage) NN_NOEXCEPT
{
    // Pingソケット
    {
        EditSocketBase::SetupArg setupArg;
        setupArg.portNumber = detail::EDIT_PING_PORT_NUMBER;
        setupArg.channelName = detail::EDIT_PING_CHANNEL_NAME;

        ViewerResult result = m_pPingSocket->Initialize(setupArg);
        if (result != ViewerResult_Success)
        {
            return result;
        }
    }

    m_SendPingPacket.block.codePage = codePage;
    // TODO: 本当は uniqueID を 64 bit にした方がいい
    m_SendPingPacket.block.uniqueID = static_cast<uint32_t>(reinterpret_cast<uintptr_t>(this));
    m_SendPingPacket.block.state = PingState_Normal;

    m_SendPingPacket.block.flag = PingFlagMask_None;
#ifdef NN_BUILD_CONFIG_ADDRESS_64
    m_SendPingPacket.block.flag |= PlatformAddressType_Address64;
#else
    m_SendPingPacket.block.flag |= PlatformAddressType_Address32;
#endif

    return ViewerResult_Success;
}

PingMonitor::~PingMonitor() NN_NOEXCEPT
{
    Close();

    if (m_pPingSocket)
    {
        m_pPingSocket->~EditSocketBase();
        m_pAllocator->Free(m_pPingSocket);
    }
}

bool
PingMonitor::RequestOpen() NN_NOEXCEPT
{
    m_pPingSocket->RequestOpen();
    return true;
}

void
PingMonitor::Close() NN_NOEXCEPT
{
    ClearState();
    m_pPingSocket->Close();
    m_IsCloseRequested = false;
    m_IsClosing = false;
}

void PingMonitor::ClearState() NN_NOEXCEPT
{
    m_IsBeginFreezeEnabled = false;
    m_IsCommSyncEnabled = true;

    m_SendPingPacket.block.state = PingState_Normal;
}

void
PingMonitor::Poll() NN_NOEXCEPT
{
    if (!m_IsWritePingSucceeded)
    {
        // 前回の書き込みに失敗している場合は再度書き込みをトライする
        NN_G3D_VIEWER_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
        m_IsWritePingSucceeded = m_pPingSocket->WriteASync(&m_SendPingPacket, sizeof(m_SendPingPacket));
        if (m_IsWritePingSucceeded)
        {
            m_IsWriteStarted = true;
            if (m_IsClosing)
            {
                Close();
            }
        }
    }

    if (!m_pPingSocket->IsWriting())
    {
        m_pPingSocket->ResetWriteFlag();
        m_IsWriteStarted = false;
    }

    m_pPingSocket->Poll();

    if (!m_pPingSocket->IsConnected())
    {
        m_pPingSocket->ResetWriteFlag();
        m_pPingSocket->ResetReadFlag();
        m_IsReadStarted = false;
    }

    if (!m_IsReadStarted && !m_IsWriteStarted)
    {
        m_IsReadStarted = m_pPingSocket->ReadASync(&m_RecvPingPacket, sizeof(m_RecvPingPacket));
    }

    if (m_IsReadStarted && !m_IsWriteStarted)
    {
        AnalyzePingPacket();
    }
}

void
PingMonitor::AnalyzePingPacket() NN_NOEXCEPT
{
    if (m_pPingSocket->IsReading())
    {
        return;
    }

    switch(m_RecvPingPacket.header.command)
    {
    case SYSTEM_PING_RECV_COMMAND_FLAG:
        {
            if (m_IsCloseRequested)
            {
                m_SendPingPacket.block.state = PingState_Closed;
                m_IsClosing = true;
            }
            else if (m_IsCommSyncEnabled)
            {
                if (m_IsBeginFreezeEnabled)
                {
                    if (m_SendPingPacket.block.state & PingState_Lock)
                    {
                        m_SendPingPacket.block.state |= PingState_Keeping;
                        NN_G3D_VIEWER_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
                    }
                }
                else if (m_SendPingPacket.block.state & PingState_Unlock)
                {
                    m_SendPingPacket.block.state |= PingState_Keeping;
                    NN_G3D_VIEWER_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
                }
            }
            else
            {
                m_SendPingPacket.block.state = PingState_Normal;
            }

            m_IsWritePingSucceeded = m_pPingSocket->WriteASync(&m_SendPingPacket, sizeof(m_SendPingPacket));
            if (m_IsWritePingSucceeded)
            {
                m_IsWriteStarted = true;
                if (m_IsClosing)
                {
                    Close();
                }
            }
        }

        break;
    default:
        break;
    }

    m_pPingSocket->ResetReadFlag();
    m_IsReadStarted = false;
}

}}}} // namespace nn::g3d::viewer::detail


