﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_HostFileDevice.h"

#include <nn/fs.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>

#include <cstdio>
#include <string>

#define NN_DETAIL_G3D_VIEWER_EXECUTE_RESULT_FUNC(func)\
    {\
        nn::Result execute_result_funcresult = func; \
        if (execute_result_funcresult.IsFailure()) {\
            NN_G3D_VIEWER_DEBUG_PRINT(#func " failed: ResultInnerValue = 0x%08x\n", result.GetInnerValueForDebug());\
        }\
    }

namespace {

const size_t FileReadDevisionSize = 4 * 1024 * 1024;
const size_t HostPcPathSizeMax = 260;

bool ConvertFSPathToWinPath(char* dst, size_t dstSize, const char* src, size_t srcSize) NN_NOEXCEPT
{
    if (dstSize <= (srcSize + 1) || dstSize < 3) // コンバート元の長さに1文字追加分と3文字以下の場合も失敗
    {
        return false;
    }

    memcpy(dst, src, dstSize);
    const char* exceptFirstSlashPath = src + 1;
    dst[0] = exceptFirstSlashPath[0];
    dst[1] = ':';
    dst[2] = '/';

    return true;
}

} // anonymous namespace

namespace nn { namespace g3d { namespace viewer { namespace detail {

HostFileDeviceGeneric::HostFileDeviceGeneric() NN_NOEXCEPT
    : m_ReadingSize(0)
{
}

HostFileDeviceGeneric::~HostFileDeviceGeneric() NN_NOEXCEPT
{
}

bool
HostFileDeviceGeneric::OpenInternal(const char* fileName, OpenFlag flag) NN_NOEXCEPT
{
    // TODO: Cafe をベースにしていたので 3DEditor から送信されるパスは /c/.. のようになっている
    // nn::fs 基準のパス(WindowsホストPCのパス形式)に修正する
    char winPathBuffer[HostPcPathSizeMax];
    bool convertResult = ConvertFSPathToWinPath(
        winPathBuffer, HostPcPathSizeMax, fileName, strlen(fileName));
    if (!convertResult)
    {
        NN_G3D_VIEWER_INTERNAL_WARNING(
            "Convert to windows path failed: fileName = %s\n", fileName);
        return false;
    }

    nn::fs::OpenMode openMode;
    switch(flag)
    {
    case OpenFlag_ReadOnly:
        {
            openMode = nn::fs::OpenMode_Read;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    nn::Result result;
    NN_DETAIL_G3D_VIEWER_EXECUTE_RESULT_FUNC(
        result = nn::fs::OpenFile(&m_FileHandle, winPathBuffer, openMode));
    if (result.IsFailure())
    {
        NN_G3D_VIEWER_INTERNAL_WARNING(
            "Failed OpenFile %s: Result code 0x%08x\n", fileName, result.GetInnerValueForDebug());
        return false;
    }

    return true;
}

void
HostFileDeviceGeneric::CloseInternal() NN_NOEXCEPT
{
    nn::fs::CloseFile(m_FileHandle);
}



bool
HostFileDeviceGeneric::ReadASyncInternal(void* readBuffer, size_t readBufferSize) NN_NOEXCEPT
{
    nn::Result result;
    uint8_t* bufferPtr = static_cast<uint8_t*>(readBuffer);
    size_t totalReadSize = 0;
    size_t restOfSize;

    // バッファサイズのチェック
    {
        int64_t actualFileSize;
        result = nn::fs::GetFileSize(&actualFileSize, m_FileHandle);
        NN_UNUSED(result);
        NN_G3D_VIEWER_ASSERT(result.IsSuccess());
        NN_UNUSED(actualFileSize);
        NN_G3D_VIEWER_ASSERT_DETAIL(
            static_cast<size_t>(actualFileSize) <= readBufferSize, NN_TEXT_G3DVIEWER("3DEditor から渡されたファイルサイズが不正です"));
        restOfSize = static_cast<size_t>(actualFileSize);
    }

    // TODO: 分割読み込みに対応する
    do
    {
        size_t readSize = restOfSize < FileReadDevisionSize ? restOfSize : FileReadDevisionSize;
        size_t tmpReadSize = 0;
        NN_DETAIL_G3D_VIEWER_EXECUTE_RESULT_FUNC(
            result = nn::fs::ReadFile(&tmpReadSize, m_FileHandle, totalReadSize, bufferPtr, readSize));
        if (result.IsFailure())
        {
            return false;
        }

        totalReadSize += tmpReadSize;
        restOfSize -= tmpReadSize;
        bufferPtr += tmpReadSize;
    } while (restOfSize > 0);

    m_ReadingSize = totalReadSize;

    return true;
}

bool HostFileDeviceGeneric::IsReadingInternal() const NN_NOEXCEPT
{
    // TODO: 分割読み込みに対応する
    return false;
}



ViewerResult
HostFileDeviceGeneric::InitializeInternal() NN_NOEXCEPT
{
    // TODO: 本当はユーザへの MountHostRoot 要件をなくして、ここで任意の名前でホストPCをマウントしたい
    return ViewerResult_Success;
}

}}}} // namespace nn::g3d::edit::detail


