﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditSocket.h"
#include "g3d_ViewerDetailDefine.h"
#include "util/g3d_ScopedLock.h"
#include <nn/ige/ige_HtcsHelper.h>


namespace {
    static const int s_InvalidSocket = -1;
} // anonymous namespace

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditSocketGeneric::EditSocketGeneric() NN_NOEXCEPT
    : m_WriteMutex(false)
    , m_ReadMutex(false)
{
    m_Impl.socket = s_InvalidSocket;
    m_Impl.clientSocket = s_InvalidSocket;
}

ViewerResult EditSocketGeneric::InitializeInternal(const SetupArg& arg) NN_NOEXCEPT
{
    m_Impl.socket = s_InvalidSocket;
    m_Impl.clientSocket = s_InvalidSocket;
    m_Impl.portNumber = arg.portNumber;
    strncpy(m_ChannelName, arg.channelName, CHANNEL_NAME_LENGTH);
    m_ChannelName[CHANNEL_NAME_LENGTH] = '\0';

    return ViewerResult_Success;
}


bool EditSocketGeneric::OpenInternal() NN_NOEXCEPT
{
    if (m_Impl.socket != s_InvalidSocket)
    {
        return false;
    }

    // ソケットの作成
    m_Impl.socket = nn::ige::HtcsHelperSocket();
    if (m_Impl.socket == -1)
    {
        m_Impl.socket = s_InvalidSocket;
        return false;
    }

    // ノンブロッキングモードに設定
    nn::htcs::FcntlFlag  nonBlock = nn::htcs::HTCS_O_NONBLOCK;
    nn::ige::HtcsHelperFcntl(m_Impl.socket, nn::htcs::HTCS_F_SETFL, nonBlock);

    // ソケットの設定
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    strncpy(addr.peerName.name, nn::ige::HtcsHelperGetPeerNameAny().name, nn::htcs::PortNameBufferLength);
    strncpy(addr.portName.name, m_ChannelName, nn::htcs::PortNameBufferLength);

    nn::ige::HtcsHelperBind(m_Impl.socket, &addr);

    // TCPクライアントからの接続要求を待てる状態にする
    int listenResult = nn::ige::HtcsHelperListen(m_Impl.socket, 5);
    if (listenResult == -1)
    {
        NN_G3D_WARNING(false, "Socket open error %d\n", nn::ige::HtcsHelperGetLastError());
        CloseSocket();
        return false;
    }

    return true;
}

void EditSocketGeneric::CloseSocket() NN_NOEXCEPT
{
    if (m_Impl.socket == s_InvalidSocket)
    {
        return;
    }

    nn::ige::HtcsHelperClose(m_Impl.socket);
    m_Impl.socket = s_InvalidSocket;
}

void EditSocketGeneric::CloseClientSocket() NN_NOEXCEPT
{
    if (m_Impl.clientSocket == s_InvalidSocket)
    {
        return;
    }

    nn::ige::HtcsHelperClose(m_Impl.clientSocket);
    m_Impl.clientSocket = s_InvalidSocket;
}

void
EditSocketGeneric::CloseInternal() NN_NOEXCEPT
{
    CloseClientSocket();
    CloseSocket();
}

bool
EditSocketGeneric::WriteASyncInternal(void* buf, size_t size) NN_NOEXCEPT
{
    ScopedLock locker(m_WriteMutex);
    NN_UNUSED(locker);

    if (m_Impl.clientSocket == s_InvalidSocket)
    {
        return false;
    }

    m_IsWriteEnd = false;
    m_pWriteBuffer = static_cast<uint8_t*>(buf);
    m_WriteBufferSize = size;

    m_WriteSize = size;
    m_TotalWritingSize = m_WritingSize = 0;

    int bufferSize = static_cast<int>(m_WriteBufferSize);
    nn::htcs::ssize_t result = nn::ige::HtcsHelperSend(m_Impl.clientSocket, m_pWriteBuffer, bufferSize, 0);
    if (m_IsWriteLogEnabled)
    {
        NN_G3D_VIEWER_DEBUG_PRINT("Send: result = %zd\n", result);
    }

    if (result <= 0)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        bool isGracefulClose = result == 0;
        if (nn::ige::HtcsHelperGetLastError() != nn::htcs::HTCS_EWOULDBLOCK || isGracefulClose )
        {
            ResetWriteFlag();
            CloseClientSocket();
            return false;
        }
        result = 0;
    }

    m_WritingSize += static_cast<size_t>(result);
    m_TotalWritingSize = m_WritingSize;

    return true;
}

bool
EditSocketGeneric::ReadASyncInternal(void* buf, size_t size) NN_NOEXCEPT
{
    ScopedLock locker(m_ReadMutex);
    NN_UNUSED(locker);

    if (m_Impl.clientSocket == s_InvalidSocket)
    {
        return false;
    }

    m_IsReadEnd = false;
    m_pReadBuffer = static_cast<uint8_t*>(buf);
    m_ReadBufferSize = size;

    m_ReadSize = size;
    m_TotalReadingSize = m_ReadingSize = 0;

    nn::htcs::ssize_t result = nn::ige::HtcsHelperRecv(m_Impl.clientSocket, m_pReadBuffer, m_ReadBufferSize, 0);
    if (result <= 0)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        bool isGracefulClose = result == 0;
        if (nn::ige::HtcsHelperGetLastError() != nn::htcs::HTCS_EWOULDBLOCK || isGracefulClose )
        {
            ResetReadFlag();
            CloseClientSocket();
            return false;
        }

        result = 0;
    }

    m_ReadingSize += static_cast<size_t>(result);
    m_TotalReadingSize = m_ReadingSize;

    return true;
}

void
EditSocketGeneric::PollInternal() NN_NOEXCEPT
{
    // ソケットが無効ならば、処理を抜ける
    if (m_Impl.socket == s_InvalidSocket)
    {
        return;
    }

    // クライアントの接続ができている場合は処理を抜ける
    if (m_Impl.clientSocket != s_InvalidSocket)
    {
        ReadASyncForPoll();
        WriteASyncForPoll();
        return;
    }

    // TCPクライアントからの接続要求を受け付ける
    nn::htcs::SockAddrHtcs client;
    int clientSocket = nn::ige::HtcsHelperAccept(m_Impl.socket, &client);
    if (clientSocket != -1)
    {
        m_Impl.clientSocket = clientSocket;
    }
}

bool
EditSocketGeneric::IsConnected() const NN_NOEXCEPT
{
    return m_Impl.clientSocket != s_InvalidSocket;
}

void
EditSocketGeneric::ReadASyncForPoll() NN_NOEXCEPT
{
    ScopedLock locker(m_ReadMutex);
    NN_UNUSED(locker);

    // 読み込みが終了している場合は処理を抜ける
    if (m_IsReadEnd)
    {
        return;
    }

    NN_G3D_VIEWER_ASSERT(m_Impl.clientSocket != s_InvalidSocket);

    int bufferSize = static_cast<int>(m_ReadBufferSize - m_ReadingSize);
    nn::htcs::ssize_t result = 0;
    if (bufferSize > 0)
    {
        result = nn::ige::HtcsHelperRecv(m_Impl.clientSocket, m_pReadBuffer + m_ReadingSize, bufferSize, 0);
        if (result <= 0)
        {
            // ソケットが使えなくなっている場合は、ソケットを閉じる
            int errorCode = nn::ige::HtcsHelperGetLastError();
            bool isGracefulClose = result == 0;
            if (errorCode != nn::htcs::HTCS_EWOULDBLOCK || isGracefulClose)
            {
                ResetReadFlag();
                CloseClientSocket();
                return;
            }

            return;
        }
    }

    m_ReadingSize += static_cast<size_t>(result);
    m_TotalReadingSize += m_ReadingSize;

    if (m_ReadBufferSize <= m_TotalReadingSize)
    {
        m_IsReadEnd = true;
    }
}

void
EditSocketGeneric::WriteASyncForPoll() NN_NOEXCEPT
{
    ScopedLock locker(m_WriteMutex);
    NN_UNUSED(locker);

    // 書き込みが終了している場合は処理を抜ける
    if (m_IsWriteEnd)
    {
        return;
    }

    if (m_Impl.clientSocket == s_InvalidSocket) // ツール側から通信を切った時の対処
    {
        return;
    }

    int bufferSize = static_cast<int>(m_WriteBufferSize - m_WritingSize);
    nn::htcs::ssize_t result = 0;

    if (bufferSize > 0)
    {
        result = nn::ige::HtcsHelperSend(m_Impl.clientSocket, m_pWriteBuffer + m_WritingSize, bufferSize, 0);
        if (m_IsWriteLogEnabled)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Send: result = %zd\n", result);
        }

        if (result <= 0)
        {
            // ソケットが使えなくなっている場合は、ソケットを閉じる
            bool isGracefulClose = result == 0;
            if (nn::ige::HtcsHelperGetLastError() != nn::htcs::HTCS_EWOULDBLOCK || isGracefulClose )
            {
                ResetWriteFlag();
                CloseClientSocket();
                return;
            }
            return;
        }
    }

    m_WritingSize += static_cast<size_t>(result);
    m_TotalWritingSize += m_WritingSize;

    if (m_WriteBufferSize <= m_TotalWritingSize)
    {
        m_IsWriteEnd = true;
    }
}

}}}} // namespace nn::g3d::viewer::detail


