﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_AttachCommand.h"
#include "../g3d_EditSocket.h"
#include "../g3d_ViewerDetailDefine.h"
#include "../g3d_EditCommandManager.h"
#include "../model/g3d_EditModelObj.h"
#include "../g3d_ResourceManager.h"
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_ResShader.h>

using namespace nn::g3d::viewer::detail;

nn::g3d::viewer::detail::AttachCommand::AttachCommand(
    EditCommandManager* pEditCommandManager,
    Allocator* pAllocator,
    size_t alignment,
    ViewerKeyType key,
    AttachKind kind,
    const char* path) NN_NOEXCEPT
    : m_Key(key)
    , m_Kind(kind)
    , m_Path(nullptr)
    , m_pEditCommandManager(pEditCommandManager)
    , m_ModelObjs(pAllocator, alignment, nullptr)
    , m_Alignment(alignment)
    , m_pAllocator(pAllocator)
    , m_ExecuteState(ExecuteState_WaitPreprocess)
    , m_LastErrorCode(RUNTIME_ERROR_CODE_NO_ERROR)
{
    memset(&m_AttachPacket, 0, sizeof(AttachPacket));

    if (path != nullptr)
    {
        size_t length = strlen(path);
        NN_G3D_VIEWER_ASSERT(length < NN_G3D_EDIT_FILENAME_MAX);
        void* stringBuffer = m_pAllocator->Allocate(sizeof(DynamicLengthString), nn::g3d::detail::Alignment_Default, AllocateType_Communication);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(stringBuffer, "Memory allocation failed");
        DynamicLengthString* pathString = new (stringBuffer) DynamicLengthString(m_pAllocator);
        ViewerResult result = pathString->Assign(path);
        NN_G3D_VIEWER_ASSERT(result == ViewerResult_Success);
        m_Path = pathString;
    }
}

nn::g3d::viewer::detail::AttachCommand::~AttachCommand() NN_NOEXCEPT
{
    if (m_Path != nullptr)
    {
        m_Path->~DynamicLengthString();
        m_pAllocator->Free(m_Path);
    }
}

void nn::g3d::viewer::detail::AttachCommand::ExecutePreprocess() NN_NOEXCEPT
{
    if (m_ExecuteState != ExecuteState_WaitPreprocess)
    {
        return;
    }

    int blockIndex = m_pEditCommandManager->FindFileLoadedCommandIndex(m_Key);
    bool isLoadModelCommand = (m_Kind == ATTACH_MODEL) && (blockIndex != -1);
    if (isLoadModelCommand)
    {
        const nn::g3d::ResModel* pResModel = m_Resource.pResModel;
        ViewerKeyType resModelKey = ViewerKeyManager::GetInstance().FindKey(pResModel);
        NN_G3D_VIEWER_ASSERT_VALID_KEY(resModelKey);
        EditModelObj* pEditModelObj = m_pEditCommandManager->GetResourceManager()->FindUnattachedEditModelObj(resModelKey);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);
        // アタッチする ModelObj のキーを登録
        for (int modelIndex = 0, modelCount = this->GetModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            ModelObj* pModelObj = this->GetModelObjArrayData()[modelIndex];
            ViewerKeyManager::GetInstance().Register(pModelObj, ViewerKeyManager::DataType_ModelObj);
        }

        ModelObj* pModelObj = this->GetModelObj(0);
        ViewerKeyType modelObjKey = ViewerKeyManager::GetInstance().FindKey(pModelObj);
        NN_G3D_VIEWER_DEBUG_PRINT("Load File AttachModel [%d]\n", modelObjKey);

        FileLoadedCommand* pFileLoadedCommand;
        nn::g3d::ResFile* pResFileLoadedFrom3dEditor;
        {
            IViewerCommand* pCommand = m_pEditCommandManager->m_CommandQueue[blockIndex];
            NN_G3D_VIEWER_ASSERT(pCommand->Type() == static_cast<int>(CommandType_FileLoaded));
            pFileLoadedCommand = static_cast<FileLoadedCommand*>(pCommand);

            // 3DEditor からロードされた ResFile を取得
            ViewerKeyType firstLoadedResFileKey = pFileLoadedCommand->GetFirstLoadResFileKey();
            pResFileLoadedFrom3dEditor = ViewerKeyManager::GetInstance().FindData<nn::g3d::ResFile>(firstLoadedResFileKey);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFileLoadedFrom3dEditor);
        }

        m_pEditCommandManager->RemoveAttachingModelObjs(pModelObj->GetResource());

        NN_G3D_VIEWER_ASSERT_DETAIL(m_pEditCommandManager->GetResourceManager()->FindEditModelObj(modelObjKey) == nullptr,
            "%s has already attached\n", NN_G3D_VIEWER_RES_NAME(pResModel, GetName()));

        RuntimeErrorCode errorCode = m_pEditCommandManager->GetResourceManager()->AttachModelObjs(
            pEditModelObj, pResFileLoadedFrom3dEditor, this->GetModelObjArrayData(), this->GetModelObjCount());
        if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
        {
            m_ExecuteState = ExecuteState_ImpossibleToContinue;
            m_LastErrorCode = errorCode;
            return;
        }

        m_pEditCommandManager->GetResourceManager()->ForceBindExistingTextures(pEditModelObj);

        // ロードモデル時はここでコマンド終了
        m_ExecuteState = ExecuteState_Finish;
        pFileLoadedCommand->SetModelObjKey(modelObjKey);
    }
    else
    {
        m_ExecuteState = ExecuteState_WaitExecute;
    }
}

void nn::g3d::viewer::detail::AttachCommand::SetPacketName(const char* name) NN_NOEXCEPT
{
    size_t length = strlen(name);
    NN_G3D_VIEWER_ASSERT(length < NN_G3D_EDIT_FILENAME_MAX);
    memset(m_AttachPacket.block.fileName, 0, NN_G3D_EDIT_FILENAME_MAX);
    strncpy(reinterpret_cast<char*>(m_AttachPacket.block.fileName), name, length);
}

CommandResult nn::g3d::viewer::detail::AttachCommand::Execute(EditSocketBase* pSocket) NN_NOEXCEPT
{
    if (m_ExecuteState != ExecuteState_WaitExecute)
    {
        return CommandResult_Failed;
    }

    NN_G3D_VIEWER_DEBUG_PRINT("Write AttachCommand\n");
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pSocket);

    m_AttachPacket.header.magic = NN_G3D_EDIT_MAGIC;
    m_AttachPacket.header.verWord = NN_G3D_EDIT_VERSION;
    m_AttachPacket.header.dataSize = sizeof(AttachBlock);
    m_AttachPacket.block.attachKind = static_cast<uint16_t>(m_Kind);
    m_AttachPacket.block.attachFileName[0] = '\0'; // 全体を0で埋めるのはコストが高いので0番目で終わるように設定
    m_AttachPacket.block.flag = 0;

    switch(m_Kind)
    {
    case ATTACH_MODEL:
        {
            // ModelObj のキーを登録
            for (int modelIndex = 0; modelIndex < this->GetModelObjCount(); ++modelIndex)
            {
                nn::g3d::ModelObj* pModelObj = this->GetModelObj(modelIndex);
                m_AttachPacket.block.attachedKey = ViewerKeyManager::GetInstance().Register(
                    pModelObj, ViewerKeyManager::DataType_ModelObj);
            }

            // 先頭の ModelObj を 3DEditor との関連付けのためのキーとする
            // TODO: resource.keyで関連付けられるようにする
            nn::g3d::ModelObj* pModelObj = this->GetModelObj(0);
            m_AttachPacket.block.attachedKey = ViewerKeyManager::GetInstance().FindKey(pModelObj);
            m_AttachPacket.header.command = EDIT_SEND_ATTACH_COMMAND_FLAG;
            SetPacketName(pModelObj->GetResource()->GetName());
        }
        break;
    case DETACH_MODEL:
        {
            nn::g3d::ModelObj* pModelObj = this->GetModelObj(0);
            ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(pModelObj);
            NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
            m_AttachPacket.block.attachedKey = key;
            m_AttachPacket.header.command = EDIT_SEND_DETACH_COMMAND_FLAG;
            SetPacketName(pModelObj->GetResource()->GetName());
        }
        break;
    case ATTACH_SHADER_ARCHIVE:
        {
            m_AttachPacket.block.attachedKey = m_Key;
            m_AttachPacket.header.command = EDIT_SEND_ATTACH_COMMAND_FLAG;
            SetPacketName(m_Resource.pResShaderArchive->GetName());
            m_AttachPacket.block.flag |= (m_Resource.pResShaderArchive->HasBinary()) ? ATTACH_SHADER_ARCHIVE_IS_BINARY : 0;
        }
        break;
    case DETACH_SHADER_ARCHIVE:
        {
            m_AttachPacket.block.attachedKey = m_Key;
            m_AttachPacket.header.command = EDIT_SEND_DETACH_COMMAND_FLAG;
            SetPacketName(m_Resource.pResShaderArchive->GetName());
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (m_Path != nullptr)
    {
        const char* attachPath = m_Path->GetStr();
        size_t attachPathLength = m_Path->GetLength();
        memset(m_AttachPacket.block.attachFileName, 0, NN_G3D_EDIT_FILENAME_MAX);
        strncpy(reinterpret_cast<char*>(m_AttachPacket.block.attachFileName), attachPath, attachPathLength);
    }

    bool success = pSocket->WriteSync(&m_AttachPacket, sizeof(AttachPacket));
    if (!success)
    {
        return CommandResult_Failed;
    }

    m_ExecuteState = ExecuteState_WaitPostprocess;
    return CommandResult_Success;
}

void nn::g3d::viewer::detail::AttachCommand::ExecutePostprocess() NN_NOEXCEPT
{
    if (m_ExecuteState != ExecuteState_WaitPostprocess)
    {
        return;
    }

    // Detach 処理の場合は、ここで編集用のインスタンスを削除
    switch(m_Kind)
    {
    case ATTACH_MODEL:
        {
            ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(this->GetModelObj(0));
            NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
            NN_G3D_VIEWER_LOG("Attach Model CommandQueue : [key = %d]\n", key);
        }
        break;
    case ATTACH_SHADER_ARCHIVE:
        NN_G3D_VIEWER_LOG("Attach ShaderArchive CommandQueue : [key = %d]\n", m_Key);
        break;
    case DETACH_MODEL:
        {
            ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(this->GetModelObj(0));
            NN_G3D_VIEWER_ASSERT_VALID_KEY(key);
            NN_G3D_VIEWER_LOG("Detach Model CommandQueue : [key = %d]\n", key);
            m_pEditCommandManager->ScheduleDestructEditModelObj(key);
            m_pEditCommandManager->GetResourceManager()->ScheduleDestroyUnusedResFiles();
        }
        break;
    case DETACH_SHADER_ARCHIVE:
        {
            NN_G3D_VIEWER_LOG("Detach ShaderArchive CommandQueue : [ key = %d]\n", m_Key);
            m_pEditCommandManager->GetResourceManager()->DeleteEditShaderArchive(m_Key);
            m_pEditCommandManager->GetResourceManager()->ScheduleDestroyUnusedResFiles();
        }
        break;
    default:
        NN_G3D_VIEWER_ASSERT_DETAIL(false, "Invalid command kind");
    }

    m_ExecuteState = ExecuteState_Finish;
}
