﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "../g3d_IEditTargetObj.h"
#include "../util/g3d_FixedSizeArray.h"
#include "../util/g3d_EditWorkBuffer.h"
#include "g3d_EditAnimCurve.h"
#include "../util/g3d_ViewerUtility.h"
#include "../g3d_ViewerKeyManager.h"

#include <nn/g3d/g3d_SceneAnimObj.h>

#include <stdint.h>

namespace nn { namespace g3d {

class ResFile;
class ResSceneAnim;
class ResCameraAnim;
class ResLightAnim;
class ResFogAnim;

namespace viewer {
namespace detail {

class Allocator;
class EditCameraAnimObj;
class EditLightAnimObj;
class EditFogAnimObj;

class EditSceneAnimObj : IEditTargetObj
{
public:
    explicit EditSceneAnimObj(Allocator* allocator, nn::g3d::ResFile* resFile, ResSceneAnim* resAnim) NN_NOEXCEPT
        : m_pAllocator(allocator)
        , m_pResFile(resFile)
        , m_pResAnim(resAnim)
        , m_IsBound(false)
        , m_IsLoopAnim(false)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(resFile);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(resAnim);
    }

    bool Setup() NN_NOEXCEPT;

    bool ReloadSceneAnimObj(nn::g3d::ResFile* resFile) NN_NOEXCEPT;
    bool Attach() NN_NOEXCEPT;
    void Detach() NN_NOEXCEPT;

    void EditCameraAnimCurve(
        int cameraIndex,
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void EditLightAnimCurve(
        int lightIndex,
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void EditFogAnimCurve(
        int fogIndex,
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void Calculate() NN_NOEXCEPT;
    void SetFrame(float frame) NN_NOEXCEPT;
    void SetStep(float step) NN_NOEXCEPT;
    void SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT;

    bool IsLoopAnim() const NN_NOEXCEPT
    {
        return m_IsLoopAnim;
    }

    void SetBindFlag(bool enable) NN_NOEXCEPT
    {
        m_IsBound = enable;
    }
    bool IsBound() const NN_NOEXCEPT
    {
        return m_IsBound;
    }

    virtual bool IsResFileBound(ResFile* pResFile) const NN_NOEXCEPT
    {
        return m_pResFile == pResFile;
    }

    FixedSizeArray<CameraAnimObj*>& GetCameraAnimObjPtrArray() NN_NOEXCEPT
    {
        return m_CameraAnimObjPtrArray;
    }

    FixedSizeArray<LightAnimObj*>& GetLightAnimObjPtrArray() NN_NOEXCEPT
    {
        return m_LightAnimObjPtrArray;
    }

    FixedSizeArray<FogAnimObj*>& GetFogAnimObjPtrArray() NN_NOEXCEPT
    {
        return m_FogAnimObjPtrArray;
    }

    ViewerKeyType GetResFileKey() const NN_NOEXCEPT
    {
        return ViewerKeyManager::GetInstance().FindKey(m_pResFile);
    }

    bool ContainsLoopAnim() const NN_NOEXCEPT
    {
        return ContainsLoopAnimInCameraAnim()
            || ContainsLoopAnimInLightAnim()
            || ContainsLoopAnimInFogAnim();
    }

private:
    bool ContainsLoopAnimInCameraAnim() const NN_NOEXCEPT;
    bool ContainsLoopAnimInLightAnim() const NN_NOEXCEPT;
    bool ContainsLoopAnimInFogAnim() const NN_NOEXCEPT;

private:
    Allocator*                 m_pAllocator;
    nn::g3d::ResFile*      m_pResFile;
    nn::g3d::ResSceneAnim* m_pResAnim;
    bool                        m_IsBound;
    bool                        m_IsLoopAnim;

    FixedSizeArray<EditCameraAnimObj> m_EditCameraAnimArray;
    FixedSizeArray<CameraAnimObj*>    m_CameraAnimObjPtrArray;

    FixedSizeArray<EditLightAnimObj>  m_EditLightAnimArray;
    FixedSizeArray<LightAnimObj*>     m_LightAnimObjPtrArray;

    FixedSizeArray<EditFogAnimObj>    m_EditFogAnimArray;
    FixedSizeArray<FogAnimObj*>       m_FogAnimObjPtrArray;

private:
    NN_DISALLOW_COPY(EditSceneAnimObj);
};

class EditCameraAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init() NN_NOEXCEPT;
    void Destroy() NN_NOEXCEPT;
    bool Setup() NN_NOEXCEPT;

    void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT;
    bool IsLoopAnim() const NN_NOEXCEPT
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == AnimFrameCtrl::PlayLoop;
    }

    nn::g3d::CameraAnimObj* GetAnimObj() NN_NOEXCEPT
    {
        return &m_AnimObj;
    }
private:
    explicit EditCameraAnimObj(
        Allocator* allocator,
        nn::g3d::ResCameraAnim* resAnim) NN_NOEXCEPT
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, CameraAnimObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(resAnim);
    }

private:
    Allocator*                                 m_pAllocator;
    nn::g3d::CameraAnimObj                      m_AnimObj;
    nn::g3d::ResCameraAnim*                m_pResAnim;
    nn::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    EditWorkBuffer                m_WorkBuffer;
    FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

class EditLightAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init() NN_NOEXCEPT;
    void Destroy() NN_NOEXCEPT;
    bool Setup() NN_NOEXCEPT;

    void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT;
    bool IsLoopAnim() const NN_NOEXCEPT
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == AnimFrameCtrl::PlayLoop;
    }

    nn::g3d::LightAnimObj* GetAnimObj() NN_NOEXCEPT
    {
        return &m_AnimObj;
    }
private:
    explicit EditLightAnimObj(
        Allocator* allocator,
        nn::g3d::ResLightAnim* resAnim) NN_NOEXCEPT
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, LightAnimObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(resAnim);
    }

private:
    Allocator*                                 m_pAllocator;
    nn::g3d::LightAnimObj                       m_AnimObj;
    nn::g3d::ResLightAnim*                 m_pResAnim;
    nn::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    EditWorkBuffer                m_WorkBuffer;
    FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

class EditFogAnimObj
{
    friend class EditSceneAnimObj;
public:
    bool Init() NN_NOEXCEPT;
    void Destroy() NN_NOEXCEPT;
    bool Setup() NN_NOEXCEPT;

    void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT;

    void SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT;
    bool IsLoopAnim() const NN_NOEXCEPT
    {
        return m_AnimObj.GetDefaultFrameCtrl().GetPlayPolicy() == AnimFrameCtrl::PlayLoop;
    }

    nn::g3d::FogAnimObj* GetAnimObj() NN_NOEXCEPT
    {
        return &m_AnimObj;
    }
private:
    explicit EditFogAnimObj(
        Allocator* allocator,
        nn::g3d::ResFogAnim* resAnim) NN_NOEXCEPT
        : m_pAllocator(allocator)
        , m_pResAnim(resAnim)
        , m_WorkBuffer(allocator, FogAnimObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NN_G3D_VIEWER_RES_NAME(resAnim, GetName()));
        NN_G3D_VIEWER_ASSERT_NOT_NULL(resAnim);
    }

private:
    Allocator*                                 m_pAllocator;
    nn::g3d::FogAnimObj                         m_AnimObj;
    nn::g3d::ResFogAnim*                   m_pResAnim;
    nn::g3d::AnimFrameCtrl                      m_AnimFrameCtrl;
    EditWorkBuffer                m_WorkBuffer;
    FixedSizeArray<EditAnimCurve> m_EditAnimCurveArray;
};

}}}}

