﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Arithmetic.h>
#include <nn/g3d/g3d_ResAnimCurve.h>
#include <nn/g3d/detail/g3d_Flag.h>
#include <nn/g3d/detail/g3d_Perf.h>

#include <climits>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

// ResAnimCurveData から float, int の値を取得するためのテンプレート関数
template <typename T>
struct ResAnimCurveParameter;

template <>
struct ResAnimCurveParameter<float>
{
    static float GetOffset(const ResAnimCurve* curve)
    {
        return curve->ToData().fOffset;
    }

    static float GetDelta(const ResAnimCurve* curve)
    {
        return curve->ToData().fDelta;
    }
};

template <>
struct ResAnimCurveParameter<int>
{
    static int GetOffset(const ResAnimCurve* curve)
    {
        return curve->ToData().iOffset;
    }

    static int GetDelta(const ResAnimCurve* curve)
    {
        return curve->ToData().iDelta;
    }
};

// メンバ関数ポインターはサイズの解釈が一定でないためヘッダに出さない
class ResAnimCurve::Impl
{
public:
    static void (ResAnimCurve::* const s_pFuncFindFrame[])(AnimFrameCache*, float) const;
    static float (ResAnimCurve::* const s_pFuncEvaluateFloat[][TypeCount_Key])(float, AnimFrameCache*) const;
    static int (ResAnimCurve::* const s_pFuncEvaluateInt[][TypeCount_Key])(float, AnimFrameCache*) const;
};

void (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncFindFrame[])(AnimFrameCache*, float) const =
{
    &ResAnimCurve::FindFrame<float>,
    &ResAnimCurve::FindFrame<int16_t>,
    &ResAnimCurve::FindFrame<uint8_t>
};

float (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncEvaluateFloat[][TypeCount_Key])(float, AnimFrameCache*) const =
{
    {
        &ResAnimCurve::EvaluateCubic<float>,
        &ResAnimCurve::EvaluateCubic<int16_t>,
        &ResAnimCurve::EvaluateCubic<int8_t>
    },
    {
        &ResAnimCurve::EvaluateLinear<float>,
        &ResAnimCurve::EvaluateLinear<int16_t>,
        &ResAnimCurve::EvaluateLinear<int8_t>
    },
    {
        &ResAnimCurve::EvaluateBakedFloat<float>,
        &ResAnimCurve::EvaluateBakedFloat<int16_t>,
        &ResAnimCurve::EvaluateBakedFloat<int8_t>
    },
};

int (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncEvaluateInt[][TypeCount_Key])(float, AnimFrameCache*) const =
{
    {
        &ResAnimCurve::EvaluateStepInt<int32_t>,
        &ResAnimCurve::EvaluateStepInt<int16_t>,
        &ResAnimCurve::EvaluateStepInt<int8_t>
    },
    {
        &ResAnimCurve::EvaluateBakedInt<int32_t>,
        &ResAnimCurve::EvaluateBakedInt<int16_t>,
        &ResAnimCurve::EvaluateBakedInt<int8_t>
    },
    {
        &ResAnimCurve::EvaluateStepBool,
        &ResAnimCurve::EvaluateStepBool,
        &ResAnimCurve::EvaluateStepBool
    },
    {
        &ResAnimCurve::EvaluateBakedBool,
        &ResAnimCurve::EvaluateBakedBool,
        &ResAnimCurve::EvaluateBakedBool
    }
};

//--------------------------------------------------------------------------------------------------

float ResAnimCurve::EvaluateFloat(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateFloat");
    NN_SDK_REQUIRES(IsFloatCurve() == true);
    NN_SDK_REQUIRES_NOT_NULL(pFrameCache);

    float offset;
    frame = WrapFrame(&offset, frame);
    int curveType = GetCurveType() >> Shift_Curve;
    int keyType = GetKeyType() >> Shift_Key;
    float result = (this->*Impl::s_pFuncEvaluateFloat[curveType][keyType])(frame, pFrameCache);
    return result * ToData().fScale + offset;
}

int ResAnimCurve::EvaluateInt(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateInt");
    NN_SDK_REQUIRES(IsIntCurve() == true);
    NN_SDK_REQUIRES_NOT_NULL(pFrameCache);

    int offset;
    frame = WrapFrame(&offset, frame);
    int curveType = (GetCurveType() - CurveType_IntBase) >> Shift_Curve;
    int keyType = GetKeyType() >> Shift_Key;
    int result = (this->*Impl::s_pFuncEvaluateInt[curveType][keyType])(frame, pFrameCache);
    return result + offset;
}

template <typename T>
float ResAnimCurve::WrapFrame(T *pOutOffset, float frame) const NN_NOEXCEPT
{
    // カーブ単位でのフレームの折り返しを処理する。
    // アニメーション単位でのフレームの変更はより上位層で行う。
    *pOutOffset = ResAnimCurveParameter<T>::GetOffset(this);
    float startFrame = GetStartFrame();
    float endFrame = GetEndFrame();
    if (frame >= startFrame && frame <= endFrame )
    {
        return frame;
    }
    float frameDiff = startFrame - frame;
    WrapMode wrapMode;
    bool outWrap = false;
    if (frameDiff > 0)
    {
        wrapMode = GetPreWrapMode();
        if (WrapMode_Clamp == wrapMode)
        {
            return startFrame;
        }
    }
    else
    {
        wrapMode = GetPostWrapMode();
        if (WrapMode_Clamp == wrapMode)
        {
            return endFrame;
        }
        frameDiff = frame - endFrame;
        outWrap = true;
    }

    // クランプ以外の場合の処理。
    float duration = endFrame - startFrame;
    int count = static_cast<int>(frameDiff / duration);
    float frameMod = frameDiff - duration * count;
    bool repeat = true;
    if (wrapMode == WrapMode_RelativeRepeat)
    {
        T relative = ( count + 1 ) * ResAnimCurveParameter<T>::GetDelta(this);
        if (!outWrap)
        {
            relative *= -1;
        }
        *pOutOffset += relative;
    }
    else if (wrapMode == WrapMode_Mirror && !(count & 0x01))
    {
        repeat = false;
    }
    if (repeat != outWrap) // outWrap の場合は inWarp と逆になるので裏返す。
    {
        frame = endFrame - frameMod;
    }
    else
    {
        frame = startFrame + frameMod;
    }

    NN_SDK_ASSERT(ToData().startFrame <= frame && frame <= ToData().endFrame);
    return frame;
}

void ResAnimCurve::UpdateFrameCache(AnimFrameCache* pFrameCache, float frame) const NN_NOEXCEPT
{
    if (frame < pFrameCache->start || pFrameCache->end <= frame)
    {
        // FrameCache の区間に収まっていない場合は探索する。
        // 前回の keyIndex から探索した方が近い場合を効率的に判定できれば切り替えたい。
        float denom = nn::util::Rcp(ToData().endFrame - ToData().startFrame);
        pFrameCache->keyIndex = FastCast<uint16_t>((frame - ToData().startFrame) * denom * ToData().keyCount);

        if (pFrameCache->keyIndex > ToData().keyCount - 1)
        {
            pFrameCache->keyIndex = ToData().keyCount - 1;
        }

        int frameType = GetFrameType() >> Shift_Frame;
        (this->*Impl::s_pFuncFindFrame[frameType])(pFrameCache, frame);
    }
}


template <typename T>
void ResAnimCurve::FindFrame(AnimFrameCache *pFrameCache, float frame) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::FindFrame");
    // 長さ0のキーは終端を除いて存在しない前提。
    typedef Frame<T> FrameType;
    const FrameType* pFrameArray = GetFrameArray<FrameType>();
    T quantized = FrameType::Quantize(frame);
    int endFrameIndex = ToData().keyCount - 1;

    if (quantized >= pFrameArray[endFrameIndex].frame)
    {
        // 終端フレームは長さ0で除算しないよう end を適当に設定しておく。
        pFrameCache->keyIndex = static_cast<uint16_t>(endFrameIndex);
        pFrameCache->start = pFrameArray[pFrameCache->keyIndex].Get();
        pFrameCache->end = pFrameCache->start + 1.0f;
        return;
    }

    // 線形探索
    int keyIndex = pFrameCache->keyIndex;
    if (quantized < pFrameArray[keyIndex].frame)
    {
        do
        {
            NN_SDK_ASSERT(0 < keyIndex);
            --keyIndex;
        } while (quantized < pFrameArray[keyIndex].frame);
    }
    else
    {
        do
        {
            NN_SDK_ASSERT(keyIndex < endFrameIndex);
            ++keyIndex;
        } while (pFrameArray[keyIndex].frame <= quantized);

        // ここにきた時点で1つ通り過ぎている。
        NN_SDK_ASSERT(0 < keyIndex);
        --keyIndex;
    }

    pFrameCache->keyIndex = keyIndex;
    pFrameCache->start = pFrameArray[keyIndex].Get();
    pFrameCache->end = pFrameArray[keyIndex + 1].Get();
}

//--------------------------------------------------------------------------------------------------

template <typename T>
float ResAnimCurve::EvaluateCubic(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateCubic");

    UpdateFrameCache(pFrameCache, frame);

    // セグメント内での比率を求める。
    float start = pFrameCache->start;
    float end = pFrameCache->end;
    float ratio = (frame - start) * nn::util::Rcp(end - start);

    // インデックスと比率から求めた値を返す。
    typedef ResCubicKey<T> KeyType;
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get(ratio);
}

template <typename T>
float ResAnimCurve::EvaluateLinear(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateLinear");

    UpdateFrameCache(pFrameCache, frame);

    // セグメント内での比率を求める。
    float start = pFrameCache->start;
    float end = pFrameCache->end;
    float ratio = (frame - start) * nn::util::Rcp(end - start);

    // インデックスと比率から求めた値を返す。
    typedef ResLinearKey<T> KeyType;
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get(ratio);
}

template <typename T>
float ResAnimCurve::EvaluateBakedFloat(float frame, AnimFrameCache* /*pFrameCache*/) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateBakedFloat");
    typedef ResFloatKey<T> KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;
    float ratio = frame - (start + keyIndex);

    // keyIndex + 1 のキーが必ず存在するようコンバートされている前提。
    NN_SDK_ASSERT(keyIndex + 1 < ToData().keyCount);

    // ratio : 1-ratio で補間。
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return pKeyArray[keyIndex + 1].Get() * ratio + pKeyArray[keyIndex].Get() * (1.0f - ratio);
}

template <typename T>
int ResAnimCurve::EvaluateStepInt(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateStepInt");

    UpdateFrameCache(pFrameCache, frame);

    // インデックスと比率から求めた値を返す。
    typedef ResIntKey<T> KeyType;
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get();
}

template <typename T>
int ResAnimCurve::EvaluateBakedInt(float frame, AnimFrameCache* /*pFrameCache*/) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateBakedInt");
    typedef ResIntKey<T> KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;

    // float 以外は補間しない。
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return pKeyArray[keyIndex].Get();
}

int ResAnimCurve::EvaluateStepBool(float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateStepBool");

    UpdateFrameCache(pFrameCache, frame);

    // インデックスと比率から求めた値を返す。
    typedef uint32_t KeyType;
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return (pKeyArray[pFrameCache->keyIndex >> 5] >> (pFrameCache->keyIndex & 0x1F)) & 0x1;
}

int ResAnimCurve::EvaluateBakedBool(float frame, AnimFrameCache* /*pFrameCache*/) const NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL3("ResAnimCurve::EvaluateBakedBool");
    typedef uint32_t KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;

    // float 以外は補間しない。
    const KeyType* pKeyArray = GetKeyArray<KeyType>();
    return (pKeyArray[keyIndex >> 5] >> (keyIndex & 0x1F)) & 0x1;
}

//--------------------------------------------------------------------------------------------------

template <typename T>
void ResAnimCurve::BakeImpl(void* pBuffer, float start, int keyCount) NN_NOEXCEPT
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = keyCount - 1;
    T* pBaked = static_cast<T*>(pBuffer);

    pBaked[0] = static_cast<T>(EvaluateStepInt<T>(GetStartFrame(), &frameCache));
    for (int idxFrame = 1; idxFrame < last; ++idxFrame)
    {
        pBaked[idxFrame] = static_cast<T>(EvaluateStepInt<T>(start + idxFrame, &frameCache));
    }
    pBaked[last] = static_cast<T>(EvaluateStepInt<T>(GetEndFrame(), &frameCache));
}

template <>
void ResAnimCurve::BakeImpl<bool>(void* pBuffer, float start, int keyCount) NN_NOEXCEPT
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = keyCount - 1;
    Bit32* pBaked = static_cast<Bit32*>(pBuffer);

    SetBit(pBaked, 0, EvaluateStepBool(GetStartFrame(), &frameCache));
    for (int idxFrame = 1; idxFrame < keyCount - 1; ++idxFrame)
    {
        SetBit(pBaked, idxFrame, EvaluateStepBool(start + idxFrame, &frameCache));
    }
    SetBit(pBaked, last, EvaluateStepBool(GetEndFrame(), &frameCache));
}

template <>
void ResAnimCurve::BakeImpl<float>(void* pBuffer, float start, int keyCount) NN_NOEXCEPT
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = keyCount - 1;
    float* pBaked = static_cast<float*>(pBuffer);

    pBaked[0] = EvaluateFloat(GetStartFrame(), &frameCache);
    for (int idxFrame = 1; idxFrame < last; ++idxFrame)
    {
        pBaked[idxFrame] = EvaluateFloat(start + idxFrame, &frameCache);
    }
    pBaked[last] = EvaluateFloat(GetEndFrame(), &frameCache);

    // 両端は外挿により整数フレームの値を計算する。
    float end = start + (keyCount - 1);
    float diffFrame;
    diffFrame = GetStartFrame() - start;
    pBaked[0] = (pBaked[0] - diffFrame * pBaked[1]) / (1.0f - diffFrame);
    diffFrame = end - GetEndFrame();
    if (diffFrame < 1.0f) // 最終フレームが整数の場合、+1 フレームを計算する必要はない。
    {
        pBaked[last] = (pBaked[last] - diffFrame * pBaked[last - 1]) / (1.0f - diffFrame);
    }
}

void ResAnimCurve::BakeFloat(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer) == true);
    NN_SDK_REQUIRES(bufferSize >= CalculateBakedFloatSize());
    (void)bufferSize;

    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()) + 1.0f; // 補間するので EndFrame より大きな整数にする。
    int keyCount = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (GetCurveType() == CurveType_BakedFloat || keyCount <= 2)
    {
        return; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    BakeImpl<float>(pBuffer, start, keyCount);

    // pBuffer にバックアップをとる。
    static const size_t BAKED_BACKUP_SIZE =
        nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount) + sizeof(ToData().fOffset) + sizeof(ToData().fScale), 8)
        + sizeof(ToData().pKeyArray);
    void* backupBuffer = AddOffset(pBuffer, CalculateBakedFloatSize() - BAKED_BACKUP_SIZE);
    *static_cast<Bit16*>(backupBuffer) = ToData().flag;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().flag));
    *static_cast<uint16_t*>(backupBuffer) = ToData().keyCount;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().keyCount));
    *static_cast<float*>(backupBuffer) = ToData().fScale;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().fScale));
    *static_cast<float*>(backupBuffer) = ToData().fOffset;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().fOffset));
    nn::util::BytePtr ptr(backupBuffer);
    backupBuffer = ptr.AlignUp(8).Get();
    *reinterpret_cast<nn::util::BinPtr*>(backupBuffer) = ToData().pKeyArray;

    ToData().flag = (ToData().flag & ~(Mask_Curve | Mask_Key)) | (CurveType_BakedFloat | KeyType_Quant32);
    ToData().keyCount = static_cast<uint16_t>(keyCount);
    ToData().fScale = 1.0f;
    ToData().fOffset = 0.0f;
    ToData().pKeyArray.Set(pBuffer);
}

void ResAnimCurve::BakeInt(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer) == true);
    NN_SDK_REQUIRES(bufferSize >= CalculateBakedIntSize());
    (void)bufferSize;

    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()); // 切り捨てで評価されるので Floor でよい。
    int keyCount = static_cast<int>(end) - static_cast<int>(start) + 1;
    Bit16 flag = 0;

    if (GetCurveType() == CurveType_BakedFloat ||
        GetCurveType() == CurveType_BakedBool ||
        keyCount <= 2)
    {
        return; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    if (GetCurveType() == CurveType_StepInt)
    {
        size_t size = keyCount;
        if (GetKeyType() == KeyType_Quant8)
        {
            NN_SDK_ASSERT(bufferSize >= size);

            BakeImpl<int8_t>(pBuffer, start, keyCount);
        }
        else if (GetKeyType() == KeyType_Quant16)
        {
            size <<= 1;
            NN_SDK_ASSERT(bufferSize >= size);

            BakeImpl<int16_t>(pBuffer, start, keyCount);
        }
        else
        {
            size <<= 2;
            NN_SDK_ASSERT(bufferSize >= size);

            BakeImpl<int32_t>(pBuffer, start, keyCount);
        }

        flag = (ToData().flag & ~Mask_Curve) | CurveType_BakedInt;
    }
    else if (GetCurveType() == CurveType_StepBool)
    {
        BakeImpl<bool>(pBuffer, start, keyCount);

        flag = CurveType_BakedBool;
    }
    // pBuffer にバックアップをとる。
    static const size_t BAKED_BACKUP_SIZE =
        nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount), 8) + sizeof(ToData().pKeyArray);
    void* backupBuffer = AddOffset(pBuffer, CalculateBakedIntSize() - BAKED_BACKUP_SIZE);
    *static_cast<Bit16*>(backupBuffer) = ToData().flag;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().flag));
    *static_cast<uint16_t*>(backupBuffer) = ToData().keyCount;
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().keyCount));
    nn::util::BytePtr ptr(backupBuffer);
    backupBuffer = ptr.AlignUp(8).Get();
    *reinterpret_cast<nn::util::BinPtr*>(backupBuffer) = ToData().pKeyArray;

    ToData().flag = flag;
    ToData().keyCount = static_cast<uint16_t>(keyCount);
    ToData().pKeyArray.Set(pBuffer);
}

size_t ResAnimCurve::CalculateBakedFloatSize() const NN_NOEXCEPT
{
    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()) + 1.0f; // 補間するので EndFrame より大きな整数にする。
    int keyCount = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (keyCount <= 2)
    {
        return 0; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    static const size_t BAKED_BACKUP_SIZE =
        nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount) + sizeof(ToData().fOffset) + sizeof(ToData().fScale), 8)
        + sizeof(ToData().pKeyArray);
    return nn::util::align_up(sizeof(float) * keyCount, 8) + BAKED_BACKUP_SIZE;
}

size_t ResAnimCurve::CalculateBakedIntSize() const NN_NOEXCEPT
{
    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()); // 切り捨てで評価されるので Floor でよい。
    int keyCount = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (keyCount <= 2)
    {
        return 0; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    size_t size = 0;
    if (GetCurveType() == CurveType_StepInt ||
        GetCurveType() == CurveType_BakedInt)
    {
        if (GetKeyType() == KeyType_Quant8)
        {
            size = keyCount;
        }
        else if (GetKeyType() == KeyType_Quant16)
        {
            size = (keyCount << 1);
        }
        else
        {
            size = (keyCount << 2);
        }
    }
    else if (GetCurveType() == CurveType_StepBool ||
             GetCurveType() == CurveType_BakedBool)
    {
        size = nn::util::align_up(keyCount, sizeof(Bit32) * CHAR_BIT) / CHAR_BIT;
    }

    static const size_t BAKED_BACKUP_SIZE =
        nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount), 8) + sizeof(ToData().pKeyArray);
    return nn::util::align_up(size, 8) + BAKED_BACKUP_SIZE;
}

void ResAnimCurve::ResetFloat() NN_NOEXCEPT
{
    if (GetCurveType() != CurveType_BakedFloat)
    {
        return;
    }
    void* pBuffer = ToData().pKeyArray.Get();

    static const size_t BAKED_BACKUP_SIZE = nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount) + sizeof(ToData().fOffset) + sizeof(ToData().fScale), 8)
                                          + sizeof(ToData().pKeyArray);

    // pBuffer にとったバックアップから復元する
    void* backupBuffer = AddOffset(pBuffer, CalculateBakedFloatSize() - BAKED_BACKUP_SIZE);
    ToData().flag = *static_cast<uint16_t*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().flag));
    ToData().keyCount = *static_cast<uint16_t*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().keyCount));
    ToData().fScale = *static_cast<float*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().fScale));
    ToData().fOffset = *static_cast<float*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().fOffset));
    nn::util::BytePtr ptr(backupBuffer);
    backupBuffer = ptr.AlignUp(8).Get();
    ToData().pKeyArray = *(reinterpret_cast<nn::util::BinPtr*>(backupBuffer));
}

void ResAnimCurve::ResetInt() NN_NOEXCEPT
{
    if (GetCurveType() != CurveType_BakedInt &&
        GetCurveType() != CurveType_BakedBool)
    {
        return;
    }
    void* pBuffer = ToData().pKeyArray.Get();

    static const size_t BAKED_BACKUP_SIZE = nn::util::align_up(sizeof(ToData().flag) + sizeof(ToData().keyCount), 8) + sizeof(ToData().pKeyArray);

    // pBuffer にとったバックアップから復元する
    void* backupBuffer = AddOffset(pBuffer, CalculateBakedIntSize() - BAKED_BACKUP_SIZE);
    ToData().flag = *static_cast<uint16_t*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().flag));
    ToData().keyCount = *static_cast<uint16_t*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, sizeof(ToData().keyCount));
    nn::util::BytePtr ptr(backupBuffer);
    backupBuffer = ptr.AlignUp(8).Get();
    ToData().pKeyArray = *(reinterpret_cast<nn::util::BinPtr*>(backupBuffer));
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
