﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_Billboard.h>

namespace nn { namespace g3d {

void Billboard::CalculateWorld(nn::util::Matrix4x3fType* pWorldView, const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
    //   2. Y 軸の向きは通常のワールドビュー行列と同じまま維持する

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(wv.m01,  wv.m11,  wv.m21);
        vz.Set(0.0f,    0.0f,    1.0f);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // この処理はSetSR 内で行っている
        vx.Normalize(vx);       // y,zが正規化されているため不要

        vy.Cross(vz, vx);       // 計算簡略化可能 vy={-vx.y, vx.x, 0.0f}
                                //                  ={ vy.x, vy.y, 0.0f}
                                // よって vy.z を 0.0f とする事で不要
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/

    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, view);

    nn::util::Vector3fType vectorY;
    MatrixGetAxisY(&vectorY, worldView);
    VectorSet(&vectorY, VectorGetX(vectorY), VectorGetY(vectorY), 0.0f);

    if (!(VectorNormalize(&vectorY, vectorY) > 0.0f))
    {
        // 'world' ビルボードの Y 軸が視線軸と平行です
        NN_G3D_WARNING( false, "Y axis of a 'world' billboard is parallel to eye axis." );
        VectorSet(&vectorY, 0.0f, 1.0f, 0.0f);
    }

    nn::util::Vector3fType transVector;
    nn::util::MatrixGetAxisW(&transVector, *pWorldView);

    MatrixSet(pWorldView,
              // row0
              VectorGetY(vectorY),
              -VectorGetX(vectorY),
              0.0f,
              // row1
              VectorGetX(vectorY),
              VectorGetY(vectorY),
              0.0f,
              // row2
              0.0f,
              0.0f,
              1.0f,
              // row3
              VectorGetX(transVector),
              VectorGetY(transVector),
              VectorGetZ(transVector));
}

void Billboard::CalculateWorldViewpoint(nn::util::Matrix4x3fType* pWorldView, const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの方を向くようにする
    //   2. Y 軸の向きは通常のワールドビュー行列と同じまま維持する

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set( wv.m01,  wv.m11,  wv.m21);
        vz.Set(-wv.m03, -wv.m13, -wv.m23);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vy.Cross(vz, vx);       //
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/
    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, view);

    nn::util::Vector3fType vectorX;
    nn::util::Vector3fType vectorY;
    nn::util::Vector3fType vectorZ;
    nn::util::Vector3fType scale;

    VectorSet(&scale, -1.0f, -1.0f, -1.0f);

    MatrixGetAxisY(&vectorY, worldView);
    MatrixGetAxisW(&vectorZ, worldView);
    VectorMultiply(&vectorZ, vectorZ, scale);
    if (!(VectorNormalize(&vectorZ, vectorZ) > 0.0f))
    {
        // カメラがモデルと同じ座標に位置しています
        NN_G3D_WARNING( false, "The camera is locating at a 'world_viewpoint' billboard." );
        VectorSet(&vectorZ, 0.0f, 0.0f, 1.0f);
    }

    VectorCross(&vectorX, vectorY, vectorZ);
    if (VectorNormalize(&vectorX, vectorX) > 0.0f)
    {
        VectorCross(&vectorY, vectorZ, vectorX);
    }
    else
    {
        // カメラが 'world_viewpoint' ビルボードの Y 軸上に位置しています
        NN_G3D_WARNING( false, "The camera is locating on Y axis of a 'world_viewpoint' billboard." );
        VectorSet(&vectorX, VectorGetZ(vectorZ), 0.0f, -VectorGetX(vectorZ));
        if (VectorNormalize(&vectorX, vectorX) > 0.0f)
        {
            VectorCross(&vectorY, vectorZ, vectorX);
        }
        else
        {
            // z 軸が (0, 1, 0) と平行
            VectorSet(&vectorX, 1.0f, 0.0f, 0.0f);
            VectorSet(&vectorY, 0.0f, 0.0f, -VectorGetY(vectorZ)); // vz と vx の外積
        }
    }
    MatrixSetAxisX(pWorldView, vectorX);
    MatrixSetAxisY(pWorldView, vectorY);
    MatrixSetAxisZ(pWorldView, vectorZ);
}

void Billboard::CalculateYAxis(nn::util::Matrix4x3fType* pWorldView, const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. Y 軸の向きは通常のワールドビュー行列と同じまま維持する
    //   2. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(wv.m01,  wv.m11,  wv.m21);
        vz.Set(0.0f,    0.0f,    1.0f);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // よって vx をあらかじめ指定することで不要
        vx.Normalize(vx);       //

        vz.Cross(vx, vy);       //
        vz.Normalize(vz);       // x,yが正規化されているため不要
    である。
*/
    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, view);

    nn::util::Vector3fType vectorX;
    nn::util::Vector3fType vectorY;
    nn::util::Vector3fType vectorZ;

    MatrixGetAxisY(&vectorY, worldView);
    VectorSet(&vectorX, VectorGetY(vectorY), -VectorGetX(vectorY), 0.0f); // vy vzの外積
    if (!(VectorNormalize(&vectorY, vectorY) > 0.0f))
    {
        // 行列が縮退しています
        NN_G3D_WARNING( false, "Y axis of a 'yaxis' billboard is degenerated." );
        VectorSet(&vectorY, 0.0f, 1.0f, 0.0f);
        VectorSet(&vectorX, 1.0f, 0.0f, 0.0f);
    }

    if (VectorNormalize(&vectorX, vectorX) > 0.0f)
    {
        VectorCross(&vectorZ, vectorX, vectorY);
    }
    else
    {
        // 'yaxis' ビルボードの Y 軸が視線軸と平行です
        NN_G3D_WARNING( false, "Y axis of a 'yaxis' billboard is parallel to eye axis." );
        VectorSet(&vectorX, 1.0f, 0.0f, 0.0f);
        VectorSet(&vectorZ, 0.0f, -VectorGetZ(vectorY), 0.0f); // vx と vy の外積
    }
    MatrixSetAxisX(pWorldView, vectorX);
    MatrixSetAxisY(pWorldView, vectorY);
    MatrixSetAxisZ(pWorldView, vectorZ);
}

void Billboard::CalculateYAxisViewpoint(nn::util::Matrix4x3fType* pWorldView, const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. Y 軸の向きは通常のワールドビュー行列と同じまま維持する
    //   2. モデル座標系の Z 軸がカメラの方を向くようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set( wv.m01,  wv.m11,  wv.m21);
        vz.Set(-wv.m03, -wv.m13, -wv.m23);

        vy.Normalize(vy);       //
        vz.Normalize(vz);       // 外積計算後に正規化するため不要

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vz.Cross(vx, vy);       //
        vz.Normalize(vz);       // x,yが正規化されているため不要
    である。
*/
    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, view);

    nn::util::Vector3fType vectorX;
    nn::util::Vector3fType vectorY;
    nn::util::Vector3fType vectorZ;
    nn::util::Vector3fType scale;

    VectorSet(&scale, -1.0f, -1.0f, -1.0f);

    MatrixGetAxisY(&vectorY, worldView);
    MatrixGetAxisW(&vectorZ, worldView);
    VectorMultiply(&vectorZ, vectorZ, scale);
    if (!(VectorNormalize(&vectorY, vectorY) > 0.0f))
    {
        // 行列が縮退しています
        NN_G3D_WARNING( false, "Y axis of a 'yaxis_viewpoint' billboard is degenerated." );
        VectorSet(&vectorY, 0.0f, 1.0f, 0.0f);
    }

    VectorCross(&vectorX, vectorY, vectorZ);
    if (VectorNormalize(&vectorX, vectorX) > 0.0f)
    {
        VectorCross(&vectorZ, vectorX, vectorY);
    }
    else
    {
        // カメラが 'yaxis_viewpoint' ビルボードの Y 軸上に位置しています
        NN_G3D_WARNING( false, "The camera is locating on Y axis of a 'yaxis_viewpoint' billboard." );
        VectorSet(&vectorX, VectorGetY(vectorY), -VectorGetX(vectorY), 0.0f); // vy と (0, 0, 1) の外積
        if (VectorNormalize(&vectorX, vectorX) > 0.0f)
        {
            VectorCross(&vectorZ, vectorX, vectorY);
        }
        else
        {
            // y 軸が (0, 0, 1) と平行
            VectorSet(&vectorX, 1.0f, 0.0f, 0.0f);
            VectorSet(&vectorZ, 0.0f, -VectorGetZ(vectorY), 0.0f); // vx と vy の外積
        }
    }

    MatrixSetAxisX(pWorldView, vectorX);
    MatrixSetAxisY(pWorldView, vectorY);
    MatrixSetAxisZ(pWorldView, vectorZ);
}

void Billboard::CalculateScreen(nn::util::Matrix4x3fType* pWorldView, const nn::util::Matrix4x3fType& local) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
    //   2. ワールド座標系の Y 軸がビュー座標系のY軸と平行になるようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(local.m01, local.m11, local.m21);
        vz.Set(     0.0f,      0.0f,      1.0f);

        vy.Normalize(vy);       //
        vz.Normalize(vz);       // 既に正規化されているので不要

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // この処理はSetSR 内で行っている
        vx.Normalize(vx);       // y,zが正規化されているため不要

        vy.Cross(vz, vx);       // 計算簡略化可能 vy={-vx.y, vx.x, 0.0f}
                                //                  ={ vy.x, vy.y, 0.0f}
                                // よって vy.z を 0.0f とする事で不要
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/
    nn::util::Float4x3 floatLocal;
    MatrixStore(&floatLocal, local);

    nn::util::Vector3fType vectorY;
    VectorSet(&vectorY, floatLocal.m[1][0], floatLocal.m[1][1], 0.0f);
    if (!(VectorNormalize(&vectorY, vectorY) > 0.0f))
    {
        // 'screen' ビルボードの Y 軸が視線軸と平行です
        NN_G3D_WARNING( false, "Y axis of a 'screen' billboard is parallel to eye axis." );
        VectorSet(&vectorY, 0.0f, 1.0f, 0.0f);
    }

    nn::util::Vector3fType transVector;
    nn::util::MatrixGetAxisW(&transVector, *pWorldView);

    MatrixSet(pWorldView,
              // row0
              VectorGetY(vectorY),
              -VectorGetX(vectorY),
              0.0f,
              // row1
              VectorGetX(vectorY),
              VectorGetY(vectorY),
              0.0f,
              // row2
              0.0f,
              0.0f,
              1.0f,
              // row3
              VectorGetX(transVector),
              VectorGetY(transVector),
              VectorGetZ(transVector));
}

void Billboard::CalculateScreenViewpoint(nn::util::Matrix4x3fType* pWorldView,
        const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world, const nn::util::Matrix4x3fType& local) NN_NOEXCEPT
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの方を向くようにする
    //   2. ワールド座標系の Y 軸がビュー座標系のY軸と平行になるようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Vec3 vx, vy, vz;
        vy.Set(local.m01, local.m11, local.m21);
        vz.Set(  -wv.m03,   -wv.m13,   -wv.m23);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vy.Cross(vz, vx);       //
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/
    nn::util::Matrix4x3fType worldView;
    MatrixMultiply(&worldView, world, view);

    nn::util::Vector3fType vectorX;
    nn::util::Vector3fType vectorY;
    nn::util::Vector3fType vectorZ;
    nn::util::Vector3fType scale;

    VectorSet(&scale, -1.0f, -1.0f, -1.0f);

    MatrixGetAxisY(&vectorY, local);
    VectorSet(&vectorY, VectorGetX(vectorY), VectorGetY(vectorY), 0.0f);
    MatrixGetAxisW(&vectorZ, worldView);
    VectorMultiply(&vectorZ, vectorZ, scale);
    if (!(VectorNormalize(&vectorZ, vectorZ) > 0.0f))
    {
        // カメラがモデルと同じ座標に位置しています
        NN_G3D_WARNING( false, "The camera is locating at a 'screen_viewpoint' billboard." );
        VectorSet(&vectorZ, 0.0f, 0.0f, 1.0f);
    }

    VectorCross(&vectorX, vectorY, vectorZ);
    if (VectorNormalize(&vectorX, vectorX) > 0.0f)
    {
        VectorCross(&vectorY, vectorZ, vectorX);
    }
    else
    {
        // カメラが 'screen_viewpoint' ビルボードの Y 軸上に位置しています
        NN_G3D_WARNING( false, "The camera is locating on Y axis of a 'screen_viewpoint' billboard." );
        VectorSet(&vectorX, VectorGetZ(vectorZ), 0.0f, -VectorGetX(vectorZ)); // (0, 1, 0) と vz の外積
        if (VectorNormalize(&vectorX, vectorX) > 0.0f)
        {
            VectorCross(&vectorY, vectorZ, vectorX);
        }
        else
        {
            // z 軸が (0, 1, 0) と平行
            VectorSet(&vectorX, 1.0f, 0.0f, 0.0f);
            VectorSet(&vectorY, 0.0f, 0.0f, -VectorGetY(vectorZ)); // vz と vx の外積
        }
    }
    MatrixSetAxisX(pWorldView, vectorX);
    MatrixSetAxisY(pWorldView, vectorY);
    MatrixSetAxisZ(pWorldView, vectorZ);
}

void Billboard::Calculate(Bit32 billboardMode, nn::util::Matrix4x3fType* pWorldView,
    const nn::util::Matrix4x3fType& view, const nn::util::Matrix4x3fType& world, const nn::util::Matrix4x3fType& local) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(billboardMode, static_cast<uint32_t>(ResBone::Flag_BillboardWorldViewVector <= billboardMode),
                          static_cast<uint32_t>(ResBone::Flag_BillboardMax + 1));
    // Billboard クラスの関数群はスケールを維持しないので退避する。
    nn::util::Vector3fType vectorScale;
    MatrixExtractScaleBase(&vectorScale, world);

    static void (* const pCalcBillboard[])(nn::util::Matrix4x3fType*, const nn::util::Matrix4x3fType&,
                                           const nn::util::Matrix4x3fType&, const nn::util::Matrix4x3fType&) = {
        &Billboard::CalculateWorld,
        &Billboard::CalculateWorldViewpoint,
        &Billboard::CalculateScreen,
        &Billboard::CalculateScreenViewpoint,
        &Billboard::CalculateYAxis,
        &Billboard::CalculateYAxisViewpoint
    };

    int funcIndex = (billboardMode - ResBone::Flag_BillboardWorldViewVector) >> ResBone::Shift_Billboard;
    pCalcBillboard[funcIndex](pWorldView, view, world, local);

    // 退避しておいたスケールを復元する。
    MatrixScaleBase(pWorldView, *pWorldView, vectorScale);
}

}} // namespace nn::g3d
