﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_AnimObj.h>
#include <limits>
#include <nn/g3d/g3d_SkeletonObj.h>

namespace nn { namespace g3d {

// NaN の扱いはコンパイラによって厳密さが異なるので infinity を使用する。
const float AnimFrameCtrl::InvalidFrame = std::numeric_limits<float>::infinity();

void AnimFrameCtrl::Initialize(float startFrame, float endFrame, PlayPolicy pPlayPolicy) NN_NOEXCEPT
{
    m_Frame = 0.0f;
    SetFrameRange(startFrame, endFrame);
    m_Step = 1.0f;
    m_pPlayPolicy = pPlayPolicy;
    m_pUserPtr = NULL;
}

float AnimFrameCtrl::PlayOneTime(float inputFrame, float startFrame, float endFrame, void* /*pUserData*/) NN_NOEXCEPT
{
    if (inputFrame >= endFrame)
    {
        return endFrame;
    }
    else if (inputFrame < startFrame)
    {
        return startFrame;
    }
    else
    {
        return inputFrame;
    }
}

float AnimFrameCtrl::PlayLoop(float inputFrame, float startFrame, float endFrame, void* /*pUserData*/) NN_NOEXCEPT
{
    float frameDiff, sign, offset;
    if (inputFrame >= endFrame)
    {
        frameDiff = inputFrame - endFrame;
        sign = 1.0f;
        offset = startFrame;
    }
    else if (inputFrame < startFrame)
    {
        frameDiff = startFrame - inputFrame;
        sign = -1.0f;
        offset = endFrame;
    }
    else
    {
        return inputFrame;
    }

    float duration = endFrame - startFrame;
    if (duration == 0.0f)
    {
        return startFrame;
    }

    float frameMod = frameDiff - duration * static_cast<int>(frameDiff / duration);
    return sign * frameMod + offset;
}

//--------------------------------------------------------------------------------------------------

void AnimBindTable::Initialize(Bit32* pBindArray, int tableSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBindArray != NULL || tableSize == 0);
    m_pBindArray = pBindArray;
    m_Flag = 0;
    m_Size = static_cast<uint16_t>(tableSize);
    m_AnimCount = 0;
    m_TargetCount = 0;
}

void AnimBindTable::ClearAll(int targetCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(targetCount, 0, m_Size + 1);
    m_TargetCount = static_cast<uint16_t>(targetCount);
    int clearSize = m_TargetCount > m_AnimCount ? m_TargetCount : m_AnimCount;
    for (int bindIndex = 0; bindIndex < clearSize; ++bindIndex)
    {
        m_pBindArray[bindIndex] = Bit32(Flag_NotBound | Flag_ReverseNotBound | Flag_Disabled);
    }
}

void AnimBindTable::BindAll(const uint16_t* pBindIndexArray) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBindIndexArray != NULL || m_AnimCount == 0);
    for (int animIndex = 0; animIndex < m_AnimCount; ++animIndex)
    {
        uint16_t targetIndex = pBindIndexArray[animIndex];
        if (targetIndex < Flag_NotBound)
        {
            NN_SDK_ASSERT_RANGE(targetIndex, 0U, m_TargetCount + 1);
            Bind(animIndex, targetIndex);
        }
    }
}

//--------------------------------------------------------------------------------------------------

void AnimContext::Initialize(AnimFrameCache* pFrameCacheArray, int arraySize) NN_NOEXCEPT
{
    // コンテクストを使用しない場合は pKeyIndexArray == NULL && arraySize >= 0 となる。
    m_pFrameCacheArray = pFrameCacheArray;
    m_Size = (pFrameCacheArray == NULL) ? 0 : arraySize;
    m_CurveCount = 0;
}

//--------------------------------------------------------------------------------------------------

void AnimObj::ResetFrameCtrl(int frameCount, bool loop) NN_NOEXCEPT
{
    // デフォルトでない FrameCtrl は管理下に無いのでリセットしない。
    m_DefaultFrameCtrl.Initialize(0.0f, static_cast<float>(frameCount),
        loop ? AnimFrameCtrl::PlayLoop : AnimFrameCtrl::PlayOneTime);
}

//--------------------------------------------------------------------------------------------------
void ModelAnimObj::SetBindFlagImpl(int targetIndex, BindFlag flag) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsTargetBound() == true);
    int idxAnim = m_BindTable.GetAnimIndex(targetIndex);
    if (idxAnim != AnimBindTable::Flag_NotBound)
    {
        m_BindTable.SetBindFlag(idxAnim, static_cast<AnimBindTable::BindFlag>(flag));
    }
}

AnimObj::BindFlag ModelAnimObj::GetBindFlagImpl(int targetIndex) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsTargetBound() == true);
    int idxAnim = m_BindTable.GetAnimIndex(targetIndex);
    if (idxAnim != AnimBindTable::Flag_NotBound)
    {
        return static_cast<AnimObj::BindFlag>(m_BindTable.GetBindFlag(idxAnim));
    }
    else
    {
        return AnimObj::BindFlag_Disabled;
    }
}

}} // namespace nn::g3d
