﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/font/font_TextWriterBase.h>
#include <nn/font/font_TagProcessorBase.h>

namespace nn {
namespace font {


/* ------------------------------------------------------------------------
        クラス変数定義
   ------------------------------------------------------------------------ */

template <typename CharType>
TagProcessorBase<CharType> TextWriterBase<CharType>::g_DefaultTagProcessor;

template <typename CharType>
ExtendedTagProcessorBase<CharType> TextWriterBase<CharType>::g_ExtendedTagProcessor;

namespace
{

/*!--------------------------------------------------------------------------*
  @brief        基点が中央であるときのカーソル位置補正の値を求めます。
                半分の値が浮動小数点にならないように整数値に切り上げます。

  @param[in]    value  考慮する幅/高さ
  @param[in]    isCeil 半分の値が浮動小数点にならないように整数値に
                       切り上げる場合は真。

  @return       基点が中央であるときのカーソル位置補正の値を返します。
 *---------------------------------------------------------------------------*/
inline
float
AdjustCenterValue(float value, bool isCeil)
{
    float ret = value / 2;
    return isCeil ? std::ceilf(ret): ret;
}

// 印字されない文字かどうかを判定する
bool IsPrintableChar(uint32_t code)
{
    // http://unicode.org/reports/tr9/ に記載されている双方向テキストは
    // 印字されない文字として扱う。

    // ALM はアラビア文字用で通常はテキストに入り込まないためチェックしない。
    // if (code == 0x061c /*ALM*/)
    // {
    //     return true;
    // }

    // 高速化のために 0x20XX かどうかで一旦ふるいにかける。
    if ((code & 0xffffff00) == 0x2000)
    {
        if ((0x200e <= code && code <= 0x200f) /*LRM,RLM*/ ||
            (0x202a <= code && code <= 0x202e) /*LRE,RLE,PDF,LRO,RLO*/ ||
            (0x2066 <= code && code <= 0x2069) /*LRI,RLI,FSI,PDI*/)
        {
            return false;
        }
    }

    return true;
}

}   // namespace


/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

template <typename CharType>
TextWriterBase<CharType>::TextWriterBase()
: CharWriter()
,m_WidthLimit(std::numeric_limits<float>::max())
,m_CharSpace(0)
,m_LineSpace(0)
,m_WidthLimitOffset(0)
,m_TabWidth (4)
,m_DrawFlag(DefaultDrawFlag)
,m_pTagProcessor(&g_DefaultTagProcessor)
,m_CenterCeilingEnabled(false)
,m_LinefeedKerningEnabled(false)
{
}

template <typename CharType>
TextWriterBase<CharType>::~TextWriterBase()
{
}





/* ------------------------------------------------------------------------
        行間/文字間/タブ幅
   ------------------------------------------------------------------------ */

template <typename CharType>
void
TextWriterBase<CharType>::SetLineHeight(float height)
{
    const Font* pFont = GetFont();
    const int linefeed = pFont != NULL ? pFont->GetLineFeed(): 0;

    m_LineSpace = height - linefeed * GetScaleY();
}

template <typename CharType>
float
TextWriterBase<CharType>::GetLineHeight() const
{
    const Font* pFont = GetFont();
    const int linefeed = pFont != NULL ? pFont->GetLineFeed(): 0;

    return linefeed * GetScaleY() + m_LineSpace;
}






/* ------------------------------------------------------------------------
        文字列サイズ計算
   ------------------------------------------------------------------------ */

template <typename CharType>
float
TextWriterBase<CharType>::CalculateFormatStringWidth(
    const CharType*  pFormat,
    ...
) const
{
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    nn::font::Rectangle rect;
    std::va_list        vargs;
    va_start(vargs, pFormat);

    CalculateVStringRect(&rect, pFormat, vargs);

    va_end(vargs);
    return rect.GetWidth();
}

template <typename CharType>
float
TextWriterBase<CharType>::CalculateFormatStringHeight(
    const CharType*  pFormat,
    ...
) const
{
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    nn::font::Rectangle rect;
    std::va_list        vargs;
    va_start(vargs, pFormat);

    CalculateVStringRect(&rect, pFormat, vargs);

    va_end(vargs);
    return rect.GetHeight();
}

template <typename CharType>
void
TextWriterBase<CharType>::CalculateFormatStringRect(
    nn::font::Rectangle*    pRect,
    const CharType*         pFormat,
    ...
) const
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    std::va_list vargs;
    va_start(vargs, pFormat);

    CalculateVStringRect(pRect, pFormat, vargs);

    va_end(vargs);
}

template <typename CharType>
void
TextWriterBase<CharType>::CalculateVStringRect(
    nn::font::Rectangle*    pRect,
    const CharType*         pFormat,
    std::va_list            args
) const
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    CharType buffer[DefaultFormatBufferSize];
    int length = this->VSNPrintf(buffer, DefaultFormatBufferSize, pFormat, args);
    length = std::min(length, static_cast<int>(DefaultFormatBufferSize - 1));

    CalculateStringRect(pRect, buffer, length);
}

template <typename CharType>
float
TextWriterBase<CharType>::CalculateStringWidth(
    const CharType* pStr,
    int         length
) const
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    nn::font::Rectangle rect;

    CalculateStringRect(&rect, pStr, length);

    return rect.GetWidth();
}

template <typename CharType>
float
TextWriterBase<CharType>::CalculateStringHeight(
    const CharType* pStr,
    int         length
) const
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    nn::font::Rectangle rect;

    CalculateStringRect(&rect, pStr, length);

    return rect.GetHeight();
}

template <typename CharType>
void
TextWriterBase<CharType>::CalculateStringRect(
    nn::font::Rectangle*    pRect,
    const CharType*         pStr,
    int                     length
) const
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_FONT_MIN_ASSERT(length, 0);

    if (pStr == NULL)
    {
        pRect->left = 0.0f;
        pRect->top = 0.0f;
        pRect->right = 0.0f;
        pRect->bottom = 0.0f;
        return;
    }

    TextWriterBase<CharType> myCopy = *this;

    m_pTagProcessor->BeginCalculateRectWhole(this, pStr, pStr + length);

    myCopy.CalculateStringRectImpl(pRect, pStr, length);

    m_pTagProcessor->EndCalculateRectWhole(this, pStr, pStr + length);
}






/* ------------------------------------------------------------------------
        文字列描画
   ------------------------------------------------------------------------ */

template <typename CharType>
float
TextWriterBase<CharType>::Printf(
    const CharType* pFormat,
    ...
)
{
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    std::va_list vargs;
    va_start(vargs, pFormat);

    float width = VPrintf(pFormat, vargs);

    va_end(vargs);
    return width;
}

template <typename CharType>
float
TextWriterBase<CharType>::VPrintf(
    const CharType* pFormat,
    std::va_list    args
)
{
    NN_SDK_ASSERT_NOT_NULL(pFormat);

    CharType buffer[DefaultFormatBufferSize];
    int length = this->VSNPrintf(buffer, DefaultFormatBufferSize, pFormat, args);
    length = std::min(length, static_cast<int>(DefaultFormatBufferSize - 1));

    float width = Print(buffer, length);

    return width;
}

template <typename CharType>
float
TextWriterBase<CharType>::Print(
    const CharType* pStr,
    int         length
)
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    return Print(pStr, length, 0, NULL, NULL);
}

template <typename CharType>
float
TextWriterBase<CharType>::Print(
    const CharType* pStr,
    int         length,
    int             lineWidthOffsetCount,
    const float*    pLineOffset,
    const float*    pLineWidth
)
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    TextWriterBase<CharType> myCopy = *this;

    float width = myCopy.PrintImpl(pStr, length, lineWidthOffsetCount, pLineOffset, pLineWidth);

    SetCursor( myCopy.GetCursorX(), myCopy.GetCursorY() );

    return width;
}

/* =======================================================================
        private
   ======================================================================== */

/* ------------------------------------------------------------------------
        文字列処理
   ------------------------------------------------------------------------ */

template <typename CharType>
float
TextWriterBase<CharType>::CalculateLineWidth(
    StreamType  pStr,
    int         length
)
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    nn::font::Rectangle rect;
    TextWriterBase<CharType> myCopy = *this;

    myCopy.SetCursor(0, 0);
    myCopy.CalculateLineRectImpl(&rect, &pStr, length);

    return rect.GetWidth();
}

template <typename CharType>
const CharType*
TextWriterBase<CharType>::FindPosOfWidthLimit(
    const CharType* pStr,
    int length) const
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    nn::font::Rectangle rect;
    TextWriterBase<CharType> myCopy = *this;

    myCopy.SetCursor(0, 0);
    myCopy.CalculateLineRectImpl(&rect, &pStr, length);

    return pStr;
}

template <typename CharType>
bool
TextWriterBase<CharType>::CalculateLineRectImpl(
    nn::font::Rectangle*    pRect,
    StreamType*             pStr,
    int                     length
)
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_SDK_ASSERT_NOT_NULL(*pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    const StreamType pString        = *pStr;
    const StreamType pEnd           = pString + length;
    const bool bUseLimit            = m_WidthLimit < std::numeric_limits<float>::max();
    PrintContext<CharType> context(this, pString, pEnd, 0, 0, GetScaleX(), GetScaleY(), 0);
    float limitLeft                   = 0;
    float limitRight                  = 0;
    bool bCharSpace                 = false;
    bool bOverLimit                 = false;
    StreamType pPrevStreamPos        = NULL;
    nn::font::Rectangle             prevRect;

    /*
        文字列の途中でエンコーディングが変わる事は想定していない
    */
    NN_SDK_ASSERT_NOT_NULL(GetFont());
    CharStrmReader reader = GetFont()->GetCharStrmReader(CharType(0));

    const float fontHeight = context.writer->GetFontHeight();
    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = std::min(0.0f, fontHeight);
    pRect->bottom   = std::max(0.0f, fontHeight);
    prevRect = *pRect;

    reader.Set(pString);
    pPrevStreamPos = NULL;

    m_pTagProcessor->BeginCalculateRect(&context);

    uint32_t code = reader.Next();
    while (reinterpret_cast<StreamType>(reader.GetCurrentPos()) <= pEnd)
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;

            // TODO: m_LinefeedKerningEnabled の削除。
            // m_LinefeedKerningEnabled は過去との互換モードで実行できるために念のために残してあり、
            // アプリ側での問題ないことが確認でき次第削除を予定しています。
            if (code == '\n' && !m_LinefeedKerningEnabled)
            {
                // 行末の palt 処理
                limitRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleX();
                // 改行時に前後の文字のカーニングが引き継がれないようにリセットする
                context.prevCode = 0;
            }

            nn::font::Rectangle rect(limitRight, 0, 0, 0);

            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: ContextFlag_NoCharSpace;
            SetCursorX(limitRight);

            //---- 折り返しの判定
            if ( bUseLimit
              && code != '\n'
              && pPrevStreamPos != NULL
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                nn::font::Rectangle rect2;

                context2.writer = &myCopy;
                m_pTagProcessor->CalculateRect(&rect2, &context2, code);

                if ( rect2.GetWidth() > 0.0f
                  && (myCopy.GetCursorX() - context.xOrigin > m_WidthLimit)
                )
                {
                    bOverLimit = true;
                    code       = '\n';
                    reader.Set(pPrevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = m_pTagProcessor->CalculateRect(&rect, &context, code);

            NN_SDK_ASSERT_NOT_NULL(context.str);
            reader.Set(context.str);

            pRect->left     = std::min(pRect->left,      rect.left);
            pRect->top      = std::min(pRect->top,       rect.top);
            pRect->right    = std::max(pRect->right,     rect.right);
            pRect->bottom   = std::max(pRect->bottom,    rect.bottom);
            limitRight = GetCursorX();

            if (operation == TagProcessor::Operation_EndDraw)
            {
                //---- 全部読み進んだ事にする
                *pStr += length;
                return false;
            }
            else if (operation == TagProcessor::Operation_NoCharSpace)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::Operation_CharSpace)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::Operation_NextLine)
            {
                break;
            }
        }
        else if (IsPrintableChar(code))
        {
            //---- 通常の文字

            float crntRight = limitRight;

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += GetFont()->GetCharWidth(code) * GetScaleX();
            }
            crntRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, code) * GetScaleX());

            float kerningLast = static_cast<float>(GetFont()->GetKerning(code, 0) * GetScaleX());
            crntRight += kerningLast;

            //---- 折り返しの判定
            if (bUseLimit && pPrevStreamPos != NULL)
            {
                float width = crntRight - limitLeft;
                if (width + m_WidthLimitOffset > m_WidthLimit)
                {
                    bOverLimit = true;
                    code       = '\n';
                    reader.Set(pPrevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;
            pRect->left  = std::min(pRect->left,  limitRight);
            pRect->right = std::max(pRect->right, limitRight);
            limitRight -= kerningLast;
            bCharSpace = true;
            context.prevCode = code;
        }

        if (bUseLimit)
        {
            pPrevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }

    // 末尾の文字のカーニングの適用
    {
        limitRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleX();
        pRect->left = std::min(pRect->left, limitRight);
        pRect->right = std::max(pRect->right, limitRight);
    }

    *pStr = reinterpret_cast<StreamType>(reader.GetCurrentPos());

    m_pTagProcessor->EndCalculateRect(&context);

    return bOverLimit;
}// NOLINT(impl/function_size)

template <typename CharType>
void
TextWriterBase<CharType>::CalculateStringRectImpl(
    nn::font::Rectangle*    pRect,
    StreamType              pStr,
    int                     length
)
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    const StreamType pEnd = pStr + length;
    int remain = length;
    StreamType pos = pStr;

    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = 0;
    pRect->bottom   = 0;

    SetCursor(0, 0);

    do
    {
        nn::font::Rectangle rect;
        CalculateLineRectImpl(&rect, &pos, remain);
        remain   = static_cast<int>(pEnd - pos);

        pRect->left     = std::min(pRect->left,      rect.left);
        pRect->top      = std::min(pRect->top,       rect.top);
        pRect->right    = std::max(pRect->right,     rect.right);
        pRect->bottom   = std::max(pRect->bottom,    rect.bottom);
    } while (remain > 0);
}

template <typename CharType>
float
TextWriterBase<CharType>::PrintImpl(
    StreamType  pStr,
    int         length,
    int         lineWidthOffsetCount,
    const float*    pLineOffset,
    const float*    pLineWidth
)
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_SDK_ASSERT_NOT_NULL(GetFont());
    NN_FONT_MIN_ASSERT(length, 0);

    float xOrigin    = GetCursorX();
    float yOrigin    = GetCursorY();
    float limitLeft  = 0;
    float limitRight = 0;
    const bool bUseLimit = m_WidthLimit < std::numeric_limits<float>::max();
    const float orgCursorY = yOrigin;
    bool bCharSpace  = false;
    StreamType pPrevStreamPos = pStr;
    StreamType pPrevNewLinePos = pStr;
    int line = 0;

    m_pTagProcessor->BeginPrintWhole(this, pStr, pStr + length);

    float textWidth  = AdjustCursor(&xOrigin, &yOrigin, pStr, length);
    float yCursorAdj = orgCursorY - GetCursorY();

    PrintContext<CharType> context(this, pStr, pStr + length, xOrigin, yOrigin, GetScaleX(), GetScaleY(), 0);
    if (lineWidthOffsetCount != 0)
    {
        float lineOffset = GetLineOffset(lineWidthOffsetCount, pLineOffset, line);
        SetCursorX(lineOffset);
    }
    CharStrmReader reader = GetFont()->GetCharStrmReader(CharType(0));
    reader.Set(pStr);

    m_pTagProcessor->BeginPrint(&context);

    uint32_t code = reader.Next();
    while (reinterpret_cast<StreamType>(reader.GetCurrentPos()) - pStr <= length)
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;
            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: ContextFlag_NoCharSpace;

            //---- 折り返しの判定
            // 行ごとの幅とオフセットが設定されていない行で 1 文字目の場合は改行処理を飛ばします。
            if ( bUseLimit
              && code != '\n'
              && (line < lineWidthOffsetCount || pPrevStreamPos != pPrevNewLinePos)
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                nn::font::Rectangle rect;

                context2.writer = &myCopy;
                operation = m_pTagProcessor->CalculateRect(&rect, &context2, code);

                if ( rect.GetWidth() > 0.0f
                  && myCopy.GetCursorX() - context.xOrigin > m_WidthLimit + GetLineWidth(lineWidthOffsetCount, pLineWidth, line)
                )
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(pPrevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = m_pTagProcessor->Process(code, &context);
            limitRight = GetCursorX() - xOrigin;

            // TODO: m_LinefeedKerningEnabled の削除。
            // m_LinefeedKerningEnabled は過去との互換モードで実行できるために念のために残してあり、
            // アプリ側での問題ないことが確認でき次第削除を予定しています。
            if (code == '\n' && !m_LinefeedKerningEnabled)
            {
                // 改行時に前後の文字のカーニングが引き継がれないようにリセットする
                context.prevCode = 0;
            }

            if (operation == TagProcessor::Operation_NextLine)
            {
                NN_SDK_ASSERT_NOT_NULL(context.str);

                line++;
                float lineOffset = GetLineOffset(lineWidthOffsetCount, pLineOffset, line);

                //---- 次行描画開始位置Xの補正
                if (IsDrawFlagSet(PositionFlag_HorizontalAlignMask, PositionFlag_HorizontalAlignCenter))
                {
                    const int   remain  = length - static_cast<int>(context.str - pStr);
                    const float   width   = CalculateLineWidth( context.str, remain );
                    const float   offset  = AdjustCenterValue(textWidth, m_CenterCeilingEnabled) - AdjustCenterValue(width, m_CenterCeilingEnabled);
                    SetCursorX( context.xOrigin + offset + lineOffset );
                }
                else if (IsDrawFlagSet(PositionFlag_HorizontalAlignMask, PositionFlag_HorizontalAlignRight))
                {
                    const int   remain  = length - static_cast<int>(context.str - pStr);
                    const float   width   = CalculateLineWidth( context.str, remain );
                    const float   offset  = textWidth - width;
                    SetCursorX( context.xOrigin + offset + lineOffset );
                }
                else
                {
                    //---- 最大幅の更新
                    const float width = GetCursorX() - context.xOrigin;
                    textWidth = std::max(textWidth, width);

                    SetCursorX( context.xOrigin + lineOffset );
                }

                if (bUseLimit)
                {
                    pPrevNewLinePos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
                }
                bCharSpace = false;
            }
            else if (operation == TagProcessor::Operation_NoCharSpace)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::Operation_CharSpace)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::Operation_EndDraw)
            {
                break;
            }

            NN_SDK_ASSERT_NOT_NULL(context.str);
            reader.Set(context.str);
        }
        else if (IsPrintableChar(code))
        {
            //---- 通常の文字

            const float baseY = GetCursorY();

            float crntRight = limitRight;
            float kerning = static_cast<float>(GetFont()->GetKerning(context.prevCode, code)) * GetScaleX();
            Glyph glyph;
            GetFont()->GetGlyph(&glyph, code);

            if (glyph.height > glyph.texHeight)
            {
                // セルの高さが TextureCache の高さを超えることはまず無いため
                // 不正な文字として警告を出力し、処理を飛ばす。
                NN_DETAIL_FONT_WARN("The 0x%04x character's height (%d) which is over the texture cache's height (%d) is wrong.\n", code, glyph.height, glyph.texHeight);
                code = reader.Next();
                continue;
            }

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += glyph.widths.charWidth * GetScaleX();
            }
            crntRight += kerning;

            float kerningLast = static_cast<float>(GetFont()->GetKerning(code, 0)) * GetScaleX();
            crntRight += kerningLast;

            //---- 折り返しの判定
            // 行ごとの幅とオフセットが設定されていない行で 1 文字目の場合は改行処理を飛ばします。
            if (bUseLimit && (line < lineWidthOffsetCount || pPrevStreamPos != pPrevNewLinePos))
            {
                float width = crntRight - limitLeft;
                if (width > m_WidthLimit + GetLineWidth(lineWidthOffsetCount, pLineWidth, line))
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(pPrevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;
            limitRight -= kerningLast;


            if (bCharSpace)
            {
                MoveCursorX( GetCharSpace() );
            }
            bCharSpace = true;

            MoveCursorX( kerning );

            //---- カーソルはベースラインにあるのでセル上端へ移動する
            {
                const Font* pFont = GetFont();
                const float adj = -(pFont->GetBaselinePos() + static_cast<int>(glyph.baselineDifference)) * GetScaleY();
                MoveCursorY( adj );
            }

            PrintGlyph(glyph);

            // 戻す
            SetCursorY( baseY );

            context.prevCode = code;
        }

        if (bUseLimit)
        {
            pPrevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }

    // 末尾の文字のカーニングの適用
    {
        float kerning = static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleX();
        MoveCursorX( kerning );
    }

    //---- 最大幅の更新
    {
        const float width = GetCursorX() - context.xOrigin;
        textWidth = std::max(textWidth, width);
    }

    //---- カーソル位置Yを描画フラグに従って移動
    // AdjustCursorでベースライン上に移動したカーソル位置を元に戻す
    if ( IsDrawFlagSet(PositionFlag_VerticalOriginMask, PositionFlag_VerticalOriginMiddle)
      || IsDrawFlagSet(PositionFlag_VerticalOriginMask, PositionFlag_VerticalOriginBottom)
    )
    {
        SetCursorY(orgCursorY);
    }
    else
    {
        MoveCursorY(yCursorAdj);
    }

    m_pTagProcessor->EndPrint(&context);
    m_pTagProcessor->EndPrintWhole(this, pStr, pStr + length);

    return textWidth;
}// NOLINT(impl/function_size)

template <typename CharType>
float
TextWriterBase<CharType>::AdjustCursor(
    float*        pXOrigin,
    float*        pYOrigin,
    StreamType  pStr,
    int         length
)
{
    float textWidth  = 0;
    float textHeight = 0;

    {
        uint32_t  flagMask  = PositionFlag_HorizontalAlignMask | PositionFlag_HorizontalOriginMask | PositionFlag_VerticalOriginMask;
        uint32_t  blineFlag = PositionFlag_HorizontalAlignLeft | PositionFlag_HorizontalOriginLeft | PositionFlag_VerticalOriginBaseLine;
        uint32_t  topFlag   = PositionFlag_HorizontalAlignLeft | PositionFlag_HorizontalOriginLeft | PositionFlag_VerticalOriginTop;

        if ( ! IsDrawFlagSet(flagMask, blineFlag)
          && ! IsDrawFlagSet(flagMask, topFlag)
        )
        {
            nn::font::Rectangle textRect;

            NN_FONT_MIN_ASSERT(length, 0);
            if (pStr == NULL)
            {
                textRect.left = 0.0f;
                textRect.top = 0.0f;
                textRect.right = 0.0f;
                textRect.bottom = 0.0f;
            }
            else
            {
                TextWriterBase<CharType> myCopy = *this;
                myCopy.CalculateStringRectImpl(&textRect, pStr, length);
            }

            textWidth  = textRect.left + textRect.right;
            textHeight = textRect.top  + textRect.bottom;
        }
    }

    //---- 描画開始位置Xの補正
    // 各々の位置から文字列の左端に統一
    if (IsDrawFlagSet(PositionFlag_HorizontalOriginMask, PositionFlag_HorizontalOriginCenter))
    {
        *pXOrigin -= AdjustCenterValue(textWidth, m_CenterCeilingEnabled);
    }
    else if (IsDrawFlagSet(PositionFlag_HorizontalOriginMask, PositionFlag_HorizontalOriginRight))
    {
        *pXOrigin -= textWidth;
    }

    //---- 描画開始位置Yの補正
    // 各々の位置から 1 行目のアセンダラインに統一
    if (IsDrawFlagSet(PositionFlag_VerticalOriginMask, PositionFlag_VerticalOriginMiddle))
    {
        *pYOrigin -= AdjustCenterValue(textHeight, m_CenterCeilingEnabled);
    }
    else if (IsDrawFlagSet(PositionFlag_VerticalOriginMask, PositionFlag_VerticalOriginBottom))
    {
        *pYOrigin -= textHeight;
    }

    //---- 1行目描画開始位置Xの補正
    // 文字列の左端からそれぞれの寄せ位置へカーソルを移動
    if (IsDrawFlagSet(PositionFlag_HorizontalAlignMask, PositionFlag_HorizontalAlignCenter))
    {
        const float width = CalculateLineWidth(pStr, length);
        const float offset = AdjustCenterValue(textWidth, m_CenterCeilingEnabled) - AdjustCenterValue(width, m_CenterCeilingEnabled);
        SetCursorX( *pXOrigin + offset );
    }
    else if (IsDrawFlagSet(PositionFlag_HorizontalAlignMask, PositionFlag_HorizontalAlignRight))
    {
        const float width = CalculateLineWidth(pStr, length);
        const float offset = textWidth - width;
        SetCursorX( *pXOrigin + offset );
    }
    else
    {
        SetCursorX( *pXOrigin );
    }

    //---- 1行目描画開始位置Yの補正
    // 1 行目のベースライン位置にカーソルを移動
    if (IsDrawFlagSet(PositionFlag_VerticalOriginMask, PositionFlag_VerticalOriginBaseLine))
    {
        // pYOrigin はベースラインであるのでそのままセット
        SetCursorY( *pYOrigin );
    }
    else
    {
        // pYOrigin はアセンダラインであるのでベースラインにカーソルを移動
        SetCursorY( *pYOrigin + GetFontAscent() );
    }

    return textWidth;
}

/*---------------------------------------------------------------------------*/
template <typename CharType>
float TextWriterBase<CharType>::GetLineOffset(int lineOffsetCount, const float* pLineOffset, int line) const
{
    if (line >= lineOffsetCount)
    {
        return 0.0f;
    }
    return pLineOffset[line];
}

/*---------------------------------------------------------------------------*/
template <typename CharType>
float TextWriterBase<CharType>::GetLineWidth(int lineWidthCount, const float* pLineWidth, int line) const
{
    if (line >= lineWidthCount)
    {
        return 0.0f;
    }
    return pLineWidth[line];
}

template <typename CharType>
void TextWriterBase<CharType>::UpdateTextWriterWithTags(const CharType* pStr, int length)
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    NN_FONT_MIN_ASSERT(length, 0);

    // TextWriter の状態を更新する目的のため、取得した範囲は使いません。
    nn::font::Rectangle rect;
    CalculateStringRectImpl(&rect, pStr, length);
}

/* =======================================================================
        明示的実体化
   ======================================================================== */

template class TextWriterBase<char>;
template class TextWriterBase<uint16_t>;



}   // namespace font
}   // namespace nn

