﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/font/font_TagProcessorBase.h>
#include <nn/font/font_TextWriterBase.h>


namespace nn {
namespace font {

namespace detail {

const char* AcquireNextPrinableCharImpl(bool* pIsPrintable, const char* position)
{
    char buffer[4] = { 0 };
    nn::util::CharacterEncodingResult result;
    result = nn::util::PickOutCharacterFromUtf8String(buffer, &position);
    NN_SDK_ASSERT(result == nn::util::CharacterEncodingResult_Success, "Wrong character code of UTF-8.");

    *pIsPrintable = true;
    return position;
}

const uint16_t* AcquireNextPrinableCharImpl(bool* pIsPrintable, const uint16_t* position)
{
    *pIsPrintable = true;
    return position + 1;
}

} // namespace nn::font::detail

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

template <typename CharType>
TagProcessorBase<CharType>::TagProcessorBase()
{
}

template <typename CharType>
TagProcessorBase<CharType>::~TagProcessorBase()
{
}

/* ------------------------------------------------------------------------
        タグ処理
   ------------------------------------------------------------------------ */

template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::Process(
    uint32_t              code,
    PrintContext<CharType>* pContext
)
{
    NN_SDK_ASSERT(code < ' ');
    NN_SDK_ASSERT_NOT_NULL(pContext);

    switch (code)
    {
    case '\n':
        ProcessLinefeed(pContext);
        return Operation_NextLine;

    case '\t':
        ProcessTab(pContext);
        return Operation_NoCharSpace;

    default:
        // nothing to do;
        break;
    }

    return Operation_Default;
}

template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::CalculateRect(
    nn::font::Rectangle*    pRect,
    PrintContext<CharType>* pContext,
    uint32_t                code
)
{
    NN_SDK_ASSERT_NOT_NULL(pRect);
    NN_SDK_ASSERT(code < ' ');
    NN_SDK_ASSERT_NOT_NULL(pContext);

    switch (code)
    {
    case '\n':
        {
            TextWriterBase<CharType>& writer = *pContext->writer;

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();

            ProcessLinefeed(pContext);

            pRect->left     = writer.GetCursorX();

            // ProcessLinefeed はカーソルを移動するだけで次の行の高さは含まれていないので
            // 次の行の高さとして FontHeight を足す。
            pRect->bottom   = writer.GetCursorY() + pContext->writer->GetFontHeight();

            pRect->Normalize();
        }
        return Operation_NextLine;

    case '\t':
        {
            TextWriterBase<CharType>& writer = *pContext->writer;

            pRect->left     = writer.GetCursorX();

            ProcessTab(pContext);

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();
            pRect->bottom   = pRect->top + writer.GetFontHeight();

            pRect->Normalize();
        }
        return Operation_NoCharSpace;

    default:
        // nothing to do;
        break;
    }

    return Operation_Default;
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginPrint(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndPrint(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginCalculateRect(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndCalculateRect(PrintContext<CharType>* /* pContext */)
{
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginPrintWhole(const TextWriterBase<CharType>* pTextWriter, const CharType* pString, const CharType* pEnd)
{
    NN_UNUSED(pTextWriter);
    NN_UNUSED(pString);
    NN_UNUSED(pEnd);
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndPrintWhole(const TextWriterBase<CharType>* pTextWriter, const CharType* pString, const CharType* pEnd)
{
    NN_UNUSED(pTextWriter);
    NN_UNUSED(pString);
    NN_UNUSED(pEnd);
}

template <typename CharType>
void
TagProcessorBase<CharType>::BeginCalculateRectWhole(const TextWriterBase<CharType>* pTextWriter, const CharType* pString, const CharType* pEnd)
{
    NN_UNUSED(pTextWriter);
    NN_UNUSED(pString);
    NN_UNUSED(pEnd);
}

template <typename CharType>
void
TagProcessorBase<CharType>::EndCalculateRectWhole(const TextWriterBase<CharType>* pTextWriter, const CharType* pString, const CharType* pEnd)
{
    NN_UNUSED(pTextWriter);
    NN_UNUSED(pString);
    NN_UNUSED(pEnd);
}

template <typename CharType>
const CharType*
TagProcessorBase<CharType>::AcquireNextPrintableChar(bool* pIsPrintable, const CharType* position)
{
    return detail::AcquireNextPrinableCharImpl(pIsPrintable, position);
}

/* ------------------------------------------------------------------------
        private タグ処理
   ------------------------------------------------------------------------ */
template <typename CharType>
void
TagProcessorBase<CharType>::ProcessLinefeed(ContextType* pContext) const
{
    NN_SDK_ASSERT_NOT_NULL(pContext);

    TextWriterBase<CharType>& writer = *pContext->writer;

    const float x   = pContext->xOrigin;
    const float y   = writer.GetCursorY() + writer.GetLineHeight();

    writer.SetCursor(x, y);
}

template <typename CharType>
void
TagProcessorBase<CharType>::ProcessTab(ContextType* pContext) const
{
    NN_SDK_ASSERT_NOT_NULL(pContext);

    TextWriterBase<CharType>& writer = *pContext->writer;
    const int tabWidth = writer.GetTabWidth();

    if (tabWidth > 0)
    {
        const float aCharWidth  = writer.IsWidthFixed() ? writer.GetFixedWidth(): writer.GetFontWidth();
        const float dx          = writer.GetCursorX() - pContext->xOrigin;
        const float tabPixel    = tabWidth * aCharWidth;
        const int numTab      = static_cast<int>(dx / tabPixel) + 1;
        const float x           = pContext->xOrigin + tabPixel * numTab;

        writer.SetCursorX(x);
    }
}




/* ------------------------------------------------------------------------
        明示的実体化
   ------------------------------------------------------------------------ */

template class TagProcessorBase<char>;
template class TagProcessorBase<uint16_t>;


}   // namespace font
}   // namespace nn
