﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdint.h>

#include <nn/font/font_ScalableFontUtil.h>
#include <nn/font/font_ScalableFont.h>
#include <nn/ui2d/ui2d_TextBox.h>
#include <nn/font/font_PairFont.h>

namespace nn
{
namespace font
{

namespace detail
{

// RegisterGlyphInScalableFont に渡すパラメータ
struct RegisterGlyphInScalableFontParam
{
    const nn::ui2d::TextBox* pTextBox;
    int lockGroup;
    uint32_t count;
};

// 見つけたスケーラブルフォントにテキストボックスのグリフを登録するコールバック関数
void RegisterGlyphInScalableFont(const ScalableFont* pFont, void* pParam)
{
    RegisterGlyphInScalableFontParam* pCallbackParam = static_cast<RegisterGlyphInScalableFontParam*>(pParam);
    const nn::ui2d::TextBox* pTextBox = pCallbackParam->pTextBox;
    if (pTextBox->GetUtf8())
    {
        const char* string = pTextBox->GetStringUtf8();
        if (string != NULL)
        {
            const char* end = string + pTextBox->GetStringLength();
            while (string < end)
            {
                uint32_t c = nn::font::ConvertCharUtf8ToUtf32AndIncrement(&string);
                if (c < ' ')
                {
                    continue;
                }
                if (pFont->RegisterGlyph(c, pCallbackParam->lockGroup))
                {
                    pCallbackParam->count++;
                }
            }
        }
    }
    else
    {
        const uint16_t* string = pTextBox->GetString();
        if (string != NULL)
        {
            int length = static_cast<int>(pTextBox->GetStringLength());
            for (int i = 0; i < length; i++)
            {
                uint32_t c = string[i];
                if (c < ' ')
                {
                    continue;
                }
                if (pFont->RegisterGlyph(c, pCallbackParam->lockGroup))
                {
                    pCallbackParam->count++;
                }
            }
        }
    }
}

}

void SearchScalableFont(const Font* pFont, void(*pCallback)(const ScalableFont*, void*), void* pParam)
{
    const ScalableFont* pScalableFont = nn::font::DynamicCast<const ScalableFont*>(pFont);
    if (pScalableFont)
    {
        pCallback(pScalableFont, pParam);
        return;
    }

    const PairFont* pPairFont = nn::font::DynamicCast<const PairFont*>(pFont);
    if (pPairFont)
    {
        SearchScalableFont(pPairFont->GetFontFirst(), pCallback, pParam);
        SearchScalableFont(pPairFont->GetFontSecond(), pCallback, pParam);
    }
}

uint32_t RegisterGlyphFromTextBox(const nn::ui2d::TextBox* pTextBox, int lockGroup)
{
    detail::RegisterGlyphInScalableFontParam param;
    param.pTextBox = pTextBox;
    param.lockGroup = lockGroup;
    param.count = 0;
    SearchScalableFont(pTextBox->GetFont(), detail::RegisterGlyphInScalableFont, &param);
    return param.count;
}

uint32_t RegisterGlyphFromTextBox(const nn::ui2d::TextBox* pTextBox)
{
    return RegisterGlyphFromTextBox(pTextBox, -1);
}

uint32_t RegisterGlyphFromTextBoxRecursive(const nn::ui2d::Pane* pPane, int lockGroup)
{
    uint32_t ret = 0;
    const nn::ui2d::TextBox* pTextBox = nn::font::DynamicCast<const nn::ui2d::TextBox*>(pPane);
    if (pTextBox)
    {
        ret += RegisterGlyphFromTextBox(pTextBox, lockGroup);
    }

    nn::ui2d::PaneList::const_iterator it_end = pPane->GetChildList().end();
    for (nn::ui2d::PaneList::const_iterator it = pPane->GetChildList().begin(); it != it_end; ++it)
    {
        ret += RegisterGlyphFromTextBoxRecursive(reinterpret_cast<const nn::ui2d::Pane*>(&(*it)), lockGroup);
    }

    return ret;
}

uint32_t RegisterGlyphFromTextBoxRecursive(const nn::ui2d::Pane* pPane)
{
    return RegisterGlyphFromTextBoxRecursive(pPane, -1);
}


} // namespace nn::font
} // namespace nn
