﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/font/font_ScalableFont.h>

namespace nn
{
namespace font
{

void ScalableFont::InitializeArg::SetDefault()
{
    pTextureCache = NULL;
    fontSize = 40;
    fontFace = 0;
    isAlternateCharSpaceWithOriginalWidth = false;
    isAlternateCharSpaceWithOriginalWidthForNotReadyChar = false;
    alternateChar = '?';
    lineFeedOffset = 0;
    isDrawingHyphenMinusInsteadOfNonBreakingHyphen = true;
}

ScalableFont::ScalableFont()
: font::Font()
, m_pTextureCache(NULL)
, m_FontSize(40)
, m_FontHeight(50)
, m_FontAscent(40)
, m_FontBaselinePos(40)
, m_FontLineFeed(50)
, m_AlternateCharMode(AlternateCharMode_UseAlternateChar)
, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen(true)
, m_FontFace(0)
, m_FontAlternateChar('?')
{
    m_FontDefaultCharWidth.left = 0;
    m_FontDefaultCharWidth.glyphWidth = 40;
    m_FontDefaultCharWidth.charWidth = 40;
}

ScalableFont::~ScalableFont()
{
}

void ScalableFont::Initialize(const InitializeArg& arg)
{
    if (arg.pTextureCache == NULL)
    {
        NN_SDK_ASSERT(false, "arg.pTextureCache must be set.");
        return;
    }

    NN_SDK_ASSERT(arg.pTextureCache ->GetTextureObject()->IsDescriptorSlotForTextureReady(),
        "You must call TextureCache::RegisterTextureViewToDescriptorPool() before you use it.");

    m_pTextureCache = arg.pTextureCache;
    m_FontSize = arg.fontSize;
    m_FontFace = arg.fontFace;
    {
        const TextureCache::FontMetrics& metrics = m_pTextureCache->GetFontMetrics(m_FontFace);
        // ResFontの表示に近くなるように、切り捨てではなく四捨五入する
        m_FontAscent = static_cast<int>(metrics.ascent_ratio * m_FontSize + 0.5f);
        m_FontHeight = static_cast<int>(metrics.height_ratio * m_FontSize + 0.5f);
        m_FontBaselinePos = static_cast<int>(metrics.boundingBoxAscentRatio * m_FontSize + 0.5f);
    }
    m_FontLineFeed = m_FontHeight + arg.lineFeedOffset;
    m_FontDefaultCharWidth.left = 0;
    m_FontDefaultCharWidth.glyphWidth = static_cast<uint8_t>(m_FontSize);
    m_FontDefaultCharWidth.charWidth = static_cast<uint8_t>(m_FontSize);

    if (arg.isAlternateCharSpaceWithOriginalWidth) {
        m_AlternateCharMode = AlternateCharMode_UseWhiteSpace;
        m_FontAlternateChar = CharCodeAlternateWhiteSpace;
    } else if (arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar) {
        m_AlternateCharMode = AlternateCharMode_UseWhiteSpaceForNotReadyChar;
        m_FontAlternateChar = arg.alternateChar;
    } else {
        m_AlternateCharMode = AlternateCharMode_UseAlternateChar;
        m_FontAlternateChar = arg.alternateChar;
    }

    m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen = arg.isDrawingHyphenMinusInsteadOfNonBreakingHyphen;

    // 代替文字を登録
    RegisterAlternateCharGlyph();
}

void ScalableFont::RegisterAlternateCharGlyph() const
{
    RegisterGlyph(m_FontAlternateChar);
    m_pTextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace)->SetSystemReserved(true);
    if (m_AlternateCharMode == AlternateCharMode_UseWhiteSpaceForNotReadyChar) {
        // フォントにあってまだプロットされていない文字が表示する文字の幅の空白になるモードでは、通常の代替文字
        // の他に空白のプロットとシステム予約も必要
        RegisterGlyph(CharCodeAlternateWhiteSpace);
        nn::font::GlyphNode* pGlyphNode = m_pTextureCache->FindGlyphNode(CharCodeAlternateWhiteSpace, m_FontSize, m_FontFace);
        if (pGlyphNode != NULL)
        {
            m_pTextureCache->FindGlyphNode(CharCodeAlternateWhiteSpace, m_FontSize, m_FontFace)->SetSystemReserved(true);
        }
    }
}

int ScalableFont::GetWidth() const
{
    return m_FontSize;
}

int ScalableFont::GetHeight() const
{
    return m_FontHeight;
}

int ScalableFont::GetAscent() const
{
    return m_FontAscent;
}

int ScalableFont::GetDescent() const
{
    return m_FontHeight - m_FontAscent;
}

int ScalableFont::GetMaxCharWidth() const
{
    return m_FontSize;
}

font::Font::Type ScalableFont::GetType() const
{
    return font::Font::Type_Scalable;
}

font::TexFmt ScalableFont::GetTextureFormat() const
{
    return font::FontSheetFormat_A8;
}

int ScalableFont::GetLineFeed() const
{
    return m_FontLineFeed;
}

const font::CharWidths ScalableFont::GetDefaultCharWidths() const
{
    return m_FontDefaultCharWidth;
}

void ScalableFont::SetLineFeed(int linefeed)
{
    m_FontLineFeed = linefeed;
}

void ScalableFont::SetDefaultCharWidths(const font::CharWidths& widths)
{
    m_FontDefaultCharWidth = widths;
}

bool ScalableFont::SetAlternateChar(uint32_t c)
{
    c = ConvertNonBreakingHyphenToHyphenMinus(c, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    // 代替文字として文字幅と同じスペースを使う設定のときは、代替文字は変更不可能
    if (m_AlternateCharMode != AlternateCharMode_UseWhiteSpace && m_pTextureCache->IsGlyphExistInFont(c, m_FontFace))
    {
        m_FontAlternateChar = c;
        return true;
    }
    else
    {
        return false;
    }
}

int ScalableFont::GetCharWidth(uint32_t c) const
{
    c = ConvertNonBreakingHyphenToHyphenMinus(c, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    const GlyphNode* pNode = m_pTextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    if (pNode && ! pNode->IsPlottingOrNotInFont())
    {
        return pNode->GetAdvanceX();
    }
    else if (m_AlternateCharMode != AlternateCharMode_UseAlternateChar && m_pTextureCache->IsGlyphExistInFont(c, m_FontFace))
    {
        // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
        // フォント内にあれば、直接スケーラブルフォントエンジンから取得する
        return m_pTextureCache->CalculateCharWidth(c, m_FontSize, m_FontFace);
    }
    else
    {
        // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
        // 代替文字をテクスチャキャッシュから探して幅を返す
        pNode = m_pTextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
        return pNode->GetAdvanceX();
    }
}

const font::CharWidths ScalableFont::GetCharWidths(uint32_t c) const
{
    c = ConvertNonBreakingHyphenToHyphenMinus(c, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    font::CharWidths widths;
    const GlyphNode* pNode = m_pTextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    if (pNode == NULL || pNode->IsPlottingOrNotInFont())
    {
        if (m_AlternateCharMode != AlternateCharMode_UseAlternateChar && m_pTextureCache->IsGlyphExistInFont(c, m_FontFace))
        {
            // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
            // フォント内にあれば、直接スケーラブルフォントエンジンから取得する
            int width = m_pTextureCache->CalculateCharWidth(c, m_FontSize, m_FontFace);
            widths.left = 0;
            widths.glyphWidth = static_cast<uint8_t>(width);
            widths.charWidth = static_cast<uint8_t>(width);
            return widths;
        }
        else
        {
            // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
            // 代替文字をテクスチャキャッシュから探す
            pNode = m_pTextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
        }
    }

    widths.left = static_cast<int8_t>(pNode->GetLeftOffset());
    widths.glyphWidth = static_cast<uint8_t>(pNode->GetGlyphWidth());
    widths.charWidth = static_cast<uint8_t>(pNode->GetAdvanceX());
    return widths;
}

void ScalableFont::GetGlyph(font::Glyph* pGlyph, uint32_t c) const
{
    c = ConvertNonBreakingHyphenToHyphenMinus(c, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    GlyphNode* pNode = m_pTextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    int charWidth;
    if (pNode == NULL || ! pNode->IsUsable())
    {
        if (m_AlternateCharMode != AlternateCharMode_UseAlternateChar && m_pTextureCache->IsGlyphExistInFont(c, m_FontFace))
        {
            // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
            // フォント内にあれば、スケーラブルフォントエンジンから文字幅を取得する
            charWidth = m_pTextureCache->CalculateCharWidth(c, m_FontSize, m_FontFace);
            // 文字としては空白を使用する
            pNode = m_pTextureCache->FindGlyphNode(CharCodeAlternateWhiteSpace, m_FontSize, m_FontFace);
        }
        else
        {
            // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
            // 代替文字をテクスチャキャッシュから探す
            pNode = m_pTextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
            NN_SDK_ASSERT(pNode != NULL, "The glyph of the alternate char [0x%x] was not registered.", m_FontAlternateChar);
            charWidth = pNode->GetAdvanceX();
        }
    }
    else
    {
        charWidth = pNode->GetAdvanceX();
    }

    pGlyph->widths.left = pNode->GetLeftOffset();
    pGlyph->widths.glyphWidth = pNode->GetGlyphWidth();
    pGlyph->widths.charWidth = static_cast<uint16_t>(charWidth);
    pGlyph->widths.rawWidth = pNode->GetCacheWidth();
    // 文字の高さをそのまま渡すとテクスチャ補間の影響で下が切れてしまうため、+1する
    pGlyph->height = static_cast<uint16_t>(pNode->GetGlyphHeight() + 1);
    pGlyph->rawHeight = static_cast<uint16_t>(pNode->GetCacheHeight() + 1);
    // 文字の左座標をそのまま渡すとテクスチャ補間の影響で左にノイズが出るため、+1する
    pGlyph->cellX = pNode->GetCachePosX() + 1;
    pGlyph->cellY = pNode->GetCachePosY();
    pGlyph->pTexture = m_pTextureCache->GetTextureCacheBitMap();
    pGlyph->sheetIndex = 0;
    pGlyph->isSheetUpdated = false;
    pGlyph->texFormat = font::FontSheetFormat_A8;
    pGlyph->texWidth = static_cast<uint16_t>(m_pTextureCache->GetTextureCacheWidth());
    pGlyph->texHeight = static_cast<uint16_t>(m_pTextureCache->GetTextureCacheHeight());
    pGlyph->pTextureObject = m_pTextureCache->GetTextureObject();
    // 高さのスケールによって生じるベースラインの差分を設定する
    const uint16_t baselineDifference = static_cast<int16_t>(pNode->GetBaselineOffset() - m_FontBaselinePos);
    pGlyph->baselineDifference = baselineDifference;
    pNode->SetUsed(true);
}

bool ScalableFont::HasGlyph(uint32_t c) const
{
    c = ConvertNonBreakingHyphenToHyphenMinus(c, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    GlyphNode* pNode = m_pTextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    return (pNode && pNode->IsUsable());
}

int ScalableFont::GetKerning(uint32_t c0, uint32_t c1) const
{
    c0 = ConvertNonBreakingHyphenToHyphenMinus(c0, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    c1 = ConvertNonBreakingHyphenToHyphenMinus(c1, m_IsDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    if (IsKerningEnabled())
    {
        return m_pTextureCache->CalculateKerning(c0, c1, m_FontSize, m_FontFace);
    }
    else
    {
        return 0;
    }
}

font::CharacterCode ScalableFont::GetCharacterCode() const
{
    return font::CharacterCode_Unicode;
}

int ScalableFont::GetBaselinePos() const
{
    return m_FontBaselinePos;
}

int ScalableFont::GetCellHeight() const
{
    return m_FontHeight;
}

int ScalableFont::GetCellWidth() const
{
    return m_FontSize;
}

void ScalableFont::SetLinearFilterEnabled(bool atSmall, bool atLarge)
{
    (void)atSmall;
    (void)atLarge;
}

bool ScalableFont::IsLinearFilterEnabledAtSmall() const
{
    return true;
}

bool ScalableFont::IsLinearFilterEnabledAtLarge() const
{
    return true;
}

uint32_t  ScalableFont::GetTextureWrapFilterValue() const
{
    // 使われていない
    return 0;
}

bool ScalableFont::IsColorBlackWhiteInterpolationEnabled() const
{
    return true;
}

bool ScalableFont::IsBorderEffectEnabled() const
{
    return (m_pTextureCache) ? m_pTextureCache->IsBorderEffectEnabled(m_FontFace) : false;
}

void ScalableFont::SetColorBlackWhiteInterpolationEnabled(bool /* flag */)
{
}

} // namespace nn::font
} // namespace nn
