﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/font/font_ResFont.h>
#include <nn/font/detail/font_Log.h>

namespace nn {
namespace font {

void
ResFont::RevertResource(void* pResource)
{
    nn::font::detail::BinaryFileHeader* pFileHeader =
        reinterpret_cast<nn::font::detail::BinaryFileHeader*>(pResource);
    NN_SDK_ASSERT_NOT_NULL(pFileHeader);
    NN_SDK_ASSERT(detail::IsAligned(pFileHeader, GlyphDataAlignment));

    // 初期化済みの場合すぐ返る。
    if (pFileHeader->signature != BinFileSignatureFontResoleved) { return; }

    //----------------------------------------
    // オフセット値を戻す
    nn::font::detail::BinaryBlockHeader* pBlockHeader;
    FontInformation* pInfo = NULL;
    NN_UNUSED(pInfo); // ASSERT でだけ利用されます。
    int nBlocks = 0;

    pBlockHeader = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pFileHeader))
                    .Advance(pFileHeader->headerSize).Get<nn::font::detail::BinaryBlockHeader>();

    while (nBlocks < pFileHeader->dataBlocks)
    {
        NN_SDK_ASSERT_NOT_NULL(pBlockHeader);
        switch (pBlockHeader->kind)
        {
        //--------------------------------------------------
        // INFO ブロック
        case BinBlockSignatureFinf:
            // INFOブロックは1つでなければならない
            NN_SDK_ASSERT(pInfo == NULL);
            {
                pInfo = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                        .Advance(sizeof(*pBlockHeader)).Get<FontInformation>();

                NN_SDK_ASSERT(pInfo->fontType == FontType_FontTypeTexture || pInfo->fontType == FontType_FontTypePackedTexture);
                NN_SDK_ASSERT(pInfo->alterCharIndex != GlyphIndexNotFound);

                // pGlyph は必須
                NN_SDK_ASSERT(pInfo->pGlyph != 0);
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BinBlockSignatureTglp:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* pGlyph = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                            .Advance(sizeof(*pBlockHeader)).Get<FontTextureGlyph>();
                NN_UNUSED(pGlyph);
                NN_SDK_ASSERT(pGlyph->sheetImage != 0);
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BinBlockSignatureCwdh:
            {
                FontWidth* pWidth = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                    .Advance(sizeof(*pBlockHeader)).Get<FontWidth>();

                NN_UNUSED(pWidth);
                NN_SDK_ASSERT( pWidth->indexBegin <= pWidth->indexEnd );
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BinBlockSignatureCmap:
            {
                FontCodeMap* pCodeMap = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                        .Advance(sizeof(*pBlockHeader)).Get<FontCodeMap>();

                NN_UNUSED(pCodeMap);
                NN_SDK_ASSERT( pCodeMap->codeBegin <= pCodeMap->codeEnd );
                NN_SDK_ASSERT( (pCodeMap->mappingMethod == FontMapMethod_Direct)
                                || (pCodeMap->mappingMethod == FontMapMethod_Table)
                                || (pCodeMap->mappingMethod == FontMapMethod_Scan) );
            }
            break;

        //--------------------------------------------------
        // KRNGブロック
        case BinBlockSignatureKrng:
            {
                // ここでは何も行わない
            }
            break;

        //--------------------------------------------------
        // GLGR ブロック
        case BinBlockSignatureGlgr:
            {
                // ResFont では GLGR ブロックは参照しないので何もする必要はない
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
            NN_SDK_ASSERT(false, "The font has unknown block('%c%c%c%c').",
                (pBlockHeader->kind >> 24) & 0xFF,
                (pBlockHeader->kind >> 16) & 0xFF,
                (pBlockHeader->kind >>  8) & 0xFF,
                (pBlockHeader->kind >>  0) & 0xFF
                );
            return;
        }

        pBlockHeader = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader)).
                        Advance(pBlockHeader->size).Get<nn::font::detail::BinaryBlockHeader>();
        nBlocks++;
    }

    //---- シグネチャを変更して元に戻す
    pFileHeader->signature = BinFileSignatureFont;
}

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

ResFont::ResFont()
: ResFontBase()
{
}

ResFont::~ResFont()
{
    NN_SDK_ASSERT(IsManaging(NULL));
}

void
ResFont::Finalize(nn::gfx::Device* pDevice)
{
    if (!IsManaging(NULL))
    {
        RemoveResource(pDevice);
    }
}


/* ------------------------------------------------------------------------
        構築
   ------------------------------------------------------------------------ */

bool
ResFont::SetResource(
    nn::gfx::Device*
    pDevice, void* pBfntData,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize)
{
    NN_SDK_ASSERT_NOT_NULL(pBfntData);
    NN_SDK_ASSERT(detail::IsAligned(pBfntData, GlyphDataAlignment));

    FontInformation* pFontInfo = NULL;
    nn::font::detail::BinaryFileHeader* pFileHeader =
        reinterpret_cast<nn::font::detail::BinaryFileHeader*>(pBfntData);

    //---- 既にリソースがセットされていないか
    if (! IsManaging(NULL))
    {
        NN_DETAIL_FONT_WARN("Font resource already atached.");
        return false;
    }

    //---- 正しいファイルヘッダを持っているか

    // サポートしてるバージョンかチェック
    if (! IsValidBinaryFile(pFileHeader, BinFileSignatureFont, FontFileVersion, 2))
    {
        // サポートしていない形式
        NN_SDK_ASSERT(false, "Invalid font resource.");
        return false;
    }

    // 再構築する
    pFontInfo = Rebuild(pFileHeader);
    SetResourceOffsetBasePointer(reinterpret_cast<uint8_t *>(pFileHeader));

    if (pFontInfo == NULL)
    {
        return false;
    }

    SetResourceBuffer(pBfntData, pFontInfo, static_cast<nn::font::FontKerningTable *>(FindBlock(pFileHeader, BinBlockSignatureKrng)), pMemoryPool, memoryPoolOffset, memoryPoolSize);

    GenTextureNames(pDevice);

    return true;
}

void*
ResFont::RemoveResource(nn::gfx::Device* pDevice)
{
    if (IsManaging(NULL))
    {
        NN_DETAIL_FONT_WARN("ResFont::RemoveResource(): Res font is not loaded.\n");
        return 0;
    }

    DeleteTextureNames(pDevice);

    return RemoveResourceBuffer();
}

void
ResFont::Unrelocate(void* pBfntData)
{
    NN_SDK_ASSERT_NOT_NULL(pBfntData);

    // static メソッドのため ResFontBase の機能が使用できない。
    // 自力で解析してテクスチャを探す。

    nn::font::detail::BinaryFileHeader* pFileHeader =
        reinterpret_cast<nn::font::detail::BinaryFileHeader*>(pBfntData);

    nn::font::detail::BinaryBlockHeader* pBlockHeader;
    FontInformation* pInfo = NULL;
    int nBlocks = 0;

    pBlockHeader = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pFileHeader))
                    .Advance(pFileHeader->headerSize).Get<nn::font::detail::BinaryBlockHeader>();

    while (nBlocks < pFileHeader->dataBlocks)
    {
        NN_SDK_ASSERT_NOT_NULL(pBlockHeader);
        switch (pBlockHeader->kind)
        {
        //--------------------------------------------------
        // INFO ブロック
        case BinBlockSignatureFinf:
            // INFOブロックは1つでなければならない
            NN_SDK_ASSERT(pInfo == NULL);
            {
                pInfo = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                        .Advance(sizeof(*pBlockHeader)).Get<FontInformation>();

                NN_SDK_ASSERT(pInfo->fontType == FontType_FontTypeTexture || pInfo->fontType == FontType_FontTypePackedTexture);
                NN_SDK_ASSERT(pInfo->alterCharIndex != GlyphIndexNotFound);

                // pGlyph は必須
                NN_SDK_ASSERT(pInfo->pGlyph != 0);
            }
            break;
        default:
            break;
        }

        if (pInfo != NULL)
        {
            break;
        }
    }

    void* pResourceOffsetBase = pBfntData;

    const FontTextureGlyph& tg = *nn::util::ConstBytePtr(pResourceOffsetBase).Advance(pInfo->pGlyph).Get<FontTextureGlyph>();
    void *const pImage = nn::util::BytePtr(pResourceOffsetBase).Advance(tg.sheetImage).Get<uint8_t>();

    ::nn::gfx::ResTextureFileData* pData = static_cast< ::nn::gfx::ResTextureFileData* >(pImage);

    if (pData->fileHeader.IsRelocated()) {
        pData->fileHeader.GetRelocationTable()->Unrelocate();
        pData->fileHeader.SetRelocated(false);
    }
}



FontInformation*
ResFont::Rebuild(nn::font::detail::BinaryFileHeader* pFileHeader)
{
    NN_SDK_ASSERT_NOT_NULL(pFileHeader);
    NN_SDK_ASSERT(detail::IsAligned(pFileHeader, GlyphDataAlignment));

    nn::font::detail::BinaryBlockHeader* pBlockHeader;
    FontInformation* pInfo = NULL;
    int nBlocks = 0;

    pBlockHeader = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pFileHeader))
                    .Advance(pFileHeader->headerSize).Get<nn::font::detail::BinaryBlockHeader>();

    while (nBlocks < pFileHeader->dataBlocks)
    {
        NN_SDK_ASSERT_NOT_NULL(pBlockHeader);
        switch (pBlockHeader->kind)
        {
        //--------------------------------------------------
        // INFO ブロック
        case BinBlockSignatureFinf:
            // INFOブロックは1つでなければならない
            NN_SDK_ASSERT(pInfo == NULL);
            {
                pInfo = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                        .Advance(sizeof(*pBlockHeader)).Get<FontInformation>();

                NN_SDK_ASSERT(pInfo->fontType == FontType_FontTypeTexture || pInfo->fontType == FontType_FontTypePackedTexture);
                NN_SDK_ASSERT(pInfo->alterCharIndex != GlyphIndexNotFound);

                // pGlyph は必須
                NN_SDK_ASSERT(pInfo->pGlyph != 0);
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BinBlockSignatureTglp:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* pGlyph = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                            .Advance(sizeof(*pBlockHeader)).Get<FontTextureGlyph>();
                NN_UNUSED(pGlyph);
                NN_SDK_ASSERT(pGlyph->sheetImage != 0);

                //---- 32x32 の I4 が最小、1024x1024 の RGBA8 が最大
                NN_FONT_MIN_ASSERT       ( pGlyph->cellWidth,     1 );
                NN_FONT_MIN_ASSERT       ( pGlyph->cellHeight,    1 );
                // sheetSize は単純計算できるものではなく、1024x1024 の RGBA8 で作成したものが
                // 1024 * 1024 * 4 のサイズに収まらないケースもあったので、assert しません。
                // NN_FONT_MINMAX_ASSERT    ( pGlyph->sheetSize,     32 * 32 / 2,    1024 * 1024 * 4 );
                NN_FONT_MIN_ASSERT       ( pGlyph->sheetCount,      1 );
                NN_FONT_MIN_ASSERT       ( pGlyph->sheetRow,      1 );
                NN_FONT_MIN_ASSERT       ( pGlyph->sheetLine,     1 );
                NN_FONT_MINMAX_ASSERT    ( pGlyph->sheetWidth,    32,             1024 );
                NN_FONT_MINMAX_ASSERT    ( pGlyph->sheetHeight,   32,             1024 );
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BinBlockSignatureCwdh:
            {
                FontWidth* pWidth = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                    .Advance(sizeof(*pBlockHeader)).Get<FontWidth>();
                NN_UNUSED(pWidth);
                NN_SDK_ASSERT( pWidth->indexBegin <= pWidth->indexEnd );
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BinBlockSignatureCmap:
            {
                FontCodeMap* pCodeMap = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                                        .Advance(sizeof(*pBlockHeader)).Get<FontCodeMap>();
                NN_UNUSED(pCodeMap);
                NN_SDK_ASSERT( pCodeMap->codeBegin <= pCodeMap->codeEnd );
                NN_SDK_ASSERT( (pCodeMap->mappingMethod == FontMapMethod_Direct)
                                || (pCodeMap->mappingMethod == FontMapMethod_Table)
                                || (pCodeMap->mappingMethod == FontMapMethod_Scan) );
            }
            break;

        //--------------------------------------------------
        // KRNGブロック
        case BinBlockSignatureKrng:
            {
                // ここでは何も行わない
            }
            break;

        //--------------------------------------------------
        // GLGR ブロック
        case BinBlockSignatureGlgr:
            {
                // ResFont では GLGR ブロックは参照しないので何もする必要はない
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
            NN_SDK_ASSERT(false, "The font has unknown block('%c%c%c%c').",
                (pBlockHeader->kind >> 24) & 0xFF,
                (pBlockHeader->kind >> 16) & 0xFF,
                (pBlockHeader->kind >>  8) & 0xFF,
                (pBlockHeader->kind >>  0) & 0xFF
                );
            return NULL;
        }

        pBlockHeader = nn::util::BytePtr(reinterpret_cast<uint8_t *>(pBlockHeader))
                        .Advance(pBlockHeader->size).Get<nn::font::detail::BinaryBlockHeader>();
        nBlocks++;
    }

    return pInfo;
}// NOLINT(impl/function_size)

}   // namespace font
}   // namespace nn
