﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>
#include <nn/font/font_GpuBuffer.h>
#include <nn/font/font_Types.h>

namespace nn {
namespace font {

//----------------------------------------
bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg
    )
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);

    // 計測モードに必要なパラメータのみ設定する。
    // バッファのアライメント値を保存
    m_Alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // 計測モード時はフラグを設定するだけで抜ける。
    if (arg.m_AnalyseModeFlag)
    {
        NN_SDK_ASSERT(!arg.m_AllocateSyncFlag, "Syncrounous allocation is not supported at the Memory usage analyse mode.");
        m_Flags |= Flags_AnalyseMode;
        return true;
    }

    NN_SDK_ASSERT(arg.m_BufferSize > 0);
    NN_SDK_ASSERT(arg.m_BufferCount >= 1);
    NN_SDK_ASSERT_NOT_NULL(arg.m_pAllocateFunction);
    NN_SDK_ASSERT_NOT_NULL(arg.m_pMemoryPool);

    m_Size = arg.m_BufferSize;
    m_Count = arg.m_BufferCount;

    info.SetSize(m_Size);

    // nn::gfx::Buffer 用のメモリをバッファ数に応じて確保する
    m_pBuffers = static_cast<nn::gfx::Buffer*>((*arg.m_pAllocateFunction)(sizeof(nn::gfx::Buffer) * m_Count, 4, arg.m_pAllocateFunctionUserData));
    m_pGpuAddresses = static_cast<nn::gfx::GpuAddress*>((*arg.m_pAllocateFunction)(sizeof(nn::gfx::GpuAddress) * m_Count, 4, arg.m_pAllocateFunctionUserData));

    for(uint32_t index = 0; index < m_Count;++index)
    {
        // nn::gfx::Buffer のコンストラクタ呼び出し
        new (&m_pBuffers[index]) nn::gfx::Buffer();
        size_t  alignedSize = nn::util::align_up(m_Size, m_Alignment);
        m_pBuffers[index].Initialize(pDevice, info, arg.m_pMemoryPool, arg.m_MemoryPoolOffset + alignedSize * index, m_Size);

        // GetGpuAddress は高頻度で呼ばれる処理なのであらかじめ取得しておく
        m_pBuffers[index].GetGpuAddress(&m_pGpuAddresses[index]);
    }



    // 渡されたメモリプールオフセットのアライメントチェック
    NN_SDK_ASSERT(detail::IsAligned(arg.m_MemoryPoolOffset, m_Alignment));

    if (arg.m_AllocateSyncFlag)
    {
        m_Flags |= Flags_AllocateWithSync;
        m_AllocateHeadSize.m_pSync = static_cast<std::atomic<size_t>*>((*arg.m_pAllocateFunction)(sizeof(std::atomic<size_t>), 4, arg.m_pAllocateFunctionUserData));
        m_AllocateTailSize.m_pSync = static_cast<std::atomic<size_t>*>((*arg.m_pAllocateFunction)(sizeof(std::atomic<size_t>), 4, arg.m_pAllocateFunctionUserData));
    }

    return true;
}

//----------------------------------------
void GpuBuffer::Finalize(
    nn::gfx::Device* pDevice,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pFreeFunctionUserData)
{
    NN_UNUSED(pDevice);

    if (m_Flags & Flags_AllocateWithSync)
    {
        (*pFreeFunction)(m_AllocateHeadSize.m_pSync, pFreeFunctionUserData);
        (*pFreeFunction)(m_AllocateTailSize.m_pSync, pFreeFunctionUserData);
    }

    if (m_pBuffers != NULL)
    {
        for (uint32_t index = 0; index < m_Count; ++index)
        {
            m_pBuffers[index].Finalize(pDevice);
        }

        // バッファを開放する
        (*pFreeFunction)(m_pBuffers, pFreeFunctionUserData);
    }

    if (m_pGpuAddresses != NULL)
    {
        (*pFreeFunction)(m_pGpuAddresses, pFreeFunctionUserData);
    }

    // 再利用に備えてすべてのパラメータをクリアする。
    m_pBuffers = NULL;
    m_pGpuAddresses = NULL;
    m_Size = 0;
    m_Alignment = 0;
    m_Count = 0;
    m_MappedBufferIndex = -1;
    m_GpuAccessBufferIndex = 0;
    m_pMappedPointer = NULL;
}

//----------------------------------------
void GpuBuffer::Map(int bufferIndex)
{
    if(m_MappedBufferIndex >= 0)
    {
        return ;
    }

    m_MappedBufferIndex = bufferIndex;

    if (m_pBuffers != NULL)
    {
        m_pMappedPointer = m_pBuffers[m_MappedBufferIndex].Map();
    }

    if (m_Flags & Flags_AllocateWithSync)
    {
        *(m_AllocateHeadSize.m_pSync) = 0;
        *(m_AllocateTailSize.m_pSync) = 0;
    }
    else
    {
        m_AllocateHeadSize.m_Raw = 0;
        m_AllocateTailSize.m_Raw = 0;
    }
}

//----------------------------------------
void GpuBuffer::Unmap()
{
    if(m_MappedBufferIndex < 0)
    {
        return ;
    }

    if (m_pBuffers != NULL)
    {
        m_pBuffers[m_MappedBufferIndex].Unmap();
    }

    m_pMappedPointer = NULL;
    m_MappedBufferIndex = -1;

    if (m_Flags & Flags_AllocateWithSync)
    {
        *(m_AllocateHeadSize.m_pSync) = 0;
        *(m_AllocateTailSize.m_pSync) = 0;
    }
    else
    {
        m_AllocateHeadSize.m_Raw = 0;
        m_AllocateTailSize.m_Raw = 0;
    }
}

}   // namespace font
}   // namespace nn
