﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <nn/atk2/detail/fnd/atk2_Thread.h>
#include <nn/nn_TimeSpan.h>

namespace nn { namespace atk2 { namespace detail { namespace fnd {

NN_DEFINE_STATIC_CONSTANT( const int64_t Thread::InvalidId );
NN_DEFINE_STATIC_CONSTANT( const int Thread::DefaultThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::MinThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::MaxThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int Thread::StackAlignment );
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
NN_DEFINE_STATIC_CONSTANT( const int Thread::InvalidCoreNumber );
#endif

//---------------------------------------------------------------------------
Thread::RunArgs::RunArgs() NN_NOEXCEPT :
name(""),
stack(NULL),
stackSize(0),
idealCoreNumber(nn::os::IdealCoreDontCare),
affinityMask(Thread::AffinityMask_CoreDefault),
priority(Thread::DefaultThreadPriority),
param(NULL),
handler(NULL)
{ }

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const NN_NOEXCEPT
{
    if(stack == NULL) { return false; }
    if(stackSize == 0) { return false; }
    if(priority > MaxThreadPriority) { return false; }
    if(handler == NULL) { return false; }
    return true;
}

//---------------------------------------------------------------------------
Thread::~Thread() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
bool
Thread::Run(const RunArgs& args) NN_NOEXCEPT
{
    if(!args.IsValid())
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return false;
    }

    m_Param = args.param;
    m_Handler = args.handler;

    if(!Create(m_Handle, m_Id, args))
    {
        return false;
    }

    SetName(args.name);

    if(args.affinityMask != AffinityMask_CoreDefault)
    {
        SetAffinityMask(args.idealCoreNumber, args.affinityMask);
    }

    m_Priority = args.priority;

    Resume();

    return true;
}

//---------------------------------------------------------------------------
void
Thread::WaitForExit() NN_NOEXCEPT
{
    Join();
}

//---------------------------------------------------------------------------
void
Thread::Release() NN_NOEXCEPT
{
    switch(m_State)
    {
    case State_Exited:
        break;

    case State_Running:
        NN_SDK_ASSERT(false, "must not Release() for running thread.\n");
        return;

    default:
        return;
    }

    Detach();
    SetState(State_Released);
}

//---------------------------------------------------------------------------
int32_t
Thread::GetPriority() const NN_NOEXCEPT
{
    return m_Priority;
}

//---------------------------------------------------------------------------
Thread::State
Thread::GetState() const NN_NOEXCEPT
{
    return static_cast<State>(m_State);
}

//---------------------------------------------------------------------------
void
Thread::SetState(State value) NN_NOEXCEPT
{
    m_State = value;
}

//---------------------------------------------------------------------------
void
Thread::OnRun() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_NotRun || m_State == State_Released);
    SetState(State_Running);
}

//---------------------------------------------------------------------------
void
Thread::OnExit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == State_Running);
    SetState(State_Exited);
}

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(void* ptrArg) NN_NOEXCEPT
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NN_SDK_ASSERT_NOT_NULL(owner);
        NN_SDK_ASSERT_NOT_NULL(owner->m_Handler);

        owner->m_IsTerminated = false;
        owner->OnRun();


#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
        owner->m_CoreNumber = nn::os::GetCurrentCoreNumber();
#endif
        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
        owner->m_IsTerminated = true;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() NN_NOEXCEPT :
m_State(State_NotRun),
m_Id(InvalidId),
m_Priority(DefaultThreadPriority),
#if defined(NN_BUILD_CONFIG_OS_WIN32)
m_Name(""),
#endif
m_Handler(NULL),
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
m_CoreNumber(InvalidCoreNumber),
#endif
m_IsTerminated(false)
{
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(int32_t value) NN_NOEXCEPT
{
    nn::os::ChangeThreadPriority(&m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(int64_t sleepNanoSec) NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(sleepNanoSec));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, int64_t& id, const RunArgs& args) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::os::GetThreadAvailableCoreMask() & (1ULL << args.idealCoreNumber), "Unsupported core number[%d].", args.idealCoreNumber);
    NN_UNUSED(id);

    if (!nn::os::CreateThread(
        &handle,
        ThreadMain::Run,
        this,
        args.stack,
        args.stackSize,
        args.priority,
        args.idealCoreNumber).IsSuccess()
    )
    {
        m_Id = InvalidId;
        return false;
    }

    nn::os::StartThread(&handle);

    // OSThread へのポインタを ID として利用します。
    m_Id = reinterpret_cast<uint64_t>(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsTerminated()); // 事前に Join が呼ばれていれば ThreadMain::Run() の処理は完了しているはず

    nn::os::DestroyThread(&m_Handle);
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
    m_CoreNumber = InvalidCoreNumber;
#endif
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name) NN_NOEXCEPT
{
    nn::os::SetThreadNamePointer(&m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(int idealCoreNumber, AffinityMask value) NN_NOEXCEPT
{
    nn::os::SetThreadCoreMask(&m_Handle, idealCoreNumber, value);
}

//---------------------------------------------------------------------------
void
Thread::Resume() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
void
Thread::Join() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_Handle);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const NN_NOEXCEPT
{
    return m_IsTerminated;
}

}}}}
