﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/detail/atk2_VoiceCommandManager.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_BytePtr.h>

#include <nn/audio/audio_Common.h>

#include <nn/atk2/atk2_AudioEngine.h>

namespace nn { namespace atk2 { namespace detail {

void VoiceCommandManager::GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
{
    NN_UNUSED(platformArg);
    arg.waveBufferPacketCount = 512;
}

size_t VoiceCommandManager::GetRequiredBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
{
    NN_UNUSED(platformArg);
    NN_SDK_ASSERT_GREATER_EQUAL( arg.waveBufferPacketCount, 0 );
    size_t result = 0;
    result += arg.commandBufferSize;
    return result;
}

size_t VoiceCommandManager::GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
{
    NN_UNUSED(platformArg);
    NN_SDK_ASSERT_GREATER_EQUAL( arg.waveBufferPacketCount, 0 );
    size_t result = 0;
    result = GetRequiredBufferSizeForWaveBufferPacket(arg.waveBufferPacketCount);
    return result;
}

void VoiceCommandManager::Initialize(InitArg arg, PlatformInitArg platformArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(arg.waveBufferPacketCount, 0);
    nn::util::BytePtr workBuffer( arg.workBuffer );
    nn::util::BytePtr workBufferForMemoryPool( platformArg.workBufferForMemoryPool );
    m_pWaveBufferPacket = reinterpret_cast<WaveBufferPacket*>(workBufferForMemoryPool.Get());
    for(auto i = 0; i< m_WaveBufferPacketCount; i++)
    {
        m_pWaveBufferPacket[i].adpcmContext = AdpcmContext();
    }
    workBufferForMemoryPool.Advance(GetRequiredBufferSizeForWaveBufferPacket(arg.waveBufferPacketCount));
    m_WaveBufferPacketCount = arg.waveBufferPacketCount;

    m_CommandBuffer = workBuffer.Get();
    m_CommandBufferSize = arg.commandBufferSize;
    workBuffer.Advance( arg.commandBufferSize );

    CommandManager::Initialize(arg.workBuffer, arg.commandBufferSize);
    m_pAudioEngine = arg._pAudioEngine;
}

void VoiceCommandManager::Initialize( void* commandBuffer, size_t commandBufferSize, void* waveBuffer, size_t waveBufferSize, int waveBufferPacketCount ) NN_NOEXCEPT
{
    NN_UNUSED( waveBufferSize );
    NN_SDK_REQUIRES_NOT_NULL( waveBuffer );
    //NN_SDK_REQUIRES_GREATER_EQUAL( waveBufferSize, GetRequiredBufferSize(waveBufferPacketCount) );
    NN_SDK_ASSERT_GREATER_EQUAL( waveBufferPacketCount, 0 );

    //  WaveBufferPacket の AdpcmContext を nn::audio::BufferAlignSize アラインします
    NN_SDK_REQUIRES_ALIGNED( waveBuffer, nn::audio::BufferAlignSize );
    //NN_STATIC_ASSERT( sizeof( WaveBufferPacket ) % nn::audio::BufferAlignSize == 0 );

    m_WaveBufferPacketCount = waveBufferPacketCount;
    m_pWaveBufferPacket = reinterpret_cast<WaveBufferPacket*>( waveBuffer );
    for ( auto i = 0; i < m_WaveBufferPacketCount; i++ )
    {
        m_pWaveBufferPacket[i].adpcmContext = AdpcmContext();
        //m_pWaveBufferPacket[i].waveBuffer.Initialize();
    }

    CommandManager::Initialize( commandBuffer, commandBufferSize );
}

void VoiceCommandManager::VoiceDisposeCallback( RendererVoice* /* voice */, void* arg ) NN_NOEXCEPT
{
    // TODO: キャストは clang ビルドエラー回避のための暫定対応
    uint32_t voiceId = static_cast<uint32_t>(reinterpret_cast<uint64_t>(arg));
    m_pAudioEngine->GetVirtualVoiceManager().SetRendererVoice(voiceId, nullptr);
}

// private

VoiceCommandManager::WaveBufferPacket* VoiceCommandManager::GetFreeWaveBuffer() NN_NOEXCEPT
{
    for ( int i = 0; i < m_WaveBufferPacketCount; i++ )
    {
        WaveBufferPacket* packet = &m_pWaveBufferPacket[i];
        if ( packet->waveBuffer.state == WaveBufferState_Free ||
             packet->waveBuffer.state == WaveBufferState_Done )
        {
            return packet;
        }
    }
    return nullptr;
}

void VoiceCommandManager::ProcessCommandList( Command* commandList ) NN_NOEXCEPT
{
    Command* command = commandList;

    while( command != nullptr )
    {
        switch( command->id )
        {
        case VoiceCommandId_AllocVoice:
        {
            VoiceCommandAlloc* c = reinterpret_cast<VoiceCommandAlloc*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetRendererVoiceManager().AllocVoice();
            if ( pRendererVoice == nullptr ) {
                break;
            }

            pRendererVoice->SetVoicePriority( c->priority );
            pRendererVoice->SetVoice( reinterpret_cast<Voice*>(c->userId) );

            m_pAudioEngine->GetVirtualVoiceManager().SetRendererVoice( c->voiceId, pRendererVoice );
            break;
        }
        case VoiceCommandId_FreeVoice:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            RendererVoice* pVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pVoice == nullptr ) break;

            m_pAudioEngine->GetRendererVoiceManager().FreeVoice(pVoice);
            m_pAudioEngine->GetVirtualVoiceManager().SetRendererVoice(c->voiceId, nullptr);
            break;
        }
        case VoiceCommandId_SetPriority:
        {
            VoiceCommandPriority* c = reinterpret_cast<VoiceCommandPriority*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;

            //pRendererVoice->SetPriority( c->priority );
            break;
        }
        case VoiceCommandId_Play:
        {
            VoiceCommandPlay* c = reinterpret_cast<VoiceCommandPlay*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;

            pRendererVoice->SetSampleRate( c->sampleRate );
            pRendererVoice->SetSampleFormat( c->sampleFormat );
            pRendererVoice->SetAdpcmParam( &c->adpcmParam ); //TODO: c->adpcmParam
            pRendererVoice->SetOutputReceiver( c->pOutputReceiver );
            pRendererVoice->SetVoiceState( VoiceState_Play );
            break;
        }
        case VoiceCommandId_Pause:
        {
            VoiceCommandPause* c = reinterpret_cast<VoiceCommandPause*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;

            pRendererVoice->SetVoiceState( VoiceState_Pause );
            break;
        }
        case VoiceCommandId_Stop:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;

            pRendererVoice->ClearWaveBufferList();
            pRendererVoice->SetVoiceState( VoiceState_Stop );
            pRendererVoice->Update(); // TODO: 解放されると Update が呼ばれなくなる
            break;
        }
        case VoiceCommandId_AppendWaveBuffer:
        {
            VoiceCommandAppendWaveBuffer* c = reinterpret_cast<VoiceCommandAppendWaveBuffer*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;

            WaveBufferPacket* packet = GetFreeWaveBuffer();
            if ( packet == nullptr ) break;

            WaveBuffer* waveBuffer = &(packet->waveBuffer);
            waveBuffer->bufferAddress = c->bufferAddress;
            waveBuffer->bufferSize = c->bufferSize;
            waveBuffer->sampleLength = c->sampleLength;
            waveBuffer->sampleOffset = c->sampleOffset;
            if ( c->adpcmContextEnable )
            {
                packet->adpcmContext = c->adpcmContext;
                waveBuffer->pAdpcmContext = &(packet->adpcmContext);
            }
            else
            {
                waveBuffer->pAdpcmContext = nullptr;
            }
            waveBuffer->loopFlag = c->loopFlag;
            waveBuffer->userParam = c->tag;
            pRendererVoice->AppendWaveBufferList( waveBuffer );
            break;
        }
        case VoiceCommandId_UpdateParam:
        {
            VoiceCommandParam* c = reinterpret_cast<VoiceCommandParam*>(command);
            RendererVoice* pRendererVoice = m_pAudioEngine->GetVirtualVoiceManager().GetRendererVoice(c->voiceId);
            if ( pRendererVoice == nullptr ) break;
            pRendererVoice->SetVoiceVolume(c->volume);
            pRendererVoice->SetVoicePitch(c->pitch);
            //pRendererVoice->SetVoiceBiquadFilterParameter(c->biquadFilterCoefficients);
            // TODO: ミックス音量の暫定処理
            pRendererVoice->SetVoiceMixVolume(0.5f, 0, 0);
            pRendererVoice->SetVoiceMixVolume(0.5f, 0, 1);
            //pRendererVoice->SetVoiceParam( c->voiceParam );
            break;
        }
        default:
            break;

        } // switch

        command = command->next;
    } // while

} // NOLINT(impl/function_size)

size_t VoiceCommandManager::GetRequiredBufferSizeForWaveBufferPacket(int waveBufferPacketCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL( waveBufferPacketCount, 0 );
    return waveBufferPacketCount * sizeof(WaveBufferPacket);
}

bool VoiceCommandManager::Update() NN_NOEXCEPT
{
    Update(1);
    return true;
}

void VoiceCommandManager::Update(uint32_t frameCount) NN_NOEXCEPT
{
    // VoiceCommandProcess 相当の処理
    //if (g_IsInitialized == false)
    //{
    //    return;
    //}

    //detail::driver::SoundThread& soundThread = detail::driver::SoundThread::GetInstance();
    //detail::LowLevelVoiceCommand& lowLevelVoiceCommand = detail::LowLevelVoiceCommand::GetInstance();

    while (GetCommandListCount() < frameCount)
    {
        //soundThread.FrameProcess( updateType );

        RecvCommandReply();
        FlushCommand( true, false );
    }
}

}}}
