﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/detail/atk2_RendererManager.h>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

namespace {

    const int8_t AudioChannelIndexSettings[nn::atk2::detail::RendererManager::ChannelCount] =
    {
        nn::audio::ChannelMapping_FrontLeft, nn::audio::ChannelMapping_FrontRight,
        nn::audio::ChannelMapping_RearLeft, nn::audio::ChannelMapping_RearRight,
        nn::audio::ChannelMapping_FrontCenter, nn::audio::ChannelMapping_LowFrequency,
    };

}

namespace nn { namespace atk2 { namespace detail {

    void RendererManager::GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        NN_UNUSED(platformArg);
        nn::audio::InitializeAudioRendererParameter(&platformArg.rendererParameter);
    }

    RendererManager::Result RendererManager::Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_ALIGNED(platformArg.workBuffer, nn::os::MemoryPageSize);
        NN_SDK_REQUIRES_GREATER_EQUAL(platformArg.workBufferSize, GetRequiredPlatformBufferSize(arg, platformArg));

        nn::util::BytePtr platformWorkBuffer(platformArg.workBuffer);

        if (!nn::audio::IsValidAudioRendererParameter(platformArg.rendererParameter))
        {
            NN_SDK_LOG("Invalid renderer parameters assigned.\n" );
            return Result_InvalidParameter;
        }

        // レンダラのオープン
        size_t rendererWorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(platformArg.rendererParameter);
        void* rendererWorkBuffer = platformWorkBuffer.Get();
        nn::Result result = nn::audio::OpenAudioRenderer(&m_RendererHandle, &m_RendererEvent, platformArg.rendererParameter, rendererWorkBuffer, rendererWorkBufferSize);
        if (result.IsFailure())
        {
            NN_SDK_LOG("Failed to open renderer.\n" );
            return Result_FailedOpenRenderer;
        }
        m_pRendererWorkBuffer = rendererWorkBuffer;
        platformWorkBuffer.Advance(rendererWorkBufferSize);
        NN_SDK_LOG("[atk2Debug]AudioRenderer 0x%08x %zd\n", rendererWorkBuffer, rendererWorkBufferSize);

        // レンダラコンフィグの初期化
        size_t rendererConfigBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(platformArg.rendererParameter);
        void* rendererConfigBuffer = platformWorkBuffer.Get();
        nn::audio::InitializeAudioRendererConfig(&m_RendererConfig, platformArg.rendererParameter, rendererConfigBuffer, rendererConfigBufferSize);
        m_pRendererConfigWorkBuffer = rendererConfigBuffer;
        platformWorkBuffer.Advance(rendererConfigBufferSize);
        NN_SDK_LOG("[atk2Debug]AudioRendererConfig 0x%08x %zd\n", rendererConfigBuffer, rendererConfigBufferSize);

        if (!m_FinalMix.Initialize(&m_RendererConfig, ChannelCount))
        {
            NN_SDK_LOG("Failed to acquire final mix.\n" );
            return Result_ErrorUnknown;
        }
        result = nn::audio::AddDeviceSink(&m_RendererConfig, &m_DeviceSinkType, m_FinalMix.GetAudioFinalMixInstance(), AudioChannelIndexSettings, ChannelCount, "MainAudioOut");
        if (result.IsFailure())
        {
            NN_SDK_LOG("Failed to add device sink.\n" );
            return Result_ErrorUnknown;
        }

        // レンダラのスタート
        result = nn::audio::StartAudioRenderer(m_RendererHandle);
        if (result.IsFailure())
        {
            NN_SDK_LOG("Failed to start audio renderer.\n" );
            return Result_ErrorUnknown;
        }

        m_RendererParameter = platformArg.rendererParameter;
        m_pAudioEngine = arg._pAudioEngine;
        m_IsNeedRendererUpdate = true;

        return Result_Success;
    }

    void RendererManager::Finalize() NN_NOEXCEPT
    {
        nn::audio::StopAudioRenderer(m_RendererHandle);
        nn::audio::RemoveDeviceSink(&m_RendererConfig, &m_DeviceSinkType, m_FinalMix.GetAudioFinalMixInstance());
        nn::audio::CloseAudioRenderer(m_RendererHandle);
        nn::os::DestroySystemEvent(m_RendererEvent.GetBase());
        m_IsNeedRendererUpdate = false;
    }

    bool RendererManager::Update() NN_NOEXCEPT
    {
        // TODO: RequestUpdateAudioRenderer の呼び出し最適化
        //if (m_IsNeedRendererUpdate)
        {
            Lock();
            nn::Result result = nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_RendererConfig);
            Unlock();
            if (result.IsFailure())
            {
                NN_SDK_LOG("Failed to request update audio renderer.\n");
                return false;
            }
            m_IsNeedRendererUpdate = false;
        }
        return true;
    }

    void RendererManager::Lock() NN_NOEXCEPT
    {
        m_UpdateCriticalSection.Lock();
    }

    void RendererManager::Unlock() NN_NOEXCEPT
    {
        m_UpdateCriticalSection.Unlock();
    }

    // プラットフォーム固有
    size_t RendererManager::GetRequiredPlatformBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        if (!nn::audio::IsValidAudioRendererParameter(platformArg.rendererParameter))
        {
            NN_SDK_LOG("Invalid renderer param.\n" );
            return 0;
        }

        size_t bufferSize = 0;
        bufferSize += nn::audio::GetAudioRendererWorkBufferSize(platformArg.rendererParameter);
        bufferSize += nn::audio::GetAudioRendererConfigWorkBufferSize(platformArg.rendererParameter);
        return bufferSize;
    }

    size_t RendererManager::GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        NN_UNUSED(platformArg);
        size_t bufferSize = 0;
        return bufferSize;
    }

    nn::audio::AudioRendererConfig& RendererManager::GetAudioRendererConfig() NN_NOEXCEPT
    {
        return m_RendererConfig;
    }

    nn::audio::AudioRendererParameter& RendererManager::GetAudioRendererParameter() NN_NOEXCEPT
    {
        return m_RendererParameter;
    }

    const nn::audio::AudioRendererConfig& RendererManager::GetAudioRendererConfig() const NN_NOEXCEPT
    {
        return m_RendererConfig;
    }

    const nn::audio::AudioRendererParameter& RendererManager::GetAudioRendererParameter() const NN_NOEXCEPT
    {
        return m_RendererParameter;
    }

    FinalMix& RendererManager::GetFinalMix() NN_NOEXCEPT
    {
        return m_FinalMix;
    }

    const FinalMix& RendererManager::GetFinalMix() const  NN_NOEXCEPT
    {
        return m_FinalMix;
    }

    void RendererManager::WaitUpdateAudioRendererEvent() NN_NOEXCEPT
    {
        m_RendererEvent.Wait();
    }

    bool RendererManager::TimedWaitUpdateAudioRendererEvent(int64_t waitTimeMsec) NN_NOEXCEPT
    {
        nn::TimeSpan timeSpan;
        return m_RendererEvent.TimedWait(timeSpan.FromMilliSeconds(waitTimeMsec));
    }

    int RendererManager::GetChannelCount() const NN_NOEXCEPT
    {
        return ChannelCount;
    }
}}}
