﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/hio/atk_HioPacketStream.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/viewer/detail/hio/atk_HioStream.h>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//----------------------------------------------------------
HioPacketStream::HioPacketStream() NN_NOEXCEPT :
m_ReadState(ReadState_NotRead),
m_Stream(NULL),
m_ReadBodySize(0)
{
}

//----------------------------------------------------------
HioResult
HioPacketStream::Initialize(HioStream& stream, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    if(IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is already initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    if(buffer == NULL ||
        bufferSize < sizeof(HioPacket))
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return HioResult(HioResultType_Failed);
    }

    m_Stream = &stream;
    m_Buffer = reinterpret_cast<HioPacket*>(buffer);
    m_BufferSize = bufferSize;
    m_ReadState = ReadState_NotRead;

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
void
HioPacketStream::Finalize() NN_NOEXCEPT
{
    m_Stream = NULL;
    m_Buffer = NULL;
    m_BufferSize = 0;
    m_ReadState = ReadState_NotRead;
}

//----------------------------------------------------------
bool
HioPacketStream::IsAvailable() const NN_NOEXCEPT
{
    return IsInitialized() && m_Stream->IsAvailable();
}

//----------------------------------------------------------
size_t
HioPacketStream::GetReadableBytes() const NN_NOEXCEPT
{
    return m_Stream->GetReadableBytes();
}

//----------------------------------------------------------
const HioPacketHeader*
HioPacketStream::GetReadHeader() const NN_NOEXCEPT
{
    if(!IsInitialized() ||
        m_ReadState == ReadState_NotRead)
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return NULL;
    }

    return &m_Buffer->GetHeader();
}

//----------------------------------------------------------
HioPacket*
HioPacketStream::GetReadingPacket() const NN_NOEXCEPT
{
    if(!IsInitialized() ||
        m_ReadState == ReadState_NotRead)
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return NULL;
    }

#if defined(NN_ATKVIEWER_HIO_PACKET_DEBUG_DUMP)
    m_Buffer->Dump();
#endif

    return m_Buffer;
}

//----------------------------------------------------------
const HioPacket*
HioPacketStream::GetReadPacket() const NN_NOEXCEPT
{
    if(!IsInitialized() ||
        m_ReadState != ReadState_BodyRead)
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return NULL;
    }

#if defined(NN_ATKVIEWER_HIO_PACKET_DEBUG_DUMP)
    m_Buffer->Dump();
#endif

    return m_Buffer;
}

//----------------------------------------------------------
bool
HioPacketStream::CanReadNewPacket() const NN_NOEXCEPT
{
    return m_ReadState == ReadState_NotRead || m_ReadState == ReadState_BodyRead;
}

//----------------------------------------------------------
bool
HioPacketStream::CanReadBody() const NN_NOEXCEPT
{
    return m_ReadState == ReadState_HeaderRead || m_ReadState == ReadState_BodyReading;
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadPacket() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // 中途半端な手動読み込み中は失敗させます。
    if(m_ReadState == ReadState_BodyReading)
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    // メッセージヘッダを受信します。
    if(CanReadNewPacket())
    {
        HioResult result = TryReadHeader();

        if(!result.IsTrue())
        {
            return result;
        }
    }

    const HioPacketHeader* header = GetReadHeader();
    NN_SDK_ASSERT_NOT_NULL(header);

    // メッセージボディを受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < header->bodySize)
    {
        return HioResult(HioResultType_False);
    }

    // メモリ不足の場合は、そのメッセージ全体をスキップします。
    if(m_BufferSize < sizeof(HioPacketHeader) + header->bodySize)
    {
        SkipBody();
        return HioResult(HioResultType_TargetOutOfMemory);
    }

    // メッセージボディを受信します。
    return TryReadBody();
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadHeader() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    if(!CanReadNewPacket())
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    m_ReadState = ReadState_NotRead;
    m_ReadBodySize = 0;

    // 受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < sizeof(HioPacketHeader))
    {
        return HioResult(HioResultType_False);
    }

    size_t readSize = m_Stream->Read(m_Buffer, sizeof(HioPacketHeader));

    if(readSize < sizeof(HioPacketHeader))
    {
        return HioResult(HioResultType_TargetRecieveError);
    }

    if(!m_Buffer->GetHeader().IsValid())
    {
        NN_ATK_WARNING("[nn::atk::viewer::detail::HioPacketStream] receive invalid packet header.");
        return HioResult(HioResultType_TargetRecieveError);
    }

    m_ReadState = ReadState_HeaderRead;

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadBody() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // ReadBodyPart() による読み込み途中の場合は失敗させます。
    if(m_ReadState != ReadState_HeaderRead)
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    const HioPacketHeader* header = GetReadHeader();
    NN_SDK_ASSERT_NOT_NULL(header);

    NN_SDK_ASSERT(header->bodySize >= m_ReadBodySize);
    size_t restBodySize = header->bodySize - m_ReadBodySize;

    // 受信しきれない場合は処理しません。
    if(m_Stream->GetReadableBytes() < restBodySize)
    {
        return HioResult(HioResultType_False);
    }

    // メモリ不足の場合は処理しません。
    if(m_BufferSize < sizeof(HioPacketHeader) + header->bodySize)
    {
        SkipBody();
        return HioResult(HioResultType_TargetOutOfMemory);
    }

    // メッセージボディを受信します。
    if(restBodySize > 0)
    {
        size_t readSize = m_Stream->Read(
            util::BytePtr(m_Buffer->GetBody(), m_ReadBodySize).Get(),
            restBodySize);

        if(readSize != restBodySize)
        {
            return HioResult(HioResultType_TargetRecieveError);
        }
    }

    m_ReadBodySize = header->bodySize;
    m_ReadState = ReadState_BodyRead;

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
HioPacketStream::TryReadBodyPart(void* buffer, size_t readLength) NN_NOEXCEPT
{
    if(GetReadableBytes() < readLength)
    {
        return HioResult(HioResultType_False);
    }

    return ReadBodyPart(buffer, readLength);
}

//----------------------------------------------------------
HioResult
HioPacketStream::ReadBodyPart(void* buffer, size_t readLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    if(!CanReadBody())
    {
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    if(readLength == 0)
    {
        return HioResult(HioResultType_False);
    }

    const HioPacketHeader* header = GetReadHeader();
    NN_SDK_ASSERT_NOT_NULL(header);

    NN_SDK_ASSERT(header->bodySize > m_ReadBodySize);
    size_t restBodySize = header->bodySize - m_ReadBodySize;

    // パケットサイズを超える場合は受信しません。
    if(readLength > restBodySize)
    {
        return HioResult(HioResultType_Failed);
    }

    // メッセージボディを受信します。
    size_t readSize = m_Stream->Read(buffer, readLength);

    if(readSize != readLength)
    {
        return HioResult(HioResultType_TargetRecieveError);
    }

    m_ReadBodySize += readLength;

    if(m_ReadBodySize == header->bodySize)
    {
        m_ReadState = ReadState_BodyRead;
    }

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipHeader() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // パケット読み込みが完了していない場合は失敗させます。
    switch(m_ReadState)
    {
    case ReadState_NotRead:
    case ReadState_BodyRead:
        break;

    default:
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    size_t skipedSize = m_Stream->Skip(sizeof(HioPacketHeader));

    if(skipedSize != sizeof(HioPacketHeader))
    {
        return HioResult(HioResultType_TargetRecieveError);
    }

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipBody() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // ヘッダ読み込みが完了していない場合は失敗させます。
    switch(m_ReadState)
    {
    case ReadState_HeaderRead:
    case ReadState_BodyReading:
    case ReadState_BodyRead:
        break;

    default:
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    const HioPacketHeader* header = GetReadHeader();
    NN_SDK_ASSERT_NOT_NULL(header);
    NN_SDK_ASSERT(header->bodySize >= m_ReadBodySize);
    size_t skipSize = header->bodySize - m_ReadBodySize;
    size_t skipedSize = m_Stream->Skip(skipSize);

    m_ReadBodySize += skipedSize;

    if(skipedSize != skipSize)
    {
        m_ReadState = ReadState_Error;
        return HioResult(HioResultType_TargetRecieveError);
    }

    m_ReadState = ReadState_BodyRead;
    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
HioPacketStream::SkipBodyPart(size_t length) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // ボディ読み込み中でない場合は失敗させます。
    switch(m_ReadState)
    {
    case ReadState_HeaderRead:
    case ReadState_BodyReading:
        break;

    default:
        NN_SDK_ASSERT(false, "HioPacketStream has invalid state.\n");
        return HioResult(HioResultType_Failed);
    }

    const HioPacketHeader* header = GetReadHeader();
    NN_SDK_ASSERT_NOT_NULL(header);
    NN_SDK_ASSERT(header->bodySize >= m_ReadBodySize + length);
    size_t skipedSize = m_Stream->Skip(length);

    m_ReadBodySize += skipedSize;

    if(skipedSize != length)
    {
        m_ReadState = ReadState_Error;
        return HioResult(HioResultType_TargetRecieveError);
    }

    if(m_ReadBodySize == header->bodySize)
    {
        m_ReadState = ReadState_BodyRead;
    }

    return HioResult(HioResultType_True);
}


//----------------------------------------------------------
HioResult
HioPacketStream::WritePacket(const HioPacket& packet) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "HioPacketStream is not initialized.\n");
        return HioResult(HioResultType_TargetNotInitialized);
    }

    size_t packetSize = packet.GetSize();

    if(m_Stream->Write(&packet, packetSize) != packetSize)
    {
        return HioResult(HioResultType_TargetSendError);
    }

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
void
HioPacketStream::ClearBuffer() NN_NOEXCEPT
{
    m_ReadState = ReadState_NotRead;
    m_ReadBodySize = 0;
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
