﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/atk_SoundControllerImpl.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_SequenceSoundHandle.h>
#include <nn/atk/atk_SoundArchive.h>
#include <nn/atk/atk_SoundArchivePlayer.h>

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

namespace
{

bool
ReadGlobalVariable(
    SequenceSoundHandle* /*handle*/,
    atk::detail::fnd::BinS32 /*trackNo*/,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32* var
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( var );

    short nWork = -1;
    bool  br    = SequenceSoundHandle::ReadGlobalVariable( &nWork, varNo );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteGlobalVariable(
    SequenceSoundHandle* /*handle*/,
    atk::detail::fnd::BinS32 /*trackNo*/,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32 var
) NN_NOEXCEPT
{
    if( detail::SequenceVariableTraits::InvalidValue == var ) return true;
    return SequenceSoundHandle::WriteGlobalVariable( varNo, (short)var );
}

bool ReadLocalVariable(
    SequenceSoundHandle* handle,
    atk::detail::fnd::BinS32 /*trackNo*/,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32* var
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( handle );
    NN_SDK_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadVariable( &nWork, varNo );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteLocalVariable(
    SequenceSoundHandle* handle,
    atk::detail::fnd::BinS32 /*trackNo*/,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32 var
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteVariable( varNo, (short)var );
}

bool ReadTrackVariable(
    SequenceSoundHandle* handle,
    atk::detail::fnd::BinS32 trackNo,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32* var
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( handle );
    NN_SDK_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = detail::SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadTrackVariable( &nWork, trackNo, varNo );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteTrackVariable(
    SequenceSoundHandle* handle,
    atk::detail::fnd::BinS32 trackNo,
    atk::detail::fnd::BinS32 varNo,
    atk::detail::fnd::BinS32 var
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteTrackVariable( trackNo, varNo, (short)var );
}

}


//----------------------------------------------------------
SoundControllerImpl::SoundControllerImpl() NN_NOEXCEPT :
m_SoundHandle(NULL),
m_SoundArchivePlayer(NULL),
m_SoundId(SoundArchive::InvalidId),
m_State(State_Disabled),
m_PreplayAction(NULL),
m_UserParam(NULL)
{
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::Initialize(SoundHandle* soundHandle, SoundArchivePlayer* soundArchivePlayer) NN_NOEXCEPT
{
    if(soundHandle == NULL ||
        soundArchivePlayer == NULL)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return viewer::Result(viewer::ResultType_Failed);
    }

    m_SoundHandle = soundHandle;
    m_SoundArchivePlayer = soundArchivePlayer;

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
void
SoundControllerImpl::Finalize() NN_NOEXCEPT
{
    m_SoundHandle = NULL;
    m_SoundArchivePlayer = NULL;
}

//----------------------------------------------------------
uint32_t
SoundControllerImpl::GetSoundIndex() const NN_NOEXCEPT
{
    // HACK : インデックスを抽出する正式な方法が見つかれば置き換える
    return nn::atk::detail::Util::GetItemIndex(m_SoundId);
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::SetSoundIndex(uint32_t index) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    return SetSoundId(m_SoundArchivePlayer->GetSoundArchive().GetSoundIdFromIndex(index));
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::SetSoundId(SoundArchive::ItemId soundId) NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    m_SoundId = soundId;
    SetParameterDirty(true);

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
const char*
SoundControllerImpl::GetLabel() const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return "";
    }

    if(m_SoundId == nn::atk::SoundArchive::InvalidId)
    {
        return "";
    }

    const char* result = m_SoundArchivePlayer->GetSoundArchive().GetItemLabel(m_SoundId);
    return result == NULL ? "" : result;
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::SetLabel(const char* label) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    return SetSoundId(m_SoundArchivePlayer->GetSoundArchive().GetItemId(label));
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::Play(const SoundStartable::StartInfo * pStartInfo) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    Stop();

    if( m_PreplayAction != nullptr )
    {
        m_PreplayAction(*this, m_SoundId, m_UserParam);
    }

    nn::atk::SoundStartable::StartInfo startInfo;
    if( pStartInfo != nullptr )
    {
        startInfo = *pStartInfo;
    }

    //  m_Parameters.startOffset を反映させます
    startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
    startInfo.startOffset = m_Parameters.startOffset;
    startInfo.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;

    SoundStartable::StartResult result = m_SoundArchivePlayer->StartSound(m_SoundHandle, m_SoundId, &startInfo);
    UpdateState();

    return result.IsSuccess() ?
        viewer::Result(viewer::ResultType_True) : viewer::Result(viewer::ResultType_Failed);
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::Stop() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    if(!m_SoundHandle->IsAttachedSound())
    {
        return viewer::Result(viewer::ResultType_False);
    }

    m_SoundHandle->Stop(0);
    UpdateState();

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
viewer::Result
SoundControllerImpl::Pause() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return viewer::Result(viewer::ResultType_NotInitialized);
    }

    if(!m_SoundHandle->IsAttachedSound())
    {
        return viewer::Result(viewer::ResultType_False);
    }

    m_SoundHandle->Pause(!m_SoundHandle->IsPause(), 0);
    UpdateState();

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
void
SoundControllerImpl::Update() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    UpdateState();
    UpdateParameters();
}

//----------------------------------------------------------
void
SoundControllerImpl::SetParameterDirty(bool isDirty) NN_NOEXCEPT
{
    m_IsParameterDirty = m_IsParameterDirty || isDirty;
}

//----------------------------------------------------------
void
SoundControllerImpl::UpdateState() NN_NOEXCEPT
{
    State newState = m_State;

    if( m_SoundHandle == NULL ||
        m_SoundId == SoundArchive::InvalidId)
    {
        newState = State_Disabled;
    }
    else if( !m_SoundHandle->IsAttachedSound())
    {
        newState = State_Stopped;
    }
    else if( m_SoundHandle->IsPause())
    {
        newState = State_Paused;
    }
    else
    {
        newState = State_Playing;
    }

    if(m_State == newState)
    {
        return;
    }

    m_State = newState;
}

//----------------------------------------------------------
bool
SoundControllerImpl::GetVariables(ToolSoundSequenceVariableContainer* pVariables) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVariables );

    switch( pVariables->header.variableType )
    {
    case ToolSoundSequenceVariableType_Global:
        return GetVariables( pVariables, ReadGlobalVariable );

    case ToolSoundSequenceVariableType_Local:
        return GetVariables( pVariables, ReadLocalVariable );

    case ToolSoundSequenceVariableType_Track:
        return GetVariables( pVariables, ReadTrackVariable );

    default:
        break;
    }

    InvalidateVariables( pVariables );
    return false;
}

//----------------------------------------------------------
bool
SoundControllerImpl::SetVariables(const ToolSoundSequenceVariableContainer* pVariables) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVariables );

    bool result = false;

    switch( pVariables->header.variableType ){
    case ToolSoundSequenceVariableType_Global:
        {
            SequenceVariableContainer globalVariables;
            globalVariables.Parse( pVariables );
            result = SetVariables( globalVariables, WriteGlobalVariable );
        }
        break;

    case ToolSoundSequenceVariableType_Local:
        {
            SequenceVariableContainer& localVariables = GetParameters().localVariables;
            localVariables.Parse( pVariables );
            result = SetVariables( localVariables, WriteLocalVariable );
        }
        break;

    case ToolSoundSequenceVariableType_Track:
        if( !SequenceVariableTraits::ValidateTrackNo( pVariables->header.trackNo ) ){
            NN_SDK_ASSERT( false );
            break;
        }
        {
            SequenceVariableContainer* trackVariables = GetParameters().trackVariables;
            trackVariables[ pVariables->header.trackNo ].Parse( pVariables );
            result = SetVariables( trackVariables[ pVariables->header.trackNo ], WriteTrackVariable );
        }
        break;

    default:
        break;

    }

    SetParameterDirty(result);

    return result;
}

//----------------------------------------------------------
void
SoundControllerImpl::InvalidateAllVariables() NN_NOEXCEPT
{
    GetParameters().localVariables.InvalidateAll();

    for( int i = 0; i < SequenceVariableTraits::VariableCount; i++ )
    {
        GetParameters().trackVariables[ i ].InvalidateAll();
    }
}

//----------------------------------------------------------
void
SoundControllerImpl::ApplySequenceVariables() NN_NOEXCEPT
{
    SetVariables( GetParameters().localVariables, WriteLocalVariable );

    for( int32_t i = 0; i < SequenceVariableTraits::TrackCount; i++ )
    {
        SetVariables( GetParameters().trackVariables[ i ], WriteTrackVariable );
    }
}

//----------------------------------------------------------
bool
SoundControllerImpl::GetVariables(ToolSoundSequenceVariableContainer* pVariables, ReadVariableFunc pFunc) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVariables );
    NN_SDK_ASSERT_NOT_NULL( pFunc );

    SequenceSoundHandle seqSoundHandle( m_SoundHandle );

    // シーケンス変数の値を取得する（割り込み禁止）
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_Lock);

        for( int i = 0; i < SequenceVariableTraits::VariableCount; i++ )
        {
            pFunc( &seqSoundHandle, (int)pVariables->header.trackNo, i, &pVariables->variables[ i ] );
        }
    }

    return true;
}

//----------------------------------------------------------
bool
SoundControllerImpl::SetVariables(SequenceVariableContainer& Variables, WriteVariableFunc pFunc) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pFunc );

    SequenceSoundHandle seqSoundHandle( m_SoundHandle );

    // シーケンス変数の値を更新する（割り込み禁止）
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_Lock);

        for( int i = 0; i < SequenceVariableTraits::VariableCount; i++ )
        {
            pFunc( &seqSoundHandle, (int)Variables.GetTrackNo(), i, Variables[ i ].GetValue() );
        }
    }

    return true;
}

//----------------------------------------------------------
void
SoundControllerImpl::InvalidateVariables(ToolSoundSequenceVariableContainer* pVariables) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVariables );

    // シーケンス変数の値を無効値に設定する
    for( int i = 0; i < SequenceVariableTraits::VariableCount; i++ )
    {
        pVariables->variables[ i ] = SequenceVariableTraits::InvalidValue;
    }
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
