﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Config.h>
#include "nn/atk/viewer/detail/atk_SequenceVariable.h"

#ifdef NN_ATK_CONFIG_ENABLE_DEV

namespace nn        {
namespace atk       {
namespace viewer      {
namespace detail  {

NN_DEFINE_STATIC_CONSTANT( const uint32_t SequenceVariableTraits::TrackCount );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SequenceVariableTraits::VariableCount );
NN_DEFINE_STATIC_CONSTANT( const int32_t SequenceVariableTraits::InvalidValue );
NN_DEFINE_STATIC_CONSTANT( const int32_t SequenceVariableTraits::MaxValue );
NN_DEFINE_STATIC_CONSTANT( const int32_t SequenceVariableTraits::MinValue );

//----------------------------------------------------------
bool
SequenceVariableTraits::ValidateTrackNo(uint32_t nTrackNo) NN_NOEXCEPT
{
    return ( SequenceVariableTraits::TrackCount > nTrackNo );
}

//----------------------------------------------------------
bool
SequenceVariableTraits::ValidateValue(int32_t lValue) NN_NOEXCEPT
{
    if( SequenceVariableTraits::InvalidValue == lValue ) return true;

    if( SequenceVariableTraits::MaxValue < lValue ) return false;
    if( SequenceVariableTraits::MinValue > lValue ) return false;

    return true;
}

//----------------------------------------------------------
SequenceVariable::SequenceVariable() NN_NOEXCEPT
: m_lValue( SequenceVariableTraits::InvalidValue )
{
}

//----------------------------------------------------------
SequenceVariable::~SequenceVariable() NN_NOEXCEPT
{
}

//----------------------------------------------------------
int32_t
SequenceVariable::GetValue() NN_NOEXCEPT
{
    return m_lValue;
}

//----------------------------------------------------------
void
SequenceVariable::SetValue(int32_t value) NN_NOEXCEPT
{
    NN_SDK_ASSERT( SequenceVariableTraits::ValidateValue( value ) );

    m_lValue = value;
}

//----------------------------------------------------------
SequenceVariableContainer::SequenceVariableContainer() NN_NOEXCEPT
: m_nType( ToolSoundSequenceVariableType_Global )
, m_nPreviewSoundIndex( 0 )
, m_nTrackNo( 0 )
{
}

//----------------------------------------------------------
bool
SequenceVariableContainer::Parse(const ToolSoundSequenceVariableContainer* pContainer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pContainer );

    // コンテナの内容をパラメータに展開する
    m_nType               = static_cast<ToolSoundSequenceVariableType>((uint32_t)pContainer->header.variableType);
    m_nPreviewSoundIndex  = pContainer->header.previewSoundIndex;
    m_nTrackNo            = pContainer->header.trackNo;

    for( int32_t i = 0; i < SequenceVariableTraits::VariableCount; i++ ){
        m_variables[ i ].SetValue( pContainer->variables[ i ] );
    }

    return true;
}

//----------------------------------------------------------
void
SequenceVariableContainer::InvalidateAll() NN_NOEXCEPT
{
    for( int32_t i = 0; i < SequenceVariableTraits::VariableCount; i++ ){
        m_variables[ i ].SetValue( SequenceVariableTraits::InvalidValue );
    }
}

//----------------------------------------------------------
uint32_t
SequenceVariableContainer::GetCount() NN_NOEXCEPT
{
    return SequenceVariableTraits::VariableCount;
}

//----------------------------------------------------------
ToolSoundSequenceVariableType
SequenceVariableContainer::GetType() NN_NOEXCEPT
{
    return m_nType;
}

//----------------------------------------------------------
uint32_t
SequenceVariableContainer::GetPreviewPlayerIndex() NN_NOEXCEPT
{
    return m_nPreviewSoundIndex;
}

//----------------------------------------------------------
uint32_t
SequenceVariableContainer::GetTrackNo() NN_NOEXCEPT
{
    return m_nTrackNo;
}

//----------------------------------------------------------
SequenceVariable
SequenceVariableContainer::operator [](int32_t index) NN_NOEXCEPT
{
    return m_variables[ index ];
}

} // namespace detail
} // namespace viewer
} // namespace atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
