﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/atk_SoundEditSession.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/fnd/basis/atkfnd_Memory.h>
#include <nn/atk/viewer/atk_Config.h>
#include <nn/atk/viewer/atk_SoundArchiveEditor.h>
#include <nn/atk/viewer/detail/atk_IErrorProvider.h>
#include <nn/atk/viewer/detail/atk_SoundEditConnection.h>
#include <nn/atk/viewer/detail/hio/atk_HioSyncChannel.h>
#include <nn/atk/viewer/detail/hio/atk_HioAsyncChannel.h>
#include <nn/atk/viewer/detail/protocol/atk_SyncPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemsPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemInfoPacket.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_COM_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {

NN_DEFINE_STATIC_CONSTANT( const size_t SoundEditSession::DefaultChannelStreamBufferSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundEditSession::DefaultSyncTimeout );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundEditSession::DefaultCacheSyncInterval );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundEditSession::DefaultSendTimeout );
NN_DEFINE_STATIC_CONSTANT( const int SoundEditSession::DefaultMaxItemName );

//----------------------------------------------------------
SoundEditSession::SoundEditSession() NN_NOEXCEPT :
// HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
m_Interim_IsSoundPlayer(false),
#endif
m_State(State_NotInitialized),
m_Port0(0),
m_Port1(0),
m_SyncTimeout(0),
m_SoundArchiveEditor(NULL)
{
}

//----------------------------------------------------------
SoundEditSession::~SoundEditSession() NN_NOEXCEPT
{
    // Finalize() 済みであることを確認します。
    NN_SDK_ASSERT(!IsInitialized(), "SoundEditSession is not finalized.\n");
}

//----------------------------------------------------------
void
SoundEditSession::Finalize() NN_NOEXCEPT
{
    Close();

    UnregisterSoundArchiveEditor(NULL);

    FinalizeFuncChannel();
    FinalizeSyncChannel();

    m_HioManager.Finalize();

    m_Connection.Finalize();

    m_State = State_NotInitialized;
    m_SyncTimeout = 0;
}

//----------------------------------------------------------
Result
SoundEditSession::RegisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundEditSession is not initialized.\n");
    NN_SDK_ASSERT(m_SoundArchiveEditor == NULL, "SoundArchiveEditor is already registered.\n");

    if(soundArchiveEditor == NULL ||
        !soundArchiveEditor->IsInitialized())
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    m_SoundArchiveEditor = soundArchiveEditor;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundEditSession::UnregisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor) NN_NOEXCEPT
{
    // TODO : 複数サウンドアーカイブに対応したら利用します。
    (void)soundArchiveEditor;

    NN_SDK_ASSERT(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
        m_SoundArchiveEditor = NULL;
    }
}

//----------------------------------------------------------
void
SoundEditSession::Open() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(IsOpened())
    {
        return;
    }

    ClearBuffer();
    m_State = State_SyncRequesting;

    RequestSync();
    m_SyncStopWatch.Start();
}

//----------------------------------------------------------
void
SoundEditSession::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(IsOpened())
    {
        Disconnect();
    }

    m_SyncStopWatch.Stop();

    m_State = State_Stopped;
}

//----------------------------------------------------------
void
SoundEditSession::Update() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(!IsOpened())
    {
        return;
    }

    if(!m_SyncChannel.IsOpened())
    {
        if(!m_SyncChannel.Open(GetChannelInfo(detail::HioChannelType_SndeditSync)))
        {
            return;
        }
    }

    if(!m_FuncChannel.IsOpened())
    {
        if(!m_FuncChannel.Open(GetChannelInfo(detail::HioChannelType_SndeditFunc)))
        {
            return;
        }
    }

    m_HioManager.Update();

    // SYNC タイムアウトを経過したら...
    // SYNC が未完了なら Close() します。
    // SYNC が完了していたら、再度 SYNC 要求します。
    if(m_SyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_SyncTimeout)
    {
        if(m_State != State_SyncCompleted)
        {
            if(m_Connection.IsOpened())
            {
                Disconnect();
            }
        }
        else
        {
            m_State = State_SyncUpdating;
        }

        RequestSync();
    }

    m_Connection.Update();
}

//----------------------------------------------------------
Result
SoundEditSession::InitializeSyncChannel(
    atk::detail::fnd::FrameHeap& allocator,
    size_t recvStreamBufferSize,
    size_t recvPacketBufferSize) NN_NOEXCEPT
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);
    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return Result(ResultType_OutOfMemory);
    }

    detail::HioResult result = m_SyncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );

    if(result.IsFailed())
    {
        return Result(ResultType(result));
    }

    InitializeSyncHandlers();

    m_HioManager.RegisterChannel(m_SyncChannel);

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeSyncChannel() NN_NOEXCEPT
{
    m_HioManager.UnregisterChannel(m_SyncChannel);

    FinalizeSyncHandlers();

    m_SyncChannel.Close();
    m_SyncChannel.Finalize();
}

//----------------------------------------------------------
void
SoundEditSession::InitializeSyncHandlers() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SyncChannel.IsInitialized());

    m_SyncReplyHandler.Initialize(*this);
    m_SyncChannel.RegisterMessageHandler(m_SyncReplyHandler);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeSyncHandlers() NN_NOEXCEPT
{
    if(m_SyncChannel.IsInitialized())
    {
        m_SyncChannel.UnregisterMessageHandler(m_SyncReplyHandler);
        m_SyncReplyHandler.Finalize();
    }
}

//----------------------------------------------------------
Result
SoundEditSession::InitializeFuncChannel(
    atk::detail::fnd::FrameHeap& allocator,
    size_t recvStreamBufferSize,
    size_t recvPacketBufferSize) NN_NOEXCEPT
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);
    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return Result(ResultType_OutOfMemory);
    }

    detail::HioResult result = m_FuncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );

    if(result.IsFailed())
    {
        return Result(ResultType(result));
    }

    InitializeFuncHandlers();

    m_HioManager.RegisterChannel(m_FuncChannel);

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeFuncChannel() NN_NOEXCEPT
{
    m_HioManager.UnregisterChannel(m_FuncChannel);

    FinalizeFuncHandlers();

    m_FuncChannel.Close();
    m_FuncChannel.Finalize();
}

//----------------------------------------------------------
void
SoundEditSession::InitializeFuncHandlers() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_FuncChannel.IsInitialized());
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeFuncHandlers() NN_NOEXCEPT
{
    if(m_FuncChannel.IsInitialized())
    {
    }
}

//----------------------------------------------------------
size_t
SoundEditSession::GetSyncChannelRecvPacketBufferSize(int maxItemName) const NN_NOEXCEPT
{
    size_t result = 0;

    // TODO : SYNC チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : SYNC チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = std::max(result, detail::SyncPacket::GetRequiredSize());
    result = std::max(result, detail::SyncReplyPacket::GetRequiredSize());
    result = std::max(result, detail::QueryItemsPacket::GetRequiredSize());
    result = std::max(result, detail::QueryItemsReplyPacket::GetRequiredSize(maxItemName));

    return nn::util::align_up( result, atk::detail::fnd::MemoryTraits::DefaultAlignment );
}

//----------------------------------------------------------
size_t
SoundEditSession::GetFuncChannelRecvPacketBufferSize(int maxItemName, size_t maxItemInfoSize) const NN_NOEXCEPT
{
    size_t result = 0;

    // TODO : FUNC チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : FUNC チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = std::max(result, detail::QueryItemInfoPacket::GetRequiredSize(maxItemName));
    result = std::max(result, detail::QueryItemInfoReplyPacket::GetRequiredSize(maxItemInfoSize));

    return nn::util::align_up( result, atk::detail::fnd::MemoryTraits::DefaultAlignment );
}

//----------------------------------------------------------
Result
SoundEditSession::Connect() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditSession is not initialized.\n");
        Disconnect();
        return Result(ResultType_NotInitialized);
    }

    NN_SDK_ASSERT_NOT_NULL(m_SoundArchiveEditor);

    if(IsConnected() && m_SoundArchiveEditor->IsStarted())
    {
        m_State = State_SyncCompleted;

#if defined(NN_ATK_ENABLE_COM_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[sndedit] SYNC completed : time=%d.\n",
            m_SyncStopWatch.GetElapsedTime().ToMilliSeconds());
#endif

        return Result(ResultType_True);
    }

    // もし不正な状態ならば、それを正すために Disconect() します。
    Disconnect();

    Result result = m_Connection.Open(*m_SoundArchiveEditor);

    if(!result.IsTrue())
    {
        Disconnect();
        return result;
    }

    m_State = State_SyncCompleted;

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    if(IsConnected())
    {
        NN_DETAIL_ATK_INFO("[sndedit] connected.\n");
    }
#endif

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundEditSession::Disconnect() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditSession is not initialized.\n");
        return;
    }

    // TODO : ★マルチサウンドアーカイブ対応 : すべての SoundArchiveEditor を停止します。
    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
    }

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    if(IsConnected())
    {
        NN_DETAIL_ATK_INFO("[sndedit] disconnected.\n");
    }
#endif

    m_Connection.Close();
    ClearBuffer();

    m_State = State_SyncRequesting;
}

//----------------------------------------------------------
void
SoundEditSession::ClearBuffer() NN_NOEXCEPT
{
    m_SyncChannel.ClearBuffer();
    m_FuncChannel.ClearBuffer();
}

//----------------------------------------------------------
bool
SoundEditSession::RequestSync() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    // HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
    detail::SyncPacket packet(m_Interim_IsSoundPlayer);
#else
    detail::SyncPacket packet;
#endif

    packet.GetBody().SetMemoryUsage(static_cast<uint32_t>(m_SoundArchiveEditor->GetMemoryUsage()));
    packet.GetBody().SetMemoryMax(static_cast<uint32_t>(m_SoundArchiveEditor->GetMemoryMax()));
    packet.GetBody().SetEditItemCount(m_SoundArchiveEditor->GetEditItemCount());
    packet.GetBody().SetMaxEditableItemCount(m_SoundArchiveEditor->GetEditableItemCountMax());
    packet.GetBody().SetEditFileCount(m_SoundArchiveEditor->GetEditFileCount());
    packet.GetBody().SetMaxEditableFileCount(m_SoundArchiveEditor->GetEditableFileCountMax());
    packet.GetBody().SetIsOutOfMemory(m_SoundArchiveEditor->IsOutOfMemory());
    packet.GetBody().SetIsItemsOverflow(m_SoundArchiveEditor->IsItemInfosOverflow());
    packet.GetBody().SetIsFilesOverflow(m_SoundArchiveEditor->IsFilesOverflow());

    m_SyncStopWatch.Reset();
    return m_SyncChannel.Send(packet).IsSucceeded();
}

//----------------------------------------------------------
Result
SoundEditSession::Initialize(
     void* buffer,
     size_t bufferLength,
     const Configs& configs) NN_NOEXCEPT
{
    if(buffer == NULL ||
        bufferLength == 0 ||
        configs.channelStreamBufferSize == 0 ||
        configs.syncTimeout == 0 ||
        configs.cacheSyncInterval == 0 ||
        configs.sendTimeout == 0 ||
        configs.maxItemName == 0)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

//----------------------------------------------------------
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // バッファサイズをチェックします。
    if(bufferLength < GetRequiredMemorySize(configs))
    {
        NN_SDK_ASSERT(false, "bufferLength too small.\n");
        return Result(ResultType_Failed);
    }
#endif // defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
//----------------------------------------------------------

    if(IsInitialized())
    {
        return Result(ResultType_False);
    }

    m_Port0 = configs.port0;
    m_Port1 = configs.port1;

    if(m_SyncChannel.IsInitialized() ||
        m_FuncChannel.IsInitialized() ||
        m_SyncChannel.IsOpened() ||
        m_FuncChannel.IsOpened())
    {
        return Result(ResultType_Failed);
    }

    if(!m_HioManager.Initialize())
    {
        return Result(ResultType_Failed);
    }

    atk::detail::fnd::FrameHeap allocator;
    allocator.Initialize(buffer, bufferLength);

    Result result = InitializeSyncChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetSyncChannelRecvPacketBufferSize(configs.maxItemName));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    result = InitializeFuncChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetFuncChannelRecvPacketBufferSize(
            configs.maxItemName,
            detail::ResItemInfoUtility::GetMaxItemInfoSize(configs.maxItemName, detail::Limits::MaxFilePath)));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    {
        size_t bufferForConnectionLength = m_Connection.GetRequiredMemorySize(configs.maxItemName);
        void* bufferForConnection = allocator.Alloc(bufferForConnectionLength);

        if(bufferForConnection == NULL)
        {
            Finalize();
            return Result(ResultType_OutOfMemory);
        }

        detail::SoundEditConnection::InitializeArgs initializeArgs;
        initializeArgs.buffer = bufferForConnection;
        initializeArgs.bufferLength = bufferForConnectionLength;
        initializeArgs.cacheSyncInterval = configs.cacheSyncInterval;
        initializeArgs.sendTimeout = configs.sendTimeout;
        initializeArgs.hioManager = &m_HioManager;
        initializeArgs.syncChannel = &m_SyncChannel;
        initializeArgs.funcChannel = &m_FuncChannel;
        initializeArgs.maxItemName = configs.maxItemName;

        result = m_Connection.Initialize(initializeArgs);

        if(result.IsFailed())
        {
            Finalize();
            return result;
        }
    }

    m_SyncTimeout = configs.syncTimeout;
    m_State = State_Stopped;

    return Result(ResultType_True);
}

//----------------------------------------------------------
size_t
SoundEditSession::GetRequiredMemorySize(const Configs& configs) const NN_NOEXCEPT
{
    size_t result = m_Connection.GetRequiredMemorySize(configs.maxItemName);

    // SYNC, FUNC チャンネルの受信用ストリームバッファサイズ
    result += nn::util::align_up( configs.channelStreamBufferSize, atk::detail::fnd::MemoryTraits::DefaultAlignment ) * 2;

    // SYNC, FUNC チャンネルの受信用パケットバッファサイズ
    result += GetSyncChannelRecvPacketBufferSize(configs.maxItemName);
    result += GetFuncChannelRecvPacketBufferSize(
        configs.maxItemName,
        detail::ResItemInfoUtility::GetMaxItemInfoSize(configs.maxItemName, detail::Limits::MaxFilePath));
    result += GetRequiredWorkBufferSize() * 2;

    return result;
}

//----------------------------------------------------------
size_t
SoundEditSession::GetRequiredWorkBufferSize() const NN_NOEXCEPT
{
    return detail::HioChannel::DefaultWorkBufferLength;
}

//----------------------------------------------------------
detail::HioStream::ChannelType
SoundEditSession::GetChannelInfo(detail::HioChannelType channel) const NN_NOEXCEPT
{
    switch(channel)
    {
        case detail::HioChannelType_SndeditSync:
            return "NN_ATK_SYNC";
        case detail::HioChannelType_SndeditFunc:
            return "NN_ATK_FUNC";
        default:
            NN_SDK_ASSERT(false);
            return "INVALID_CH";

    }
}

} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
