﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/atk_SoundEdit.h>
#include <nn/atk/detail/atk_Macro.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

namespace nn {
namespace atk {
namespace viewer {

NN_DEFINE_STATIC_CONSTANT( const size_t SoundEdit::DefaultSndeditHeapSize );
NN_DEFINE_STATIC_CONSTANT( const size_t SoundEdit::DefaultThreadStackSize );
NN_DEFINE_STATIC_CONSTANT( const PORT SoundEdit::DefaultSndeditPort0 );
NN_DEFINE_STATIC_CONSTANT( const PORT SoundEdit::DefaultSndeditPort1 );
NN_DEFINE_STATIC_CONSTANT( const PORT SoundEdit::DefaultSndeditPort2 );
NN_DEFINE_STATIC_CONSTANT( const int SoundEdit::DefaultSndeditWorkerThreadCoreNumber );

//----------------------------------------------------------
SoundEdit::SoundEdit() NN_NOEXCEPT
: m_IsInitialized(false)
, m_IsStarted(false)
, m_pMemoryForUpdateCachesThreadStack(NULL)
, m_UpdateCachesThreadStackSize(0)
, m_pMemoryForSoundArchiveEditor(NULL)
, m_pMemoryForSoundEditSession(NULL)
, m_pMemoryForSoundControlSession(NULL)
, m_WorkerThreadCoreNumber(DefaultSndeditWorkerThreadCoreNumber)
{
}

//----------------------------------------------------------
SoundEdit::~SoundEdit() NN_NOEXCEPT
{
    // Finalize() 済みであることを確認します。
    NN_SDK_ASSERT(!IsInitialized(), "SoundEdit is not finalized.\n");
}

//----------------------------------------------------------
void
SoundEdit::Initialize(const InitializeArg& args, const Option& param) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(args.buffer);
    NN_SDK_ASSERT(args.bufferSize > 0);
    NN_SDK_ASSERT_NOT_NULL(args.soundArchive);
    NN_SDK_ASSERT_NOT_NULL(args.soundArchivePlayer);

    NN_SDK_ASSERT(param.threadStackSize > 0);
    NN_SDK_ASSERT(param.sndEditHeapSize > 0);
    NN_SDK_ASSERT(args.workerThreadCoreNumber >= 0);

    if (m_IsInitialized)
    {
        return;
    }

    // アロケータの初期化
    m_Allocator.Initialize( args.buffer, args.bufferSize );

    // キャッシュヘルパー用のスタックメモリの確保
    m_UpdateCachesThreadStackSize = param.threadStackSize;
    m_pMemoryForUpdateCachesThreadStack = static_cast<char*>( m_Allocator.Alloc(m_UpdateCachesThreadStackSize, nn::atk::detail::fnd::Thread::StackAlignment) );

    // サウンドアーカイブエディタの初期化
    {
        nn::atk::viewer::SoundArchiveEditor::Configs configs;

        m_pMemoryForSoundArchiveEditor = m_Allocator.Alloc( param.sndEditHeapSize );
        NN_SDK_ASSERT_NOT_NULL( m_pMemoryForSoundArchiveEditor );

        nn::atk::viewer::Result result = m_SoundArchiveEditor.Initialize(
            m_pMemoryForSoundArchiveEditor,
            param.sndEditHeapSize,
            args.soundArchive,
            args.soundArchivePlayer,
            configs );
        NN_SDK_ASSERT( result.IsTrue() );
    }

    // サウンド編集セッションの初期化
    {
        nn::atk::viewer::SoundEditSession::Configs configs;
        configs.port0 = args.port0;
        configs.port1 = args.port1;
        size_t memorySize = m_SoundEditSession.GetRequiredMemorySize(configs);

        m_pMemoryForSoundEditSession = m_Allocator.Alloc( memorySize );
        NN_SDK_ASSERT_NOT_NULL( m_pMemoryForSoundEditSession );

        nn::atk::viewer::Result result = m_SoundEditSession.Initialize(
            m_pMemoryForSoundEditSession,
            memorySize,
            configs );
        if (!result.IsTrue())
        {
            NN_ATK_WARNING("SoundEditSession::Initialize is failed (%s)", result.ToString());
            return;
        }
    }

    // サウンドオブジェクトコントローラの初期化
    {
        nn::atk::viewer::Result result = m_SoundObjectController.Initialize(args.soundArchivePlayer);
        NN_SDK_ASSERT( result.IsTrue() );
    }

    // サウンド制御セッションの初期化
    {
        nn::atk::viewer::SoundControlSession::Configs configs;
        configs.port = args.port2;
        size_t memorySize = m_SoundControlSession.GetRequiredMemorySize(configs);

        m_pMemoryForSoundControlSession = m_Allocator.Alloc( memorySize );
        NN_SDK_ASSERT_NOT_NULL( m_pMemoryForSoundControlSession );

        nn::atk::viewer::Result result = m_SoundControlSession.Initialize(
            m_pMemoryForSoundControlSession,
            memorySize,
            configs );
        if (!result.IsTrue())
        {
            NN_ATK_WARNING("SoundControlSession::Initialize is failed (%s)", result.ToString());
            return;
        }
    }

    m_WorkerThreadCoreNumber = args.workerThreadCoreNumber;
    m_IsInitialized = true;
}

//----------------------------------------------------------
void
SoundEdit::Finalize() NN_NOEXCEPT
{
    if (m_SoundControlSession.IsInitialized())
    {
        m_SoundControlSession.Finalize();
    }
    m_SoundObjectController.Finalize();
    if (m_SoundEditSession.IsInitialized())
    {
        m_SoundEditSession.Finalize();
    }

    // 各種オブジェクトの終了処理
    if (m_SoundArchiveEditor.IsInitialized())
    {
        m_SoundArchiveEditor.Finalize();
    }

    // アロケータ終了処理
    m_Allocator.Clear();
    m_Allocator.Finalize();

    m_WorkerThreadCoreNumber = DefaultSndeditWorkerThreadCoreNumber;
    m_IsInitialized = false;
}

//----------------------------------------------------------
size_t
SoundEdit::GetRequiredMemorySize(const Option& param) const NN_NOEXCEPT
{
    size_t memorySize = 0;

    memorySize += param.threadStackSize + nn::os::StackRegionAlignment;
    memorySize += param.sndEditHeapSize;
    {
        nn::atk::viewer::SoundEditSession::Configs configs;
        memorySize += m_SoundEditSession.GetRequiredMemorySize(configs);
    }
    {
        nn::atk::viewer::SoundControlSession::Configs configs;
        memorySize += m_SoundControlSession.GetRequiredMemorySize(configs);
    }

    return memorySize;
}

//----------------------------------------------------------
void
SoundEdit::Start() NN_NOEXCEPT
{
    if (!m_IsInitialized)
    {
        return;
    }

    if (IsStarted())
    {
        return;
    }

    nn::atk::viewer::Result result;

    result = m_SoundEditSession.RegisterSoundArchiveEditor(&m_SoundArchiveEditor);
    NN_SDK_ASSERT( result.IsTrue() );

    result = m_SoundControlSession.RegisterSoundObjectController(&m_SoundObjectController);
    NN_SDK_ASSERT( result.IsTrue() );

    m_SoundEditSession.Open();
    m_SoundControlSession.Open();

    result = m_UpdateCachesHelper.Start(
        &m_SoundArchiveEditor,
        m_pMemoryForUpdateCachesThreadStack,
        m_UpdateCachesThreadStackSize,
        UpdateCachesHelper::AffinityMask_CoreDefault,
        atk::detail::fnd::Thread::DefaultThreadPriority,
        m_WorkerThreadCoreNumber
    );
    NN_SDK_ASSERT(result.IsSucceeded());

    m_IsStarted = true;
}

//----------------------------------------------------------
void
SoundEdit::Stop() NN_NOEXCEPT
{
    if (!m_IsInitialized)
    {
        return;
    }

    if (!IsStarted())
    {
        return;
    }

    m_UpdateCachesHelper.Stop();

    m_SoundControlSession.Close();
    m_SoundEditSession.Close();

    m_SoundControlSession.UnregisterSoundObjectController(&m_SoundObjectController );
    m_SoundEditSession.UnregisterSoundArchiveEditor(&m_SoundArchiveEditor );

    m_IsStarted = false;
}

//----------------------------------------------------------
void
SoundEdit::Update() NN_NOEXCEPT
{
    if (!m_IsInitialized)
    {
        return;
    }

    m_SoundEditSession.Update();
    m_SoundObjectController.Update();
    m_SoundControlSession.Update();
}

#if defined(NN_ATK_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
//----------------------------------------------------------
int
SoundEdit::GetWorkerThreadCoreNumber() const NN_NOEXCEPT
{
    return m_UpdateCachesHelper.GetThreadCoreNumber();
}
#endif

} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
