﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/atk/atk_TaskProfileReader.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

namespace nn {
namespace atk {

nn::TimeSpan TaskProfile::LoadStreamBlock::GetTotalTime() const NN_NOEXCEPT
{
    return nn::os::Tick( m_EndTick - m_BeginTick ).ToTimeSpan();
}

nn::os::Tick TaskProfile::LoadStreamBlock::GetBeginTick() const NN_NOEXCEPT
{
    return nn::os::Tick( m_BeginTick );
}

nn::os::Tick TaskProfile::LoadStreamBlock::GetEndTick() const NN_NOEXCEPT
{
    return nn::os::Tick( m_EndTick );
}

float TaskProfile::LoadStreamBlock::GetRemainingCachePercentage() const NN_NOEXCEPT
{
    if (m_CachedLength == 0 || m_CacheCurrentPosition < m_CacheStartPosition)
    {
        return 0.f;
    }

    return ((static_cast<float>(m_CacheStartPosition) + m_CachedLength - m_CacheCurrentPosition) / m_CachedLength) * 100.f;
}

size_t TaskProfile::LoadStreamBlock::GetCachedLength() const NN_NOEXCEPT
{
    return m_CachedLength;
}

nn::atk::detail::driver::StreamSoundPlayer* TaskProfile::LoadStreamBlock::GetStreamSoundPlayer() const NN_NOEXCEPT
{
    return m_pPlayer;
}

void TaskProfile::LoadStreamBlock::SetData(const nn::os::Tick& beginTick, const nn::os::Tick& endTick, const detail::IStreamDataDecoder::CacheProfile& cacheProfile) NN_NOEXCEPT
{
    m_BeginTick = beginTick.GetInt64Value();
    m_EndTick = endTick.GetInt64Value();
    m_CacheStartPosition = cacheProfile.cacheStartPosition;
    m_CachedLength = cacheProfile.cachedLength;
    m_CacheCurrentPosition = cacheProfile.cacheCurrentPosition;
    m_pPlayer = cacheProfile.player;
}



nn::TimeSpan TaskProfile::LoadOpusStreamBlock::GetTotalTime() const NN_NOEXCEPT
{
    return nn::os::Tick( m_EndTick - m_BeginTick ).ToTimeSpan();
}

nn::os::Tick TaskProfile::LoadOpusStreamBlock::GetBeginTick() const NN_NOEXCEPT
{
    return nn::os::Tick( m_BeginTick );
}

nn::os::Tick TaskProfile::LoadOpusStreamBlock::GetEndTick() const NN_NOEXCEPT
{
    return nn::os::Tick( m_EndTick );
}

float TaskProfile::LoadOpusStreamBlock::GetRemainingCachePercentage() const NN_NOEXCEPT
{
    if (m_CachedLength == 0 || m_CacheCurrentPosition < m_CacheStartPosition)
    {
        return 0.f;
    }

    return ((static_cast<float>(m_CacheStartPosition) + m_CachedLength - m_CacheCurrentPosition) / m_CachedLength) * 100.f;
}

size_t TaskProfile::LoadOpusStreamBlock::GetCachedLength() const NN_NOEXCEPT
{
    return m_CachedLength;
}

nn::TimeSpan TaskProfile::LoadOpusStreamBlock::GetDecodeTime() const NN_NOEXCEPT
{
    return nn::os::Tick( m_DecodeTick ).ToTimeSpan();
}

int TaskProfile::LoadOpusStreamBlock::GetDecodedSampleCount() const NN_NOEXCEPT
{
    return m_DecodedSampleCount;
}

nn::TimeSpan TaskProfile::LoadOpusStreamBlock::GetFsAccessTime() const NN_NOEXCEPT
{
    return nn::os::Tick( m_FsAccessTick ).ToTimeSpan();
}

std::size_t TaskProfile::LoadOpusStreamBlock::GetFsReadSize() const NN_NOEXCEPT
{
    return m_FsReadSize;
}

nn::atk::detail::driver::StreamSoundPlayer* TaskProfile::LoadOpusStreamBlock::GetStreamSoundPlayer() const NN_NOEXCEPT
{
    return m_pPlayer;
}

void TaskProfile::LoadOpusStreamBlock::SetData(const nn::os::Tick& beginTick, const nn::os::Tick& endTick, const detail::IStreamDataDecoder::DecodeProfile& decodeProfile, const detail::IStreamDataDecoder::CacheProfile& cacheProfile) NN_NOEXCEPT
{
    m_BeginTick = beginTick.GetInt64Value();
    m_EndTick = endTick.GetInt64Value();
    m_DecodeTick = decodeProfile.decodeTick.GetInt64Value();
    m_FsAccessTick = decodeProfile.fsAccessTick.GetInt64Value();
    m_FsReadSize = decodeProfile.fsReadSize;
    m_DecodedSampleCount = decodeProfile.decodedSampleCount;
    m_CacheStartPosition = cacheProfile.cacheStartPosition;
    m_CachedLength = cacheProfile.cachedLength;
    m_CacheCurrentPosition = cacheProfile.cacheCurrentPosition;
    m_pPlayer = cacheProfile.player;
}



TaskProfileLogger::TaskProfileLogger() NN_NOEXCEPT
    : m_IsProfilingEnabled( false )
{
}

void TaskProfileLogger::Record(const TaskProfile& profile) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_Lock );
    for( auto itr = m_List.begin(); itr != m_List.end(); ++itr )
    {
        itr->Record( profile );
    }
}

void TaskProfileLogger::RegisterReader(TaskProfileReader& reader) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_Lock );
    m_List.push_back( reader );
}

void TaskProfileLogger::UnregisterReader(const TaskProfileReader& reader) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_Lock );
    m_List.erase( m_List.iterator_to( reader ) );
}

void TaskProfileLogger::SetProfilingEnabled(bool isEnabledProfiling) NN_NOEXCEPT
{
    m_IsProfilingEnabled = isEnabledProfiling;
}

void TaskProfileLogger::Finalize() NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_Lock );
    m_List.clear();
}

} // namespace nn::atk
} // namespace nn

